/* --------------------------------------------------------------------------
 * ''Enlightenment Sound Daemon (ESD)'' specific audio driver interface.
 * --------------------------------------------------------------------------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <unistd.h>

#include <esd.h>

#include "AudioDriverESD.h"

//#define DOJDEBUG

#ifdef DOJDEBUG
#include <iostream>
#endif

AudioDriverESD::AudioDriverESD() :
  audioHd(-1),
  swapEndian(false)
{
#ifdef DOJDEBUG
  clog << "AudioDriverESD()" << endl;
#endif
}

AudioDriverESD::~AudioDriverESD()
{
#ifdef DOJDEBUG
  clog << "~AudioDriverESD()" << endl;
#endif
  unload();
}

bool AudioDriverESD::reset()
{
#ifdef DOJDEBUG
  clog << "AudioDriverESD::reset()" << endl;
#endif
  return true;
}

bool AudioDriverESD::open(const AudioConfig& inConfig)
{
#ifdef DOJDEBUG
  clog << "AudioDriverESD::open()" << endl;
#endif

  static AudioConfig lastConfig;

  if(audioHd>=0 && inConfig == config && inConfig == lastConfig)
    return true;
  else
    unload();

  config = inConfig;

  esd_format_t format=ESD_STREAM|ESD_PLAY;

  switch(config.precision)
    {
    case AudioConfig::BITS_8:
        format|=ESD_BITS8;
        config.encoding = AudioConfig::UNSIGNED_PCM;
        break;
    case AudioConfig::BITS_16:
        format|=ESD_BITS16;
        config.encoding = AudioConfig::SIGNED_PCM;
        break;
    default: errorString="sample bits were not 8 or 16"; return false;
    }

  switch(config.channels)
    {
    case AudioConfig::MONO: format|=ESD_MONO; break;
    case AudioConfig::STEREO: format|=ESD_STEREO; break;
    default: errorString="channels were not 1 or 2"; return false;
    }

  lastConfig = config;

#ifdef DOJDEBUG
  clog << 3 << endl;
#endif
  audioHd=esd_play_stream_fallback(format, config.frequency, getenv("ESPEAKER"), "XSidPlay");
#ifdef DOJDEBUG
  clog << 4 << endl;
#endif

  if(audioHd < 0)
    {
      errorString="could neither open ESD device, nor failback /dev/dsp. Perhaps you forgot to set your ESPEAKER environment?";
      return false;
    }

  errorString = "OK";
  return true;
}

void AudioDriverESD::close()
{
#ifdef DOJDEBUG
  clog << "AudioDriverESD::close()" << endl;
#endif
}

void AudioDriverESD::unload()
{
  if (audioHd >= 0)
    esd_close(audioHd);
  audioHd = -1;
}

void AudioDriverESD::play(void* buffer, unsigned long int bufferSize)
{
    if (audioHd < 0)
        return;
    write(audioHd, buffer, bufferSize);
}
