/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// A textfile class that does a _little_ bit more than getline().

#ifndef TEXTFILE_H
#define TEXTFILE_H

class ifstream;
class ofstream;

class TextFile
{
 public:  // --------------------------------------------------------- public

    TextFile(const int maxLineLen = 4096);
    virtual ~TextFile();

    virtual bool open(const char* fileName);
    virtual void close();

    virtual bool getLine();
    virtual int getLineLen() const;
    virtual int getLineNum() const;
    virtual const char* getLineBuf() const;
    virtual const char* getCurLineBuf() const;
    virtual bool isEOF() const;

    // Return a buffer that contains the current line without spaces.
    virtual const char* getParseBuf();

    // Return a buffer that contains the current line without spaces
    // at current parse position.
    virtual const char* getCurParseBuf();

    // Check whether the first non-space character is a ``#'' or ``;''.
    virtual bool isComment();

    // Check whether the line does not contain any non-space characters.
    virtual bool isBlank();

    // Check whether the first characters are equal to a given keyword
    // string followed by an equal sign. Example: Keyword=
    // By default a possibly matching keyword and the equal sign
    // will be skipped.
    virtual bool isKey(const char* key, bool advance = true);
    // Same as above, but checks current comparison position for
    // a string without the trailing equal sign.
    virtual bool isValue(const char* value, bool advance = true);

 protected:  // --------------------------------------------------- protected

    virtual ofstream& writeKey(ofstream& toFile, const char* key);
    virtual bool createParseCopy();

    ifstream* inFile;   // our input file stream

    char* lineBuf;      // current line buffer
    
    char* parseBuf;        // line without white-space (see flag below)
    char* curParseBuf;  // points to current pos in parseBuf

    int lineLen;        // actual number of chars till end of line
    int lineNum;        // line number in file

    int keyLen;         // length of last matched key +1 ('=')
    
    bool haveParseCopy;         // true, if line contents were copied to an
                                // extra buffer via a white-space eating
                                // string stream
    bool status;
    bool isGood;                // to prevent from using a bad inFile pointer

    const int maxLineLen;

 private:  // prevent copying
    TextFile(const TextFile&);
    TextFile& operator=(TextFile&);  
};

#endif  /* TEXTFILE_H */
