/*-
# MOTIF/X-BASED RUBIK'S CUBE(tm)
#
#  xrubik.c
#
###
#
#  Copyright (c) 1993 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/12/15 Xt/Motif
  Version 5: 95/10/06 Xt/Motif
  Version 4: 94/04/07 Xt
  Version 3: 93/05/20 Motif
  Version 2: 92/01/16 XView
  Version 1: 91/01/16 SunView
*/

#ifdef WINVER
#include "RubikP.h"
#include "wrubik.h"
#define TITLE "wrubik"

static RubikRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->rubik.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Rubik.h"
#include "Rubik2d.h"
#include "Rubik3d.h"
#ifdef HAVE_OPENGL
#include "RubikGL.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/rubik.xpm"
#endif
#include "icons/rubik.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.3\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n"
	"[-size{x|y|z} {int}] [-[no]orient] [-[no]practice] "
	"[-delay msecs] [-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The original puzzle  has 9 squares per face  (size = 3).  "
	"The puzzle was  designed by Erno Rubik and called the\n"
	"Rubik's Cube.  This has 8!*12!*3^8*2^12/12 or 4.3 * 10^19 "
	"different combinations.\n"
	"The Pocket Cube  has 4 squares per face  (size = 2) also "
	"designed by  Erno Rubik.  This has 7!*3^6 or 3,674,160\n"
	"different combinations.\n"
	"Rubik's Revenge has 16 squares per face (size = 4) also "
	"designed by Erno Rubik.  This has 7!*3^6*24!*24!/(4!)^6\n"
	"or 7.4 * 10^46 different combinations.\n"
	"5x5x5 Cube.  This has 8!*12!*3^7*2^10*(24!)^3/(4!)^12 "
	"or 2.83 * 10^74 different combinations.\n"
	"There is also the Magic Domino 3x3x2 cube which has "
	"(8!)^2/4 or 406,425,600 combinations.\n"
	"A physical 6x6x6 cube is possible but to my knowledge no "
	"one has been too successful in building one.  7x7x7 is\n"
	"also possible, but here one  must make the center most "
	"cubes smaller than the  outside cubes, so the corners do\n"
	"not fall off when turned.\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a piece.  Release "
	"\"mouse-left\" button on a piece on the same face and\n"
	"in the same row.  The pieces will then turn towards "
	"where the mouse button was released.\n"
	"Click \"mouse-center\" button,  or press \"P\" or \"p\" "
	"keys,  to toggle the practice mode  (in practice mode\n"
	"the record should say \"practice\").  This is good for learning "
	"learning moves and experimenting.\n"
	"Click \"mouse-right\" button,  or press \"R\" or \"r\" "
	"keys, to randomize the puzzle (this must be done first\n"
	"to set a new record).\n"
	"Press \"I\" or \"i\" keys to increase the number of \"cubies\".\n"
	"Press \"D\" or \"d\" keys to decrease the number of \"cubies\".\n"
	"Press \"x\" key to increase the number of \"cubies\" along "
	"the x axis.\n"
	"Press \"X\" key to decrease the number of \"cubies\" along "
	"the x axis.\n"
	"Press \"y\" key to increase the number of \"cubies\" along "
	"the y axis.\n"
	"Press \"Y\" key to decrease the number of \"cubies\" along "
	"the y axis.\n"
	"Press \"z\" key to increase the number of \"cubies\" along "
	"the z axis.\n"
	"Press \"Z\" key to decrease the number of \"cubies\" along "
	"the z axis.\n"
	"Press \"O\" or \"o\" keys to toggle  the orient mode.  One "
	"has to  orient the faces in orient mode, besides\n"
	"getting all  the faces to be the  same color.  To do this "
	"one  has to get the  lines to be  oriented in\n"
	"the same direction,  this only matters  with center "
	"\"cubies\",  if at all (i.e.  those \"cubies\" not on a\n"
	"corner or edge).  This does add complexity so there are "
	"2 sets of records.\n"
	"Press \"S\" or \"s\" keys to start auto-solver.  Only works "
	"on 1x1x1, 2x2x2, and 3x3x3 cubes (3x3x3 cube in\n"
	"non-orient mode).\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"mouse clicks.\n"
	"Key pad is defined for Rubik2d as:\n"
	"  /    Counterclockwise\n"
	"  8    Up\n"
	"  ^\n"
	"4<5>6  Left, Clockwise, Right\n"
	"  v\n"
	"  2    Down\n"
	"Key pad for Rubik3d, use must use your intuition (is this "
	"a cop out or what?).  The key pad  is defined\n"
	"differently  depending on which side  of the cube your mouse "
	"is pointing at.  One thing  that stays the\n"
	"same is \"5\" is Clockwise and \"/\" is Counterclockwise.\n"
	"Use the control key  and the left mouse button, keypad, or "
	"arrow keys  to move the whole cube.  This is\n"
	"not recorded as a turn.\n"
};
static const char referencesHelp[] = {
	"Inside Rubik's Cube and Beyond by Christoph Bandelow, "
	"Birkhauser, 1982 pp 44, 45, 88, 89.\n"
	"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow.\n"
	"The Simple Solution To Rubik's Cube, James G. Nourse, "
	"June 1981.\n"
	"Rubik's Cube Newsletter by Ideal Aug 1982 Vol.1 No. 2.\n"
	"Rubik's Cube The Solution, Ideal Toy Corporation, 1981.\n"
	"Rubik's Revenge Puzzle The Solution, Ideal Toy "
	"Corporation, 1982.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, only implemented for size nxnxn where "
	"n < 4.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "rubik.scores"
#endif

#define MAXFACETS 6
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int         score;
	char        name[USERNAMELEN];
} GameRecord;

static GameRecord rubikRecord[2][MAXFACETS - MINFACETS + 1][MAXFACETS - MINFACETS + 1][MAXFACETS - MINFACETS + 1];
static int  movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";
#ifndef HAVE_MOTIF
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message, cubesx, cubesy, cubesz,
	orientSwitch, practiceSwitch, speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
static Widget shell;
#ifdef HAVE_OPENGL
static Widget shellGL;
#endif
#endif
static Pixmap rubikIcon = None;
static Widget topLevel, rubik2d, rubik3d;
#ifdef HAVE_OPENGL
static Widget rubikGL;
#endif
static Arg arg[6];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";

/* There's probably a better way to assure that they are the same but I don't
   know it off hand. */
static void
MakeEquivalent(String * username, int *sizex, int *sizey, int *sizez, Boolean * orient, Boolean * practice, int *delay)
{
	Boolean mono, reverse;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAXFACES];

	XtVaGetValues(rubik2d,
		XtNuserName, username,
		XtNsizex, sizex,
		XtNsizey, sizey,
		XtNsizez, sizez,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverse, &reverse,
		XtNdelay, delay,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]), NULL);
	XtVaSetValues(rubik2d,
		XtNdirection, RUBIK_IGNORE,
		XtNstart, False, NULL);
	XtVaSetValues(rubik3d,
		XtNuserName, *username,
		XtNsizex, *sizex,
		XtNsizey, *sizey,
		XtNsizez, *sizez,
		XtNorient, *orient,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdelay, *delay,
		XtNdirection, RUBIK_IGNORE,
		XtNpractice, *practice,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(rubikGL,
		XtNuserName, *username,
		XtNsizex, *sizex,
		XtNsizey, *sizey,
		XtNsizez, *sizez,
		XtNorient, *orient,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdelay, *delay,
		XtNdirection, RUBIK_IGNORE,
		XtNpractice, *practice,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#endif
}
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int dim, int sizeX, int sizeY, int sizeZ, int moves, char *msg)
{
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) sprintf(titleDsp, "%sGL: %dx%dx%d@ (%d/%s) - %s",
			progDsp, sizeX, sizeY, sizeZ,
			moves, recordDsp, msg);
	else
#endif
		(void) sprintf(titleDsp, "%s%dd: %dx%dx%d@ (%d/%s) - %s",
			progDsp, dim, sizeX, sizeY, sizeZ,
			moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i, j, k, orient;

	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAXFACETS - MINFACETS + 1; i++)
			for (j = i; j < MAXFACETS - MINFACETS + 1; j++)
				for (k = j; k < MAXFACETS - MINFACETS + 1; k++) {
					rubikRecord[orient][k][j][i].score = rubikRecord[orient][k][i][j].score =
						rubikRecord[orient][j][k][i].score = rubikRecord[orient][j][i][k].score =
						rubikRecord[orient][i][k][j].score = rubikRecord[orient][i][j][k].score = NEVER;
					(void) strncpy(rubikRecord[orient][k][j][i].name,
						NOACCESS, USERNAMELEN);
					(void) strncpy(rubikRecord[orient][k][i][j].name,
						NOACCESS, USERNAMELEN);
					(void) strncpy(rubikRecord[orient][j][k][i].name,
						NOACCESS, USERNAMELEN);
					(void) strncpy(rubikRecord[orient][j][i][k].name,
						NOACCESS, USERNAMELEN);
					(void) strncpy(rubikRecord[orient][i][k][j].name,
						NOACCESS, USERNAMELEN);
					(void) strncpy(rubikRecord[orient][i][j][k].name,
						NOACCESS, USERNAMELEN);
				}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, k, n, orient;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAXFACETS - MINFACETS + 1; i++)
			for (j = i; j < MAXFACETS - MINFACETS + 1; j++)
				for (k = j; k < MAXFACETS - MINFACETS + 1; k++) {
					(void) fscanf(fp, "%d %s\n", &n, username);
					if (n <= rubikRecord[orient][i][j][k].score ||
					    rubikRecord[orient][i][j][k].score <= NEVER) {
						rubikRecord[orient][k][j][i].score = rubikRecord[orient][k][i][j].score =
							rubikRecord[orient][j][k][i].score = rubikRecord[orient][j][i][k].score =
							rubikRecord[orient][i][k][j].score = rubikRecord[orient][i][j][k].score = n;
						(void) strncpy(rubikRecord[orient][k][j][i].name,
							username, USERNAMELEN);
						(void) strncpy(rubikRecord[orient][k][i][j].name,
							username, USERNAMELEN);
						(void) strncpy(rubikRecord[orient][j][k][i].name,
							username, USERNAMELEN);
						(void) strncpy(rubikRecord[orient][j][i][k].name,
							username, USERNAMELEN);
						(void) strncpy(rubikRecord[orient][i][k][j].name,
							username, USERNAMELEN);
						(void) strncpy(rubikRecord[orient][i][j][k].name,
							username, USERNAMELEN);
					}
				}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, k, orient;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < MAXFACETS - MINFACETS + 1; i++) {
				for (j = i; j < MAXFACETS - MINFACETS + 1; j++) {
					for (k = j; k < MAXFACETS - MINFACETS + 1; k++)
						(void) fprintf(fp, "%d %s\n",
							rubikRecord[orient][i][j][k].score,
							rubikRecord[orient][i][j][k].name);
					(void) fprintf(fp, "\n");
				}
				(void) fprintf(fp, "\n");
			}

			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int sizex, int sizey, int sizez, Boolean orient, Boolean practice)
{
	int i = sizex - MINFACETS, j = sizey - MINFACETS, k = sizez - MINFACETS;
	int l = (orient) ? 1 : 0;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (sizex > MAXFACETS || sizey > MAXFACETS || sizez > MAXFACETS)
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	else if (rubikRecord[l][i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			rubikRecord[l][i][j][k].score,
			rubikRecord[l][i][j][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int sizex, int sizey, int sizez, Boolean orient)
{
	int i = sizex - MINFACETS, j = sizey - MINFACETS, k = sizez - MINFACETS;
	int l = (orient) ? 1 : 0;

	if (sizex <= MAXFACETS && sizey <= MAXFACETS && sizez <= MAXFACETS &&
	    (counter < rubikRecord[l][i][j][k].score ||
	    rubikRecord[l][i][j][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		rubikRecord[l][i][j][k].score = counter;
		(void) strncpy(rubikRecord[l][i][j][k].name, usernameDsp,
			USERNAMELEN);
		if ((sizex <= 2 && sizey <= 2) || (sizey <= 2 && sizez <= 2) ||
		    (sizez <= 2 && sizex <= 2) || (orient &&
			       (counter < rubikRecord[!l][i][j][k].score ||
			       rubikRecord[!l][i][j][k].score <= NEVER))) {
			rubikRecord[!l][i][j][k].score = counter;
			(void) strncpy(rubikRecord[!l][i][j][k].name, usernameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(sizex, sizey, sizez, orient, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
RubikWidget w, HBRUSH brush
#else
void
#endif
)
{
	int sizex, sizey, sizez;
	Boolean orient, practice;
	char *username;
#ifdef WINVER
	int dim;

	InitializeRubik(w, brush);

	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	username = w->rubik.username;
	dim = w->rubik.dim;
	SET_STARTED(w, False);
#else
	int delay;

	MakeEquivalent(&username, &sizex, &sizey, &sizez,
		&orient, &practice, &delay);
#ifdef HAVE_MOTIF
	if (sizex > MAXFACETS)
		XtVaSetValues(cubesx, XmNmaximum, sizex, NULL);
	XmScaleSetValue(cubesx, sizex);
	if (sizey > MAXFACETS)
		XtVaSetValues(cubesy, XmNmaximum, sizey, NULL);
	XmScaleSetValue(cubesy, sizey);
	if (sizez > MAXFACETS)
		XtVaSetValues(cubesz, XmNmaximum, sizez, NULL);
	XmScaleSetValue(cubesz, sizez);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
#endif
#endif
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		/* The NOACCESS is not necessary, but it may stop silliness. */
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", NOBODY);
#endif
	}
	PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
	PrintState(dim, sizex, sizey, sizez, movesDsp, messageDsp);
#else
#ifndef HAVE_MOTIF
	PrintState(XtParent(rubik2d), 2, sizex, sizey, sizez,
		movesDsp, messageDsp);
	PrintState(XtParent(rubik3d), 3, sizex, sizey, sizez,
		movesDsp, messageDsp);
#ifdef HAVE_OPENGL
	PrintState(XtParent(rubikGL), 4, sizex, sizey, sizez,
		movesDsp, messageDsp);
#endif
#endif
#endif
}

#ifdef WINVER
void
SetRubik(RubikWidget w, int reason)
#else
static void
CallbackRubik(Widget w, caddr_t clientData, rubikCallbackStruct * callData)
#endif
{
	int sizex, sizey, sizez;
	Boolean orient, practice, start, cheat;
#ifdef WINVER
	int dim = 0;
#else
	int reason = callData->reason;
	Widget otherw1 = (Widget) NULL;
#ifdef HAVE_OPENGL
	Widget otherw2 = (Widget) NULL;
#endif
#ifndef HAVE_MOTIF
	int dim = 0, otherdim1 = 0;
#ifdef HAVE_OPENGL
	int otherdim2 = 0;
#endif
#endif

	if (w == rubik2d) {
		otherw1 = rubik3d;
#ifdef HAVE_OPENGL
		otherw2 = rubikGL;
#endif
#ifndef HAVE_MOTIF
		dim = 2;
		otherdim1 = 3;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	} else if (w == rubik3d) {
		otherw1 = rubik2d;
#ifdef HAVE_OPENGL
		otherw2 = rubikGL;
#endif
#ifndef HAVE_MOTIF
		dim = 3;
		otherdim1 = 2;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	}
#ifdef HAVE_OPENGL
	else if (w == rubikGL) {
		otherw1 = rubik2d;
		otherw2 = rubik3d;
#ifndef HAVE_MOTIF
		dim = 4;
		otherdim1 = 2;
		otherdim2 = 3;
#endif
	}
#endif
#endif
	(void) strcpy(messageDsp, "");
#ifdef WINVER
	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
	start = w->rubik.started;
	dim = w->rubik.dim;
#else
	XtVaGetValues(w,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case RUBIK_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#ifndef HAVE_MOTIF
			(void) XIconifyWindow(XtDisplay(shell),
				XtWindow(shell),
				XScreenNumberOfScreen(XtScreen(shell)));
#ifdef HAVE_OPENGL
			(void) XIconifyWindow(XtDisplay(shellGL),
				XtWindow(shellGL),
				XScreenNumberOfScreen(XtScreen(shellGL)));
#endif
#endif
#endif
			break;
#ifndef WINVER
		case RUBIK_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
			XtSetValues(rubik2d, arg, 1);
#endif
			break;
		case RUBIK_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(rubik2d, arg, 1);
#endif
			break;
#endif
		case RUBIK_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case RUBIK_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, RUBIK_RESTORE);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case RUBIK_CLEAR:
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, RUBIK_CLEAR);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case RUBIK_RESET:
			movesDsp = 0;
			break;
		case RUBIK_ILLEGAL:
			if (practice || start)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			break;
#ifndef WINVER
		case RUBIK_MOVED:
			if (!callData->control)
				movesDsp++;
			XtSetArg(arg[0], XtNface, callData->face);
			XtSetArg(arg[1], XtNpos, callData->position);
			XtSetArg(arg[2], XtNdirection, callData->direction);
			XtSetArg(arg[3], XtNcontrol, callData->control);
			XtSetArg(arg[4], XtNfast, callData->fast);
			XtSetArg(arg[5], XtNstart, True);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, (callData->control) ? 6 : 5);
#endif
			XtSetValues(otherw1, arg, (callData->control) ? 6 : 5);
			XtSetValues(w, arg, 1);
			break;
#endif
		case RUBIK_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, sizex, sizey, sizez, orient))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);

#ifdef WINVER
			SET_STARTED(w, False);
#else
			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#endif
			break;
		case RUBIK_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.practice = practice;
			w->rubik.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 2);
#endif
			XtSetValues(otherw1, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case RUBIK_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->rubik.practice = practice;
			w->rubik.started = False;
#else
			XtSetArg(arg[0], XtNpractice, False);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 2);
#endif
			XtSetValues(otherw1, arg, 2);
#endif
			break;
		case RUBIK_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case RUBIK_DECX:
			movesDsp = 0;
			sizex--;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizex = sizex;
#else
			XtSetArg(arg[0], XtNsizex, sizex);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(cubesx, sizex);
			if (sizex >= MAXFACETS)
				XtVaSetValues(cubesx, XmNmaximum, sizex, NULL);
#endif
#endif
			break;
		case RUBIK_INCX:
			movesDsp = 0;
			sizex++;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizex = sizex;
#else
			XtSetArg(arg[0], XtNsizex, sizex);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			if (sizex > MAXFACETS)
				XtVaSetValues(cubesx, XmNmaximum, sizex, NULL);
			XmScaleSetValue(cubesx, sizex);
#endif
#endif
			break;
		case RUBIK_DECY:
			movesDsp = 0;
			sizey--;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizey = sizey;
#else
			XtSetArg(arg[0], XtNsizey, sizey);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(cubesy, sizey);
			if (sizey >= MAXFACETS)
				XtVaSetValues(cubesy, XmNmaximum, sizey, NULL);
#endif
#endif
			break;
		case RUBIK_INCY:
			movesDsp = 0;
			sizey++;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizey = sizey;
#else
			XtSetArg(arg[0], XtNsizey, sizey);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			if (sizey > MAXFACETS)
				XtVaSetValues(cubesy, XmNmaximum, sizey, NULL);
			XmScaleSetValue(cubesy, sizey);
#endif
#endif
			break;
		case RUBIK_DECZ:
			movesDsp = 0;
			sizez--;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizez = sizez;
#else
			XtSetArg(arg[0], XtNsizez, sizez);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(cubesz, sizez);
			if (sizez >= MAXFACETS)
				XtVaSetValues(cubesz, XmNmaximum, sizez, NULL);
#endif
#endif
			break;
		case RUBIK_INCZ:
			movesDsp = 0;
			sizez++;
			PrintRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
			w->rubik.sizez = sizez;
#else
			XtSetArg(arg[0], XtNsizez, sizez);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			if (sizez > MAXFACETS)
				XtVaSetValues(cubesz, XmNmaximum, sizez, NULL);
			XmScaleSetValue(cubesz, sizez);
#endif
#endif
			break;
		case RUBIK_COMPUTED:
#ifdef WINVER
			SET_STARTED(w, False);
#else
			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#endif
			break;
#ifndef WINVER
		case RUBIK_UNDO:
			if (!callData->control)
				movesDsp--;
			XtSetArg(arg[0], XtNface, callData->face);
			XtSetArg(arg[1], XtNpos, callData->position);
			XtSetArg(arg[2], XtNdirection, callData->direction);
			XtSetArg(arg[3], XtNcontrol, callData->control);
			XtSetArg(arg[4], XtNfast, callData->fast);
			XtSetArg(arg[5], XtNstart, True);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, (callData->control) ? 6 : 5);
#endif
			XtSetValues(otherw1, arg, (callData->control) ? 6 : 5);
			XtSetValues(w, arg, 1);
			break;
#endif
#ifdef WINVER
		case RUBIK_DIM:
			dim++;
#ifdef HAVE_OPENGL
			if (dim > 4)
#else
			if (dim > 3)
#endif
				dim = 2;
			w->rubik.dim = dim;
			break;
#endif
	}
#ifdef WINVER
	PrintState(dim, sizex, sizey, sizez, movesDsp, messageDsp);
#else
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(XtParent(w), dim, sizex, sizey, sizez,
		movesDsp, messageDsp);
	PrintState(XtParent(otherw1), otherdim1, sizex, sizey, sizez,
		movesDsp, messageDsp);
#ifdef HAVE_OPENGL
	PrintState(XtParent(otherw2), otherdim2, sizex, sizey, sizez,
		movesDsp, messageDsp);
#endif
#endif
#endif
}

#ifdef WINVER
void
SetRubikMove(RubikWidget w, int reason, int face, int position, int direction,
	Boolean control, int fast)
{
	int sizex, sizey, sizez;
	/* Boolean orient, practice, cheat; */
	int dim;

	(void) strcpy(messageDsp, "");
	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
#if 0
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
#endif
	dim = w->rubik.dim;
	switch (reason) {
		case RUBIK_MOVED:
			if (!control) {
				movesDsp++;
				SET_STARTED(w, True);
			}
			break;
		case RUBIK_UNDO:
			if (!control) {
				movesDsp--;
				SET_STARTED(w, True);
			}
			break;
	}
	PrintState(dim, sizex, sizey, sizez, movesDsp, messageDsp);
}

static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterRubik(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveRubik(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyRubik(brush);
			break;
		case WM_SIZE:
			ResizeRubik(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeRubik(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeRubik(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectRubik(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseRubik(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MoveRubikInput(&widget,
						cursor.x - origin.x, cursor.y - origin.y,
						TOP,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MoveRubikInput(&widget,
						cursor.x - origin.x, cursor.y - origin.y,
						BOTTOM,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetRubik(&widget);
					ResizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteRubik(&widget);
					break;
				case IDM_EXIT:
					DestroyRubik(brush);
					break;
				case IDM_HIDE:
					HideRubik(&widget);
					break;
				case IDM_CLEAR:
					ClearRubik(&widget);
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoRubik(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeRubik(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeRubik(&widget);
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveRubik(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeRubik(&widget);
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIM:
					(void) DimRubik(&widget);
					ResizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_TOP:
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BR:
				case IDM_BOTTOM:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_TL:
				case IDM_CW:
				case IDM_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveRubikInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_TOP,
							FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_TR:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_LEFT:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveRubikInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_CONTROL_TOP,
							TRUE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_DEC:
					DecrementRubik(&widget);
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INC:
					IncrementRubik(&widget);
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DECX:
					IncrementAxisRubik(&widget, 'X');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INCX:
					IncrementAxisRubik(&widget, 'x');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DECY:
					IncrementAxisRubik(&widget, 'Y');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INCY:
					IncrementAxisRubik(&widget, 'y');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DECZ:
					IncrementAxisRubik(&widget, 'Z');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INCZ:
					IncrementAxisRubik(&widget, 'z');
					SizeRubik(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else
static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n");
	(void) fprintf(stderr,
		"\t[-size{x|y|z} {int}] [-[no]orient] [-[no]practice]\n");
	(void) fprintf(stderr,
		"\t[-delay msecs] [-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*rubik.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*rubik.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*rubik.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*rubik.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*rubik.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*rubik.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*rubik.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*rubik.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*rubik.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*rubik.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*rubik.sizex", XrmoptionSepArg, NULL},
	{(char *) "-sizey", (char *) "*rubik.sizey", XrmoptionSepArg, NULL},
	{(char *) "-sizez", (char *) "*rubik.sizez", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-delay", (char *) "*rubik.delay", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*rubik.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackRubikPractice(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
		XtSetValues(rubik2d, arg, 1);
	}
}

static void
CallbackRubikRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(rubik2d, arg, 1);
	}
}

static void
CubesXSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int sizex = cbs->value, old, sizey, sizez;
	Boolean orient, practice;

	XtVaGetValues(rubik2d,
		XtNsizex, &old,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizex) {
		XtVaSetValues(rubik2d,
			XtNsizex, sizex, NULL);
		XtVaSetValues(rubik3d,
			XtNsizex, sizex, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(rubikGL,
			XtNsizex, sizex, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
CubesYSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int sizey = cbs->value, old, sizex, sizez;
	Boolean orient, practice;

	XtVaGetValues(rubik2d,
		XtNsizex, &sizex,
		XtNsizey, &old,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizey) {
		XtVaSetValues(rubik2d,
			XtNsizey, sizey, NULL);
		XtVaSetValues(rubik3d,
			XtNsizey, sizey, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(rubikGL,
			XtNsizey, sizey, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
CubesZSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int sizez = cbs->value, old, sizex, sizey;
	Boolean orient, practice;

	XtVaGetValues(rubik2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizez) {
		XtVaSetValues(rubik2d,
			XtNsizez, sizez, NULL);
		XtVaSetValues(rubik3d,
			XtNsizez, sizez, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(rubikGL,
			XtNsizez, sizez, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int sizex, sizey, sizez;
	Boolean orient = cbs->set, practice;

	XtVaGetValues(rubik2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNpractice, &practice, NULL);
	XtVaSetValues(rubik2d,
		XtNorient, orient, NULL);
	XtVaSetValues(rubik3d,
		XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(rubikGL,
		XtNorient, orient, NULL);
#endif
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizex, sizey, sizez, orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int sizex, sizey, sizez;
	Boolean orient, practice = cbs->set;

	XtVaSetValues(rubik2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(rubik3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(rubikGL,
		XtNpractice, practice,
		XtNstart, False, NULL);
#endif
	XtVaGetValues(rubik2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNpractice, &orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizex, sizey, sizez, orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(rubik2d,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
#ifdef HAVE_OPENGL
		XtVaSetValues(rubikGL, XtNdelay, delay, NULL);
#endif
		XtVaSetValues(rubik3d, XtNdelay, delay, NULL);
		XtVaSetValues(rubik2d, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(rubik2d, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(rubik2d, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, cubesRowCol, switchRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, practiceString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Rubik",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) rubik_xpm, &rubikIcon, NULL,
			&xpmAttributes);
	}
	if (rubikIcon == (Pixmap) NULL)
#endif
		rubikIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) rubik_bits,
			rubik_width, rubik_height);
	XtSetArg(arg[0], XtNiconPixmap, rubikIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Solve");
	orientString = XmStringCreateSimple((char *) "Orientize");
	practiceString = XmStringCreateSimple((char *) "Practice");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(practiceString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("FeaTures",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackRubikPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackRubikRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("Rowcol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	cubesRowCol = XtVaCreateManagedWidget("cubesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	cubesx = XtVaCreateManagedWidget("cubesx",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size X", 7,
		XmNminimum, MINFACETS,
		XmNmaximum, MAXFACETS,
		XmNvalue, MINFACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesx, XmNvalueChangedCallback,
		(XtCallbackProc) CubesXSlider, (XtPointer) NULL);
	cubesy = XtVaCreateManagedWidget("cubesy",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size Y", 7,
		XmNminimum, MINFACETS,
		XmNmaximum, MAXFACETS,
		XmNvalue, MINFACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesy, XmNvalueChangedCallback,
		(XtCallbackProc) CubesYSlider, (XtPointer) NULL);
	cubesz = XtVaCreateManagedWidget("cubesz",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size Z", 7,
		XmNminimum, MINFACETS,
		XmNmaximum, MAXFACETS,
		XmNvalue, MINFACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesz, XmNvalueChangedCallback,
		(XtCallbackProc) CubesZSlider, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	speed = XtVaCreateManagedWidget("speed",
		xmScaleWidgetClass, switchRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - 10,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Rubik's Cube! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	rubik2d = XtCreateManagedWidget("rubik",
		rubik2dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(rubik2d,
		XtNheight, 200, NULL);
	XtAddCallback(rubik2d, XtNselectCallback,
		(XtCallbackProc) CallbackRubik, (XtPointer) NULL);
	rubik3d = XtCreateManagedWidget("rubik",
		rubik3dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(rubik3d,
		XtNheight, 200, NULL);
#ifdef HAVE_OPENGL
	rubikGL = XtCreateManagedWidget("rubik",
		rubikGLWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(rubikGL,
		XtNheight, 200, NULL);
#endif
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	shell = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtSetValues(shell, arg, 2);
#ifdef HAVE_OPENGL
	shellGL = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtSetValues(shellGL, arg, 2);
#endif
	rubik2d = XtCreateManagedWidget("rubik",
		rubik2dWidgetClass, topLevel, NULL, 0);
	XtAddCallback(rubik2d, XtNselectCallback,
		(XtCallbackProc) CallbackRubik, (XtPointer) NULL);
	rubik3d = XtCreateManagedWidget("rubik",
		rubik3dWidgetClass, shell, NULL, 0);
#ifdef HAVE_OPENGL
	rubikGL = XtCreateManagedWidget("rubik",
		rubikGLWidgetClass, shellGL, NULL, 0);
#endif
#endif
	XtAddCallback(rubik3d, XtNselectCallback,
		(XtCallbackProc) CallbackRubik, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	XtAddCallback(rubikGL, XtNselectCallback,
		(XtCallbackProc) CallbackRubik, (XtPointer) NULL);
#endif
	Initialize();
	XtRealizeWidget(topLevel);
#ifndef HAVE_MOTIF
	XtRealizeWidget(shell);
#ifdef HAVE_OPENGL
	XtRealizeWidget(shellGL);
#endif
#endif
	XGrabButton(XtDisplay(rubik2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(rubik2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(rubik2d),
		XCreateFontCursor(XtDisplay(rubik2d), XC_crosshair));
	XGrabButton(XtDisplay(rubik3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(rubik3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(rubik3d),
		XCreateFontCursor(XtDisplay(rubik3d), XC_crosshair));
#ifdef HAVE_OPENGL
	XGrabButton(XtDisplay(rubikGL), (unsigned int) AnyButton, AnyModifier,
		XtWindow(rubikGL), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(rubikGL),
		XCreateFontCursor(XtDisplay(rubikGL), XC_crosshair));
#endif
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
