/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_focus.cc,v 1.5 2001/07/11 15:43:54 bbarrett Exp $
 *
 *	Function:	- process focus dialog
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <Xm/ArrowB.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/SeparatoG.h>
#include <X11/cursorfont.h>

#include "blktype.h"
#include "xmpi.h"
#include "xmpi_focus.h"
#include "xmpi_misc.h"
#include "xmpi_view.h"
#include "xmpi_dtype.h"
#include "xmpi_pophelp.h"
#include "xmpi_dbase.h"

#include "Bitmaps/light.xbm"
#include "Bitmaps/red.xbm"
#include "Bitmaps/yellow.xbm"
#include "Bitmaps/green.xbm"
#include "Bitmaps/mesg.xbm"
#include "Bitmaps/tack.xbm"
#include "Bitmaps/comm.xbm"
#include "Bitmaps/next.xbm"
#include "Bitmaps/data.xbm"

/*
 * external functions
 */
extern void _XEditResCheckMessages();

/*
 * local functions
 */
static void focusbuild(Widget, int);
static void focusmsgfill(int, struct xmfocus*);
static void destroy_cb(Widget, XtPointer, XtPointer);
static void showproccomm_cb(Widget, XtPointer, XtPointer);
static void showdt_cb(Widget, XtPointer, XtPointer);
static void showmsgdt_cb(Widget, XtPointer, XtPointer);
static void showmsgcomm_cb(Widget, XtPointer, XtPointer);
static void showcomm(int, int);
static void msgflip_cb(Widget, XtPointer, XtPointer);
static void position_cb(Widget, XtPointer, XtPointer);
static void createpixlabels(Widget);
static void setmsgcnt(struct xmfocus*);
static void getcomm(int, int, struct _gps*, int*,
		    int*, int*, char**);
static int group_index(int, struct _gps*, int);
static void mpiblk_state_out(int, struct xmproc*);
static void formatproc(int4, int4, char*);
static void formattag(int, char*);

/*
 * local constants and macros
 */
#define	nullify(s)	((s)[0] = 0)

/*
 * external variables
 */
extern int fl_dbase;		       /* driven by dbase flag */
extern struct _gps *xmpi_app_procs;    /* appl. GPS array */

/*
 * local variables
 */
static struct xmfocus *fo_procs = 0;   /* array of process info */
static int fo_nprocs = 0;	       /* # processes */
static int yfocus = 0;		       /* focus vert. pos. */
static char fmtbuf[130];	       /* formatting buffer */
static char task_str[32];	       /* task formatting buffer */
static char func_str[64];	       /* function formatting buf */
static char peer_str[32];	       /* peer formatting buffer */
static char root_str[32];	       /* root formatting buffer */
static char tag_str[16];	       /* tag formatting buffer */
static char cid_str[130];	       /* cid formatting buffer */
static char count_str[16];	       /* count formatting buffer */
static Pixmap blkpix = XmUNSPECIFIED_PIXMAP;
static Pixmap runpix = XmUNSPECIFIED_PIXMAP;
static Pixmap syspix = XmUNSPECIFIED_PIXMAP;
static Pixmap undefpix = XmUNSPECIFIED_PIXMAP;
static Pixmap mesgpix = XmUNSPECIFIED_PIXMAP;
static Pixmap tackpix = XmUNSPECIFIED_PIXMAP;
static Pixmap armtackpix = XmUNSPECIFIED_PIXMAP;
static Pixmap commpix = XmUNSPECIFIED_PIXMAP;
static Pixmap armcommpix = XmUNSPECIFIED_PIXMAP;
static Pixmap offcommpix = XmUNSPECIFIED_PIXMAP;
static Pixmap nextpix = XmUNSPECIFIED_PIXMAP;
static Pixmap armnextpix = XmUNSPECIFIED_PIXMAP;
static Pixmap offnextpix = XmUNSPECIFIED_PIXMAP;
static Pixmap datapix = XmUNSPECIFIED_PIXMAP;
static Pixmap armdatapix = XmUNSPECIFIED_PIXMAP;
static Pixmap offdatapix = XmUNSPECIFIED_PIXMAP;

/*
 *	xmpi_fo_toggle
 *
 *	Function:	- toggle a focus window (build or destroy)
 *	Accepts:	- parent widget
 *			- process rank
 *			- process entry
 */
void
xmpi_fo_toggle(Widget parent_w, int rank, struct xmproc *pproc)
{
  int i;			       /* favourite index */

  struct xmfocus *p;		       /* favourite pointer */

/*
 * Allocate process array if needed.
 */
  if (fo_procs == 0) {
    fo_nprocs = xmpi_vw_getnprocs();

    fo_procs = (struct xmfocus *)
      malloc((unsigned) fo_nprocs * sizeof(struct xmfocus));

    if (fo_procs == 0)
      xmpi_fail((char*) "xmpi (malloc)");

    memset(fo_procs, 0, sizeof((unsigned) fo_nprocs * sizeof(struct xmfocus)));


    for (i = 0, p = fo_procs; i < fo_nprocs; ++i, ++p) {
      p->xmf_myrank = i;
      p->xmf_window = 0;
    }
  }
/*
 * Toggle process focus window.
 */
  p = fo_procs + rank;

  if (p->xmf_window) {
    xmpi_fo_destroy(rank);
  } else {
    p->xmf_proc = pproc;
    focusbuild(parent_w, rank);
  }
}

/*
 *	focusbuild
 *
 *	Function:	- creates process focus subwindow
 *	Accepts:	- parent widget
 *			- process rank
 */
static void
focusbuild(Widget parent_w, int rank)
{
  Widget focus_w;		       /* focus widget */

  Widget form_w;		       /* form window widget */

  Widget lbl_w;			       /* label widget */

  Widget main_w;		       /* main focus widget */

  Widget area_w;		       /* area widget */

  Widget w;			       /* favourite widget */

  Pixmap pix;			       /* traffic light pixmap */

  Dimension width, height;	       /* focus widget dimensions */

  struct xmproc *proc;		       /* process entry */

  struct xmfocus *p;		       /* favourite pointer */

/*
 * Set up datatype subwindow.
 */
  xmpi_dt_create(parent_w);

  p = fo_procs + rank;
  proc = p->xmf_proc;
/*
 * Create a focus window.
 */
  focus_w = XtVaCreatePopupShell("focus",
				 xmDialogShellWidgetClass, parent_w,
				 XmNdefaultPosition, False,
				 XmNmappedWhenManaged, True,
				 XmNdeleteResponse, XmDESTROY,
				 XmNtitle, "XMPI Focus",
				 NULL);

  XtAddCallback(focus_w, XmNpopupCallback, (XtCallbackProc) position_cb, NULL);
  XtAddCallback(focus_w, XmNdestroyCallback,
		(XtCallbackProc) destroy_cb, (void *) &p->xmf_myrank);

#ifdef DEBUG
  XtAddEventHandler(focus_w, (EventMask) 0, True,
		    _XEditResCheckMessages, NULL);
#endif

  p->xmf_window = focus_w;
/*
 * Create the high-level form.
 */
  form_w = XtVaCreateWidget("focus_mgr",
			    xmFormWidgetClass, focus_w,
			    XmNmarginHeight, 2,
			    XmNmarginWidth, 2,
			    XmNverticalSpacing, 2,
			    NULL);
/*
 * Create the label area window.
 */
  lbl_w = XtVaCreateWidget("focus_label",
			   xmFormWidgetClass, form_w,
			   XmNtopAttachment, XmATTACH_FORM,
			   XmNleftAttachment, XmATTACH_FORM,
			   XmNrightAttachment, XmATTACH_FORM,
			   NULL);
/*
 * tack button (cache and re-use pixmaps)
 */
  if (tackpix == XmUNSPECIFIED_PIXMAP) {
    w = xmpi_mkpixbutton(lbl_w, destroy_cb,
			 (char *) &p->xmf_myrank,
			 (XtCallbackProc) NULL, NULL,
			 (XtCallbackProc) NULL, NULL,
			 tack_bits, tack_width, tack_height, False);

    XtVaGetValues(w, XmNlabelPixmap, &tackpix,
		  XmNarmPixmap, &armtackpix, NULL);
  } else {
    w = xmpi_mkpixbutton(lbl_w, destroy_cb,
			 (char *) &p->xmf_myrank, 
			 (XtCallbackProc) NULL, NULL,
			 (XtCallbackProc) NULL, NULL,
			 NULL, 0, 0, False);

    XtVaSetValues(w, XmNlabelType, XmPIXMAP,
		  XmNlabelPixmap, tackpix,
		  XmNarmPixmap, armtackpix, NULL);
  }

  XtVaSetValues(w, XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM, NULL);

  xmpi_add_pophelp(w, (char*) "Dismiss focus window");
/*
 * header label
 */
  p->xmf_rank = xmpi_mklabel(lbl_w, (char*) "banner",
			     XmALIGNMENT_CENTER, False);

  XtVaSetValues(p->xmf_rank,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNleftOffset, 3,
		NULL);

  sprintf(fmtbuf, "%d", rank);
  xmpi_setlabel(p->xmf_rank, fmtbuf);

  XtManageChild(lbl_w);
/*
 * Create the main focus window.
 */
  main_w = XtVaCreateWidget("focus_main",
			    xmFormWidgetClass, form_w,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, lbl_w,
			    NULL);
/*
 * function label
 */
  p->xmf_func = xmpi_mklabel(main_w, (char*) "fo_func",
			     XmALIGNMENT_CENTER, False);
  XtVaSetValues(p->xmf_func,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
/*
 * Create the traffic light area.
 */
  area_w = XtVaCreateWidget("traffic_mgr",
			    xmFormWidgetClass, main_w,
			    XmNfractionBase, 24,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, p->xmf_func,
			    NULL);

  createpixlabels(area_w);

  switch (proc->xmp_state) {
  case XMPI_SRUN:
    pix = runpix;
    break;
  case XMPI_SBLOCK:
    pix = blkpix;
    break;
  case XMPI_SSYSTEM:
    pix = syspix;
    break;
  default:
    pix = undefpix;
    break;
  }

  p->xmf_light = xmpi_mklabel(area_w, (char*) "traffic_lbl",
			      XmALIGNMENT_CENTER, False);
  xmpi_formattach(p->xmf_light, 0, 24, 0, 4);

  XtVaSetValues(p->xmf_light, XmNlabelType, XmPIXMAP,
		XmNlabelPixmap, pix, NULL);
/*
 * Create the traffic light labels.
 */
  p->xmf_pr_rt = xmpi_mklabel(area_w, (char*) "peer", XmALIGNMENT_END, False);
  xmpi_formattach(p->xmf_pr_rt, 0, 8, 4, 8);

  p->xmf_peer = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_peer), 0, 8, 8, 24);

  w = xmpi_mklabel(area_w, (char*) "comm", XmALIGNMENT_END, False);
  xmpi_formattach(w, 8, 16, 4, 8);
  p->xmf_com = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_com), 8, 16, 8, -1);

  w = xmpi_mklabel(area_w, (char*) "tag", XmALIGNMENT_END, False);
  xmpi_formattach(w, 16, 24, 4, 8);
  p->xmf_tag = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_tag), 16, 24, 8, 13);

  w = xmpi_mklabel(area_w, (char*) "cnt", XmALIGNMENT_END, False);
  xmpi_formattach(w, 16, 24, 13, 16);
  p->xmf_cnt = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_cnt), 16, 24, 16, -1);
/*
 * Create the communicator button.
 */
  if (commpix == XmUNSPECIFIED_PIXMAP) {
    p->xmf_compb = xmpi_mkpixbutton(area_w,
				    showproccomm_cb, 
				    (char *) &p->xmf_myrank,
				    (XtCallbackProc) NULL, 
				    NULL, 
				    (XtCallbackProc) NULL, NULL,
				    comm_bits, comm_width, 
				    comm_height, True);

    XtVaGetValues(p->xmf_compb, XmNlabelPixmap, &commpix,
		  XmNlabelInsensitivePixmap, &offcommpix,
		  XmNarmPixmap, &armcommpix, NULL);
  } else {
    p->xmf_compb = xmpi_mkpixbutton(area_w,
				    showproccomm_cb, 
				    (char *) &p->xmf_myrank,
				    (XtCallbackProc) NULL,
				    NULL, 
				    (XtCallbackProc) NULL, 
				    NULL,
				    NULL, 0, 0, False);

    XtVaSetValues(p->xmf_compb, XmNlabelType, XmPIXMAP,
		  XmNlabelPixmap, commpix,
		  XmNlabelInsensitivePixmap, offcommpix,
		  XmNarmPixmap, armcommpix, NULL);
  }

  xmpi_formattach(p->xmf_compb, 8, 16, -1, 24);

  XtVaSetValues(XtParent(p->xmf_com),
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, p->xmf_compb, NULL);

  xmpi_add_pophelp(p->xmf_compb, (char*) "Show communicator");
/*
 * Create the datatype button.
 */
  if (datapix == XmUNSPECIFIED_PIXMAP) {
    p->xmf_dtb = xmpi_mkpixbutton(area_w,
				  showdt_cb, (char *) &p->xmf_myrank,
				  (XtCallbackProc) NULL, 
				  NULL, 
				  (XtCallbackProc) NULL, 
				  NULL,
				  data_bits, 
				  data_width, 
				  data_height, 
				  True);

    XtVaGetValues(p->xmf_dtb, XmNlabelPixmap, &datapix,
		  XmNlabelInsensitivePixmap, &offdatapix,
		  XmNarmPixmap, &armdatapix, NULL);
  } else {
    p->xmf_dtb = xmpi_mkpixbutton(area_w,
				  showdt_cb, 
				  (char *) &p->xmf_myrank,
				  (XtCallbackProc) NULL, 
				  NULL,
				  (XtCallbackProc) NULL, 
				  NULL,
				  NULL, 
				  0, 
				  0, 
				  False);

    XtVaSetValues(p->xmf_dtb, XmNlabelType, XmPIXMAP,
		  XmNlabelPixmap, datapix,
		  XmNlabelInsensitivePixmap, offdatapix,
		  XmNarmPixmap, armdatapix, NULL);
  }

  XtVaSetValues(p->xmf_dtb, XmNnavigationType, XmTAB_GROUP, NULL);
  xmpi_formattach(p->xmf_dtb, 16, 24, -1, 24);

  XtVaSetValues(XtParent(p->xmf_cnt),
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, p->xmf_dtb, NULL);

  xmpi_add_pophelp(p->xmf_dtb, (char*) "Show datatype");

  XtManageChild(area_w);
/*
 * Add a separator.
 */
  area_w = XtVaCreateManagedWidget("focus_sep",
				   xmSeparatorGadgetClass, main_w,
				   XmNseparatorType, XmSHADOW_ETCHED_OUT,
				   XmNleftAttachment, XmATTACH_FORM,
				   XmNrightAttachment, XmATTACH_FORM,
				   XmNtopAttachment, XmATTACH_WIDGET,
				   XmNtopWidget, area_w,
				   XmNtopOffset, 4,
				   NULL);
/*
 * Create the message area.
 */
  area_w = XtVaCreateWidget("message_mgr",
			    xmFormWidgetClass, main_w,
			    XmNfractionBase, 24,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNbottomOffset, 2,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, area_w,
			    XmNtopOffset, 4,
			    NULL);

  w = xmpi_mklabel(area_w, (char*) "mesg_lbl", XmALIGNMENT_CENTER, False);
  xmpi_formattach(w, 0, 24, 0, 4);
  XtVaSetValues(w, XmNlabelType, XmPIXMAP,
		XmNlabelPixmap, mesgpix, NULL);
/*
 * Create the message labels.
 */
  w = xmpi_mklabel(area_w, (char*) "src", XmALIGNMENT_END, False);
  xmpi_formattach(w, 0, 6, 4, 8);
  p->xmf_msgsrc = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_msgsrc), 0, 6, 8, 24);

  w = xmpi_mklabel(area_w, (char*) "comm", XmALIGNMENT_END, False);
  xmpi_formattach(w, 6, 12, 4, 8);
  p->xmf_msgcom = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_msgcom), 6, 12, 8, -1);

  w = xmpi_mklabel(area_w, (char*) "tag", XmALIGNMENT_END, False);
  xmpi_formattach(w, 12, 18, 4, 8);
  p->xmf_msgtag = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_msgtag), 12, 18, 8, 13);

  w = xmpi_mklabel(area_w, (char*) "cnt", XmALIGNMENT_END, False);
  xmpi_formattach(w, 12, 18, 13, 16);
  p->xmf_msgcnt = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_msgcnt), 12, 18, 16, -1);

  w = xmpi_mklabel(area_w, (char*) "copy", XmALIGNMENT_END, False);
  xmpi_formattach(w, 18, 24, 4, 8);
  p->xmf_nmsg = xmpi_mklabel(area_w, (char*) " ", XmALIGNMENT_CENTER, True);
  xmpi_formattach(XtParent(p->xmf_nmsg), 18, 24, 8, -1);
/*
 * Create the communicator button.
 */
  p->xmf_msgcompb = xmpi_mkpixbutton(area_w,
				     showmsgcomm_cb, 
				     (char *) &p->xmf_myrank,
				     (XtCallbackProc) NULL, 
				     NULL, 
				     (XtCallbackProc) NULL, 
				     NULL,
				     NULL, 
				     0, 
				     0, 
				     False);
  
  XtVaSetValues(p->xmf_msgcompb, XmNlabelType, XmPIXMAP,
		XmNlabelPixmap, commpix,
		XmNlabelInsensitivePixmap, offcommpix,
		XmNarmPixmap, armcommpix, NULL);

  xmpi_formattach(p->xmf_msgcompb, 6, 12, -1, 24);

  XtVaSetValues(XtParent(p->xmf_msgcom),
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, p->xmf_msgcompb, NULL);

  xmpi_add_pophelp(p->xmf_msgcompb, (char*) "Show communicator");
/*
 * Create the message count button.
 */
  if (nextpix == XmUNSPECIFIED_PIXMAP) {
    p->xmf_nmsgpb = xmpi_mkpixbutton(area_w,
				     msgflip_cb,
				     (char *) &p->xmf_myrank,
				     (XtCallbackProc) NULL,
				     NULL,
				     (XtCallbackProc) NULL,
				     NULL,
				     next_bits,
				     next_width,
				     next_height,
				     True);

    XtVaGetValues(p->xmf_nmsgpb, XmNlabelPixmap, &nextpix,
		  XmNlabelInsensitivePixmap, &offnextpix,
		  XmNarmPixmap, &armnextpix, NULL);
  } else {
    p->xmf_nmsgpb = xmpi_mkpixbutton(area_w,
				     msgflip_cb, 
				     (char *) &p->xmf_myrank,
				     (XtCallbackProc) NULL, 
				     NULL, 
				     (XtCallbackProc) NULL,
				     NULL,
				     NULL,
				     0,
				     0,
				     False);

    XtVaSetValues(p->xmf_nmsgpb, XmNlabelType, XmPIXMAP,
		  XmNlabelPixmap, nextpix,
		  XmNlabelInsensitivePixmap, offnextpix,
		  XmNarmPixmap, armnextpix, NULL);
  }

  XtVaSetValues(p->xmf_nmsgpb, XmNnavigationType, XmTAB_GROUP, NULL);
  xmpi_formattach(p->xmf_nmsgpb, 18, 24, -1, 24);

  xmpi_add_pophelp(p->xmf_nmsgpb, (char*) "Next message aggregate");

  XtVaSetValues(XtParent(p->xmf_nmsg),
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, p->xmf_nmsgpb, NULL);
  setmsgcnt(p);
/*
 * Create the datatype button.
 */
  p->xmf_msgdtb = xmpi_mkpixbutton(area_w,
				   showmsgdt_cb,
				   (char *) &p->xmf_myrank,
				   (XtCallbackProc) NULL, 
				   NULL,
				   (XtCallbackProc) NULL,
				   NULL,
				   NULL,
				   0,
				   0,
				   False);

  XtVaSetValues(p->xmf_msgdtb, XmNlabelType, XmPIXMAP,
		XmNlabelPixmap, datapix,
		XmNlabelInsensitivePixmap, offdatapix,
		XmNarmPixmap, armdatapix, NULL);

  XtVaSetValues(p->xmf_msgdtb, XmNnavigationType, XmTAB_GROUP, NULL);
  xmpi_formattach(p->xmf_msgdtb, 12, 18, -1, 24);

  xmpi_add_pophelp(p->xmf_msgdtb, (char*) "Show datatype");

  XtVaSetValues(XtParent(p->xmf_msgcnt),
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, p->xmf_msgdtb, NULL);

  XtManageChild(area_w);
  XtManageChild(main_w);
  XtManageChild(form_w);

#ifdef AIX
/*
 * AIX Motif 1.2 does not seem to honor the alignment resource
 * of child labels unless the we force a resize!?
 */
  {
    Dimension w;

    XtVaGetValues(focus_w, XmNwidth, &w, NULL);
    XtVaSetValues(focus_w, XmNwidth, w + 2, NULL);
  }
#endif

  XtPopup(focus_w, XtGrabNone);

  XtVaGetValues(focus_w, XmNwidth, &width, XmNheight, &height, NULL);
  XtVaSetValues(focus_w,
		XmNminWidth, width,
		XmNminHeight, height, XmNmaxHeight, height,
		NULL);
}

/*
 *	xmpi_fo_update
 *
 *	Function:	- update focus window
 *	Accepts:	- process rank
 */
void
xmpi_fo_update(int rank)
{
  Pixmap pix;			       /* favourite pixmap */

  struct xmproc *pproc;		       /* process entry */

  struct xmfocus *pfocus;	       /* focus entry */

  if (fo_procs == 0)
    return;

  pfocus = fo_procs + rank;
  pproc = pfocus->xmf_proc;

  if (pfocus->xmf_window == 0)
    return;
/*
 * If the process is running or undefined, show blanks.
 */
  if ((pproc->xmp_state == XMPI_SRUN)
      || (pproc->xmp_state == XMPI_SUNDEF)) {

    pix = (pproc->xmp_state == XMPI_SRUN) ? runpix : undefpix;
    XtVaSetValues(pfocus->xmf_light, XmNlabelPixmap, pix, NULL);

    sprintf(fmtbuf, "%d   %s", rank, pproc->xmp_prog);
    xmpi_setlabel(pfocus->xmf_rank, fmtbuf);
    xmpi_dt_setrank(rank, fmtbuf);

    strcpy(fmtbuf, " ");
    xmpi_setlabel(pfocus->xmf_peer, fmtbuf);
    xmpi_setlabel(pfocus->xmf_pr_rt, fmtbuf);
    xmpi_setlabel(pfocus->xmf_com, fmtbuf);
    xmpi_setlabel(pfocus->xmf_tag, fmtbuf);
    xmpi_setlabel(pfocus->xmf_cnt, fmtbuf);
    xmpi_setlabel(pfocus->xmf_func, fmtbuf);

    XtVaSetValues(pfocus->xmf_compb, XmNsensitive, False, NULL);
    XtVaSetValues(pfocus->xmf_dtb, XmNsensitive, False, NULL);
  }
/*
 * The process is blocked or in system overhead.
 */
  else {
    pix = (pproc->xmp_state == XMPI_SBLOCK) ? blkpix : syspix;
    XtVaSetValues(pfocus->xmf_light, XmNlabelPixmap, pix, NULL);

    if (blktype(pproc->xmp_func)) {

      mpiblk_state_out(rank, pproc);

      xmpi_setlabel(pfocus->xmf_rank, task_str);
      xmpi_dt_setrank(rank, task_str);
      if (*root_str == 0) {
	xmpi_setlabel(pfocus->xmf_peer, peer_str);
	xmpi_setlabel(pfocus->xmf_pr_rt, (char*) "peer");
      } else {
	xmpi_setlabel(pfocus->xmf_root, root_str);
	xmpi_setlabel(pfocus->xmf_pr_rt, (char*) "root");
      }
      xmpi_setlabel(pfocus->xmf_tag, tag_str);
      xmpi_setlabel(pfocus->xmf_func, func_str);
      xmpi_setlabel(pfocus->xmf_com, cid_str);
      xmpi_setlabel(pfocus->xmf_cnt, count_str);

      XtVaSetValues(pfocus->xmf_compb, XmNsensitive,
		    (*cid_str) ? True : False, NULL);

      XtVaSetValues(pfocus->xmf_dtb, XmNsensitive,
		    (pproc->xmp_dtype != 0) ? True : False, NULL);
    } else {
      sprintf(fmtbuf, "%d   %s", rank, pproc->xmp_prog);
      xmpi_setlabel(pfocus->xmf_rank, fmtbuf);
      xmpi_dt_setrank(rank, fmtbuf);

      strcpy(fmtbuf, " ");
      xmpi_setlabel(pfocus->xmf_peer, fmtbuf);
      xmpi_setlabel(pfocus->xmf_pr_rt, fmtbuf);
      xmpi_setlabel(pfocus->xmf_com, fmtbuf);
      xmpi_setlabel(pfocus->xmf_tag, fmtbuf);
      xmpi_setlabel(pfocus->xmf_cnt, fmtbuf);
      xmpi_setlabel(pfocus->xmf_func, fmtbuf);

      XtVaSetValues(pfocus->xmf_compb, XmNsensitive, False,
		    NULL);
      XtVaSetValues(pfocus->xmf_dtb, XmNsensitive, False,
		    NULL);
    }
  }
/*
 * Update the datatype window and fill the message area.
 */
  xmpi_dt_proc(rank, pfocus, 0);
  focusmsgfill(rank, pfocus);
}

/*
 *	focusmsgfill
 *
 *	Function:	- fill the focus message area
 *	Accepts:	- process global rank
 *			- focus entry
 */
static void
focusmsgfill(int rank, struct xmfocus *pfocus)
{
  struct xmproc *pproc;		       /* process entry */

  struct xmmsg *pmsg;		       /* message entry */

  int fl_empty = 1;		       /* empty flag */

  int nmsg = -1;		       /* # messages */

  pproc = pfocus->xmf_proc;

  if (pproc->xmp_msgs && (al_count(pproc->xmp_msgs) > 0)) {
    fl_empty = 0;
    pmsg = pproc->xmp_curmsg;
    nmsg = pmsg->xmm_nmsg;
  }
  if (fl_empty || (nmsg < 0)) {
    strcpy(fmtbuf, " ");

    xmpi_setlabel(pfocus->xmf_msgsrc, fmtbuf);
    xmpi_setlabel(pfocus->xmf_msgcom, fmtbuf);
    xmpi_setlabel(pfocus->xmf_msgtag, fmtbuf);
    xmpi_setlabel(pfocus->xmf_msgcnt, fmtbuf);

    XtVaSetValues(pfocus->xmf_msgcompb, XmNsensitive, False, NULL);
    XtVaSetValues(pfocus->xmf_msgdtb, XmNsensitive, False, NULL);
  } else {
    sprintf(fmtbuf, "%d / %d", pmsg->xmm_gsrc, pmsg->xmm_lsrc);
    xmpi_setlabel(pfocus->xmf_msgsrc, fmtbuf);

    sprintf(fmtbuf, "%d", pmsg->xmm_tag);
    xmpi_setlabel(pfocus->xmf_msgtag, fmtbuf);

    sprintf(fmtbuf, "%d", pmsg->xmm_cnt);
    xmpi_setlabel(pfocus->xmf_msgcnt, fmtbuf);

    strcpy(fmtbuf, pmsg->xmm_commname);
    xmpi_setlabel(pfocus->xmf_msgcom, fmtbuf);

    XtVaSetValues(pfocus->xmf_msgcompb, XmNsensitive, True, NULL);
    XtVaSetValues(pfocus->xmf_msgdtb, XmNsensitive, True, NULL);
  }

  setmsgcnt(pfocus);

  xmpi_dt_msg(rank, pfocus, 0);
}

/*
 *	xmpi_fo_destroy
 *
 *	Function:	- destroy a process focus window
 *	Accepts:	- process rank
 */
void
xmpi_fo_destroy(int rank)
{
  struct xmfocus *p;		       /* favourite pointer */

  if (fo_procs) {
    p = fo_procs + rank;

    if (p->xmf_window) {
      XtUnmapWidget(p->xmf_window);
      XtDestroyWidget(p->xmf_window);
      p->xmf_window = 0;
    }
  }
}

/*
 *	xmpi_fo_destroy_all
 *
 *	Function:	- destroy all process focus windows
 */
void
xmpi_fo_destroy_all()
{
  int i;			       /* favourite index */

  if (fo_procs == 0)
    return;

  for (i = 0; i < fo_nprocs; i++)
    xmpi_fo_destroy(i);

  free((char *) fo_procs);
  fo_procs = 0;
  fo_nprocs = 0;
}

/*
 *	destroy_cb
 *
 *	Function:	- callback to destroy a focus window
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
destroy_cb(Widget, XtPointer cdata, XtPointer)
{
  xmpi_fo_destroy(*((int *) cdata));
}

/*
 *	showproccomm_cb
 *
 *	Function:	- show the process' communicator
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
showproccomm_cb(Widget, XtPointer cdata, XtPointer)
{
  int rank;

  struct xmfocus *pfocus;

  rank = *((int *) cdata);
  pfocus = fo_procs + rank;

  showcomm(pfocus->xmf_proc->xmp_cid, rank);
}

/*
 *	showdt_cb
 *
 *	Function:	- show the process datatype
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
showdt_cb(Widget, XtPointer cdata, XtPointer)
{
  int rank;
  struct xmfocus *pfocus;	       /* focus entry */

  rank = *((int *) cdata);
  pfocus = fo_procs + rank;

  xmpi_dt_update(rank, pfocus, XMPI_SHOWPROC);
}

/*
 *	showmsgdt_cb
 *
 *	Function:	- show the message datatype
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
showmsgdt_cb(Widget, XtPointer cdata, XtPointer)
{
  int rank;
  struct xmfocus *pfocus;	       /* focus entry */

  rank = *((int *) cdata);
  pfocus = fo_procs + rank;

  xmpi_dt_update(rank, pfocus, XMPI_SHOWMESG);
}

/*
 *	showmsgcomm_cb
 *
 *	Function:	- show the message's communicator
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
showmsgcomm_cb(Widget, XtPointer cdata, XtPointer)
{
  int rank;

  struct xmfocus *pfocus;

  rank = *((int *) cdata);
  pfocus = fo_procs + rank;

  if (pfocus->xmf_proc->xmp_curmsg) {
    showcomm(pfocus->xmf_proc->xmp_curmsg->xmm_cid, rank);
  }
}

/*
 *	showcomm
 *
 *	Function:	- show communicator processes
 *	Accepts:	- context ID
 *			- member global rank
 */
static void
showcomm(int cid, int rank)
{
  struct _gps *group;		       /* GPS */

  int lsize;			       /* size local group */

  int rsize;			       /* size remote group */

  int islocal;			       /* rank is in local group */

  char *trace;			       /* cid trace */

  trace = 0;

  if (fl_dbase) {
    xmpi_db_getcomm(cid, rank, &group, &lsize, &rsize, &islocal);
  } else {
    getcomm(cid, rank, group, &lsize, &rsize, &islocal, &trace);
  }

  if (lsize > 0)
    xmpi_vw_commplot(group, lsize, rsize, islocal);

  if (trace)
    free(trace);
}

/*
 *	msgflip_cb
 *
 *	Function:	- callback to flip messages
 *	Accepts:	- focus widget
 *			- client data (process rank)
 */
static void
msgflip_cb(Widget, XtPointer cdata, XtPointer)
{
  int rank;			       /* process rank */

  struct xmproc *pproc;		       /* process entry */

  struct xmfocus *pfocus;	       /* focus entry */

  rank = *((int *) cdata);

  pfocus = fo_procs + rank;
  pproc = pfocus->xmf_proc;

  if (pproc->xmp_msgs) {

    if (al_count(pproc->xmp_msgs) > 1) {
      pproc->xmp_curmsg = (xmmsg*) al_next(pproc->xmp_msgs,
				  (char *) pproc->xmp_curmsg);

      if (pproc->xmp_curmsg == 0
	  || pproc->xmp_curmsg->xmm_nmsg < 0) {
	pproc->xmp_curmsg = (xmmsg*) al_top(pproc->xmp_msgs);
      }
    }
    focusmsgfill(rank, pfocus);
  }
}

/*
 *	position_cb
 *
 *	Function:	- callback to center focus window position
 *	Accepts:	- focus widget
 */
static void
position_cb(Widget focus_w, XtPointer, XtPointer)
{
  Dimension pwidth, pheight;	       /* parent dimensions */

  Dimension width, height;	       /* focus dimensions */

  Dimension swidth, sheight;	       /* screen dimensions */

  Position px, py;		       /* parent coordinates */

  Position x, y;		       /* focus coordinates */

  Widget w;			       /* favourite widget */

/*
 * Get the size/coordinates of the top parent and the focus window.
 * Get the size of the screen.
 */
  w = focus_w;

  while (!XtIsTopLevelShell(w))
    w = XtParent(w);

  XtVaGetValues(w, XmNx, &px, XmNy, &py,
		XmNwidth, &pwidth, XmNheight, &pheight, NULL);

  XtVaGetValues(focus_w, XmNwidth, &width, XmNheight, &height, NULL);

  swidth = WidthOfScreen(XtScreen(focus_w));
  sheight = HeightOfScreen(XtScreen(focus_w));
/*
 * Position the focus window.
 */
  pwidth += 2 * XMPI_FOBORDER;
  pheight += XMPI_FOTITLE + XMPI_FOBORDER;
  px -= XMPI_FOBORDER;
  py -= XMPI_FOTITLE;
  width += 2 * XMPI_FOBORDER;
  height += XMPI_FOTITLE + XMPI_FOBORDER;

  x = px + pwidth;
  y = py + yfocus;

  yfocus += height;
  if ((yfocus + (height / 2)) > pheight)
    yfocus = 0;

  if (x < 0)
    x = 0;
  else if ((x + width) > swidth)
    x = swidth - width;

  if (y < 0)
    y = 0;
  else if ((y + height) > sheight)
    y = sheight - height;

  XtVaSetValues(focus_w, XmNx, x, XmNy, y, NULL);
}

/*
 *	xmpi_fo_reset
 *
 *	Function:	- reset focus position
 */
void
xmpi_fo_reset()
{
  yfocus = 0;
}

/*
 *	xmpi_fo_busy
 *
 *	Function:	- set busy cursor for focus windows
 */
void
xmpi_fo_busy()
{
  int i;			       /* favourite index */

  struct xmfocus *p;		       /* favourite pointer */

  if (fo_procs) {
    for (i = 0, p = fo_procs; i < fo_nprocs; ++i, ++p) {
      if (p->xmf_window)
	xmpi_busy_widget(p->xmf_window);
    }
  }
}

/*
 *	xmpi_fo_unbusy
 *
 *	Function:	- reset busy cursor for focus windows
 */
void
xmpi_fo_unbusy()
{
  int i;			       /* favourite index */

  struct xmfocus *p;		       /* favourite pointer */

  if (fo_procs) {
    for (i = 0, p = fo_procs; i < fo_nprocs; ++i, ++p) {
      if (p->xmf_window)
	xmpi_unbusy_widget(p->xmf_window);
    }
  }
}

/*
 *	createpixlabels
 *
 *	Function:	- create pixmaps for labels
 *			- 3 traffic lights & 1 message
 *	Accepts:	- parent widget
 */
static void
createpixlabels(Widget w)
{
  Pixel fg, bg;			       /* fore/background colours */

  GC pen;			       /* graphic context */

  XGCValues gcvalues;		       /* graphic context info */

  XColor col_def;		       /* colour definition */

  XColor dev_null;		       /* unused functionality */

  Pixmap blkbm;			       /* blocked bitmap */

  Pixmap sysbm;			       /* system bitmap */

  Pixmap runbm;			       /* running bitmap */

  unsigned int depth;		       /* screen depth */

  Display *disp;		       /* display */

  Drawable root;		       /* root window */

  if (blkpix != XmUNSPECIFIED_PIXMAP) {
    return;
  }
  disp = XtDisplay(w);
  root = RootWindowOfScreen(XtScreen(w)),
    depth = DefaultDepthOfScreen(XtScreen(w));

  XtVaGetValues(w, XmNforeground, &fg, XmNbackground, &bg, NULL);
/*
 * Create the static pixmaps.
 */
  blkpix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, fg, bg, depth);

  syspix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, fg, bg, depth);

  runpix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, fg, bg, depth);

  undefpix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, fg, bg, depth);

  mesgpix = XCreatePixmapFromBitmapData(disp, root,
	     (char *) mesg_bits, mesg_width, mesg_height, fg, bg, depth);

/*
 * Create the light bitmaps.
 */
  blkbm = XCreatePixmapFromBitmapData(disp, root,
				(char *) red_bits, red_width, red_height,
				      (Pixel) 1, (Pixel) 0, 1);

  sysbm = XCreatePixmapFromBitmapData(disp, root,
		       (char *) yellow_bits, yellow_width, yellow_height,
				      (Pixel) 1, (Pixel) 0, 1);

  runbm = XCreatePixmapFromBitmapData(disp, root,
			  (char *) green_bits, green_width, green_height,
				      (Pixel) 1, (Pixel) 0, 1);
/*
 * Overlay the coloured lights on top of the traffic light.
 */
  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(w)),
		   XMPI_BLKCOLOUR, &col_def, &dev_null);

  gcvalues.foreground = col_def.pixel;
  gcvalues.background = bg;
  gcvalues.clip_mask = blkbm;
  pen = XCreateGC(disp, root,
		  (GCForeground | GCBackground | GCClipMask), &gcvalues);

  XCopyPlane(disp, blkbm, blkpix, pen, 0, 0,
	     red_width, red_height, 0, 0, (unsigned long) 1);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(w)),
		   XMPI_SYSCOLOUR, &col_def, &dev_null);

  gcvalues.foreground = col_def.pixel;
  gcvalues.clip_mask = sysbm;
  XChangeGC(disp, pen, (GCForeground | GCClipMask), &gcvalues);

  XCopyPlane(disp, sysbm, syspix, pen, 0, 0,
	     yellow_width, yellow_height, 0, 0, (unsigned long) 1);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(w)),
		   XMPI_RUNCOLOUR, &col_def, &dev_null);

  gcvalues.foreground = col_def.pixel;
  gcvalues.clip_mask = runbm;
  XChangeGC(disp, pen, (GCForeground | GCClipMask), &gcvalues);

  XCopyPlane(disp, runbm, runpix, pen, 0, 0,
	     green_width, green_height, 0, 0, (unsigned long) 1);
/*
 * Free the light bitmaps and the pen.
 */
  XFreePixmap(disp, blkbm);
  XFreePixmap(disp, sysbm);
  XFreePixmap(disp, runbm);
  XFreeGC(disp, pen);
}

/*
 *	setmsgcnt
 *
 *	Function:	- set message count
 *	Accepts:	- focus entry
 */
static void
setmsgcnt(struct xmfocus *pfocus)
{
  int ncopies;			       /* # copies */

  int nmsg;			       /* # messages */

  int flag;			       /* sensitivity flag */

  struct xmproc *pproc;		       /* process entry */

  strcpy(fmtbuf, " ");
  flag = False;

  pproc = pfocus->xmf_proc;
  if (pproc->xmp_msgs && pproc->xmp_curmsg) {

    ncopies = pproc->xmp_curmsg->xmm_nmsg;
    nmsg = pproc->xmp_nmsg;

    if ((ncopies < 0) && (al_count(pproc->xmp_msgs) == 1)) {
      strcpy(fmtbuf, "?");
    } else {
      flag = True;
      sprintf(fmtbuf, "%d%s of %d", ncopies,
	      (pproc->xmp_more) ? "+" : " ", nmsg);
    }
  }
  xmpi_setlabel(pfocus->xmf_nmsg, fmtbuf);
  XtVaSetValues(pfocus->xmf_nmsgpb, XmNsensitive, flag, NULL);
}

/*
 *	getcomm
 *
 *	Function:	- get communicator processes
 *	Accepts:	- context ID
 *			- member global rank
 *			- array of GPS (out)
 *			- size local group  (out)
 *			- size remote group (out)
 *			- local/remote flag (out)
 *			- trace buffer (out)
 *
 *	If no matching communicator processes can be found the size of the
 *	local group is returned as zero.
 *  	In case of an intra-comm the size of the group
 *  	is returned in the local group size parameter and 0
 *  	is returned for the size of the remote group parameter.  In
 *  	case of an inter-comm flag is returned as 1 if member global
 *  	rank is in the local group and 0 if in the remote group.
 */
static void
getcomm(int cid, int grank, struct _gps *pgps, int *lsize,
	int *rsize, int *islocal, char **trace)
{
  struct trcid *t;		       /* cid trace */

  int i;			       /* favourite index */

  if ((*trace = (char*) xmpi_sys_comm(xmpi_app_procs + grank, cid)) == 0) {
    *lsize = 0;
    return;
  }
  t = (struct trcid *) * trace;

  *lsize = t->trc_nlg;
  *rsize = t->trc_nrg;
  pgps = (struct _gps *) (t + 1);

  i = group_index(grank, pgps, *lsize);

  *islocal = (i >= 0);
}

/*
 *	group_index
 *
 *	Function:	- find index of process in a group
 *	Accepts:	- global rank of process
 *			- group GPS
 *			- size of group
 *	Returns:	- index in group or -1 if not in group
 */
static int
group_index(int rank, struct _gps *group, int n)
{
  int i;

  struct _gps *pg;

  for (i = 0, pg = group; i < n; ++i, ++pg) {

    if (pg->gps_grank == rank)
      return (i);
  }

  return (-1);
}

/*
 *	mpiblk_format
 *
 *	Function:	- format state of a process blocked in MPI
 *	Accepts:	- process global rank
 *			- process
 */
static void
mpiblk_state_out(int rank, struct xmproc *p)
{
  formatproc(rank, p->xmp_lrank, task_str);
  strcat(task_str, "   ");
  strcat(task_str, p->xmp_prog);

  sprintf(func_str, "%s", blktype(p->xmp_func));
  nullify(peer_str);
  nullify(root_str);
  nullify(tag_str);
  nullify(cid_str);
  nullify(count_str);
/*
 * Format according to the function blocked upon.
 */
  switch (p->xmp_func) {
/*
 * These functions show peer/tag/contextid/count.
 */
  case BLKMPIRECV:
  case BLKMPISEND:
  case BLKMPISSEND:
  case BLKMPIRSEND:
  case BLKMPIBSEND:

  case BLKMPIIRECV:
  case BLKMPIISEND:
  case BLKMPIISSEND:
  case BLKMPIIRSEND:
  case BLKMPIIBSEND:

    formatproc(p->xmp_gpeer, p->xmp_lpeer, peer_str);
    formattag(p->xmp_tag, tag_str);
    (void) strcpy(cid_str, p->xmp_commname);
    sprintf(count_str, "%d", p->xmp_cnt);

    break;
/*
 * These functions show peer/tag/contextid.
 */
  case BLKMPIPROBE:
  case BLKMPIIPROBE:

    sprintf(func_str, "%s", blktype(p->xmp_func));
    formatproc(p->xmp_gpeer, p->xmp_lpeer, peer_str);
    formattag(p->xmp_tag, tag_str);
    (void) strcpy(cid_str, p->xmp_commname);

    break;
/*
 * These functions show peer/contextid.
 */
  case BLKMPIICOMMCREATE:

    formatproc(p->xmp_gpeer, p->xmp_lpeer, peer_str);
    (void) strcpy(cid_str, p->xmp_commname);

    break;
/*
 * These functions show root/contextid/count.
 */
  case BLKMPIBCAST:
  case BLKMPIGATHER:
  case BLKMPIGATHERV:
  case BLKMPISCATTER:
  case BLKMPISCATTERV:
  case BLKMPIREDUCE:

    formatproc(p->xmp_groot, p->xmp_lroot, root_str);
    sprintf(count_str, "%d", p->xmp_cnt);
    (void) strcpy(cid_str, p->xmp_commname);

    break;
/*
 * These functions show contextid/count.
 */
  case BLKMPIALLGATHER:
  case BLKMPIALLGATHERV:
  case BLKMPIALLTOALL:
  case BLKMPIALLREDUCE:
  case BLKMPISCAN:
  case BLKMPIREDUCESCATTER:

    (void) strcpy(cid_str, p->xmp_commname);
    sprintf(count_str, "%d", p->xmp_cnt);

    break;
/*
 * These functions show contextid/root.
 */
  case BLKMPILSPAWN:
  case BLKMPICOMMSPAWN:
  case BLKMPICOMMSPAWNMULT:
  case BLKMPICOMMCONNECT:
  case BLKMPICOMMACCEPT:

    formatproc(p->xmp_groot, p->xmp_lroot, root_str);
    (void) strcpy(cid_str, p->xmp_commname);
    break;
/*
 * These functions show contextid.
 */
  case BLKMPIALLTOALLV:
  case BLKMPIBARRIER:
  case BLKMPICOMMCREATE:
  case BLKMPIICOMMMERGE:
  case BLKMPICOMMSPLIT:
  case BLKMPICOMMDUP:
  case BLKMPICARTCREATE:
  case BLKMPICARTSUB:
  case BLKMPIGRAPHCREATE:
  case BLKMPILTRACEON:
  case BLKMPILTRACEOFF:

    (void) strcpy(cid_str, p->xmp_commname);
    break;
/*
 * These are oddballs. In the send/recv parts the top function
 * is the send/recv so we don't get here. We get here inside sendrecv but
 * outside the send/recv.
 */
  case BLKMPISENDRECV:
  case BLKMPISENDRECVREP:

    p->xmp_dtype = 0;
    break;
/*
 * MPI_Init shows only the function. We also clear some process information
 * since the FYI in this case is not accurate.
 */
  case BLKMPIINIT:

    p->xmp_wfunc = 0;
    p->xmp_dtype = 0;
    break;
  }
/*
 * Add the wrapper function if any.
 */
  if (p->xmp_wfunc != 0) {
    sprintf(fmtbuf, " (%s)", blktype(p->xmp_wfunc));
    strcat(func_str, fmtbuf);
  }
}

/*
 *	formatproc
 *
 *	Function:	- formats a processes task
 *	Accepts:	- process global rank
 *			- process local rank
 *			- buffer to format into (out)
 */
static void
formatproc(int4 global, int4 local, char *buf)
{
  if (global == XMPI_ANY_SOURCE)
    strcpy(buf, "MPI_ANY_SOURCE");
  else if (local == -1)
    sprintf(buf, "%d", global);
  else
    sprintf(buf, "%d / %d", global, local);
}


/*
 *	formattag
 *
 *	Function:	- formats a tag for output
 *	Accepts:	- tag
 *			- buffer to format into (out)
 */
static void
formattag(int tag, char *str)
{
  if (tag == XMPI_ANY_TAG) {
    strcpy(str, "ANY");
  } else {
    sprintf(str, "%d", tag);
  }
}
