//
// $COPYRIGHT$
//
//	$Id: build_run.cc,v 1.4 2001/05/18 20:23:02 bbarrett Exp $
//
//	Function:	- provides option menu for build-run window
//			- parses schema for build-run window
//                      - adds schema for build-run window
//

#include <stdlib.h>
#include <stdio.h>

#include <string>

#include <args.h>
#include <all_list.h>
#include <all_opt.h>
#include <app_mgmt.h>
#include <app_schema.h>
#include <freq.h>
#include <laminternal.h>
#include <lamnet.h>
#include <kio.h>
#include <portable.h>
#include <priority.h>
#include <terror.h>
#include <trreq.h>
#include <typical.h>
#include <mpi.h>
#include <mpisys.h>
#include <net.h>

#include "xmpi/xmpi_sys.h"
#include "xmpi_asc_parse.h"

using namespace std;

// function in nodeslist.cc.  Only used here, so I am doing this...
extern int get_node_names(char ***, int *);
extern int get_nodes_list(xmpi_node*, int, char**);
extern int fill_nodes(LIST*, xmpi_node**, int*);

static int get_args(char**, char**, int*, struct aschema*);

// XMPI_SYS_OPTIONS
//
// PURPOSE: provide the description of the vendor pane of the
//          build run window
//
int
xmpi_sys_get_vendor_info(xmpi_menu_item **run_options, 
			 int *run_options_length,
			 xmpi_menu_item **schema_options,
			 int *schema_options_length,
			 xmpi_node **nodes, int *nodes_length)
{
  xmpi_menu_item *tmp_menu_item;

  //
  // Create the run_options menu...
  //
  *run_options_length = 10;
  tmp_menu_item = (xmpi_menu_item*) malloc((sizeof(xmpi_menu_item) * 
					    *run_options_length));

  // -c2c or not to C2C
  tmp_menu_item[0].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[0].current_val.xmpi_toggle = 1;
  tmp_menu_item[0].default_val.xmpi_toggle = 1;
  tmp_menu_item[0].xmpi_label = strdup("Client to Client mode");

  // -f use default I/O file descriptors
  tmp_menu_item[1].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[1].current_val.xmpi_toggle = 0;
  tmp_menu_item[1].default_val.xmpi_toggle = 0;
  tmp_menu_item[1].xmpi_label = strdup("Use default I/O file descriptors");

  // -O homogeneous lam
  tmp_menu_item[2].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[2].current_val.xmpi_toggle = 0;
  tmp_menu_item[2].default_val.xmpi_toggle = 0;
  tmp_menu_item[2].xmpi_label = strdup("Homogeneous LAM");

  // -D use location of binary as location of cwd
  tmp_menu_item[3].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[3].current_val.xmpi_toggle = 0;
  tmp_menu_item[3].default_val.xmpi_toggle = 0;
  tmp_menu_item[3].xmpi_label = strdup("Set CWD to location of binary");

  // -ger enable Guaranteed Envelope Resources
  tmp_menu_item[4].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[4].current_val.xmpi_toggle = 0;
  tmp_menu_item[4].default_val.xmpi_toggle = 0;
  tmp_menu_item[4].xmpi_label = strdup("Use GER");

  // -nsigs Have lam catch signals
  tmp_menu_item[5].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[5].current_val.xmpi_toggle = 1;
  tmp_menu_item[5].default_val.xmpi_toggle = 1;
  tmp_menu_item[5].xmpi_label = strdup("LAM catches signals");

  // -nx
  tmp_menu_item[6].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[6].current_val.xmpi_toggle = 1;
  tmp_menu_item[6].default_val.xmpi_toggle = 1;
  tmp_menu_item[6].xmpi_label = strdup("Export LAM env. variables");

  // -pty
  tmp_menu_item[7].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[7].current_val.xmpi_toggle = 1;
  tmp_menu_item[7].default_val.xmpi_toggle = 1;
  tmp_menu_item[7].xmpi_label = strdup("Enable pseudo-tty support");

  // -tracing on / off at all
  tmp_menu_item[8].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[8].current_val.xmpi_toggle = 1;
  tmp_menu_item[8].default_val.xmpi_toggle = 1;
  tmp_menu_item[8].xmpi_label = strdup("LAM Tracing Enabled");

  // tracing on / off initially
  tmp_menu_item[9].xmpi_type = XMPI_MENU_TOGGLE;
  tmp_menu_item[9].current_val.xmpi_toggle = 1;
  tmp_menu_item[9].default_val.xmpi_toggle = 1;
  tmp_menu_item[9].xmpi_label = strdup("LAM Tracing Initially On");

  *run_options = tmp_menu_item;

  //
  // Now create the individual schema line options...
  //
  *schema_options_length = 2;
  tmp_menu_item = (xmpi_menu_item*) malloc((sizeof(xmpi_menu_item) * 
					    *schema_options_length));  

  tmp_menu_item[0].xmpi_type = XMPI_MENU_TEXTBOX;
  tmp_menu_item[0].current_val.xmpi_text = NULL;
  tmp_menu_item[0].default_val.xmpi_text = NULL;
  tmp_menu_item[0].xmpi_label = strdup("Source Node: ");

  tmp_menu_item[1].xmpi_type = XMPI_MENU_TEXTBOX;
  tmp_menu_item[1].current_val.xmpi_text = NULL;
  tmp_menu_item[1].default_val.xmpi_text = NULL;
  tmp_menu_item[1].xmpi_label = strdup("Working Directory: ");

  *schema_options = tmp_menu_item;

  //
  // create the nodes listsing.
  //
  char ** node_names = 0;
  int node_names_length = 0;

  if (get_node_names(&node_names, &node_names_length) != 0) {
    return 1;
  }

  *nodes_length = node_names_length + 2;
  
  if ((*nodes) != 0)
    free(*nodes);

  *nodes = (xmpi_node*) malloc (sizeof(xmpi_node) * (*nodes_length));
  
  // set up the ALL and LOCAL things...
  (*nodes)[0].name = strdup("ALL NODES");
  (*nodes)[0].exclusive = 1;
  (*nodes)[0].default_sel = 1;
  (*nodes)[0].selected = 1;
  
  (*nodes)[1].name = strdup("LOCAL NODES");
  (*nodes)[1].exclusive = 1;
  (*nodes)[1].default_sel = 0;
  (*nodes)[1].selected = 0;
  
  // now copy over the nodes listing...
  for (int i = 0 ; i < node_names_length ; ++i) {
    (*nodes)[(i + 2)].name = node_names[i];
    (*nodes)[(i + 2)].exclusive = 0;
    (*nodes)[(i + 2)].default_sel = 0;
    (*nodes)[(i + 2)].selected = 0;
  }

  free(node_names);

  return 0;
}


// xmpi_sys_get_schema_line
//
// INPUT: prog                   name of program to launch
//        copy                   number of copies to launch
//        args                   contents of the args field
//        nodes                  the listing of nodes to launch on
//        nodes_length           number of struct node in nodes
//        schema_options         array of xmpi_menu_item
//        schema_options_length  length of the above array
//
// OUTPUT: app_schema_entry   pointer to the char* to hold new 
//                            schema entry
//
// RETURNS: 0    ->     success
//          LAMERROR -> otherwise
// 
// DESCRIPTION: This takes all the information from the build and run
// window and creates an application schema entry based on the
// information.
int
xmpi_sys_get_schema_line(char *prog, int copy, char *args, 
			 xmpi_node *nodes, int nodes_length,
			 xmpi_menu_item *schema_options,
			 int schema_options_length, 
			 char** app_schema_entry)
{

  string output;

  // sanity check...
  if (schema_options_length != 2)
    return LAMERROR;

  // get the nodes listing
  char *node_list;
  if (get_nodes_list(nodes, nodes_length, &node_list) != 0)
    return LAMERROR;

  output += node_list;
  free(node_list);

  // convert the copy thing to a char*...
  if (copy != 0) {
    char *copy_string = 0;
    copy_string = (char*) malloc(sizeof(char) * (sizeof(int) * 8 + 1));
    if (copy_string == 0)
      return LAMERROR;

    sprintf(copy_string, "%d", copy);

    output = output + " -c " + copy_string;
    free(copy_string);
  }

  // do we have anything in the -s field?
  if (schema_options[0].current_val.xmpi_text != NULL)
    if (strlen(schema_options[0].current_val.xmpi_text) != 0)
      output = output + " -s " + schema_options[0].current_val.xmpi_text;

  // do we have anything in the -wd field?
  if (schema_options[1].current_val.xmpi_text != NULL)
    if (strlen(schema_options[1].current_val.xmpi_text) != 0)
      output = output + " -wd " + schema_options[1].current_val.xmpi_text;

  // time to add the program
  output = output + " " + prog;

  // and the arguments to the program
  if (strlen(args) != 0)
    output = output + " " + args;

  *app_schema_entry = (char*) malloc(sizeof(char) * (output.size() + 1));
  if (*app_schema_entry == 0)
    return LAMERROR;

  strcpy(*app_schema_entry, output.c_str());
  
  return 0;
}


// xmpi_sys_parse_schema_line
//
// INPUT: app_schema_entry   pointer to the char* to hold new 
//                           schema entry
//
// OUTPUT: prog                   name of program to launch
//         copy                   number of copies to launch
//         args                   contents of the args field
//         nodes                  the listing of nodes to launch on
//         nodes_length           number of struct node in nodes
//         schema_options         array of xmpi_menu_item
//         schema_options_length  length of the above array
//
//
// RETURNS: 0    ->     success
//          LAMERROR -> otherwise
// 
// DESCRIPTION: Takes an app_schema_entry and fills in all the
// information for the build and run window.
int
xmpi_sys_parse_schema_line(char** prog, int *copy, char **args,
			   xmpi_node **nodes, int *nodes_length, 
			   xmpi_menu_item **schema_options, 
			   int schema_options_length, 
			   char* app_schema_entry)
{
  LIST *app;  // this should be the parsed app schema (i hope)
  int result;
  struct aschema *schema;

  if (schema_options_length != 2)
    return LAMERROR;

  app = asc_bufparse(app_schema_entry, strlen(app_schema_entry), 
		     &result, NULL);

  if (app == NULL) {
    return LAMERROR;
  }

  schema = (struct aschema*) al_top(app);

  if (get_args(prog, args, copy, schema) != 0) {
    return LAMERROR;
  }

  // update the nodes structure
  if (fill_nodes(schema->asc_nodelist, nodes, nodes_length) != 0) {
    return LAMERROR;
  }
  
  // I can't see how this would happen, but just in case...
  if (schema_options_length != 2)
    return LAMERROR;

  // schema[0] is source node
  if ((*schema_options)[0].current_val.xmpi_text != NULL)
    free((*schema_options)[0].current_val.xmpi_text);
  (*schema_options)[0].current_val.xmpi_text = NULL;
    
  if (schema->asc_srcnode >= 0) {
    char* tmp;
    tmp = (char*) malloc(sizeof(char) * (sizeof(int4) * 8 + 1));
    sprintf(tmp, "n%d", schema->asc_srcnode);
    
    (*schema_options)[0].current_val.xmpi_text = tmp;
  }

  // schema[1] is working dir
  if ((*schema_options)[1].current_val.xmpi_text != NULL)
    free((*schema_options)[1].current_val.xmpi_text);
  
  if (schema->asc_env->ape_wrkdir != NULL) {
    (*schema_options)[1].current_val.xmpi_text =
      strdup(schema->asc_env->ape_wrkdir);
  } else {
    (*schema_options)[1].current_val.xmpi_text = NULL;
  }

  al_free(app);
  return 0;
}

// xmpi_sys_verify_schema_line
//
// INPUT: app_schema_entry
// 
// RETURNS: 0 -> verify
//          1 -> not verify
int
xmpi_sys_verify_schema_line(char* app_schema_entry, char** message)
{
  LIST *app;  
  int result;
  
  app = asc_bufparse(app_schema_entry, strlen(app_schema_entry), 
		     &result, NULL);
  
  if (app == NULL) {
    *message = strdup("Error parsing schema line options");
    return 1;
  }
    
  return 0;
}


int
xmpi_sys_get_run_options(xmpi_menu_item *run_options, 
			 int run_options_length, 
			 char **run_list)
{
  string output;

  // sanity check...
  if (run_options_length != 10)
    return LAMERROR;

  output += string("# This file automatically generated by XMPI\n");
  output += string("# Please do not modify.\n\n");
  output += string("# To run this schema, use the command:\n");

  // add the mpirun command
  output += "# mpirun";

  // are we using c2c mode or lamd?
  if (run_options[0].current_val.xmpi_toggle == 0)
    output = output + " -lamd";
  else
    output = output + " -c2c";

  // add -f flag?
  if (run_options[1].current_val.xmpi_toggle != 0)
    output = output + " -f";

  // add -O flag?
  if (run_options[2].current_val.xmpi_toggle != 0)
    output = output + " -O";

  // -D
  if (run_options[3].current_val.xmpi_toggle != 0)
    output = output + " -D";

  // -ger
  if (run_options[4].current_val.xmpi_toggle != 0)
    output = output + " -ger";
  else
    output = output + " -nger";

  // -nsigs
  if (run_options[5].current_val.xmpi_toggle == 0)
    output = output + " -nsigs";

  // -nx
  if (run_options[6].current_val.xmpi_toggle == 0)
    output = output + " -nx";

  // -pty
  if (run_options[7].current_val.xmpi_toggle != 0)
    output = output + " -pty";

  if (run_options[8].current_val.xmpi_toggle != 0) {
    if (run_options[9].current_val.xmpi_toggle != 0)
      output = output + " -ton";
    else
      output = output + " -toff";
  }

  *run_list = (char*) malloc(sizeof(char) * (output.size() + 1));
  if (*run_list == NULL)
    return LAMERROR;
  
  strcpy(*run_list, output.c_str());

  return 0;
}


int
xmpi_sys_parse_run_options(xmpi_menu_item **run_options,
			   int *,
			   char *run_list)
{
  // thank God for the STL
  string args(run_list);
  OPT *ad;
  
  args.erase(0, args.find("# mpirun") + 8);
  
  if ((ad = xmpi_get_options((char*) args.c_str())) == 0)
    return LAMERROR;
  
  if (!ao_taken(ad, (char*) "lamd"))
    (*run_options)[0].current_val.xmpi_toggle = 1;
  else
    (*run_options)[0].current_val.xmpi_toggle = 0;
  
  if (!ao_taken(ad, (char*) "f"))
    (*run_options)[1].current_val.xmpi_toggle = 0;
  else
    (*run_options)[1].current_val.xmpi_toggle = 1;
  
  if (ao_taken(ad, (char*) "O"))
    (*run_options)[2].current_val.xmpi_toggle = 1;
  else
    (*run_options)[2].current_val.xmpi_toggle = 0;
  
  if (ao_taken(ad, (char*) "D"))
    (*run_options)[3].current_val.xmpi_toggle = 1;
  else
    (*run_options)[3].current_val.xmpi_toggle = 0;
  
  if (ao_taken(ad,(char*) "ger"))
    (*run_options)[4].current_val.xmpi_toggle = 1;
  else
    (*run_options)[4].current_val.xmpi_toggle = 0;
  
  if (!ao_taken(ad,(char*) "nsigs"))
    (*run_options)[5].current_val.xmpi_toggle = 0;
  else
    (*run_options)[5].current_val.xmpi_toggle = 1;
  
  if (ao_taken(ad, (char*) "nx"))
    (*run_options)[6].current_val.xmpi_toggle = 0;
  else
    (*run_options)[6].current_val.xmpi_toggle = 1;
  
  if (ao_taken(ad, (char*) "pty"))
    (*run_options)[7].current_val.xmpi_toggle = 1;
  else
    (*run_options)[7].current_val.xmpi_toggle = 0;

  if (ao_taken(ad, (char*) "ton") || 
      ao_taken(ad, (char*) "t") ||
      ao_taken(ad, (char*) "toff"))
    (*run_options)[8].current_val.xmpi_toggle = 1;
  else
    (*run_options)[8].current_val.xmpi_toggle = 0;

  if (ao_taken(ad, (char*) "ton") ||
      ao_taken(ad, (char*) "t"))
    (*run_options)[9].current_val.xmpi_toggle = 1;
  else
    (*run_options)[9].current_val.xmpi_toggle = 0;

  ao_free(ad);
  
  return 0;
}


// get_args
//
// DESCRIPTION: "little" helper function to parse out the schema
// struct that LAM builds
//
// INPUT: schema     a single application schema struct
// 
// OUTPUT: prog      program name
//         args      arguments to the program
//         copies    number of copies to launch
//
// RETURNS: 0        Life rocks
//          LAMERROR Life does not rock
//
int
get_args(char** prog, char** args, 
	 int* copies, struct aschema* schema)
{
  *copies = schema->asc_proc_cnt;

  if (schema->asc_args->apa_argc <= 0)
    return LAMERROR;

  if (*prog != NULL)
    free(*prog);

  *prog = strdup((schema->asc_args->apa_argv)[0]);

  string tmp;
  for (int i = 1 ; i < schema->asc_args->apa_argc; ++ i)
    tmp = tmp + schema->asc_args->apa_argv[i] + " ";

  if (*(tmp.end()) == ' ') 
    tmp.erase(tmp.end());

  if (*args != NULL)
    free(*args);

  *args = (char*) malloc(sizeof(char) * (tmp.size() + 1));
  strcpy(*args, tmp.c_str());

  return 0;
}

