
#include "WindowDevice.h"

#include "PixPort.h"


#if EG_MAC
#include <Displays.h>
#include <Gestalt.h>
#endif

#include "UtilStr.h"

#ifdef EG_SDL
#include <SDL/SDL_syswm.h>
#endif


WindowDevice::WindowDevice() :

	DC_VideoDevice( false ) {
	mContextRef = 0;
	mWind = 0;
	mDC = 0;
	mParentWind = 0;
	mCurPaletteSeed = -1;

#if EG_WIN
	mDDObj		= nil;
#endif

	Init();

	/*
	GDHandle gDevice = ::GetMainDevice();
	if ( gDevice ) {
		PixMapHandle pixMap = (**gDevice).gdPMap;
		sOSDepth = (**pixMap).pixelSize;
		if ( sOSDepth == 8 ) {
			::BlockMove( (**(**pixMap).pmTable).ctTable, sOSPalette, 256 * sizeof( ColorSpec ) );
		} }
	else
		sOSDepth = 16;
	*/
}

WindowDevice::~WindowDevice() {

	ExitFullscreen();
}

void WindowDevice::SetOSWindow( PP_WindowPtr inWind, PP_WindowPtr inParentWind ) {

	ExitFullscreen();

	mWind = inWind;
	mParentWind = inParentWind;

	GetWindowRect( mOutputRect );
	OffsetRect( &mOutputRect, - mOutputRect.left, - mOutputRect.top );

	mOrigin.h = 0;
	mOrigin.v = 0;
}

bool WindowDevice::EnterFullscreen( long inDispID, Point& ioSize, int inBitDepth ) {

	Point defaultSize;
	bool ok = false;

#if EG_MAC
	GDHandle	prevDev;
	GWorldPtr	prevPort;
	::GetGWorld( &prevPort, &prevDev );
#endif

	if ( IsFullscreen() ) {
		ok = true;
		goto bail;
	}

#if EG_WIN
	// If the window is maximized, get it back to normal size
	if ( IsZoomed( mWind ) )
		::ShowWindow( mWind, SW_RESTORE );
#endif

	// Store the positon of our win for when we exit fullscreen mode
	GetWindowRect( mWindRectHolder );

	// By spec, f the user gives us zero as a resolution, we use the defualt setting
	GetDisplaySize( inDispID, defaultSize );
	if ( ioSize.h == 0 )
		ioSize.h = defaultSize.h;
	if ( ioSize.v == 0 )
		ioSize.v = defaultSize.v;

#if EG_WIN
	if ( mParentWind ) {
		//::ShowWindow( mParentWind, SW_HIDE | SW_MINIMIZE );
		::ShowWindow( mParentWind, SW_MINIMIZE );
	}
#elif EG_MAC
	::HideWindow( mWind );

	if ( DMGetGDeviceByDisplayID( inDispID, &mFS_GDevice, false ) != noErr )
		mFS_GDevice = nil;
#endif

	// Safety first...
	if ( inBitDepth != 8 && inBitDepth != 16 && inBitDepth != 32 )
		inBitDepth = 32;
	if ( inDispID < 0 )
		inDispID = 0;

	mDispID		= inDispID;

#if EG_MAC
#if USE_DISP_MGR
	#pragma unused( inFreq )

	Rect		r;
	GDHandle	theGDevice;

	::HideCursor();
	::HideWindow( inWin );


	// Hide that pesky menubar...
	RgnHandle grayRgn;
	grayRgn = ::LMGetGrayRgn();
	mMenuBarHeight	= ::LMGetMBarHeight();
	::LMSetMBarHeight( 0 );
	r = qd.screenBits.bounds;
	r.bottom = r.top + mMenuBarHeight;
	mMenuBarRgn	= ::NewRgn();
	::RectRgn( mMenuBarRgn, &r );
	::UnionRgn( grayRgn, mMenuBarRgn, grayRgn );

	// Fetch a ptr to the device given by inDispNum
	if ( ::DMGetGDeviceByDisplayID( inDispNum, &theGDevice, false ) != noErr )
		theGDevice = nil;
	if ( ! theGDevice )
		theGDevice = ::GetMainDevice();

	// Use RequestVideo.c to get the Disp Mgr to do what we want
	VideoRequestRec requestRec;
	requestRec.screenDevice		=	theGDevice;
	requestRec.reqBitDepth		=	inBitDepth;
	requestRec.reqHorizontal	=	ioSize.h;
	requestRec.reqVertical		=	ioSize.v;
	requestRec.displayMode		=	nil;					// must init to nil
	requestRec.depthMode		=	nil;					// must init to nil
	requestRec.requestFlags		=	0;
	if ( RVRequestVideoSetting( &requestRec ) == noErr ) {
		if ( RVSetVideoRequest( &requestRec ) == noErr ) {
			outSize.h = requestRec.availHorizontal;
			outSize.v = requestRec.availVertical;
			ok = true;
		}
	}

	if ( ok ) {

		// Make the window cover the device
		::MoveWindow( inWin, 0, 0, true );
		::SizeWindow( inWin, outSize.h, outSize.v, true );
		::ShowWindow( inWin );

		// Setup the window as the main grafport
		mContextRef = inWin;	mIsFullscreen = true;
		mX			= outSize.h;
		mY			= outSize.v;
		::SetRect( &r, 0, 0, mX, mY+2 );
		::NewGWorld( &mWorld, inBitDepth, &r, nil, nil, useTempMem );
		mBM = ::GetGWorldPixMap( mWorld );
		mBytesPerRow	= (**mBM).rowBytes & 0x1FFF;
		mBytesPerPix	= (**mBM).pixelSize / 8;
	}


#elif USE_DRAW_SPROCKETS

	DSpContextReference		ref;
	OSStatus				err;
	DSpContextAttributes	context;
	long					bestWidth = 0x7FFFFFFF;
	bool 					isInitted = false;

	err = ::DSpStartup();
	if ( ! err ) {
		err = ::DSpGetFirstContext( inDispID, &ref );

		// Look for smallest size w/ for given depth
		while ( ! err && ref ) {
			err = DSpContext_GetAttributes( ref, &context );
			if ( ! err && ref ) {
				if ( context.displayBestDepth == inBitDepth ) {
					if ( context.displayWidth == ioSize.h && context.displayHeight == ioSize.v ) {
						mContextRef = ref;
						isInitted = true;
						break; }
					else if ( context.displayWidth <= bestWidth && context.displayWidth >= 640 ) {
						mContextRef = ref;
						isInitted = true;
						bestWidth = context.displayWidth;
					}
				}

				// Try the next context for this display
				err = ::DSpGetNextContext( ref, &ref );
			}
		}

		if ( ! isInitted ) {
			mContextRef = 0;
			::DSpShutdown();
			goto bail;
		}

		::DSpContext_GetAttributes( mContextRef, &mContext );
		ioSize.h = mContext.displayWidth;
		ioSize.v = mContext.displayHeight;

		mContext.contextOptions 			= kDSpContextOption_DontSyncVBL;
		mContext.frequency					= 0;
		mContext.reserved1					= 0;
		mContext.reserved2					= 0;
		mContext.gameMustConfirmSwitch		= false;
		mContext.reserved3[0]	= 0;
		mContext.reserved3[1]	= 0;
		mContext.reserved3[2]	= 0;
		mContext.reserved3[3]	= 0;
		mContext.colorTable		= 0;
		mContext.pageCount		= 1;
		mContext.colorNeeds		= kDSpColorNeeds_Require;

		RGBColor back = { 0, 0, 0 };
		::DSpSetBlankingColor( &back );

		// Try to reserve the device
		err = ::DSpContext_Reserve( mContextRef, &mContext );
		if ( ! err ) {

			// If no errors, 'activate' the device into fullscreen
			::HideCursor();

			err = ::DSpContext_SetState( mContextRef, kDSpContextState_Active );


			if ( err && err != kDSpConfirmSwitchWarning ) {
				::DSpContext_Release( mContextRef );
				::DSpShutdown(); }
			else {
				ok = true;

				mDC = nil;
			}
		}
	}
#endif
#endif // EG_MAC

#if EG_WIN
#if USE_DIRECTX
	if ( mWind ) {
		HRESULT err;

		err = ::DirectDrawCreate( nil, &mDDObj, nil );

		if ( err == DD_OK ) {
			LPDIRECTDRAWSURFACE context;

			err = mDDObj -> SetCooperativeLevel( mWind, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN );
			if ( err == DD_OK ) {
				err = mDDObj -> SetDisplayMode( ioSize.h, ioSize.v, inBitDepth );
				if ( err == DD_OK ) {
					mDDObj -> Compact();
					DDSURFACEDESC ddsd;
					ddsd.dwSize = sizeof(ddsd);
					ddsd.dwFlags = DDSD_CAPS;
					ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
					err = mDDObj -> CreateSurface( &ddsd, &context, nil); }
				else
					mDDObj -> SetCooperativeLevel( mWind, DDSCL_NORMAL );
			}

			if ( err == DD_OK ) {
				mContextRef = context;
				//::ShowWindow( mWind, SW_SHOW );
				//::SetWindowPos( inWin, HWND_TOP, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE );
				//::SetForegroundWindow( mWind );
				//::SetCapture( inWin );

				PALETTEENTRY pal[ 256 ];
				for ( int j = 0; j < 256; j++ ) {
					pal[ j ].peRed		= 0;
					pal[ j ].peGreen	= 0;
					pal[ j ].peBlue		= 0;
					pal[ j ].peFlags	= PC_RESERVED;
				}
				mDDObj -> CreatePalette( DDPCAPS_8BIT, pal, &mFS_Palette, nil );
				mContextRef -> SetPalette( mFS_Palette );
				ok = true; }
			else {
				 mDDObj -> Release();
				 mDDObj = nil;
			}
		}
	}
#endif
#endif // EG_WIN

/*
#ifdef EG_SDL
	nFlags = SDL_SWSURFACE | SDL_FULLSCREEN;
	SDL_WM_GrabInput(SDL_GRAB_ON);

	SDL_SetVideoMode(ioSize.h, ioSize.v, inBitDepth, SDL_SWSURFACE | SDL_FULLSCREEN);
#endif // EG_SDL
*/
	if ( ok ) {
		mDepth = inBitDepth;
		SetRect( &mOutputRect, 0, 0, ioSize.h, ioSize.v ); }
	else
		mContextRef = 0;

bail:
#if EG_MAC
	::SetGWorld( prevPort, prevDev );
	::FlushEvents( 0xFFFF, 0 );
#endif

	mIsFullscreen	= mContextRef != 0;

	if ( IsFullscreen() ) {
#if EG_MAC
		::HideCursor();
#elif EG_WIN
		::SetCursor( ::LoadCursor( nil, IDC_ARROW ) );
		while ( ::ShowCursor( false ) >= 0 ) { }
#endif
	}
	else {

#if EG_WIN
		if ( mParentWind )
			::ShowWindow( mParentWind, SW_SHOWNORMAL );
		::SetForegroundWindow( mWind );
#endif

		Resize( mWindRectHolder );
	}

	return ok;
}

void WindowDevice::ExitFullscreen() {

#if EG_MAC
	GDHandle	prevDev;
	GWorldPtr	prevPort;
	::GetGWorld( &prevPort, &prevDev );
#endif

	EndFrame();
	if ( ! IsFullscreen() )
		return;

	// If it's possible we called SetEntries() then reset the palette to the sys palette
#if EG_MAC
#if TARGET_API_MAC_CARBON
		PixMapHandle pixMapHandle = GetPortPixMap(GetWindowPort(mWind));
		char state = HGetState( (Handle)pixMapHandle );
		HLock( (Handle) pixMapHandle );
		PixMap* pixMap = *pixMapHandle;
#else
		PixMap* pixMap = *((CGrafPtr)mWind) -> portPixMap;
#endif
	if ( pixMap -> pixelSize == 8 )
		::SetEntries( 0, 255, (**GetCTable( 8 )).ctTable  );
#if TARGET_API_MAC_CARBON
		HSetState( (Handle) pixMapHandle, state );
#endif
#endif

#if USE_DRAW_SPROCKETS
	//::DSpContext_FadeGamma( mContextRef, 0, nil );
	::DSpContext_SetState( mContextRef, kDSpContextState_Inactive );
	//::DSpContext_FadeGamma( mContextRef, 100, nil );
	::DSpContext_Release( mContextRef );
	::DSpShutdown();
	::InitCursor();
#endif

#if USE_DISP_MGR
	RVSetVideoAsScreenPrefs();

	// Make the menu bar visible again
	RgnHandle grayRgn;
	grayRgn = ::LMGetGrayRgn();
	::LMSetMBarHeight( mMenuBarHeight );
	::DiffRgn( grayRgn, mMenuBarRgn, grayRgn );	// remove the menu bar from the desktop
	::PaintOne( nil, mMenuBarRgn );			// redraw the menubar
	::DisposeRgn( mMenuBarRgn );

	// Restore the original color table for the main device
	if ( sOSDepth == 8 && mBytesPerPix == 1 )
		::SetEntries( 0, 255, sOSPalette );
	::InitCursor();
#endif

#if USE_DIRECTX
	if ( mContextRef )
		mContextRef -> Release();

	if ( mDDObj ) {
		//mDDObj -> RestoreDisplayMode();
		mDDObj -> SetCooperativeLevel( mWind, DDSCL_NORMAL );
		mDDObj -> Release();
		mDDObj = nil;
	}
	while ( ::ShowCursor( true ) < 0 ) { }
	if ( mParentWind )
		::ShowWindow( mParentWind, SW_SHOWNORMAL );
#endif

	mContextRef = nil;
	mDC = nil;
	mIsFullscreen = false;

	Resize( mWindRectHolder );

#if EG_WIN
	::SetForegroundWindow( mWind );
#endif

#if EG_MAC
	::SetGWorld( prevPort, prevDev );
	::FlushEvents( 0xFFFF, 0 );
#endif
}


void WindowDevice::PreFrame_Self( PixPort* inFrame ) {

	// are we in fullscreen mode?
	if ( IsFullscreen() && inFrame ) {

#if USE_DIRECTX

		// Does the fullscreen display device need an explicit palette change?
		if ( mDepth == 8 && inFrame -> GetCurPaletteSeed() != mCurPaletteSeed ) {

			/* Windows is crapware.  Someone decided to use two kinds of RGB color structures, RGBQUAD
			and PALETTEENTRY...  What's worse is that the red and blue values are switched, making them incompatible.
			So, basically, we have to burn CPU on reordering the bloody reds and blues.  Poo!  */
			RGBQUAD pal[ 256 ];
			::GetDIBColorTable( inFrame -> GetDC(), 0, 256, pal );
			for ( int i = 0; i < 256; i++ ) {
				* ( (long*) &pal[ i ] ) = pal[ i ].rgbRed | ( pal[ i ].rgbGreen << 8 )| ( pal[ i ].rgbBlue << 16 ) | ( PC_RESERVED << 24 );
			}

			// In Direct X, you have to set the palette *before* you call GetDC() or else the palette won't be changed!
			mFS_Palette -> SetEntries( 0, 0, 256, (PALETTEENTRY*) pal );
		}

		if ( mContextRef -> GetDC( &mDC ) != DD_OK )
			mDC = nil;
#endif

		}

	// Are we in window mode?
	else if ( mWind && ! IsFullscreen() ) {

#if EG_WIN
		mDC = ::GetDC( mWind );
#endif
	}

	SetDeviceEntries( inFrame );
}



void WindowDevice::BeginFrame_Self() {


	// are we in fullscreen mode?
	if ( IsFullscreen() ) {

#if USE_DRAW_SPROCKETS
		OSErr err = ::DSpContext_GetFrontBuffer( mContextRef, (CGrafPtr*) &mDC );
		if ( mDC && mFS_GDevice )
			::SetGWorld( (CGrafPtr) mDC, mFS_GDevice );
		else if ( mDC )
			::SetPort( mDC );
		else
			ExitFullscreen();
#endif

#if USE_DISP_MGR
		mBM	= ::GetGWorldPixMap( mWorld );
		fix me!
#endif
	}

	// are we in window mode?
	else if ( mWind ) {

#if EG_MAC
#if TARGET_API_MAC_CARBON
			mDC = GetWindowPort( mWind );
#else
			mDC = mWind;
#endif
		::SetPort( mDC );
#endif
	}
}



void WindowDevice::EndFrame_Self() {


	if ( mDC ) {

		if ( IsFullscreen() ) {

#if USE_DIRECTX
			if ( mContextRef )
				mContextRef -> ReleaseDC( mDC );
#endif
			}

		else if ( mWind ) {

#if EG_WIN
			::ReleaseDC( mWind, mDC );
#endif

#if EG_MAC
#endif
		}

		mDC = nil;

		// Restore the graphics environment and call EndFrame() in mirrored devices
		DC_VideoDevice::EndFrame();
	}
}


long WindowDevice::GetParentDisplayID() {
	Rect r;

	GetWindowRect( r );

	return GetDisplayID( ( r.left + r.right ) / 2, ( r.top + r.bottom ) / 2 );
}


void WindowDevice::Erase( bool inUpdateNow ) {

	if ( ! inUpdateNow )
		DC_VideoDevice::Erase( inUpdateNow );

	else {
		bool beganFrame = false;

		if ( ! mDC ) {
			BeginFrame();
			PreFrame( nil );
			beganFrame = true;
		}

#if EG_MAC
		DC_VideoDevice::Erase( true );
#endif

#if EG_WIN
		if ( mWind ) {

			RECT r;
			HRGN winRgn;
			if ( IsFullscreen() )
				::GetWindowRect( mWind, &r );
			else
				::GetClientRect( mWind, &r );

			r.right  += 20;
			r.bottom += 20;
			winRgn = CreateRectRgnIndirect( &r );
			::FillRgn( mDC, winRgn, (HBRUSH) ::GetStockObject( BLACK_BRUSH ) );
			::DeleteObject( winRgn );
		}
#endif

		if ( beganFrame )
			EndFrame();
	}
}


void WindowDevice::Resize( Rect& inRect ) {

	Rect r = inRect;

	long x = r.right - r.left;
	long y = r.bottom - r.top;

	// Resize the window to the rect we were given and redim offscreen draw port...
	if ( mWind ) {

#if EG_MAC
		x -= x % 4;
		::MoveWindow( mWind, r.left, r.top, true );
		::SizeWindow( mWind, x, y, true );
		::ShowWindow( mWind );
		::SetRect( &mOutputRect, 0, 0, x, y );
#elif EG_WIN
		RECT cr;
		// Resize the window and find the rgn we have to work with
		::MoveWindow( mWind, r.left, r.top, x, y, true );
		::GetClientRect( mWind, &cr );
		if ( cr.right % 4 > 0 )
			::MoveWindow( mWind, r.left, r.top, x - cr.right % 4, y, true );
		::GetClientRect( mWind, &cr );
		::SetRect( &mOutputRect, 0, 0, cr.right, cr.bottom );
		::ShowWindow( mWind , SW_SHOWNORMAL );
#elif EG_SDL
	SDL_SysWMinfo info;

	SDL_VERSION(&info.version);
	if (SDL_GetWMInfo(&info) > 0) {
		if (info.subsystem == SDL_SYSWM_X11) {
			info.info.x11.lock_func();
			XMoveResizeWindow (info.info.x11.display,
				info.info.x11.wmwindow, r.left, r.top, x, y);
			info.info.x11.unlock_func();
		}
	}
#endif
	}
}



void WindowDevice::GetWindowRect( Rect& outRect ) {

	if ( IsFullscreen() ) {
		outRect = mOutputRect; }
	else if ( mWind ) {

#if EG_MAC
#if TARGET_API_MAC_CARBON
			GetPortBounds( GetWindowPort( mWind ), &outRect );
			outRect.right 	+= outRect.left;
			outRect.bottom	+= outRect.top;
#else
			outRect = (**(((CGrafPtr) mWind)->portPixMap)).bounds;
			outRect.left *= -1;
			outRect.top  *= -1;
			outRect.right = outRect.left + mWind -> portRect.right;
			outRect.bottom = outRect.top + mWind -> portRect.bottom;
#endif


#elif EG_WIN
		RECT wr;
		::GetWindowRect( mWind, &wr );

		outRect.left	= wr.left;
		outRect.top		= wr.top;
		outRect.right	= wr.right;
		outRect.bottom	= wr.bottom;
#endif
		}
	else {
		__invalidate( outRect )
	}
}
