/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <gtk/gtk.h>

#include "singit/macros.h"
#include "singit/dlg_multiline.h"

#include "singit_debug.h"
#include "singit_tools.h"

enum {
	NEW_TEXT,
	BUTTON_PRESSED,
	LAST_SIGNAL
};

static guint multi_line_dialog_signals[LAST_SIGNAL] = { 0 };
static GtkWidgetClass *parent_class = NULL;

static void multi_line_dialog_destroy (GtkObject *object);

gchar* multi_line_dialog_get_text (MultiLineDialog *dialog)
{
	gchar* result = NULL;

	if (!dialog) { return NULL; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_get_text]\n"));
#endif

	result = gtk_editable_get_chars(GTK_EDITABLE(dialog->text), 0, -1);
	if (result) {
		if (strcmp(dialog->oldText, result) != 0) {
			result = tools_replace_string(result, "\n", dialog->delimiter);
		}
	}

	return result;
}

void multi_line_dialog_set_text (MultiLineDialog *dialog, gchar* text)
{
	if ((!text) || (!dialog)) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_set_text]\n"));
#endif

	if (dialog->oldText) { g_free(dialog->oldText); }
	dialog->oldText = tools_replace_string(text, dialog->delimiter, "\n");

	gtk_text_freeze(dialog->text);
	gtk_text_insert(dialog->text, NULL, &((GtkWidget*)dialog->text)->style->black, NULL,
		dialog->oldText, -1);
	gtk_text_thaw(dialog->text);
}

void multi_line_dialog_set_delimiter (MultiLineDialog *dialog, gchar* delimiter)
{
	if ((!delimiter) || (!dialog)) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_set_delimiter]\n"));
#endif

	if (dialog->delimiter != NULL) { g_free(dialog->delimiter); }
	dialog->delimiter = g_strdup(delimiter);
}

static void multi_line_ok_button_event(GtkButton *button, gpointer data)
{
	MultiLineDialog* mld;
	gchar* result = NULL;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_ok_button_event]\n"));
#endif

	g_return_if_fail (IS_MULTI_LINE_DIALOG(button));
	mld = MULTI_LINE_DIALOG (button);

	result = gtk_editable_get_chars(GTK_EDITABLE(mld->text), 0, -1);
	if (result) {
		if (strcmp(mld->oldText, result) != 0) {
			gtk_signal_emit (GTK_OBJECT (mld), multi_line_dialog_signals[NEW_TEXT]);
		}
	}

	gtk_signal_emit (GTK_OBJECT (mld), multi_line_dialog_signals[BUTTON_PRESSED]);
}

static void multi_line_cancel_button_event(GtkButton *button, gpointer data)
{
	MultiLineDialog* mld;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_cancel_button_event]\n"));
#endif

	g_return_if_fail (IS_MULTI_LINE_DIALOG(button));
	mld = MULTI_LINE_DIALOG (button);

	gtk_signal_emit (GTK_OBJECT (mld), multi_line_dialog_signals[BUTTON_PRESSED]);
}

static void multi_line_dialog_class_init (MultiLineDialogClass *class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) class;

	parent_class = gtk_type_class (gtk_widget_get_type ());

	multi_line_dialog_signals[NEW_TEXT] =
		gtk_signal_new ("new_text", GTK_RUN_FIRST, object_class->type,
			GTK_SIGNAL_OFFSET (MultiLineDialogClass, new_text),
			gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);

	multi_line_dialog_signals[BUTTON_PRESSED] =
		gtk_signal_new ("button_pressed", GTK_RUN_FIRST, object_class->type,
			GTK_SIGNAL_OFFSET (MultiLineDialogClass, button_pressed),
			gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, multi_line_dialog_signals, LAST_SIGNAL);
  	class->button_pressed = NULL;

	object_class->destroy = multi_line_dialog_destroy;
}

static void multi_line_dialog_init (MultiLineDialog *dialog)
{
	GtkWidget
		*multi_line_vbox,
			*multi_line_text_hbox, *multi_line_text_vscrollbar,
			*multi_line_button_hbox,
				*multi_line_ok_button,
				*multi_line_cancel_button;

	dialog->isVisible = FALSE;
	dialog->delimiter = g_strdup(",");
	gtk_window_set_policy(GTK_WINDOW(dialog), TRUE, TRUE, TRUE);
	gtk_container_border_width(GTK_CONTAINER(dialog), 2);
	gtk_widget_realize(GTK_WIDGET(dialog));
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	multi_line_vbox = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(dialog), multi_line_vbox);
	gtk_widget_show(multi_line_vbox);

	multi_line_text_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(multi_line_vbox), multi_line_text_hbox, TRUE, TRUE, 2);
	gtk_widget_show(multi_line_text_hbox);

	dialog->text = (GtkText*) gtk_text_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(multi_line_text_hbox), (GtkWidget*) dialog->text, TRUE, TRUE, 0);
	gtk_text_set_editable(dialog->text, TRUE);
	gtk_text_set_line_wrap(dialog->text, FALSE);
	gtk_widget_set_usize((GtkWidget*) dialog->text, 220, 120);
	gtk_widget_show ((GtkWidget*) dialog->text);

	multi_line_text_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(dialog->text)->vadj);
	gtk_box_pack_start(GTK_BOX(multi_line_text_hbox), multi_line_text_vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show(multi_line_text_vscrollbar);

	multi_line_button_hbox = gtk_hbutton_box_new();
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(multi_line_button_hbox), 5);
	gtk_box_pack_start(GTK_BOX(multi_line_vbox), multi_line_button_hbox, FALSE, FALSE, 0);
	gtk_widget_show(multi_line_button_hbox);

	multi_line_ok_button = gtk_button_new_with_label(_("Ok"));
	GTK_WIDGET_SET_FLAGS(multi_line_ok_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(multi_line_button_hbox), multi_line_ok_button, TRUE, TRUE, 0);
	gtk_widget_show(multi_line_ok_button);

	multi_line_cancel_button = gtk_button_new_with_label(_("Cancel"));
	GTK_WIDGET_SET_FLAGS(multi_line_cancel_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(multi_line_button_hbox), multi_line_cancel_button, TRUE, TRUE, 0);
	gtk_widget_grab_default(multi_line_cancel_button);
	gtk_widget_show(multi_line_cancel_button);

	gtk_signal_connect_object(GTK_OBJECT(multi_line_ok_button), "clicked",
		GTK_SIGNAL_FUNC(multi_line_ok_button_event), (gpointer) dialog);
	gtk_signal_connect_object(GTK_OBJECT(multi_line_cancel_button), "clicked",
		GTK_SIGNAL_FUNC(multi_line_cancel_button_event), (gpointer) dialog);
}

GtkType multi_line_dialog_get_type (void)
{
	static GtkType multi_line_dialog_type = 0;

	if (!multi_line_dialog_type) {

		static const GtkTypeInfo multi_line_dialog_info =
		{
			(gchar*) "MultiLineDialog",
			sizeof (MultiLineDialog),
			sizeof (MultiLineDialogClass),
			(GtkClassInitFunc) multi_line_dialog_class_init,
			(GtkObjectInitFunc) multi_line_dialog_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL,
		};

		multi_line_dialog_type = gtk_type_unique (gtk_window_get_type(), &multi_line_dialog_info);
	}

	return multi_line_dialog_type;
}

GtkWidget *multi_line_dialog_new (const gchar  *caption)
{
	GtkWidget *dialog;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_dialog_new]\n"));
#endif

	dialog = GTK_WIDGET(gtk_type_new(TYPE_MULTI_LINE_DIALOG));

	if (caption) {
		gtk_window_set_title(GTK_WINDOW(dialog), caption);
	}

	return dialog;
}

static void multi_line_dialog_destroy (GtkObject *object)
{
	MultiLineDialog *mld;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_dialog_destroy]\n"));
#endif

	g_return_if_fail (mld = MULTI_LINE_DIALOG (object));

	if (mld->oldText != NULL) {
		g_free(mld->oldText);
		mld->oldText = NULL;
	}

	if (mld->delimiter != NULL) {
		g_free(mld->delimiter);
		mld->delimiter = NULL;
	}

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void multi_line_dialog_show (MultiLineDialog* dialog, GtkWindow *parent)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_dialog_show]\n"));
#endif

	if (dialog->isVisible == TRUE) { return; }
	gtk_window_set_transient_for (GTK_WINDOW(dialog), GTK_WINDOW(parent));
	dialog->isVisible = TRUE;
	gtk_widget_show (GTK_WIDGET(dialog));

//	while (dialog->isVisible) { gtk_main_iteration(); }
}

void multi_line_dialog_hide (MultiLineDialog* dialog)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_multiline.c [multi_line_dialog_hide]\n"));
#endif

	if (dialog->isVisible == FALSE) { return; }
	dialog->isVisible = FALSE;
	gtk_widget_hide (GTK_WIDGET(dialog));
}
