/*
 * GtkGLMaterialDialog
 * Copyright (C) 1999,2000  Rune Lillesveen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gtkglmaterialdialog.h"
#include <GL/gl.h>
#include <GL/glu.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkhseparator.h>
#include <gtk/gtkaspectframe.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkcolorsel.h>
#include <libintl.h>

static void gtk_gl_material_dialog_class_init(GtkGLMaterialDialogClass *klass);
static void gtk_gl_material_dialog_init(GtkGLMaterialDialog *gl_mat_dlg);

static gint gl_init(GtkWidget *widget);
static gint gl_draw(GtkWidget *widget, GdkEventExpose *event);
static gint gl_reshape(GtkWidget *widget, GdkEventConfigure *event);
static void gl_render_sphere();

static void redraw_widget(GtkWidget *widget);

static gint draw_ambient(GtkWidget *widget, GdkEventExpose *event, gpointer data);
static gint draw_diffuse(GtkWidget *widget, GdkEventExpose *event, gpointer data);
static gint draw_specular(GtkWidget *widget, GdkEventExpose *event, gpointer data);
static gint draw_emissive(GtkWidget *widget, GdkEventExpose *event, gpointer data);

static void on_destroy(GtkWidget *widget, gpointer data);
static gint on_delete(GtkWidget *widget, GdkEvent *event, gpointer data);

static void on_destroy_colorsel(GtkWidget *widget, gpointer data);

static gint on_click_ambient(GtkWidget *widget, gpointer data);
static gint on_click_diffuse(GtkWidget *widget, gpointer data);
static gint on_click_specular(GtkWidget *widget, gpointer data);
static gint on_click_emission(GtkWidget *widget, gpointer data);

static gint color_changed_cb(GtkWidget *widget, gpointer data);
static gint on_shine_changed(GtkWidget *widget, gpointer data);

static gint on_ok(GtkWidget *widget, gpointer data);
static gint on_cancel(GtkWidget *widget, gpointer data);

static void create_colorsel_dialog(GtkGLMaterialDialog *mat_dlg);
static void restore_colors(GtkGLMaterialDialog *mat_dlg);

GtkType gtk_gl_material_dialog_get_type()
{
  static GtkType gl_mat_dlg_type = 0;

  if (!gl_mat_dlg_type) {
    GtkTypeInfo gl_mat_dlg_info = {
      (gchar*)"GtkGLMaterialDialog",
      sizeof(GtkGLMaterialDialog),
      sizeof(GtkGLMaterialDialogClass),
      (GtkClassInitFunc) gtk_gl_material_dialog_class_init,
      (GtkObjectInitFunc) gtk_gl_material_dialog_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL
    };
    gl_mat_dlg_type = gtk_type_unique(gtk_window_get_type(), &gl_mat_dlg_info);
  }
  return gl_mat_dlg_type;
}

enum {
  MATERIAL_CHANGED_SIGNAL=0,
  LAST_SIGNAL
};

static gint gl_mat_dlg_signals[LAST_SIGNAL] = { 0 };

static void gtk_gl_material_dialog_class_init(GtkGLMaterialDialogClass *klass)
{
  GtkObjectClass *object_class;
  object_class = (GtkObjectClass*) klass;

  gl_mat_dlg_signals[MATERIAL_CHANGED_SIGNAL] = gtk_signal_new("material_changed",
							       GTK_RUN_FIRST,
							       object_class->type,
							       GTK_SIGNAL_OFFSET(GtkGLMaterialDialogClass, material_changed),
							       gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);


  gtk_object_class_add_signals(object_class, gl_mat_dlg_signals, LAST_SIGNAL);

  klass->material_changed = NULL;
}

static int visual_attributes[] = { GDK_GL_RGBA,
				   GDK_GL_DOUBLEBUFFER,
				   GDK_GL_DEPTH_SIZE, 1,
				   GDK_GL_NONE };

static void gtk_gl_material_dialog_init(GtkGLMaterialDialog *gl_mat_dlg)
{
  GtkWidget *vbox, *separator, *hbox1, *hbox2, *hbox3, *hbox_shine, *aframe, *color_vbox,
    *ambient_hbox, *diffuse_hbox, *specular_hbox, *emission_hbox,
    *ambient_button, *diffuse_button, *specular_button, *emission_button,
    *ambient_label, *diffuse_label, *specular_label, *emission_label, *shininess_label,
    *spec_slider;

  GtkObject *spec_adj;

  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg), "delete_event",
		     GTK_SIGNAL_FUNC(on_delete), NULL);
  gtk_signal_connect (GTK_OBJECT (gl_mat_dlg), "destroy",
		      GTK_SIGNAL_FUNC (on_destroy), NULL);

  create_colorsel_dialog(gl_mat_dlg);

  /* Set initial material */
  gl_mat_dlg->material.ambient[0] = 0.2;
  gl_mat_dlg->material.ambient[1] = 0.2;
  gl_mat_dlg->material.ambient[2] = 0.2;
  gl_mat_dlg->material.ambient[3] = 1.0;

  gl_mat_dlg->material.diffuse[0] = 0.8;
  gl_mat_dlg->material.diffuse[1] = 0.8;
  gl_mat_dlg->material.diffuse[2] = 0.8;
  gl_mat_dlg->material.diffuse[3] = 1.0;

  gl_mat_dlg->material.specular[0] = 0.0;
  gl_mat_dlg->material.specular[1] = 0.0;
  gl_mat_dlg->material.specular[2] = 0.0;
  gl_mat_dlg->material.specular[3] = 1.0;

  gl_mat_dlg->material.emission[0] = 0.0;
  gl_mat_dlg->material.emission[1] = 0.0;
  gl_mat_dlg->material.emission[2] = 0.0;
  gl_mat_dlg->material.emission[3] = 1.0;

  gl_mat_dlg->material.shininess = 0.0;

  /*******************************************************************
   *
   * Create widgets in material dialog
   *
   *******************************************************************/

  /* Check if OpenGL is supported */
  if (gdk_gl_query() == FALSE) {
    g_print("OpenGL not supported\n");
    return;
  }

  /* Create vbox */
  vbox = gtk_vbox_new(FALSE, 3);
  gtk_container_add(GTK_CONTAINER(gl_mat_dlg), vbox);

  /* Create gl_area for preview of material on an OpenGL sphere */
  hbox1 = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox1,
		     TRUE, TRUE, 3);

  aframe = gtk_aspect_frame_new(NULL, 0.5, 0.5, 1, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox1), aframe,
		     TRUE, TRUE, 3);

  gl_mat_dlg->gl_sphere_area = gtk_gl_area_new(visual_attributes);
  gtk_widget_set_events(GTK_WIDGET(gl_mat_dlg->gl_sphere_area),
			GDK_EXPOSURE_MASK|GDK_BUTTON_PRESS_MASK);
  gtk_object_set_data(GTK_OBJECT(gl_mat_dlg->gl_sphere_area), "material", &gl_mat_dlg->material);

  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->gl_sphere_area), "expose_event",
                     GTK_SIGNAL_FUNC(gl_draw), NULL);
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->gl_sphere_area), "configure_event",
                     GTK_SIGNAL_FUNC(gl_reshape), NULL);
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->gl_sphere_area), "realize",
                     GTK_SIGNAL_FUNC(gl_init), NULL);

  gtk_widget_set_usize(gl_mat_dlg->gl_sphere_area, 64, 64);
  gtk_container_add(GTK_CONTAINER(aframe), gl_mat_dlg->gl_sphere_area);


  /* Add color widgets */
  color_vbox = gtk_vbox_new(FALSE, 3);

  gtk_box_pack_start(GTK_BOX(hbox1), color_vbox,
		     FALSE, FALSE, 3);

  /* ambient color widgets */
  ambient_hbox = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(color_vbox), ambient_hbox,
		     TRUE, FALSE, 3);
  ambient_label = gtk_label_new(dgettext("gtkglmatdlg", "Ambient color:"));
  gtk_label_set_justify(GTK_LABEL(ambient_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(ambient_hbox), ambient_label,
		     TRUE, FALSE, 3);
  ambient_button = gtk_button_new();
  gtk_widget_set_usize(ambient_button, 24, 16);
  gtk_box_pack_start(GTK_BOX(ambient_hbox), ambient_button,
		     FALSE, FALSE, 3);
  gl_mat_dlg->ka_draw_area = gtk_drawing_area_new();
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->ka_draw_area), "expose_event",
                     GTK_SIGNAL_FUNC(draw_ambient), &gl_mat_dlg->material.ambient[0]);
  gtk_container_add(GTK_CONTAINER(ambient_button), gl_mat_dlg->ka_draw_area);
  gtk_signal_connect(GTK_OBJECT(ambient_button), "clicked",
		     GTK_SIGNAL_FUNC(on_click_ambient), gl_mat_dlg);

  /* diffuse color widgets */
  diffuse_hbox = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(color_vbox), diffuse_hbox,
		     TRUE, FALSE, 3);
  diffuse_label = gtk_label_new(dgettext("gtkglmatdlg", "Diffuse color:"));
  gtk_label_set_justify(GTK_LABEL(diffuse_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(diffuse_hbox), diffuse_label,
		     TRUE, FALSE, 3);
  diffuse_button = gtk_button_new();
  gtk_widget_set_usize(diffuse_button, 24, 16);
  gtk_box_pack_start(GTK_BOX(diffuse_hbox), diffuse_button,
		     FALSE, FALSE, 3);
  gl_mat_dlg->kd_draw_area = gtk_drawing_area_new();
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->kd_draw_area), "expose_event",
                     GTK_SIGNAL_FUNC(draw_diffuse), &gl_mat_dlg->material.diffuse[0]);
  gtk_container_add(GTK_CONTAINER(diffuse_button), gl_mat_dlg->kd_draw_area);
  gtk_signal_connect(GTK_OBJECT(diffuse_button), "clicked",
		     GTK_SIGNAL_FUNC(on_click_diffuse), gl_mat_dlg);

  /* specular color widgets */
  specular_hbox = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(color_vbox), specular_hbox,
		     TRUE, FALSE, 3);
  specular_label = gtk_label_new(dgettext("gtkglmatdlg", "Specular color:"));
  gtk_label_set_justify(GTK_LABEL(specular_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(specular_hbox), specular_label,
		     TRUE, FALSE, 3);
  specular_button = gtk_button_new();
  gtk_widget_set_usize(specular_button, 24, 16);
  gtk_box_pack_start(GTK_BOX(specular_hbox), specular_button,
		     FALSE, FALSE, 3);
  gl_mat_dlg->ks_draw_area = gtk_drawing_area_new();
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->ks_draw_area), "expose_event",
                     GTK_SIGNAL_FUNC(draw_specular), &gl_mat_dlg->material.specular[0]);
  gtk_container_add(GTK_CONTAINER(specular_button), gl_mat_dlg->ks_draw_area);
  gtk_signal_connect(GTK_OBJECT(specular_button), "clicked",
		     GTK_SIGNAL_FUNC(on_click_specular), gl_mat_dlg);

  /* emissive color widgets */
  emission_hbox = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(color_vbox), emission_hbox,
		     TRUE, FALSE, 3);
  emission_label = gtk_label_new(dgettext("gtkglmatdlg", "Emissive color:"));
  gtk_label_set_justify(GTK_LABEL(emission_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(emission_hbox), emission_label,
		     TRUE, FALSE, 3);
  emission_button = gtk_button_new();
  gtk_widget_set_usize(emission_button, 24, 16);
  gtk_box_pack_start(GTK_BOX(emission_hbox), emission_button,
		     FALSE, FALSE, 3);
  gl_mat_dlg->ke_draw_area = gtk_drawing_area_new();
  gtk_signal_connect(GTK_OBJECT(gl_mat_dlg->ke_draw_area), "expose_event",
                     GTK_SIGNAL_FUNC(draw_emissive), &gl_mat_dlg->material.emission[0]);
  gtk_container_add(GTK_CONTAINER(emission_button), gl_mat_dlg->ke_draw_area);
  gtk_signal_connect(GTK_OBJECT(emission_button), "clicked",
		     GTK_SIGNAL_FUNC(on_click_emission), gl_mat_dlg);

  /* Create the specular exponent slide bar */
  hbox_shine = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox_shine,
		     FALSE, FALSE, 3);

  shininess_label = gtk_label_new(dgettext("gtkglmatdlg", "Shininess:"));
  gtk_label_set_justify(GTK_LABEL(shininess_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(hbox_shine), shininess_label,
		     FALSE, FALSE, 3);

  spec_adj = gtk_adjustment_new(0.0, 0.0, 136.0, 0.1, 1.0, 8.0);
  spec_slider = gtk_hscale_new(GTK_ADJUSTMENT(spec_adj));
  gtk_box_pack_start(GTK_BOX(hbox_shine), spec_slider,
		     TRUE, TRUE, 3);
  gtk_signal_connect (GTK_OBJECT(spec_adj), "value_changed",
		      GTK_SIGNAL_FUNC(on_shine_changed), gl_mat_dlg);
  /* Add a separator */
  hbox2 = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox2,
		     FALSE, FALSE, 3);
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(hbox2), separator,
		     TRUE, TRUE, 3);

  /* Create ok_button */
  hbox3 = gtk_hbox_new(FALSE, 3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox3,
		     FALSE, FALSE, 3);
  gl_mat_dlg->ok_button = gtk_button_new_with_label(dgettext("gtk+", "OK"));
  gtk_widget_set_usize(gl_mat_dlg->ok_button, 74, 19);

  gtk_box_pack_start(GTK_BOX(hbox3), gl_mat_dlg->ok_button,
		     TRUE, FALSE, 3);

  /* Create cancel_button */
  gl_mat_dlg->cancel_button = gtk_button_new_with_label(dgettext("gtk+", "Cancel"));
  gtk_widget_set_usize(gl_mat_dlg->cancel_button, 74, 19);

  gtk_box_pack_start(GTK_BOX(hbox3), gl_mat_dlg->cancel_button,
		     TRUE, FALSE, 3);

  /* Show all widgets */
  gtk_widget_show(gl_mat_dlg->gl_sphere_area);
  gtk_widget_show(aframe);
  gtk_widget_show(color_vbox);
  gtk_widget_show(ambient_hbox);
  gtk_widget_show(diffuse_hbox);
  gtk_widget_show(specular_hbox);
  gtk_widget_show(emission_hbox);
  gtk_widget_show(ambient_button);
  gtk_widget_show(diffuse_button);
  gtk_widget_show(specular_button);
  gtk_widget_show(emission_button);
  gtk_widget_show(ambient_label);
  gtk_widget_show(diffuse_label);
  gtk_widget_show(specular_label);
  gtk_widget_show(emission_label);
  gtk_widget_show(gl_mat_dlg->ka_draw_area);
  gtk_widget_show(gl_mat_dlg->kd_draw_area);
  gtk_widget_show(gl_mat_dlg->ks_draw_area);
  gtk_widget_show(gl_mat_dlg->ke_draw_area);
  gtk_widget_show(hbox1);
  gtk_widget_show(separator);
  gtk_widget_show(hbox2);
  gtk_widget_show(gl_mat_dlg->ok_button);
  gtk_widget_show(gl_mat_dlg->cancel_button);
  gtk_widget_show(hbox3);
  gtk_widget_show(shininess_label);
  gtk_widget_show(spec_slider);
  gtk_widget_show(hbox_shine);
  gtk_widget_show(vbox);
}

GtkWidget *gtk_gl_material_dialog_new(const gchar *title)
{
  return GTK_WIDGET(gtk_type_new(gtk_gl_material_dialog_get_type()));
}

void gtk_gl_material_dialog_set_material(GtkGLMaterialDialog *gl_mat_dlg, GtkGLMaterial *mat)
{
  gl_mat_dlg->material = *mat;
  gl_mat_dlg->old_colors = gl_mat_dlg->material;
}

void gtk_gl_material_dialog_get_material(GtkGLMaterialDialog *gl_mat_dlg, GtkGLMaterial *mat)
{
  *mat = gl_mat_dlg->material;
}

static gint gl_init(GtkWidget *widget)
{
  float light_pos[] = { -1.0, -1.0, 2.0, 0.0 };

  if (gtk_gl_area_make_current(GTK_GL_AREA(widget))) {
    glViewport(0, 0, widget->allocation.width, widget->allocation.height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0,100,0,100,100,-100);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_DEPTH_TEST);
    glLightfv(GL_LIGHT0, GL_POSITION, light_pos);
    glShadeModel(GL_SMOOTH);
  }
  return TRUE;
}

static gint gl_draw(GtkWidget *widget, GdkEventExpose *event)
{
  /* Draw only last expose. */
  if (event->count > 0)
    return TRUE;

  if (gtk_gl_area_make_current(GTK_GL_AREA(widget))) {
    /* Draw lit sphere */

    GtkGLMaterial *material = (GtkGLMaterial*)gtk_object_get_data(GTK_OBJECT(widget), "material");

    glMaterialfv(GL_FRONT, GL_AMBIENT, material->ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, material->diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, material->specular);
    glMaterialfv(GL_FRONT, GL_EMISSION, material->emission);
    glMaterialf(GL_FRONT, GL_SHININESS, material->shininess);

    glClearColor(0,0,0,1);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();
    glTranslatef(50.0, 50.0, -50.0);
    gl_render_sphere();
    gtk_gl_area_swap_buffers(GTK_GL_AREA(widget));
  }

  return TRUE;
}

static gint gl_reshape(GtkWidget *widget, GdkEventConfigure *event)
{
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget))) {
    glViewport(0, 0, widget->allocation.width, widget->allocation.height);
  }
  return TRUE;
}

static void gl_render_sphere()
{
  static GLuint display_list = 0;
  GLUquadricObj *quad_obj = NULL;

  if (display_list == 0) {
    display_list = glGenLists(1);
    quad_obj = gluNewQuadric();
    gluQuadricDrawStyle(quad_obj, GLU_FILL);
    gluQuadricNormals(quad_obj, GLU_SMOOTH);
    gluQuadricOrientation(quad_obj, GLU_INSIDE);
    glNewList(display_list, GL_COMPILE_AND_EXECUTE);
    gluSphere(quad_obj, 40.0, 20, 20);
    glEndList();
    gluDeleteQuadric(quad_obj);
  }
  else {
    glCallList(display_list);
  }
}

static gint draw_ambient(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GdkRectangle area;
  guint32 color;
  GdkGC *gc = gdk_gc_new(widget->window);

  gfloat *ambient = (gfloat*)data;

  /* Draw only last expose. */
  if (event->count > 0)
    return TRUE;

  area.x = 0;
  area.y = 0;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;

  color = (int)(0xff*ambient[0]) << 16 | (int)(0xff*ambient[1]) << 8 | (int)(0xff*ambient[2]);

  gdk_rgb_gc_set_foreground (gc, color);
  gdk_draw_rectangle(widget->window, gc, TRUE, area.x, area.y, area.width, area.height);

  return TRUE;
}

static gint draw_diffuse(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GdkRectangle area;
  guint32 color;
  GdkGC *gc = gdk_gc_new(widget->window);

  gfloat *diffuse = (gfloat*)data;

  /* Draw only last expose. */
  if (event->count > 0)
    return TRUE;

  area.x = 0;
  area.y = 0;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;

  color = (int)(0xff*diffuse[0]) << 16 | (int)(0xff*diffuse[1]) << 8 | (int)(0xff*diffuse[2]);

  gdk_rgb_gc_set_foreground (gc, color);
  gdk_draw_rectangle(widget->window, gc, TRUE, area.x, area.y, area.width, area.height);

  return TRUE;
}

static gint draw_specular(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GdkRectangle area;
  guint32 color;
  GdkGC *gc = gdk_gc_new(widget->window);

  gfloat *specular = (gfloat*)data;

  /* Draw only last expose. */
  if (event->count > 0)
    return TRUE;

  area.x = 0;
  area.y = 0;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;

  color = (int)(0xff*specular[0]) << 16 | (int)(0xff*specular[1]) << 8 | (int)(0xff*specular[2]);

  gdk_rgb_gc_set_foreground (gc, color);
  gdk_draw_rectangle(widget->window, gc, TRUE, area.x, area.y, area.width, area.height);

  return TRUE;
}

static gint draw_emissive(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GdkRectangle area;
  guint32 color;
  GdkGC *gc = gdk_gc_new(widget->window);

  gfloat *emission = (gfloat*)data;

  /* Draw only last expose. */
  if (event->count > 0)
    return TRUE;

  area.x = 0;
  area.y = 0;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;

  color = (int)(0xff*emission[0]) << 16 | (int)(0xff*emission[1]) << 8 | (int)(0xff*emission[2]);

  gdk_rgb_gc_set_foreground (gc, color);
  gdk_draw_rectangle(widget->window, gc, TRUE, area.x, area.y, area.width, area.height);

  return TRUE;
}

static void on_destroy_colorsel(GtkWidget *widget, gpointer data)
{
  GtkGLMaterialDialog *mat_dlg = (GtkGLMaterialDialog *)data;
  if (mat_dlg->color_dialog != NULL) {
    restore_colors(mat_dlg);
    create_colorsel_dialog(mat_dlg);
  }
}

static gint on_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gtk_widget_destroy(widget);
  return TRUE;
}

static void on_destroy(GtkWidget *widget, gpointer data)
{
  GtkWidget *coldlg = GTK_GL_MATERIAL_DIALOG(widget)->color_dialog;
  GTK_GL_MATERIAL_DIALOG(widget)->color_dialog = NULL;
  gtk_widget_destroy(coldlg);
}

static gint on_click_ambient(GtkWidget *widget, gpointer data)
{
  gdouble ambient[4];
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(matdlg->color_dialog)->colorsel);

  if (matdlg->prop_cur != PROPERTY_AMBIENT) {

    ambient[0] = ((GtkGLMaterialDialog*)data)->material.ambient[0];
    ambient[1] = ((GtkGLMaterialDialog*)data)->material.ambient[1];
    ambient[2] = ((GtkGLMaterialDialog*)data)->material.ambient[2];
    ambient[3] = ((GtkGLMaterialDialog*)data)->material.ambient[3];

    gtk_color_selection_set_color(colorsel, ambient);

    if (matdlg->prop_cur == PROPERTY_NONE) {
      matdlg->old_colors = matdlg->material;
      gtk_widget_show(matdlg->color_dialog);
    }

    matdlg->prop_cur = PROPERTY_AMBIENT;
  }
  return TRUE;
}

static gint on_click_diffuse(GtkWidget *widget, gpointer data)
{
  gdouble diffuse[4];
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(matdlg->color_dialog)->colorsel);

  if (matdlg->prop_cur != PROPERTY_DIFFUSE) {

    diffuse[0] = ((GtkGLMaterialDialog*)data)->material.diffuse[0];
    diffuse[1] = ((GtkGLMaterialDialog*)data)->material.diffuse[1];
    diffuse[2] = ((GtkGLMaterialDialog*)data)->material.diffuse[2];
    diffuse[3] = ((GtkGLMaterialDialog*)data)->material.diffuse[3];

    gtk_color_selection_set_color(colorsel, diffuse);

    if (matdlg->prop_cur == PROPERTY_NONE) {
      matdlg->old_colors = matdlg->material;
      gtk_widget_show(matdlg->color_dialog);
    }

    matdlg->prop_cur = PROPERTY_DIFFUSE;
  }
  return TRUE;
}

static gint on_click_specular(GtkWidget *widget, gpointer data)
{
  gdouble specular[4];

  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(matdlg->color_dialog)->colorsel);

  if (matdlg->prop_cur != PROPERTY_SPECULAR) {

    specular[0] = ((GtkGLMaterialDialog*)data)->material.specular[0];
    specular[1] = ((GtkGLMaterialDialog*)data)->material.specular[1];
    specular[2] = ((GtkGLMaterialDialog*)data)->material.specular[2];
    specular[3] = ((GtkGLMaterialDialog*)data)->material.specular[3];

    gtk_color_selection_set_color(colorsel, specular);

    if (matdlg->prop_cur == PROPERTY_NONE) {
      matdlg->old_colors = matdlg->material;
      gtk_widget_show(matdlg->color_dialog);
    }

    matdlg->prop_cur = PROPERTY_SPECULAR;
  }
  return TRUE;
}

static gint on_click_emission(GtkWidget *widget, gpointer data)
{
  gdouble emission[4];
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);
  GtkColorSelection *colorsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(matdlg->color_dialog)->colorsel);

  if (matdlg->prop_cur != PROPERTY_EMISSION) {

    emission[0] = ((GtkGLMaterialDialog*)data)->material.emission[0];
    emission[1] = ((GtkGLMaterialDialog*)data)->material.emission[1];
    emission[2] = ((GtkGLMaterialDialog*)data)->material.emission[2];
    emission[3] = ((GtkGLMaterialDialog*)data)->material.emission[3];

    gtk_color_selection_set_color(colorsel, emission);

    if (matdlg->prop_cur == PROPERTY_NONE) {
      matdlg->old_colors = matdlg->material;
      gtk_widget_show(matdlg->color_dialog);
    }

    matdlg->prop_cur = PROPERTY_EMISSION;
  }
  return TRUE;
}

static gint color_changed_cb(GtkWidget *widget, gpointer data)
{
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);
  gdouble color[4];

  gtk_color_selection_get_color(GTK_COLOR_SELECTION(widget), color);

  switch (matdlg->prop_cur) {
  case PROPERTY_AMBIENT:
    matdlg->material.ambient[0] = color[0];
    matdlg->material.ambient[1] = color[1];
    matdlg->material.ambient[2] = color[2];
    matdlg->material.ambient[3] = color[3];
    redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->ka_draw_area);
    break;
  case PROPERTY_DIFFUSE:
    matdlg->material.diffuse[0] = color[0];
    matdlg->material.diffuse[1] = color[1];
    matdlg->material.diffuse[2] = color[2];
    matdlg->material.diffuse[3] = color[3];
    redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->kd_draw_area);
    break;
  case PROPERTY_SPECULAR:
    matdlg->material.specular[0] = color[0];
    matdlg->material.specular[1] = color[1];
    matdlg->material.specular[2] = color[2];
    matdlg->material.specular[3] = color[3];
    redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->ks_draw_area);
    break;
  case PROPERTY_EMISSION:
    matdlg->material.emission[0] = color[0];
    matdlg->material.emission[1] = color[1];
    matdlg->material.emission[2] = color[2];
    matdlg->material.emission[3] = color[3];
    redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->ke_draw_area);
    break;
  case PROPERTY_NONE:
  case PROPERTY_SHININESS:
  default:
    break;
  }

  redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->gl_sphere_area);

  return TRUE;
}

static gint on_shine_changed(GtkWidget *widget, gpointer data)
{
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);

  if (matdlg != NULL) {
    matdlg->material.shininess = GTK_ADJUSTMENT(widget)->value;
    redraw_widget(GTK_GL_MATERIAL_DIALOG(matdlg)->gl_sphere_area);
  }
  return TRUE;
}

static gint on_ok(GtkWidget *widget, gpointer data)
{
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);

  if (matdlg->color_dialog) {
    gtk_widget_hide(matdlg->color_dialog);
    matdlg->prop_cur = PROPERTY_NONE;
  }
  return TRUE;
}

static gint on_cancel(GtkWidget *widget, gpointer data)
{
  GtkGLMaterialDialog *matdlg = GTK_GL_MATERIAL_DIALOG((GtkWidget*)data);

  if (matdlg->color_dialog) {
    gtk_widget_hide(matdlg->color_dialog);
  }

  matdlg->prop_cur = PROPERTY_NONE;

  restore_colors(matdlg);

  return TRUE;
}

static void restore_colors(GtkGLMaterialDialog *mat_dlg)
{
  /* only colors should be reset */
  mat_dlg->old_colors.shininess = mat_dlg->material.shininess;
  mat_dlg->material = mat_dlg->old_colors;
  redraw_widget(mat_dlg->ka_draw_area);
  redraw_widget(mat_dlg->kd_draw_area);
  redraw_widget(mat_dlg->ks_draw_area);
  redraw_widget(mat_dlg->ke_draw_area);
  redraw_widget(mat_dlg->gl_sphere_area);
}


static void redraw_widget(GtkWidget *widget)
{
  GdkRectangle area;

  area.x = 0;
  area.y = 0;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;
  gtk_widget_draw(widget, &area);
}

static void create_colorsel_dialog(GtkGLMaterialDialog *mat_dlg)
{
  GtkWidget *colorsel;

  /* Create the color selection dialog */
  mat_dlg->prop_cur = PROPERTY_NONE;
  mat_dlg->color_dialog = gtk_color_selection_dialog_new(dgettext("gtkglmatdlg", "Change color"));

  /* We want the color to have an alpha component */
  colorsel = GTK_COLOR_SELECTION_DIALOG(mat_dlg->color_dialog)->colorsel;
  gtk_color_selection_set_opacity(GTK_COLOR_SELECTION(colorsel), TRUE);

  gtk_signal_connect(GTK_OBJECT(colorsel), "color_changed",
		     GTK_SIGNAL_FUNC(color_changed_cb), (gpointer)mat_dlg);
  gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(mat_dlg->color_dialog)->ok_button), "clicked",
		     GTK_SIGNAL_FUNC(on_ok), (gpointer)mat_dlg);
  gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(mat_dlg->color_dialog)->cancel_button), "clicked",
		     GTK_SIGNAL_FUNC(on_cancel), (gpointer)mat_dlg);
  gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(mat_dlg->color_dialog)), "destroy",
		     GTK_SIGNAL_FUNC(on_destroy_colorsel), (gpointer)mat_dlg);
}
