/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2006 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* gui_searchdialog.c - creation and destruction of the search dialog
 *
 * we create a dialog where a string can be entered. If OK is clicked this
 * string is checked against the callsigns in all logs. If there is a partial
 * match, the QSO is selected. If there is no match, a warning is displayed.
 */
 
#include <gtk/gtk.h>
#include <string.h>

#include "gui_searchdialog.h"
#include "support.h"
#include "log.h"
#include "main.h"

extern statetype state;
extern GList *searchhistory;
extern GList *logwindowlist;
extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;

#define SEARCHHISTORY 10

static void
searchok (GtkButton *button, gpointer user_data)
{
	GtkWidget *searchdialog, *searchresultdialog, *resultlabel, 
		*searchcheckbutton, *searchcombo;
	gchar *searchstr, *upsearchstr, *callsign, *current, *nr, *labeltext;
	gint i, len;
	gboolean valid, result = FALSE, searchopen;
	GList *node;
	logtype *logwindow;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreePath *path;

	searchdialog = GTK_WIDGET (user_data);
	searchcombo = lookup_widget (searchdialog, "searchcombo");
	searchcheckbutton = lookup_widget (searchdialog, "searchcheckbutton");

	searchopen = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(searchcheckbutton));
	searchstr = gtk_editable_get_chars 
		(GTK_EDITABLE (GTK_BIN(searchcombo)->child), 0, -1);
	len = strlen (searchstr);

	if (len > 0)
	{
	node = g_list_find_custom (searchhistory, searchstr, (GCompareFunc) strcmp);
	if (!node)
	{
		searchhistory =	g_list_prepend (searchhistory, g_strdup(searchstr));
		gtk_combo_box_append_text (GTK_COMBO_BOX (searchcombo), searchstr);
	}
	if (g_list_length (searchhistory) > SEARCHHISTORY)
	{
		searchhistory = g_list_remove 
			(searchhistory, g_list_last (searchhistory)->data);
		gtk_combo_box_remove_text (GTK_COMBO_BOX (searchcombo), 0);
	}

	upsearchstr = g_ascii_strup (searchstr, -1);

	for (i = 0; i < g_list_length (logwindowlist); i++)
	{
		logwindow = g_list_nth_data (logwindowlist, i);
		model = gtk_tree_view_get_model (GTK_TREE_VIEW(logwindow->treeview));
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid)
		{
			gtk_tree_model_get (model, &iter, CALL, &callsign, -1);
			if (g_strrstr (callsign, upsearchstr))
			{
				/* construct an id for this search */
				current = g_strdup_printf ("%d", i);
				gtk_tree_model_get (model, &iter, NR, &nr, -1);
				current =	g_strconcat (current, "/", nr, NULL);
				if (!g_strrstr (state.searchstr, current))
				{
					gtk_notebook_set_current_page
						(GTK_NOTEBOOK (mainnotebook), i);
					selection =	gtk_tree_view_get_selection
						(GTK_TREE_VIEW(logwindow->treeview));
					gtk_tree_selection_select_iter(selection, &iter);
					path = gtk_tree_model_get_path (model, &iter);
					gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW
						(logwindow->treeview), path, NULL, TRUE, 0.5, 0.0);
					gtk_tree_path_free (path);
					/* add id to the array */
					state.searchstr = g_strconcat
						(state.searchstr, current, ",", NULL);
					result = TRUE;
					break;
				}
			}
			valid = gtk_tree_model_iter_next (model, &iter);
		}
		if (result)
			break;
	}

	if (!result)
	{
		state.searchstr = g_strdup ("");
		searchresultdialog = gtk_dialog_new_with_buttons
			(_("xlog - searchresult"),
			GTK_WINDOW(mainwindow), GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
		labeltext =	g_strdup_printf (_("\'%s\' was not found"), searchstr);
		resultlabel = gtk_label_new (labeltext);
		g_free (labeltext);
		gtk_container_add (GTK_CONTAINER
			(GTK_DIALOG (searchresultdialog)->vbox), resultlabel);
		gtk_misc_set_padding (GTK_MISC (resultlabel), 30, 20);
		gtk_widget_show_all (searchresultdialog);
		gtk_dialog_run (GTK_DIALOG(searchresultdialog));
		gtk_editable_delete_text
			(GTK_EDITABLE (GTK_BIN(searchcombo)->child), 0, -1);
		gtk_widget_destroy (searchresultdialog);
	}
	g_free (upsearchstr);
	}
	g_free (searchstr);
	if (!searchopen)
		gtk_widget_destroy (searchdialog);
}

/* search changed, reset saved state */
static void
on_searchcombo_changed (GtkEditable * editable, gpointer user_data)
{
	state.searchstr = g_strdup ("");
}

static void
searchcancel (GtkButton *button, gpointer user_data)
{
	state.searchstr = g_strdup ("");
	gtk_widget_destroy (GTK_WIDGET(user_data));
}

void
on_menu_search_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *searchdialog, *vbox, *searchlabel, *searchcombo,
		*searchcheckbutton, *cancel_button, *ok_button;
	GdkPixbuf *dialog_icon_pixbuf;
	gint i, num;
	gchar *s;

	searchdialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (searchdialog), _("xlog - search"));
	dialog_icon_pixbuf = create_pixbuf ("xlog.png");
	if (dialog_icon_pixbuf)
	{
		gtk_window_set_icon (GTK_WINDOW (searchdialog), dialog_icon_pixbuf);
		g_object_unref (dialog_icon_pixbuf);
	}
	vbox = gtk_vbox_new (FALSE, 10);
	gtk_container_add
		(GTK_CONTAINER (GTK_DIALOG (searchdialog)->vbox), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 20);
	searchlabel = gtk_label_new (_(
"Select or enter a callsign (or part of a callsign) to search for in the log"));
	gtk_box_pack_start (GTK_BOX (vbox), searchlabel, FALSE, FALSE, 0);
	gtk_label_set_line_wrap (GTK_LABEL (searchlabel), TRUE);
	searchcombo = gtk_combo_box_entry_new_text ();
	gtk_entry_set_activates_default 
	  (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (searchcombo))), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), searchcombo, FALSE, FALSE, 0);
	searchcheckbutton = gtk_check_button_new_with_label
		(_("Keep this dialog open"));
	gtk_box_pack_start (GTK_BOX (vbox), searchcheckbutton, FALSE, FALSE, 0);

	g_signal_connect ((gpointer) searchcombo, "changed",
		G_CALLBACK (on_searchcombo_changed), NULL);

	/* Store pointers to all widgets, for use by lookup_widget(). */
	GLADE_HOOKUP_OBJECT (searchdialog, searchcombo, "searchcombo");
	GLADE_HOOKUP_OBJECT (searchdialog, searchcheckbutton, "searchcheckbutton");

	if (searchhistory)
	{
		num = g_list_length (searchhistory);
		for (i = 0; i < num; i++)
		{
			s = g_list_nth_data (searchhistory, i);
			gtk_combo_box_prepend_text (GTK_COMBO_BOX (searchcombo), s);
		}
	}

	cancel_button = gtk_dialog_add_button
		(GTK_DIALOG(searchdialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
	ok_button = gtk_dialog_add_button
		(GTK_DIALOG(searchdialog), GTK_STOCK_OK, GTK_RESPONSE_OK);
	g_signal_connect (G_OBJECT (ok_button), "clicked",
		G_CALLBACK (searchok), searchdialog);
	g_signal_connect (G_OBJECT (cancel_button), "clicked",
		G_CALLBACK (searchcancel), searchdialog);

	gtk_dialog_set_default_response (GTK_DIALOG (searchdialog),
                                             GTK_RESPONSE_OK);

	/* don't use gtk_dialog_run, so we can keep on using xlog */
	gtk_widget_show_all (searchdialog);
}
