/* stringgadget.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: stringgadget.cc,v 1.54 2005/12/12 20:45:29 ralf Exp $ */

#include "stringgadget.h"
#include "awindow.h"
#include "guielement.h"

const char *StringGadget::type="StringGadget";

StringGadget::~StringGadget()
{
  _aguix->cancelCutPaste( this );
  destroy();
  _freesafe( text );
}

StringGadget::StringGadget(AGUIX *taguix,int tx,int ty,int width,int height,const char *ttext,int tdata):GUIElement(taguix)
{
  this->data=tdata;
  _x=tx;
  _y=ty;
  if ( width > 8 )
    _w = width;
  else
    _w = 8;
  if ( height > 8 )
    _h = height;
  else
    _h = 8;
  xoffset=0;
  cursorpos=0;
  active=false;
  this->text=dupstring(ttext);

  selstart=selend=cursorpos;
  font=NULL;
  forbidPosChange=false;
  strongkeycapture=true;
  wantpaste=false;
  pasterequest=0;
  setCanHandleFocus();
  setAcceptFocus( true );
  passwordMode = false;
}

StringGadget::StringGadget(AGUIX *taguix,int tx,int ty,int width,const char *ttext,int tdata):GUIElement(taguix)
{
  int bw;

  bw = getBorderWidth();
  this->data=tdata;
  _x=tx;
  _y=ty;
  if ( width > 8 )
    _w = width;
  else
    _w = 8;

  _h = taguix->getCharHeight() + 2 * bw;
  xoffset=0;
  cursorpos=0;
  active=false;
  this->text=dupstring(ttext);

  selstart=selend=cursorpos;
  font=NULL;
  forbidPosChange=false;
  strongkeycapture=true;
  wantpaste=false;
  pasterequest=0;
  setCanHandleFocus();
  setAcceptFocus( true );
  passwordMode = false;
}

void StringGadget::resize(int tw,int th)
{
  if ( tw < 8 ) tw = 8;
  if ( th < 8 ) th = 8;
  _w=tw;
  _h=th;
  if ( isCreated() == true ) {
    _parent->resizeSubWin(win,tw,th);
  }
  updateWin();
  redraw();
}

int StringGadget::getData() const
{
  return data;
}

void StringGadget::setData(int tdata)
{
  this->data=tdata;
}

void StringGadget::redraw()
{
  int maxcol=_aguix->getMaxCols();
  int newbg;
  GC usegc;

  if ( isCreated() == true ) {
    prepareBG();

    _aguix->ClearWin(win);
    if(active==true) {
      if(maxcol>7) {
	newbg=7;
      } else {
	newbg=0;
      }
    } else {
      newbg=0;
    }
    
    if(font==NULL) usegc=0; else usegc=font->getGC();
    
    _aguix->setFG(2);
    _aguix->DrawLine(win,0,_h-1,0,0);
    _aguix->DrawLine(win,0,0,_w-1,0);
    _aguix->DrawLine(win,2,_h-1-2,_w-1-2,_h-1-2);
    _aguix->DrawLine(win,_w-1-2,_h-1-2,_w-1-2,2);
    _aguix->setFG(1);
    _aguix->DrawLine(win,0,_h-1,_w-1,_h-1);
    _aguix->DrawLine(win,_w-1,_h-1,_w-1,0);
    _aguix->DrawLine(win,2,_h-1-2,2,2);
    _aguix->DrawLine(win,2,2,_w-1-2,2);
    textRedraw();
  }
}

void StringGadget::textRedraw()
{
  int maxcol=_aguix->getMaxCols();
  GC usegc;
  int minx, maxx;
  int bw = getBorderWidth();
  int inner_x, inner_y;

  if ( isCreated() == false ) return;

  clearTextBG();

  if(font==NULL) usegc=0; else usegc=font->getGC();

  inner_x = inner_y = getBorderWidth();

  int ch;
  if(font==NULL) {
    ch=_aguix->getCharHeight();
  } else {
    ch=font->getCharHeight();
  }
  char *tstr = dupstring( text );
  int dx;
  dx=1;
  double f1 = ( ( _h - 2 * bw ) / 2 ) - ( ch / 2 );
  int dy=(int)f1;
  int x1,x2;
  int tstr_len = strlen( tstr );
  int temp_ch_pos;
  char temp_ch;

  if ( passwordMode == true ) {
    memset( tstr, '*', tstr_len );
  }

  x1=(selstart<=selend?selstart:selend);
  x2=(selstart>selend?selstart:selend);
  if((x2==(int)strlen(text))&&(selstart!=selend)) x2--;

  if ( ( x1 < 0 ) || ( x1 > tstr_len ) ) x1 = 0;
  if ( ( x2 < 0 ) || ( x2 > tstr_len ) ) x2 = 0;
  if ( x2 < x1 ) x2 = x1;
  
  int restwidth = getInnerWidth() - 2;
  int strwidth, vislen;
  bool cont = true;

  if( xoffset < x1 ) {
    // links vom x1 ist noch Text
    _aguix->setFG(usegc,1);

    temp_ch = tstr[x1];
    temp_ch_pos = x1;
    tstr[x1] = '\0';
    vislen = _aguix->getStrlen4Width( tstr + xoffset, restwidth, &strwidth, font );

    // xoffset + vislen is equal or less x1 in any case
    if ( tstr[xoffset + vislen] != '\0' ) {
      tstr[temp_ch_pos] = temp_ch;
      temp_ch = tstr[xoffset + vislen];
      temp_ch_pos = xoffset + vislen;
      tstr[temp_ch_pos] = '\0';

      // doesn't fit completely so don't draw anything else
      cont = false;
    }

    if ( vislen > 0 ) {
      if ( font == NULL ) _aguix->DrawText( win, tstr + xoffset, inner_x + dx, inner_y + dy );
      else _aguix->DrawText( win, font, tstr + xoffset, inner_x + dx, inner_y + dy );
    }

    tstr[temp_ch_pos] = temp_ch;
    
    dx += strwidth;
    restwidth -= strwidth;
  }

  if ( ( cont == true ) && ( x2 >= xoffset ) ) {
    // some part of the selection is visible

    minx = ( x1 < xoffset ) ? xoffset : x1;
    maxx = x2;
    if ( tstr[maxx] != '\0' ) maxx++;

    // draw selection from minx to maxx which can be outside screen

    temp_ch = tstr[maxx];
    temp_ch_pos = maxx;
    tstr[maxx] = '\0';

    vislen = _aguix->getStrlen4Width( tstr + minx, restwidth, &strwidth, font );

    if ( tstr[minx + vislen] != '\0' ) {
      tstr[temp_ch_pos] = temp_ch;
      temp_ch = tstr[minx + vislen];
      temp_ch_pos = minx + vislen;
      tstr[temp_ch_pos] = '\0';

      // doesn't fit completely so don't draw anything else
      cont = false;
    }

    if ( ( vislen > 0 ) || ( ( minx + vislen ) == tstr_len ) ) {
      if(active==true) {
	// draw selection area
	_aguix->setFG(usegc,1);
	_aguix->FillRectangle( win, usegc, inner_x + dx, inner_y + dy, strwidth, ch );
      }
      if(active==true) {
	if(maxcol>7) {
	  _aguix->setFG(usegc,7);
	} else {
	  _aguix->setFG(usegc,0);
	}
      } else {
	_aguix->setFG(usegc,1);
      }
      if ( font == NULL ) _aguix->DrawText( win, tstr + minx, inner_x + dx, inner_y + dy );
      else _aguix->DrawText( win, font, tstr + minx, inner_x + dx, inner_y + dy );
    }

    tstr[temp_ch_pos] = temp_ch;
    
    dx += strwidth;
    restwidth -= strwidth;
  }

  if ( ( cont == true ) && ( tstr[x2] != '\0' ) ) {
    // the rest after the selection is possibly visible
    int first_vis_char;
    
    // end of selection can be less than xoffset
    first_vis_char = a_max( x2 + 1, xoffset );
    
    vislen = _aguix->getStrlen4Width( tstr + first_vis_char, restwidth, &strwidth, font );
    
    if ( vislen > 0 ) {
      tstr[first_vis_char + vislen] = '\0';

      _aguix->setFG( usegc, 1 );
      if ( font == NULL ) _aguix->DrawText( win, tstr + first_vis_char, inner_x + dx, inner_y + dy );
      else _aguix->DrawText( win, font, tstr + first_vis_char, inner_x + dx, inner_y + dy );
    }
  }
  _freesafe(tstr);
  _aguix->Flush();
}

void StringGadget::flush()
{
}

bool StringGadget::isInside(int px,int py) const
{
/*  if((px>x)&&(px<=(x+width))) {
    if((py>y)&&(py<=(y+height))) return true;
  }*/
  return false;
}

bool StringGadget::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue;
  returnvalue=false;
  int tx,ty;
  AGMessage *agmsg;
  int oldselstart;
  
  if ( isCreated() == false ) return false;

  if(active==true) {
    if(msg->type==ButtonPress) {
      if ( msg->window != win ) {
        active=false;
        redraw();
//        returnvalue=true;
        agmsg = AGUIX_allocAGMessage();
        agmsg->type=AG_STRINGGADGET_DEACTIVATE;
        agmsg->stringgadget.sg=this;
	agmsg->stringgadget.ok = false;
        _aguix->putAGMsg(agmsg);
      } else {
        if(forbidPosChange==false) {
	  _aguix->queryPointer(msg->window,&tx,&ty);

	  int bw = getBorderWidth();
	  tx -= bw;
	  ty -= bw;

	  int l, strwidth;
	  l = _aguix->getStrlen4Width( text + xoffset, a_max( tx - 2, 0 ), &strwidth, font );
	  
	  cursorpos = l + xoffset;
	  if ( cursorpos >= (int)strlen( text ) ) cursorpos = (int)strlen( text );
	  if ( cursorpos < 0 ) cursorpos = 0;
	  selstart=selend=cursorpos;
	  if(msg->button==Button2) insertSelection();
	  redraw();
	}
      }
    } else {
      if(msg->type==KeyPress) {
	if ( isVisible() == true ) {
	  if ( _parent->isTopParent( msg->window ) == true ) {
	    ignoreRelease=false;
	    int keystate=KEYSTATEMASK(msg->keystate);
	    if((msg->key==XK_Right)||
	       (msg->key==XK_End)||
	       ((msg->key==XK_e)&&(keystate==ControlMask))||
	       ((msg->key==XK_f)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	      if(forbidPosChange==false) {
		if((((keystate==0)||(keystate==ShiftMask))&&
		    (msg->key==XK_Right))||
		   ((msg->key==XK_f)&&(keystate==ControlMask))) {
		  if(cursorpos<(int)strlen(text)) {
		    oldselstart = selstart;
		    setCursor( cursorpos + 1 );
		    selstart = oldselstart;
		    if(keystate!=ShiftMask) selstart=selend=cursorpos;
		    else {
		      selend=cursorpos;
		      applySelection();
		    }
		    textRedraw();
		    returnvalue=true;
		  } else if ( ( selstart != selend ) && ( keystate != ShiftMask ) ) {
		    // selection active and normal cursor right
		    // => deactivate selection
		    selstart = selend = cursorpos;
		    textRedraw();
		    returnvalue = true;
		  }
		} else if(((keystate&Mod1Mask)==Mod1Mask)||
			  (msg->key==XK_End)||
			  ((msg->key==XK_e)&&(keystate==ControlMask))) {
		  oldselstart = selstart;
		  setCursor( (int)strlen( text ) ); 
		  selstart = oldselstart;
		  if((keystate&ShiftMask)==0) selstart=selend=cursorpos;
		  else selend=cursorpos;
		  textRedraw();
		  returnvalue=true;
		}
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_CURSORCHANGE;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      }
	    } else if((msg->key==XK_Left)||
		      (msg->key==XK_Home)||
		      ((msg->key==XK_a)&&(keystate==ControlMask))||
		      ((msg->key==XK_b)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	      if(forbidPosChange==false) {
		if((((keystate==0)||(keystate==ShiftMask))&&
		    (msg->key==XK_Left))||
		   ((msg->key==XK_b)&&(keystate==ControlMask))) {
		  if(cursorpos>0) {
		    cursorpos--;
		    if((cursorpos-xoffset)<0) xoffset--;
		    if(keystate!=ShiftMask) selstart=selend=cursorpos;
		    else {
		      selend=cursorpos;
		      applySelection();
		    }
		    textRedraw();
		    returnvalue=true;
		  } else if ( ( selstart != selend ) && ( keystate != ShiftMask ) ) {
		    // selection active and normal cursor left
		    // => deactivate selection
		    selstart = selend = cursorpos;
		    textRedraw();
		    returnvalue = true;
		  }
		} else if(((keystate&Mod1Mask)==Mod1Mask)||
			  (msg->key==XK_Home)||
			  ((msg->key==XK_a)&&(keystate==ControlMask))) {
		  xoffset=0;
		  cursorpos=0;
		  if((keystate&ShiftMask)==0) selstart=selend=cursorpos;
		  else selend=cursorpos;
		  textRedraw();
		  returnvalue=true;
		}
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_CURSORCHANGE;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      }
	    } else if((msg->key==XK_Delete)||((msg->key==XK_d)&&(keystate==ControlMask))) {
	      int len=strlen(text);
	      if(strongkeycapture==true) returnvalue=true;
	      if(selstart!=selend) {
		removeSelection();
		textRedraw();
	      } else if((cursorpos<len)&&(len>0)) {
		char *tstr=(char*)_allocsafe(len);
		strncpy(tstr,text,cursorpos);
		tstr[cursorpos]=0;
		if(cursorpos<len) strcat(tstr,text+cursorpos+1);
		_freesafe(text);
		text=tstr;
		textRedraw();
		returnvalue=true;
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_CONTENTCHANGE;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      }
	    } else if((msg->key==XK_BackSpace)||((msg->key==XK_h)&&(keystate==ControlMask))) {
	      int len=strlen(text);
	      if(strongkeycapture==true) returnvalue=true;
	      if(selend!=selstart) {
		removeSelection();
		textRedraw();
	      } else if(cursorpos>0) {
		char *tstr=(char*)_allocsafe(len);
		tstr[0]=0;
		if(cursorpos>1) {
		  strncpy(tstr,text,cursorpos-1);
		  tstr[cursorpos-1]=0;
		}
		strcat(tstr,text+cursorpos);
		_freesafe(text);
		text=tstr;
		cursorpos--;
		if((cursorpos-xoffset)<0) xoffset--;
		selstart=selend=cursorpos;
		textRedraw();
		returnvalue=true;
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_CONTENTCHANGE;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      }
	    } else if ( msg->key == XK_Return ) {
	      if ( strongkeycapture == true ) returnvalue = true;
	    } else if ( IsModifierKey( msg->key ) ||
			IsCursorKey( msg->key ) ||
			IsPFKey( msg->key ) ||
			IsFunctionKey( msg->key ) ||
			IsMiscFunctionKey( msg->key ) ||
			( ( msg->key >= XK_BackSpace ) && ( msg->key <= XK_Escape ) ) ) {
	      // catch this special keys even they are not handled
	      // so they will not be added to the text
	    } else if ( ( strlen( msg->keybuf ) > 0 ) && ( ( keystate & ControlMask ) == 0 ) ) {
	      int len = strlen( text );
	      if ( selstart != selend ) {
		removeSelection();
		textRedraw();
	      }
	      char *tstr = (char*)_allocsafe(len + strlen( msg->keybuf ) + 1 );
	      strncpy( tstr, text, cursorpos );
	      strcpy( tstr + cursorpos, msg->keybuf );
	      strcat( tstr, text + cursorpos);
	      _freesafe( text );
	      text = tstr;
	      setCursor( cursorpos + strlen( msg->keybuf ) );
	      selstart = selend = cursorpos;
	      textRedraw();
	      returnvalue = true;
	      agmsg = AGUIX_allocAGMessage();
	      agmsg->type = AG_STRINGGADGET_CONTENTCHANGE;
	      agmsg->stringgadget.sg = this;
	      _aguix->putAGMsg( agmsg );
	    }
	  }
	}
      } else if((msg->type==KeyRelease)&&(ignoreRelease==false)) {
	if ( isVisible() == true ) {
	  if ( _parent->isTopParent( msg->window ) == true ) {
	    int keystate=KEYSTATEMASK(msg->keystate);
	    if((msg->key==XK_Return)||(msg->key==XK_Escape)) {
	      active=false;
	      selstart=selend=cursorpos;
	      redraw();
	      returnvalue=true;
	      agmsg = AGUIX_allocAGMessage();
	      agmsg->type=AG_STRINGGADGET_DEACTIVATE;
	      agmsg->stringgadget.sg=this;
	      agmsg->stringgadget.ok = ( msg->key == XK_Return ) ? true : false;
	      _aguix->putAGMsg(agmsg);
	      if(msg->key==XK_Return) {
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_OK;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      } else {
		agmsg = AGUIX_allocAGMessage();
		agmsg->type=AG_STRINGGADGET_CANCEL;
		agmsg->stringgadget.sg=this;
		_aguix->putAGMsg(agmsg);
	      }
	    } else if((msg->key==XK_Right)||
		      (msg->key==XK_End)||
		      ((msg->key==XK_e)&&(keystate==ControlMask))||
		      ((msg->key==XK_f)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	    } else if((msg->key==XK_Left)||
		      (msg->key==XK_Home)||
		      ((msg->key==XK_a)&&(keystate==ControlMask))||
		      ((msg->key==XK_b)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	    } else if((msg->key==XK_Delete)||((msg->key==XK_d)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	    } else if((msg->key==XK_BackSpace)||((msg->key==XK_h)&&(keystate==ControlMask))) {
	      if(strongkeycapture==true) returnvalue=true;
	    } else if ( IsModifierKey( msg->key ) ||
			IsCursorKey( msg->key ) ||
			IsPFKey( msg->key ) ||
			IsFunctionKey( msg->key ) ||
			IsMiscFunctionKey( msg->key ) ||
			( ( msg->key >= XK_BackSpace ) && ( msg->key <= XK_Escape ) ) ) {
	      // Some of these keys could create a char in keybuf but because we don't
	      // handle them here do nothing
	    } else if ( ( strlen( msg->keybuf ) > 0 ) && ( keystate != ControlMask ) ) {
	      if(strongkeycapture==true) returnvalue=true;
	    }
	  }
	}
      } else if(msg->type==ButtonRelease) {
        applySelection();
      } else if ( msg->type == MotionNotify &&
		  msg->window == win &&
		  ( msg->keystate & Button1Mask ) == Button1Mask ) {
        _aguix->queryPointer(msg->window,&tx,&ty);

	int bw = getBorderWidth();
	tx -= bw;
	ty -= bw;

	if ( ( tx < 0 ) && ( xoffset > 0 ) ){
	  xoffset--;
	  cursorpos = xoffset;
	} else if ( tx >= getInnerWidth() ) {
	  int l, strwidth;
	  l = _aguix->getStrlen4Width( text + xoffset, a_min( tx , getInnerWidth() ) - 2, &strwidth, font );

	  cursorpos = l + xoffset;
	  if ( cursorpos < (int)strlen( text ) ) xoffset++;
	} else {
	  int l, strwidth;
	  l = _aguix->getStrlen4Width( text + xoffset, tx - 2, &strwidth, font );

	  int ncp = l + xoffset;
	  if(ncp>=(int)strlen(text)) ncp=strlen(text);
	  if(ncp<0) ncp=0;
	  cursorpos=ncp;
	}

	selend=cursorpos;
        redraw();
      }
    }
  } else if(msg->type==ButtonPress) {
    if ( msg->window == win ) {
      takeFocus();
      active=true;
//      returnvalue=true;
      agmsg = AGUIX_allocAGMessage();
      agmsg->type=AG_STRINGGADGET_ACTIVATE;
      agmsg->stringgadget.sg=this;
      _aguix->putAGMsg(agmsg);
      if(forbidPosChange==false) {
        _aguix->queryPointer(msg->window,&tx,&ty);

	int bw = getBorderWidth();
	tx -= bw;
	ty -= bw;

	int l, strwidth;
	l = _aguix->getStrlen4Width( text + xoffset, a_max( tx - 2, 0 ), &strwidth, font );
	cursorpos = l + xoffset;
	if(cursorpos>=(int)strlen(text)) cursorpos=(int)strlen(text);
	if(cursorpos<0) cursorpos=0;
	selstart=selend=cursorpos;
	if(msg->button==Button2) insertSelection();
      }
      redraw();
    }
  }
  if(msg->type==Expose) {
    if ( msg->window == win ) {
      redraw();
    }
  }
  return returnvalue;
}

void StringGadget::setText(const char *new_text)
{
  _freesafe(text);
  text=dupstring(new_text);
  if(cursorpos>=(int)strlen(new_text)) {
    cursorpos=(int)strlen(new_text);
  }
  selstart=selend=cursorpos;
  setXOffset( xoffset );
  textRedraw();
}

const char *StringGadget::getText() const
{
  return text;
}

bool StringGadget::isActive() const
{
  return active;
}

void StringGadget::updateWin()
{
  int i1;

  if ( isCreated() == true ) {
    i1 = a_min( selstart, selend );
    setXOffset( i1 );
    textRedraw();
  }
}

void StringGadget::activate()
{
  active=true;
  ignoreRelease=true;
  redraw();
}

void StringGadget::deactivate()
{
  active=false;
  redraw();
}

void StringGadget::applySelection()
{
  int x1,x2;
  char *buffer;

  if ( passwordMode == false ) {
    x1=(selstart<=selend?selstart:selend);
    x2=(selstart>selend?selstart:selend);
    if(x2==(int)strlen(text)) x2--;
    if(x1<x2) {
      buffer=dupstring(text+x1);
      buffer[x2-x1+1]=0;
      _aguix->startCut(this,buffer);
      _freesafe(buffer);
    }
  }
}

void StringGadget::removeSelection()
{
  int x1,x2;
  x1=(selstart<=selend?selstart:selend);
  x2=(selstart>selend?selstart:selend);
  if(x2==(int)strlen(text)) x2--;
  if ( x1 <= x2 ) {
    char *tstr;
    tstr=(char*)_allocsafe(strlen(text)-x2+x1+1);
    strncpy(tstr,text,x1);
    tstr[x1]=0;
    strcat(tstr,text+x2+1);
    _freesafe(text);
    text=tstr;
    selstart=selend=cursorpos=x1;
    if(cursorpos<xoffset) xoffset=cursorpos;
  }
}

void StringGadget::insertSelection()
{
  if ( isCreated() == false ) return;
  int nbytes_return;
  char *bytes_return,*tstr;

  if(_aguix->amiOwner()==true) {
    bytes_return=dupstring(_aguix->getCutBuffer());
    nbytes_return=strlen(bytes_return);
    // an Cursorpos einfgen
    int tl=strlen(text);
    tstr=(char*)_allocsafe(tl+nbytes_return+2);
    strncpy(tstr,text,cursorpos);
    tstr[cursorpos]=0;
    strncat(tstr,bytes_return,nbytes_return);
    tstr[cursorpos+nbytes_return]=0;
    strcat(tstr,text+cursorpos);
    _freesafe(text);
    text=tstr;
    _freesafe(bytes_return);
  } else {
    _aguix->requestCut(getWindow());
    _aguix->startPaste(this);
    wantpaste=true;
    pasterequest=time(NULL);
  }
}

int StringGadget::setFont(char *fontname)
{
  font=_aguix->getFont(fontname);
  updateWin();
  if(font==NULL) return -1;
  return 0;
}

const char *StringGadget::getType() const
{
  return type;
}

bool StringGadget::isType(const char *qtype) const
{
  if(strcmp(type,qtype)==0) return true;
  return false;
}

bool StringGadget::isParent(Window child) const
{
  if ( isCreated() == false ) return false;
  if(child==win) return true;
  return false;
}

int StringGadget::getXOffset() const
{
  return xoffset;
}

void StringGadget::setXOffset( int new_pos )
{
  int l, strwidth, newx;

  xoffset = new_pos;
  if ( xoffset >= (int)strlen( text ) ) xoffset = (int)strlen( text );

  newx = xoffset;
  while ( newx >= 0 ) {
    l = _aguix->getStrlen4Width( text + newx, getInnerWidth() - 2, &strwidth, font );
    
    if ( ( l + newx ) == (int)strlen( text ) ) {
      // everything visible reduce xoffset
      // lower means not everything is visible, higher is not possible
      xoffset = newx;
      newx--;
    } else break;
  }

  if ( xoffset < 0 ) xoffset = 0;
  textRedraw();
}

int StringGadget::getCursor() const
{
  return cursorpos;
}

void StringGadget::setCursor( int new_pos )
{
  int newx, l, strwidth;

  cursorpos = new_pos;

  if ( cursorpos > (int)strlen( text ) )
    cursorpos = (int)strlen( text );
  if ( cursorpos < 0 )
    cursorpos = 0;
  selstart = selend = cursorpos;

  if ( ( cursorpos - xoffset ) < 0 )
    xoffset = cursorpos;

  newx = xoffset;
  for (;;) {
    l = _aguix->getStrlen4Width( text + newx, getInnerWidth() - 2, &strwidth, font );
    if ( ( l + newx ) == (int)strlen( text ) ) break;
    else if ( cursorpos >= ( l + newx ) ) {
      newx++;
    } else break;
  }

  setXOffset( newx );
}

bool StringGadget::isPosChangeForbidden() const
{
  return forbidPosChange;
}

void StringGadget::setForbidPosChange(bool nv)
{
  forbidPosChange=nv;
}

void StringGadget::setStrongKeyCapture(bool nv)
{
  strongkeycapture=nv;
}

void StringGadget::paste(unsigned char*buf)
{
  int nbytes_return;
  char *tstr;
  if((wantpaste==true)&&(difftime(time(NULL),pasterequest)<10.0)) {
    if(forbidPosChange==false) {
      selstart=selend=cursorpos;
      nbytes_return=strlen((char*)buf);
      // an Cursorpos einfgen
      int tl=strlen(text);
      tstr=(char*)_allocsafe(tl+nbytes_return+2);
      strncpy(tstr,text,cursorpos);
      tstr[cursorpos]=0;
      strncat(tstr,(char*)buf,nbytes_return);
      tstr[cursorpos+nbytes_return]=0;
      strcat(tstr,text+cursorpos);
      _freesafe(text);
      text=tstr;
      redraw();
    }
  }
}

void StringGadget::cancelpaste()
{
  wantpaste=false;
  pasterequest=0;
}

void StringGadget::cancelcut()
{
  selstart=selend=cursorpos;
  redraw();
}

void StringGadget::doCreateStuff()
{
  GUIElement::doCreateStuff();
}

void StringGadget::doDestroyStuff()
{
  GUIElement::doDestroyStuff();
}

void StringGadget::insertAtCursor( const char *str )
{
  if ( str == NULL ) return;
  int tl, nl;
  char *tstr;

  nl = strlen( str );
  tl = strlen( text );
  tstr = (char*)_allocsafe( tl + nl + 1);
  strncpy( tstr, text, cursorpos );
  tstr[ cursorpos ] = 0;
  strncat( tstr, str, nl );
  tstr[ cursorpos + nl ] = 0;
  strcat( tstr, text + cursorpos );
  _freesafe( text );
  text = tstr;
  textRedraw();
}

void StringGadget::selectAll()
{
  cursorpos=(int)strlen(text);
  setXOffset( cursorpos );
  selstart = 0;
  selend = cursorpos;
  applySelection();
  redraw();
}

void StringGadget::setPasswordMode( bool nv )
{
  passwordMode = nv;
  redraw();
}

void StringGadget::lostFocus()
{
  if ( active == true ) {
    active = false;
    redraw();
    AGMessage *agmsg = AGUIX_allocAGMessage();
    agmsg->type = AG_STRINGGADGET_DEACTIVATE;
    agmsg->stringgadget.sg = this;
    agmsg->stringgadget.ok = false;
    _aguix->putAGMsg( agmsg );
  }
}

void StringGadget::gotFocus()
{
  if ( active == false ) {
    activate();
    AGMessage *agmsg = AGUIX_allocAGMessage();
    agmsg->type = AG_STRINGGADGET_ACTIVATE;
    agmsg->stringgadget.sg = this;
    _aguix->putAGMsg( agmsg );
  }
}

int StringGadget::getInnerWidth() const
{
  int bw = getBorderWidth();
  return _w - 2 * bw;
}

void StringGadget::prepareBG( bool force )
{
  if ( isCreated() == false ) return;
  if ( win == 0 ) return;

  _aguix->SetWindowBG( win, _parent->getBG() );
}

int StringGadget::getBorderWidth() const
{
  int bw;

  bw = 4;
  
  return bw;
}

int StringGadget::clearTextBG()
{
  if ( isCreated() == false ) return 1;

  int bw = getBorderWidth();
  int maxcol=_aguix->getMaxCols();
  int newbg;
  GC usegc;
  
  if ( active == true ) {
    if(maxcol>7) {
      newbg=7;
    } else {
      newbg=0;
    }
  } else {
    newbg=0;
  }
  
  if(font==NULL) usegc=0; else usegc=font->getGC();
  
  _aguix->setFG( usegc, newbg );
  _aguix->FillRectangle( win, usegc, bw, bw, getInnerWidth(), _h - 2 * bw );
  return 0;
}
