# debian-linux-lib.pl
# Networking functions for Debian linux >= 2.2 (aka. potato)
# Really, this won't work with releases prior to 2.2, don't even try it.
#
# Rene Mayrhofer, July 2000
# Some code has been taken from redhat-linux-lib.pl

use File::Copy;

$network_interfaces_config = '/etc/network/interfaces';

# active_interfaces()
# Returns a list of currently ifconfig'd interfaces
sub active_interfaces
{
local(@rv, @lines, $l);
open(IFC, "ifconfig -a |");
while(<IFC>) {
	s/\r|\n//g;
	if (/^\S+/) { push(@lines, $_); }
	else { $lines[$#lines] .= $_; }
	}
close(IFC);
foreach $l (@lines) {
	local %ifc;
	$l =~ /^([^:\s]+)/; $ifc{'name'} = $1;
	$l =~ /^(\S+)/; $ifc{'fullname'} = $1;
	if ($l =~ /^(\S+):(\d+)/) { $ifc{'virtual'} = $2; }
	if ($l =~ /inet addr:(\S+)/) { $ifc{'address'} = $1; }
	else { next; }
	if ($l =~ /Mask:(\S+)/) { $ifc{'netmask'} = $1; }
	if ($l =~ /Bcast:(\S+)/) { $ifc{'broadcast'} = $1; }
	if ($l =~ /HWaddr (\S+)/) { $ifc{'ether'} = $1; }
	if ($l =~ /MTU:(\d+)/) { $ifc{'mtu'} = $1; }
	$ifc{'up'}++ if ($l =~ /\sUP\s/);
	$ifc{'edit'} = ($ifc{'name'} !~ /^ppp/);
	$ifc{'index'} = scalar(@rv);
	push(@rv, \%ifc);
	}
return @rv;
}

# activate_interface(&details)
# Create or modify an interface
sub activate_interface
{
local $a = $_[0];
local $cmd = "ifconfig $a->{'name'}";
if ($a->{'virtual'} ne "") { $cmd .= ":$a->{'virtual'}"; }
$cmd .= " $a->{'address'}";
if ($a->{'netmask'}) { $cmd .= " netmask $a->{'netmask'}"; }
if ($a->{'broadcast'}) { $cmd .= " broadcast $a->{'broadcast'}"; }
if ($a->{'mtu'}) { $cmd .= " mtu $a->{'mtu'}"; }
if ($a->{'up'}) { $cmd .= " up"; }
else { $cmd .= " down"; }
local $out = &backquote_logged("$cmd 2>&1");
if ($?) { &error($out); }
if ($a->{'ether'}) {
	$out = &backquote_logged(
		"ifconfig $a->{'name'} hw ether $a->{'ether'} 2>&1");
	if ($?) { &error($out); }
	}
}

# deactivate_interface(&details)
# Shutdown some active interface
sub deactivate_interface
{
local $cmd = "ifconfig $a->{'name'}";
if ($a->{'virtual'} ne "") { $cmd .= ":$a->{'virtual'}"; }
$cmd .= " down";
local $out = &backquote_logged("$cmd 2>&1");
if ($?) { &error($out); }
}

# boot_interfaces()
# Returns a list of interfaces brought up at boot time
sub boot_interfaces()
{
my @ifaces = &get_interface_defs();
my @rv;
foreach $iface (@ifaces) {
	my ($name, $addrfam, $method, $options) = @$iface;
	my $cfg;
	if ($addrfam eq 'inet') {
		$cfg->{'fullname'} = $name;
		if ($cfg->{'fullname'} =~ /(\S+):(\d+)/) {
			$cfg->{'name'} = $1;
			$cfg->{'virtual'} = $2;
			}
		else {
			$cfg->{'name'} = $cfg->{'fullname'};
			}
		$cfg->{'up'} = 1;
		foreach $option (@$options) {
			($param, $value) = @$option;
			if ($param eq 'noauto') { $cfg->{'up'} = 0; }
			else { $cfg->{$param} = $value; }
			}
		$cfg->{'dhcp'} = ($method eq 'dhcp');
		$cfg->{'bootp'} = ($method eq 'bootp');
		$cfg->{'edit'} = ($cfg->{'name'} !~ /^ppp|lo/);
		$cfg->{'index'} = scalar(@rv);	
		push(@rv, $cfg);
		}
	}
return @rv;
}

# save_interface(&details)
# Create or update a boot-time interface
sub save_interface
{
my $cfg = $_[0];
my $name = $cfg->{'virtual'} ne "" ? $cfg->{'name'}.":".$cfg->{'virtual'}
				       : $cfg->{'name'};
my @options;
my $method;
if ($cfg->{'dhcp'} == 1) { $method = 'dhcp'; }
elsif ($cfg->{'bootp'} == 1) { $method = 'bootp'; }
else {
	$method = 'static';
	push(@options, ['address', $cfg->{'address'}]);
	push(@options, ['netmask', $cfg->{'netmask'}]);
	push(@options, ['broadcast', $cfg->{'broadcast'}]);
	my ($ip1, $ip2, $ip3, $ip4) = split(/\./, $cfg->{'address'});
	my ($nm1, $nm2, $nm3, $nm4) = split(/\./, $cfg->{'netmask'});
	if ($cfg->{'address'} && $cfg->{'netmask'}) {
		my $network = sprintf "%d.%d.%d.%d",
					($ip1 & int($nm1))&0xff,
					($ip2 & int($nm2))&0xff,
					($ip3 & int($nm3))&0xff,
					($ip4 & int($nm4))&0xff;
		push(@options, ['network', $network]);
		}
	}
if (! $cfg->{'up'}) { push(@options, ['noauto', '']); }

my @ifaces = get_interface_defs();
my $changeit = 0;
foreach $iface (@ifaces) {
	if ($iface->[0] eq $cfg->{'fullname'}) {
		$changeit = 1;
		foreach $o (@{$iface->[3]}) {
			if ($o->[0] eq 'gateway') {
				push(@options, $o);
				}
			}
		}
	}
if ($changeit == 0) {
	new_interface_def($cfg->{'fullname'}, 'inet', $method, \@options);
	}
else {
	modify_interface_def($cfg->{'fullname'}, 'inet', $method, \@options, 0);
	}
}

# delete_interface(&details)
# Delete a boot-time interface
sub delete_interface
{
my $cfg = $_[0];
delete_interface_def($cfg->{'fullname'}, 'inet');
}

# iface_type(name)
# Returns a human-readable interface type name
sub iface_type
{
return "PPP" if ($_[0] =~ /^ppp/);
return "SLIP" if ($_[0] =~ /^sl/);
return "PLIP" if ($_[0] =~ /^plip/);
return "Ethernet" if ($_[0] =~ /^eth/);
return "Arcnet" if ($_[0] =~ /^arc/);
return "Token Ring" if ($_[0] =~ /^tr/);
return "Pocket/ATP" if ($_[0] =~ /^atp/);
return "Loopback" if ($_[0] =~ /^lo/);
return "ISDN rawIP" if ($_[0] =~ /^isdn/);
return "ISDN syncPPP" if ($_[0] =~ /^ippp/);
return $text{'ifcs_unknown'};
}

# iface_hardware(name)
# Does some interface have an editable hardware address
sub iface_hardware
{
return $_[0] =~ /^eth/;
}

# can_edit(what)
# Can some boot-time interface parameter be edited?
sub can_edit
{
return $_[0] ne "mtu";
}

# valid_boot_address(address)
# Is some address valid for a bootup interface
sub valid_boot_address
{
return &check_ipaddress($_[0]);
}

# get_dns_config()
# Returns a hashtable containing keys nameserver, domain, search & order
sub get_dns_config
{
local $dns;
open(RESOLV, "/etc/resolv.conf");
while(<RESOLV>) {
	s/\r|\n//g;
	s/#.*$//;
	if (/nameserver\s+(.*)/) {
		push(@{$dns->{'nameserver'}}, split(/\s+/, $1));
		}
	elsif (/domain\s+(\S+)/) {
		$dns->{'domain'} = [ $1 ];
		}
	elsif (/search\s+(.*)/) {
		$dns->{'domain'} = [ split(/\s+/, $1) ];
		}
	}
close(RESOLV);
open(SWITCH, "/etc/nsswitch.conf");
while(<SWITCH>) {
	s/\r|\n//g;
	if (/^\s*hosts:\s+(.*)/) {
		$dns->{'order'} = $1;
		}
	}
close(SWITCH);
return $dns;
}

# save_dns_config(&config)
# Writes out the resolv.conf and nsswitch.conf files
sub save_dns_config
{
&lock_file("/etc/resolv.conf");
open(RESOLV, "/etc/resolv.conf");
local @resolv = <RESOLV>;
close(RESOLV);
open(RESOLV, ">/etc/resolv.conf");
foreach (@{$_[0]->{'nameserver'}}) {
	print RESOLV "nameserver $_\n";
	}
if ($_[0]->{'domain'}) {
	if ($_[0]->{'domain'}->[1]) {
		print RESOLV "search ",join(" ", @{$_[0]->{'domain'}}),"\n";
		}
	else {
		print RESOLV "domain $_[0]->{'domain'}->[0]\n";
		}
	}
foreach (@resolv) {
	print RESOLV $_ if (!/^\s*(nameserver|domain|search)\s+/);
	}
close(RESOLV);
&unlock_file("/etc/resolv.conf");

&lock_file("/etc/nsswitch.conf");
open(SWITCH, "/etc/nsswitch.conf");
local @switch = <SWITCH>;
close(SWITCH);
open(SWITCH, ">/etc/nsswitch.conf");
foreach (@switch) {
	if (/^\s*hosts:\s+/) {
		print SWITCH "hosts:\t$_[0]->{'order'}\n";
		}
	else { print SWITCH $_; }
	}
close(SWITCH);
&unlock_file("/etc/nsswitch.conf");
}

$max_dns_servers = 3;

# order_input(&dns)
# Returns HTML for selecting the name resolution order
sub order_input
{
if ($_[0]->{'order'} =~ /\[/) {
	# Using a complex resolve list
	return "<input name=order size=45 value=\"$_[0]->{'order'}\">\n";
	}
else {
	# Can select by menus
	local @o = split(/\s+/, $_[0]->{'order'});
	@o = map { s/nis\+/nisplus/; s/yp/nis/; $_; } @o;
	local ($rv, $i, $j);
	local @srcs = ( "", "files", "dns", "nis", "nisplus", "db" );
	local @srcn = ( "", "Hosts", "DNS", "NIS", "NIS+", "DB" );
	for($i=1; $i<@srcs; $i++) {
		local $ii = $i-1;
		$rv .= "<select name=order_$ii>\n";
		for($j=0; $j<@srcs; $j++) {
			$rv .= sprintf "<option value=\"%s\" %s>%s\n",
					$srcs[$j],
					$o[$ii] eq $srcs[$j] ? "selected" : "",
					$srcn[$j] ? $srcn[$j] : "&nbsp;";
			}
		$rv .= "</select>\n";
		}
	return $rv;
	}
}

# parse_order(&dns)
# Parses the form created by order_input()
sub parse_order
{
if (defined($in{'order'})) {
	$in{'order'} =~ /\S/ || &error($text{'dns_eorder'});
	$_[0]->{'order'} = $in{'order'};
	}
else {
	local($i, @order);
	for($i=0; defined($in{"order_$i"}); $i++) {
		push(@order, $in{"order_$i"}) if ($in{"order_$i"});
		}
	$_[0]->{'order'} = join(" ", @order);
	}
}

# get_hostname()
sub get_hostname
{
return &get_system_hostname();
}

# save_hostname(name)
sub save_hostname
{
local %conf;
&system_logged("hostname $_[0] >/dev/null 2>&1");
&lock_file("/etc/HOSTNAME");
open(HOST, ">/etc/HOSTNAME");
print HOST $_[0],"\n";
close(HOST);
&unlock_file("/etc/HOSTNAME");
&lock_file($network_config);
&read_env_file($network_config, \%conf);
$conf{'HOSTNAME'} = $_[0];
&write_env_file($network_config, \%conf);
&unlock_file($network_config);
}

# get_domainname()
sub get_domainname
{
local $d = `domainname`;
chop($d);
return $d eq "(none)" ? "" : $d;
}

# save_domainname(domain)
sub save_domainname
{
local %conf;
system("domainname \"$_[0]\" >/dev/null 2>&1");
&read_env_file($network_config, \%conf);
if ($_[0]) {
	$conf{'NISDOMAIN'} = $_[0];
	}
else {
	delete($conf{'NISDOMAIN'});
	}
&write_env_file($network_config, \%conf);
}

# show default router and device
sub routing_input
{
local @ifaces = &get_interface_defs();
local ($router, $addr);
foreach $iface (@ifaces) {
	foreach $o (@{$iface->[3]}) {
		if ($o->[0] eq 'gateway') {
			$router = $iface;
			$addr = $o->[1];
			}
		}
	}

print "<tr> <td><b>$text{'routes_default'}</b></td>\n";
printf "<td><input type=radio name=gateway_def value=1 %s> %s\n",
	$router ? '' : 'checked', $text{'routes_none'};
printf "<input type=radio name=gateway_def value=0 %s>\n",
	$router ? 'checked' : '';
print "$text{'routes_gateway'}\n";
printf "<input name=gateway size=15 value='%s'>\n", $addr;
print "$text{'routes_device'}\n";
print "<select name=gatewaydev>\n";
foreach $iface (@ifaces) {
	next if ($iface->[0] eq 'lo');
	printf "<option %s>%s\n",
		$router eq $iface ? 'selected' : '', $iface->[0];
	}
print "</select></td> </tr>\n";
}

sub parse_routing
{
local $dev;
if (!$in{'gateway_def'}) {
	&check_ipaddress($in{'gateway'}) ||
		&error(&text('routes_egateway', $in{'gateway'}));
	$dev = $in{'gatewaydev'};
	}
local @ifaces = &get_interface_defs();
foreach $iface (@ifaces) {
	# Remove the gateway directive
	$iface->[3] = [ grep { $_->[0] ne 'gateway' } @{$iface->[3]} ];

	# Add if needed
	if ($iface->[0] eq $dev) {
		push(@{$iface->[3]}, [ 'gateway', $in{'gateway'} ]);
		}
	&modify_interface_def(@$iface);
	}
}


###############################################################################
# helper functions for file-internal use

# gets a list of interface definitions (including their options) from the
# central config file
# the returned list is an array whose contents are tupels of
# (name, addrfam, method, options) with
#    name          the interface name (e.g. eth0)
#    addrfam       the address family (e.g. inet, inet6)
#    method        the address activation method (e.g. static, dhcp, loopback)
#    options       is a list of (param, value) pairs
sub get_interface_defs()
{
local *CFGFILE;
my @ret;
open(CFGFILE, "< $network_interfaces_config") ||
	error("Unable to open $network_interfaces_config");
# read the file line by line
$line = <CFGFILE>;
while (defined $line) {
	chomp($line);
	# skip comments
	if ($line =~ /^\s*#/ || $line =~ /^\s*$/) {
		$line = <CFGFILE>;
		next;
		}

	if ($line =~ /^\s*auto/) {
		# skip auto stanzas
		$line = <CFGFILE>;
		while(defined($line) && $line !~ /^\s*(iface|mapping|auto)/) {
			$line = <CFGFILE>;
			next;
			}
		}
	elsif ($line =~ /^\s*mapping/) {
		# skip mapping stanzas
		$line = <CFGFILE>;
		while(defined($line) && $line !~ /^\s*(iface|mapping|auto)/) {
			$line = <CFGFILE>;
			next;
			}
		}
	elsif (my ($name, $addrfam, $method) = ($line =~ /^\s*iface\s+(\S+)\s+(\w+)\s+(\w+)\s*$/) ) {
		# only lines starting with "iface" are expected here
		my @iface_options;
		# now read everything until the next iface definition
		$line = <CFGFILE>;
		while (defined $line && ! ($line =~ /^\s*(iface|mapping|auto)/)) {
			# skip comments and empty lines
			if ($line =~ /^\s*#/ || $line =~ /^\s*$/) {
				$line = <CFGFILE>;
				next;
				}
			my ($param, $value);
			if ( ($param, $value) = ($line =~ /^\s*(\S+)\s+(.*)\s*$/) ) {
				push(@iface_options, [$param, $value]);
				}
			elsif ( ($param) = ($line =~ /^\s*(\S+)\s*$/) ) {
				push(@iface_options, [$param, '']);
				}
			else {
				error("Error in option line: '$line' invalid");
				}
			$line = <CFGFILE>;
			}
		push(@ret, [$name, $addrfam, $method, \@iface_options]);
		}
	else {
		error("Error reading file $pathname: unexpected line '$line'");
		}
	}
close(CFGFILE);
return @ret;
}

# modifies the options of an already stored interface definition
# the parameters should be (name, addrfam, method, options, mode)
# with options being an array of (param, value) pairs
# and mode being 0 for modify and 1 for delete
sub modify_interface_def($$$$$)
{
my ($name, $addrfam, $method, $options, $mode) = @_;
# make a backup copy
copy("$network_interfaces_config", "$network_interfaces_config~");
local *OLDCFGFILE, *NEWCFGFILE;
open(OLDCFGFILE, "< $network_interfaces_config~") ||
	error("Unable to open $network_interfaces_config");
&lock_file($network_interfaces_config);
open(NEWCFGFILE, "> $network_interfaces_config") ||
	error("Unable to open $network_interfaces_config");

my $inside_modify_region = 0;
my $iface_line = 0;
my $new_options_wrote;
while (defined ($line=<OLDCFGFILE>)) {
	if ($inside_modify_region == 0 &&
		$line =~ /^\s*iface\s+$name\s+$addrfam\s+\w+\s*$/) {
		$inside_modify_region = 1;
		$iface_line = 1;
		$new_options_wrote = 0;
		}
	elsif ($inside_modify_region == 1 &&
		$line =~ /^\s*iface\s+\w+\s+\w+\s+\w+\s*$/) {
		$inside_modify_region = 0;
		}
	# preserve comments
	if ($line =~ /^\s*#/) {
		print NEWCFGFILE $line;
		}
	# inside modify region or not ?
	elsif ($inside_modify_region == 0) {
    	print NEWCFGFILE $line;
		}
	else {
		# should the iface line be changed or the options ?
		if ($iface_line == 1 && $mode == 0) {
			print NEWCFGFILE "iface $name $addrfam $method\n";
			}
		else {
			# only write the new options and skip the old ones or just do
			# nothing if mode is delete
			# write only upon first entrance here
			if ($mode == 0 && $new_options_wrote == 0) {
				$new_options_wrote = 1;
				foreach $option (@$options) {
					my ($param, $value) = @$option;
					print NEWCFGFILE "\t$param $value\n";
					}
				}
			}
		}
	$iface_line = 0;
	}

close(OLDCFGFILE);
close(NEWCFGFILE);
&unlock_file($network_interfaces_config);
}

# creates a new interface definition in the config file
# the parameters should be (name, addrfam, method, options)
# with options being an array of (param, value) pairs
# the selection key is (name, addrfam)
sub new_interface_def($$$$)
{
# make a backup copy
copy("$network_interfaces_config", "$network_interfaces_config~");
local *CFGFILE;
&lock_file($network_interfaces_config);
open(CFGFILE, ">> $network_interfaces_config") ||
	error("Unable to open $network_interfaces_config");
local ($name, $addrfam, $method, $options) = @_;
print CFGFILE "\niface $name $addrfam $method\n";
foreach $option (@$options) {
	my ($param, $value) = @$option;
	print CFGFILE "\t$param $value\n";
	}
close(CFGFILE);
&unlock_file($network_interfaces_config);
}

# delete an already defined interface
# the parameters should be (name, addrfam)
sub delete_interface_def($$$)
{
	local ($name, $addrfam, $method) = @_;
	modify_interface_def($name, $addrfam, '', [], 1);
}

sub os_feedback_files
{
return ( $network_interfaces_config, "/etc/nsswitch.conf", "/etc/resolv.conf",
	 "/etc/HOSTNAME" );
}

1;

