/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef TYPEDIALOGS_H
#define TYPEDIALOGS_H

#include "commanddialog.h"
#include "attributes/definitioninfo.h"

class QLineEdit;
class QCheckBox;
class QRadioButton;
class QButtonGroup;
class QComboBox;

class AbstractValueEditor;
class PopEditBox;

/**
* Base class for dialogs for executing commands related to types.
*/
class TypesDialog : public CommandDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    TypesDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~TypesDialog();

protected:
    /**
    * Verify that the type name is valid.
    *
    * A warning is displayed if the name is empty or a type
    * with this name already exists.
    *
    * @param name The name to check.
    *
    * @return @c true if the name is valid, @c false otherwise.
    */
    bool checkTypeName( const QString& name );

    /**
    * Verify that the attribute name is valid.
    *
    * A warning is displayed if the name is empty or an attribute
    * with this name already exists.
    *
    * @param name The name to check.
    *
    * @param typeId Identifier of the type containing the attribute.
    *
    * @return @c true if the name is valid, @c false otherwise.
    */
    bool checkAttributeName( int typeId, const QString& name );
};

/**
* Dialog for executing the <tt>ADD TYPE</tt> command.
*/
class AddTypeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    AddTypeDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~AddTypeDialog();

public: // overrides
    void accept();

private:
    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>RENAME TYPE</tt> command.
*/
class RenameTypeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param typeId Identifier of the type to rename.
    *
    * @param parent The parent widget.
    */
    RenameTypeDialog( int typeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameTypeDialog();

public: // overrides
    void accept();

private:
    int m_typeId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>DELETE TYPE</tt> command.
*/
class DeleteTypeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param typeId Identifier of the type to delete.
    *
    * @param parent The parent widget.
    */
    DeleteTypeDialog( int typeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~DeleteTypeDialog();

public: // overrides
    void accept();

private:
    int m_typeId;
};

/**
* Base class for dialogs for adding or modifying attributes.
*/
class AttributeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    AttributeDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~AttributeDialog();

protected:
    /**
    * Create the widgets for defining the attribute.
    *
    * @param withName If @c true, also the widget for editing the name will be created.
    */
    void createWidgets( bool withName );

    /**
    * Initialize the widgets from given attribute definition.
    */
    void setDefinitionInfo( DefinitionInfo& info );

    /**
    * Create the attribute definition from the widgets.
    */
    DefinitionInfo definitionInfo();

    /**
    * Return the name of the attribute.
    */
    QString attributeName() const;

private slots:
    void typeActivated( int index );
    void detailsClicked();
    void requiredToggled( bool on );

private:
    void updateWidgets();
    void updateMetadata();

private:
    QLineEdit* m_nameEdit;
    QComboBox* m_typeCombo;
    PopEditBox* m_detailsBox;
    QCheckBox* m_requiredCheck;
    QButtonGroup* m_valueGroup;
    QRadioButton* m_valueButton1;
    QRadioButton* m_valueButton2;
    QBoxLayout* m_valueLayout;
    AbstractValueEditor* m_editor;

    DefinitionInfo m_info;
};

/**
* Dialog for executing the <tt>ADD ATTRIBUTE</tt> command.
*/
class AddAttributeDialog : public AttributeDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param typeId Identifier of the type containing the attribute.
    *
    * @param parent The parent widget.
    */
    AddAttributeDialog( int typeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~AddAttributeDialog();

public: // overrides
    void accept();

private:
    int m_typeId;
};

/**
* Dialog for executing the <tt>MODIFY ATTRIBUTE</tt> command.
*/
class ModifyAttributeDialog : public AttributeDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param attributeId Identifier of the attribute to modify.
    *
    * @param parent The parent widget.
    */
    ModifyAttributeDialog( int attributeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~ModifyAttributeDialog();

public: // overrides
    void accept();

private:
    int m_attributeId;

    QString m_oldDefinition;
};

/**
* Dialog for executing the <tt>RENAME ATTRIBUTE</tt> command.
*/
class RenameAttributeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param attributeId Identifier of the attribute to rename.
    *
    * @param parent The parent widget.
    */
    RenameAttributeDialog( int attributeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameAttributeDialog();

public: // overrides
    void accept();

private:
    int m_attributeId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>DELETE ATTRIBUTE</tt> command.
*/
class DeleteAttributeDialog : public TypesDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param attributeId Identifier of the attribute to delete.
    *
    * @param parent The parent widget.
    */
    DeleteAttributeDialog( int attributeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~DeleteAttributeDialog();

public: // overrides
    void accept();

private:
    int m_attributeId;
};

#endif
