/* Copyright (c) Mark J. Kilgard, 1997. */

/* This program is freely distributable without licensing fees  and is
   provided without guarantee or warrantee expressed or  implied. This
   program is -not- in the public domain. */

#ifndef VRENGD

#if defined(_WIN32)
#pragma warning (disable:4244)       /* disable bogus conversion warnings */
#endif

#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "texfont.h"


#ifndef GL_VERSION_1_1
#  if defined(GL_EXT_texture_object)
#    define glGenTextures glGenTexturesEXT
#    define glBindTexture glBindTextureEXT
#  else
     /* Without OpenGL 1.1 or the texture object extension, use display lists */
#    define USE_DISPLAY_LISTS
#  endif
#  if defined(GL_EXT_texture)
#    define GL_INTENSITY4 GL_INTENSITY4_EXT
     int useLuminanceAlpha = 0;
#  else
     /* Intensity texture format not in OpenGL 1.0; added by the EXT_texture
        extension and now part of OpenGL 1.1. */
     int useLuminanceAlpha = 1;
#  endif
#else
   /* OpenGL 1.1 or 1.2 cases */
   int useLuminanceAlpha = 0;
#endif


/* byte swap a 32-bit value */
#define SWAPL(x, n) { \
                 n = ((char *) (x))[0];\
                 ((char *) (x))[0] = ((char *) (x))[3];\
                 ((char *) (x))[3] = n;\
                 n = ((char *) (x))[1];\
                 ((char *) (x))[1] = ((char *) (x))[2];\
                 ((char *) (x))[2] = n; }

/* byte swap a short */
#define SWAPS(x, n) { \
                 n = ((char *) (x))[0];\
                 ((char *) (x))[0] = ((char *) (x))[1];\
                 ((char *) (x))[1] = n; }


static
TexGlyphVertexInfo * getTCVI(TexFont *txf, int c)
{
  TexGlyphVertexInfo *tgvi;

  /* Automatically substitute uppercase letters with lowercase if not
     uppercase available (and vice versa). */
lastchance:
  if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range)) {
    if ((tgvi = txf->lut[c - txf->min_glyph]) != 0)
      return tgvi;
    if (islower(c)) {
      c = toupper(c);
      if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range))
        return txf->lut[c - txf->min_glyph];
    }
    if (isupper(c)) {
      c = tolower(c);
      if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range))
        return txf->lut[c - txf->min_glyph];
    }
  }
  fprintf(stderr, "texfont: tried to access unavailable font character \"%c\" (%d)\n", isprint(c) ? c : ' ', c);
    c = ' ';
    return NULL;
    goto lastchance;
  /* NOTREACHED */
}

static const char *lastError;

const char * txfErrorString(void)
{
  return lastError;
}

static
void txfLoadFont(void *atxf, void *handle)
{
  TexFont *txf = NULL;
  TexFont **ptxf = (TexFont **) atxf;
  GLfloat w, h, xstep, ystep;
  char fileid[4], tmp;
  unsigned char *texbitmap;
  int min_glyph, max_glyph;
  int endianness, swap, format, stride, width, height;
  int i, j;
  unsigned long got;

#ifndef GL_VERSION_1_1
  trace(DBG_ZV, "GL_VERSION_1_1");
#else
  trace(DBG_ZV, "!GL_VERSION_1_1");
#endif
#if defined(GL_EXT_texture)
  trace(DBG_ZV, "GL_EXT_texture");
#else
  trace(DBG_ZV, "!GL_EXT_texture");
#endif
#if defined(GL_EXT_texture)
  trace(DBG_ZV, "GL_EXT_texture_object");
#else
  trace(DBG_ZV, "!GL_EXT_texture_object");
#endif

  if (handle == NULL) {
    *ptxf = NULL;
    return;
  }

  *ptxf = txf = (TexFont *) calloc(1, sizeof(TexFont));
  if (txf == NULL) {
    lastError = "out of memory.";
    goto error;
  }

  /* For easy cleanup in error case. */
  txf->tgi = NULL;
  txf->tgvi = NULL;
  txf->lut = NULL;
  txf->teximage = NULL;

  httpClearBuf();
  got = httpFread((char *) fileid, 1, 4, handle);
  if (got != 4 || strncmp(fileid, "\377txf", 4)) {
    lastError = "not a texture font file.";
    goto error;
  }

  /*CONSTANTCONDITION*/
  //DAX assert(sizeof(int) == 4);  /* Ensure external file format size. */
  got = httpFread((char *) &endianness, sizeof(int), 1, handle);
  if (got == 1 && endianness == 0x12345678)
    swap = 0;
  else if (got == 1 && endianness == 0x78563412)
    swap = 1;
  else {
    lastError = "not a texture font file.";
    goto error;
  }

#define EXPECT(n) if (got != (unsigned long) n) { lastError = "premature end of file."; goto error; }
  got = httpFread((char *) &format, sizeof(int), 1, handle);
  EXPECT(1);
  got = httpFread((char *) &txf->tex_width, sizeof(int), 1, handle);
  EXPECT(1);
  got = httpFread((char *) &txf->tex_height, sizeof(int), 1, handle);
  EXPECT(1);
  got = httpFread((char *) &txf->max_ascent, sizeof(int), 1, handle);
  EXPECT(1);
  got = httpFread((char *) &txf->max_descent, sizeof(int), 1, handle);
  EXPECT(1);
  got = httpFread((char *) &txf->num_glyphs, sizeof(int), 1, handle);
  EXPECT(1);

  if (swap) {
    SWAPL(&format, tmp);
    SWAPL(&txf->tex_width, tmp);
    SWAPL(&txf->tex_height, tmp);
    SWAPL(&txf->max_ascent, tmp);
    SWAPL(&txf->max_descent, tmp);
    SWAPL(&txf->num_glyphs, tmp);
  }
  trace(DBG_ZV, "txfLoadFont: fmt=%d w=%d h=%d ascent=%d descent=%d glyphs=%d", format, txf->tex_width, txf->tex_height, txf->max_ascent, txf->max_descent, txf->num_glyphs);

  txf->tgi = (TexGlyphInfo *) malloc(txf->num_glyphs * sizeof(TexGlyphInfo));
  if (txf->tgi == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  /*CONSTANTCONDITION*/
  //DAX assert(sizeof(TexGlyphInfo) == 12);  /* Ensure external file format size. */
  got = httpFread((char *) txf->tgi, sizeof(TexGlyphInfo), txf->num_glyphs, handle);
  EXPECT(txf->num_glyphs);

  if (swap) {
    for (i = 0; i < txf->num_glyphs; i++) {
      SWAPS(&txf->tgi[i].c, tmp);
      SWAPS(&txf->tgi[i].x, tmp);
      SWAPS(&txf->tgi[i].y, tmp);
    }
  }
  txf->tgvi = (TexGlyphVertexInfo *)malloc(txf->num_glyphs * sizeof(TexGlyphVertexInfo));
  if (txf->tgvi == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  w = txf->tex_width;
  h = txf->tex_height;
  xstep = 0.5 / w;
  ystep = 0.5 / h;
  for (i = 0; i < txf->num_glyphs; i++) {
    TexGlyphInfo *tgi;

    tgi = &txf->tgi[i];
    txf->tgvi[i].t0[0] = tgi->x / w + xstep;
    txf->tgvi[i].t0[1] = tgi->y / h + ystep;
    txf->tgvi[i].v0[0] = tgi->xoffset;
    txf->tgvi[i].v0[1] = tgi->yoffset;
    txf->tgvi[i].t1[0] = (tgi->x + tgi->width) / w + xstep;
    txf->tgvi[i].t1[1] = tgi->y / h + ystep;
    txf->tgvi[i].v1[0] = tgi->xoffset + tgi->width;
    txf->tgvi[i].v1[1] = tgi->yoffset;
    txf->tgvi[i].t2[0] = (tgi->x + tgi->width) / w + xstep;
    txf->tgvi[i].t2[1] = (tgi->y + tgi->height) / h + ystep;
    txf->tgvi[i].v2[0] = tgi->xoffset + tgi->width;
    txf->tgvi[i].v2[1] = tgi->yoffset + tgi->height;
    txf->tgvi[i].t3[0] = tgi->x / w + xstep;
    txf->tgvi[i].t3[1] = (tgi->y + tgi->height) / h + ystep;
    txf->tgvi[i].v3[0] = tgi->xoffset;
    txf->tgvi[i].v3[1] = tgi->yoffset + tgi->height;
    txf->tgvi[i].advance = tgi->advance;
  }

  min_glyph = txf->tgi[0].c;
  max_glyph = txf->tgi[0].c;
  for (i = 1; i < txf->num_glyphs; i++) {
    if (txf->tgi[i].c < min_glyph)
      min_glyph = txf->tgi[i].c;
    if (txf->tgi[i].c > max_glyph)
      max_glyph = txf->tgi[i].c;
  }
  txf->min_glyph = min_glyph;
  txf->range = max_glyph - min_glyph + 1;

  txf->lut = (TexGlyphVertexInfo **)
             calloc(txf->range, sizeof(TexGlyphVertexInfo *));
  if (txf->lut == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  for (i = 0; i < txf->num_glyphs; i++) {
    txf->lut[txf->tgi[i].c - txf->min_glyph] = &txf->tgvi[i];
  }

  switch (format) {
  case TXF_FORMAT_BYTE:
    if (useLuminanceAlpha) {
      unsigned char *orig;

      orig = (unsigned char *) malloc(txf->tex_width * txf->tex_height);
      if (orig == NULL) {
        lastError = "out of memory.";
        goto error;
      }
      got = httpFread((char *) orig, 1, txf->tex_width * txf->tex_height, handle);
      EXPECT(txf->tex_width * txf->tex_height);
      txf->teximage = (unsigned char *)
        malloc(2 * txf->tex_width * txf->tex_height);
      if (txf->teximage == NULL) {
        lastError = "out of memory.";
        goto error;
      }
      for (i = 0; i < txf->tex_width * txf->tex_height; i++) {
        txf->teximage[i * 2] = orig[i];
        txf->teximage[i * 2 + 1] = orig[i];
      }
      free(orig);
    } else {
      txf->teximage = (unsigned char *)
        malloc(txf->tex_width * txf->tex_height);
      if (txf->teximage == NULL) {
        lastError = "out of memory.";
        goto error;
      }
      got = httpFread((char *) txf->teximage, 1, txf->tex_width * txf->tex_height, handle);
      EXPECT(txf->tex_width * txf->tex_height);
    }
    break;
  case TXF_FORMAT_BITMAP:
    width = txf->tex_width;
    height = txf->tex_height;
    stride = (width + 7) >> 3;
    texbitmap = (unsigned char *) malloc(stride * height);
    if (texbitmap == NULL) {
      lastError = "out of memory.";
      goto error;
    }
    got = httpFread((char *) texbitmap, 1, stride * height, handle);
    EXPECT(stride * height);
    if (useLuminanceAlpha) {
      txf->teximage = (unsigned char *) calloc(width * height * 2, 1);
      if (txf->teximage == NULL) {
        lastError = "out of memory.";
        goto error;
      }
      for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
          if (texbitmap[i * stride + (j >> 3)] & (1 << (j & 7))) {
            txf->teximage[(i * width + j) * 2] = 255;
            txf->teximage[(i * width + j) * 2 + 1] = 255;
          }
        }
      }
    } else {
      txf->teximage = (unsigned char *) calloc(width * height, 1);
      if (txf->teximage == NULL) {
        lastError = "out of memory.";
        goto error;
      }
      for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
          if (texbitmap[i * stride + (j >> 3)] & (1 << (j & 7)))
            txf->teximage[i * width + j] = 255;
        }
      }
    }
    free(texbitmap);
    break;
  }
  return;

error:
  trace(DBG_FORCE, "error: %s", lastError);
  if (txf) {
    if (txf->tgi) free(txf->tgi);
    if (txf->tgvi) free(txf->tgvi);
    if (txf->lut) free(txf->lut);
    if (txf->teximage) free(txf->teximage);
    free(txf);
  }
  return;
}

GLuint
txfEstablishTexture(TexFont *txf, GLuint texobj, GLboolean setupMipmaps)
{
  if (txf->texobj == 0) {
    if (texobj == 0)
#if !defined(USE_DISPLAY_LISTS)
      glGenTextures(1, &txf->texobj);
#else
      txf->texobj = glGenLists(1);
#endif
    else
      txf->texobj = texobj;
  }

#if !defined(USE_DISPLAY_LISTS)
  glBindTexture(GL_TEXTURE_2D, txf->texobj);
#else
  glNewList(txf->texobj, GL_COMPILE);
#endif

  if (useLuminanceAlpha) {
    if (setupMipmaps) {
#if !defined(WITH_TINYGL)	/* using gluBuild2Mipmaps */
      gluBuild2DMipmaps(GL_TEXTURE_2D, GL_LUMINANCE_ALPHA,
			txf->tex_width, txf->tex_height,
			GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, txf->teximage);
#endif
    } else {
      glTexImage2D(GL_TEXTURE_2D, 0, GL_LUMINANCE_ALPHA,
		   txf->tex_width, txf->tex_height, 0,
		   GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, txf->teximage);
    }
  } else {
#if defined(GL_VERSION_1_1) || defined(GL_EXT_texture)
    /* Use GL_INTENSITY4 as internal texture format since we want to use as
       little texture memory as possible. */
    if (setupMipmaps) {
#if !defined(WITH_TINYGL)	/* using gluBuild2Mipmaps */
      gluBuild2DMipmaps(GL_TEXTURE_2D, GL_INTENSITY4,
			txf->tex_width, txf->tex_height,
			GL_LUMINANCE, GL_UNSIGNED_BYTE, txf->teximage);
#endif
    } else {
      glTexImage2D(GL_TEXTURE_2D, 0, GL_INTENSITY4,
		   txf->tex_width, txf->tex_height, 0,
		   GL_LUMINANCE, GL_UNSIGNED_BYTE, txf->teximage);
    }
#else
    trace(DBG_FORCE, "txfEstablishTexture: abort");
    abort();      /* Should not get here without EXT_texture or OpenGL 1.1. */
#endif
  }

#if defined(USE_DISPLAY_LISTS)
  glEndList();
  glCallList(txf->texobj);
#endif
  return txf->texobj;
}

void txfBindFontTexture(TexFont *txf)
{
#if !defined(USE_DISPLAY_LISTS)
  glBindTexture(GL_TEXTURE_2D, txf->texobj);
#else
  glCallList(txf->texobj);
#endif
}

void txfUnloadFont(TexFont *txf)
{
  if (txf->teximage) {
    free(txf->teximage);
  }
  free(txf->tgi);
  free(txf->tgvi);
  free(txf->lut);
  free(txf);
}

void txfGetStringMetrics(TexFont *txf, const char *string, int len, int *width,
  int *max_ascent, int *max_descent)
{
  TexGlyphVertexInfo *tgvi;
  int w, i;

  w = 0;
  for (i = 0; i < len; i++) {
    if (string[i] == 27) {
      switch (string[i + 1]) {
      case 'M':
        i += 4;
        break;
      case 'T':
        i += 7;
        break;
      case 'L':
        i += 7;
        break;
      case 'F':
        i += 13;
        break;
      }
    } else {
      if ((tgvi = getTCVI(txf, string[i])) == NULL)
        break;	//DAX
      w += (int) tgvi->advance;
    }
  }
  *width = w;
  *max_ascent = txf->max_ascent;
  *max_descent = txf->max_descent;
}

static
void txfRenderGlyph(TexFont *txf, int c)
{
#if defined(WITH_TINYGL)	/* glVertex2sv missing */
#define glVertex2sv(v) glVertex2f(v[0], v[1])
#else
  TexGlyphVertexInfo *tgvi;

  if ((tgvi = getTCVI(txf, c)) == NULL)
    return;

  glBegin(GL_QUADS);
  glTexCoord2fv(tgvi->t0);
  glVertex2sv(tgvi->v0);
  glTexCoord2fv(tgvi->t1);
  glVertex2sv(tgvi->v1);
  glTexCoord2fv(tgvi->t2);
  glVertex2sv(tgvi->v2);
  glTexCoord2fv(tgvi->t3);
  glVertex2sv(tgvi->v3);
  glEnd();
  glTranslatef(tgvi->advance, 0.0, 0.0);
#endif
}

void txfRenderString(TexFont *txf, const char *string, int len)
{
  int i;

  for (i = 0; i < len; i++) {
    txfRenderGlyph(txf, string[i]);
  }
}


/*
 * ENST: VREng specific
 */

static TexfontCacheEntry *texfont_cache_first;
static int texfont_number;


void initTexfontCache(void)
{
  texfont_cache_first = NULL;
}

TexFont *
loadTxf(const char *url)
{
  TexFont *txf;
  
  httpOpen(url, txfLoadFont, &txf, THREAD_NO_BLOCK);
  return txf;
}

TexFont *
getTexfontFromCache(const char *url)
{
  TexfontCacheEntry *tf;

  for (tf = texfont_cache_first; tf ; tf = tf->next) {
    if (!strcmp(tf->url, url)) {
      /* texfont is in the cache */
      trace(DBG_ZV, "getTexfontFromCache: %s %d in cache", url, tf->num);
      return tf->txf;
    }
  }

  /* we must download the texfont now */
  /* new entry in cache */
  if ((tf = (TexfontCacheEntry *) calloc(1, sizeof(TexfontCacheEntry))) == NULL) {
    warning("getTexfontFromCache: can't alloc tf");
    return NULL;
  }
  tf->next = texfont_cache_first;
  texfont_cache_first = tf;
  strcpy(tf->url, url);
  tf->num = ++texfont_number;
  trace(DBG_ZV, "getTexfontFromCache: loading %s", url);
  tf->txf = loadTxf(tf->url);
  return tf->txf;
}

TexfontCacheEntry *
getTexfontEntryByUrl(const char *url)
{
  TexfontCacheEntry *tf;

  for (tf = texfont_cache_first; tf ; tf = tf->next) {
    if (!strcmp(tf->url, url)) {
      return tf; /* texfont is in the cache */
    }
  }
  return NULL; /* we must dowdload the texfont latter */
}

TexfontCacheEntry *
getTexfontEntryByNumber(int num)
{
  TexfontCacheEntry *tf;

  for (tf = texfont_cache_first; tf ; tf = tf->next) {
    if (tf->num == num) {
      return tf; /* texfont is in the cache */
    }
  }
  return NULL;
}

int getCurrentTexfontNumber(void)
{
  return texfont_number;
}


#endif /* !VRENGD */
