/*
 * Interface entre vreng et zbuffer/zrender
 * (c) 1996,1997 Fabrice Bellard
 */

#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "zv.h"
#include "texture.h"	/* initTextureCache */
#include "texfont.h"	/* initTexfontCache */

#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "world.h"	/* specialRendering */
#include "list.h"

#ifndef VRENGD


/* background color */
struct _bgcolor bgcolor;

/* camera */
static struct {
  float fovy, near, far;
} camera_proj;


/*
 * Camera handling
 */

/*
 * Object viewed, called by updateObjectIn3D (all objects)
 */
void setCameraPosition(M4 *cpos)
{
  ZVContext *c = &zv_context;

  c->camera_pos = *cpos;
}

/*
 * Observer view, called by updateCameraFromObject: initw.c user.c
 */
void setCameraProjection(float fovy, float near, float far)
{
  float left, right, bottom, top, ratio;
  int viewport[4], width, height;

  glGetIntegerv(GL_VIEWPORT, viewport);
  width = viewport[2];
  height = viewport[3];
  camera_proj.fovy = fovy;
  camera_proj.near = near;
  camera_proj.far = far;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  ratio= (float) width / (float) height;
  top = near * tan(fovy * M_PI_180);
  bottom = -top;
  right = top * ratio;
  left = -right;
  glFrustum(left, right, bottom, top, near, far);
  glTranslatef(0.0, 0.0, -near); /* orig -0.4 */
}

void setCameraPerspective(float fovy, float near, float far, float ratio)
{
   float left, right, bottom, top;

   top = near * tan(fovy * M_PI / 360.0);
   bottom = -top;
   right = top * ratio;
   left = -right;
   glFrustum(left, right, bottom, top, near, far);
}

/*
 * Rendering
 */

void M4_to_GL(float *gl_mat, M4 *vr_mat)
{
  float *glmat = gl_mat;

  for (int i=0; i < 4; i++)
    for (int j=0; j < 4; j++)
      *glmat++ = vr_mat->m[j][i];
} 

float * getGLMatrix(M4 *vr_mat)
{
  float mat[16], *glmat = mat;

  M4_to_GL(mat, vr_mat);
  return glmat;
} 

static
void eyePosition(float *gl_mat, ZVContext *c)
{
  glMatrixMode(GL_MODELVIEW);
  M4_to_GL(gl_mat, &c->camera_pos);
  glLoadMatrixf(gl_mat); 
}

void RenderInit(int quality)
{
  ZVContext *c = &zv_context;

  zv_context.quality = quality;
  c->first_solid = NULL;
  c->last_solid = NULL;
  memset((char *) &bgcolor, 0, sizeof(bgcolor)); /* black */

  glFrontFace(GL_CCW);
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_LIGHTING);
  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   
  /* texture cache */
  initTextureCache(zv_context.quality);
  /* texfont cache */
  initTexfontCache();

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
}

void renderMaterials(void)
{
  static float material_black[4] = {0, 0, 0, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT, material_black);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, material_black);
  glMaterialfv(GL_FRONT, GL_SPECULAR, material_black);
  glMaterialfv(GL_FRONT, GL_EMISSION, material_black);
}

static
void renderSolids(void)
{
  ZVContext *c = &zv_context;
  Solid *s;
  float gl_mat[16];

  for (s = c->first_solid; s != NULL; s = s->next) {
    if (s->visible) {
      if (s->displaylist == NULL) {
        trace(DBG_FORCE, "s->displaylist == NULL, sid=%d", s->id);
        continue;
      }
      renderMaterials();
      //PB glLoadName((int)s);
      glPopName();
      glPushName((int)s); //PB original glLoadName(s->displaylist[s->curframe]);
      glPushMatrix();
      M4_to_GL(gl_mat, &s->posmat);
      glMultMatrixf(gl_mat);
      glCallList(s->displaylist[s->curframe]);
      glPopMatrix();
    }
  }
  renderMaterials();
  specialRendering();		/* done by wmgt.c for each object */
}

static
void renderColorBackground(void)
{
  glClearColor(bgcolor.red, bgcolor.green, bgcolor.blue, bgcolor.alpha);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
}

static
void renderLights(void)
{
  //PD static float light_position0[4] = {1, 0, -1, 0};
  static float light_position0[4] = {1, 1, 2, 0};

  //PD static float light_diffuse0[4] = {0.8, 0.8, 0.8, 1};
  static float light_diffuse0[4] = {1.0, 1.0, 1.0, 1};
  //PD static float light_ambient0[4] = {0.4, 0.4, 0.4, 1};
  static float light_ambient0[4] = {0.0, 0.0, 0.0, 1};
  static float light_specular0[4] = {1, 1, 1, 1};

  static float light_position1[4] = {-0.5, 0.7, -1, 0};

  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient0);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse0);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular0);
  glLightfv(GL_LIGHT0, GL_POSITION, light_position0);
  glLightfv(GL_LIGHT1, GL_POSITION, light_position1);
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  //DAX glDisable(GL_LIGHT2);
}


void Render(void)
{
  ZVContext *c = &zv_context;
  float gl_mat[16];

  eyePosition(gl_mat, c);	/* eye position */
  updateTextures();		/* textures */
  renderColorBackground();	/* background */
  renderLights();		/* lights */
  renderSolids();		/* all other solids */
}

void RenderClose(void)
{
  closeTextureCache();
}

/* called by init3D (gui) */
void RenderSetWindow(int width, int height)
{
  trace(DBG_ZV, "setwindow w=%d h=%d", width, height);
  glViewport(0, 0, width, height);
  setCameraProjection(camera_proj.fovy, camera_proj.near, camera_proj.far);
}

/* get solid handle by its number */  
static
Solid * getSolidByNumber(int number, ZVContext *c)
{
  /* search for solids in solidlist */
  for (Solid * s = c->first_solid; s != NULL; s = s->next) {
    //PB original// if (number == s->displaylist[s->curframe]) {
    if (number == (int)s) {
      return s;
    }
  }
  /* search for solids handled by specialRendering */
  for (ObjectList * list = mobilelist; list != NULL; list = list->next) {
    if (number == (int)list)
      return list->pobject->soh;
  }
  for (ObjectList * list = stilllist; list != NULL; list = list->next) {
    if (number == (int)list)
      return list->pobject->soh;
  }
  return NULL;	/* no solid found in selection */
}

#define SELECT_BUF_SIZE (3*1024)

Solid * RenderGetSelection(int x, int y)
{
  ZVContext *c = &zv_context;
  int hits;
  int width, height;
  GLuint zmin; //PB original: int zmin
  int number;
  float left, right, bottom, top, ratio;
  int viewport[4];
  float gl_mat[16];
  unsigned int selectbuf[SELECT_BUF_SIZE], *psel;

  glGetIntegerv(GL_VIEWPORT, viewport);
  width = viewport[2];
  height = viewport[3];
  top = camera_proj.near * tan(camera_proj.fovy * M_PI_180);
  bottom = -top;
  ratio = (float) width / (float) height;
  right = top * ratio;
  left = -right;

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();

  /* reversing y axis */
  glFrustum(left + (x-0.5)*(right-left)/(width-1),
	    left + (x+0.5)*(right-left)/(width-1),
	    top - (y+0.5)*(top-bottom)/(height-1),
	    top - (y-0.5)*(top-bottom)/(height-1),
	    camera_proj.near,
	    camera_proj.far);
  glTranslatef(0.0, 0.0, -camera_proj.near); /* orig=-0.50 */

  /* selection mode */
  glSelectBuffer(SELECT_BUF_SIZE, selectbuf);
  glRenderMode(GL_SELECT);

  glInitNames();
  glPushName(0);

  eyePosition(gl_mat, c);	/* eye position */

  /* draw the solids in the selection buffer */
  renderSolids();

  /* we put the normal mode back */
  hits = glRenderMode(GL_RENDER);
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();

  /* parcours du selection buffer, determination du hit le plus proche */
  psel = selectbuf;
  zmin = 0xFFFFFFFF; //PB original: zmin=-1;
  number = -1;
  for (int i=0; i < hits; i++) {
    if (psel[1] < zmin) { //PB orginal: if (psel[2] < zmin)
      number = psel[3];
      zmin = psel[1]; //PB orginal: zmin = psel[2];  
    }
    psel += 3 + psel[0];
  }
  trace(DBG_ZV, "selection hits=%d zmin=%d number=%d", hits, zmin, number);

  if (number == 0)
    return ((Solid *) NULL);
  return getSolidByNumber(number, c);
}

/* Converts (x,y) screen coordinates into a vector representing the direction
 * of the line from the user's eyes to the point he clicked on */
void clickToVector(int x, int y, V3 *res)
{
  GLdouble res1, res2, res3;
  GLdouble res4, res5, res6;
  GLdouble modelMatrix[16];
  GLdouble projMatrix[16];
  int viewport[4];

  glGetIntegerv(GL_VIEWPORT, viewport);
  glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);

  gluUnProject(x, viewport[3]-y, 0.0, modelMatrix, projMatrix, viewport, &res1, &res2, &res3);
  gluUnProject(x, viewport[3]-y, 1.0, modelMatrix, projMatrix, viewport, &res4, &res5, &res6);

  res->v[0]  = res1 - res4;
  res->v[1]  = res2 - res5;
  res->v[2]  = res3 - res6;
}

#endif /* !VRENGD */
