/* Copyright (c) Mark J. Kilgard, 1994. */
/**
 * (c) Copyright 1993, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED 
 * Permission to use, copy, modify, and distribute this software for 
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation, and that 
 * the name of Silicon Graphics, Inc. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission. 
 *
 * THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
 * AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL SILICON
 * GRAPHICS, INC.  BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
 * SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
 * KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
 * LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
 * THIRD PARTIES, WHETHER OR NOT SILICON GRAPHICS, INC.  HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
 * POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * US Government Users Restricted Rights 
 * Use, duplication, or disclosure by the Government is subject to
 * restrictions set forth in FAR 52.227.19(c)(2) or subparagraph
 * (c)(1)(ii) of the Rights in Technical Data and Computer Software
 * clause at DFARS 252.227-7013 and/or in similar or successor
 * clauses in the FAR or the DOD or NASA FAR Supplement.
 * Unpublished-- rights reserved under the copyright laws of the
 * United States.  Contractor/manufacturer is Silicon Graphics,
 * Inc., 2011 N.  Shoreline Blvd., Mountain View, CA 94039-7311.
 *
 * OpenGL(TM) is a trademark of Silicon Graphics, Inc.
 */

#ifndef VRENGD

#include <GL/gl.h>

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "list.h"
#include "col.h"
#include "stars.h"


const WClass Stars::wclass(STARS_TYPE, "Stars", Stars::creator);


static
void newStar(Stars *po, int n, int d)
{
  po->stars[n].x[0] = (float)(rand() % po->maxpos - po->maxpos/2);
  po->stars[n].y[0] = (float)(rand() % po->maxpos - po->maxpos/2);
  po->stars[n].z[0] = (float) (rand() % po->maxpos + d);
  po->stars[n].x[1] = po->stars[n].x[0];
  po->stars[n].y[1] = po->stars[n].y[0];
  po->stars[n].z[1] = po->stars[n].z[0];
  if (rand() % 10 == 0) {
    po->stars[n].offsetX = (float) (rand() % 100 - 100 / 2);
    po->stars[n].offsetY = (float) (rand() % 100 - 100 / 2);
    po->stars[n].offsetR = (float) (rand() % 25 - 25 / 2);
  } else {
    po->stars[n].offsetX = 0.0;
    po->stars[n].offsetY = 0.0;
    po->stars[n].offsetR = 0.0;
  }
}

static
void rotatePoint(float *x, float *y, float rotation)
{
  float tx, ty;

  tx = *x * Cos(rotation) - *y * Sin(rotation);
  ty = *y * Cos(rotation) + *x * Sin(rotation);
  *x = tx;
  *y = ty;
}

static
void moveStars(Stars *po)
{
  float offset;
  int n;

  offset = po->speed * 60.0;

  for (n = 0; n < po->count; n++) {
    po->stars[n].x[1] = po->stars[n].x[0];
    po->stars[n].y[1] = po->stars[n].y[0];
    po->stars[n].z[1] = po->stars[n].z[0];
    po->stars[n].x[0] += po->stars[n].offsetX;
    po->stars[n].y[0] += po->stars[n].offsetY;
    po->stars[n].z[0] -= offset;
    po->stars[n].rotation += po->stars[n].offsetR;
    if (po->stars[n].rotation >= STARS_MAXANGLE)
      po->stars[n].rotation = 0.0;
  }
}

static
int pointStar(Stars *po, int n)
{
  float x0, y0;

  x0 = po->stars[n].x[0] * po->pos.x * 2 / po->stars[n].z[0];
  y0 = po->stars[n].y[0] * po->pos.y * 2 / po->stars[n].z[0];
  rotatePoint(&x0, &y0, po->stars[n].rotation);
  x0 += po->pos.x;
  y0 += po->pos.y;
  if (x0 >= 0.0 && x0 < po->pos.x * 2 && y0 >= 0.0 && y0 < po->pos.y * 2)
    return TRUE;
  else
    return FALSE;
}

static
void showStar(Stars *po, int n)
{
  float x0, y0, x1, y1;
  static float stars_white[4] = {1, 1, 1, 0};

  x0 = po->stars[n].x[0] * po->pos.x * 2 / po->stars[n].z[0];
  y0 = po->stars[n].y[0] * po->pos.y * 2 / po->stars[n].z[0];
  rotatePoint(&x0, &y0, po->stars[n].rotation);
  x0 += po->pos.x;
  y0 += po->pos.y;

  if (x0 >= 0.0 && x0 < po->pos.x * 2 && y0 >= 0.0 && y0 < po->pos.y * 2) {
    x1 = po->stars[n].x[1] * po->pos.x * 2 / po->stars[n].z[1];
    y1 = po->stars[n].y[1] * po->pos.y * 2 / po->stars[n].z[1];
    rotatePoint(&x1, &y1, po->stars[n].rotation);
    x1 += po->pos.x;
    y1 += po->pos.y;

#if !defined(WITH_TINYGL)	/* glLineWidth missing */
    glLineWidth(po->maxpos / 100.0 / po->stars[n].z[0] + 1.0);
#endif
    stars_white[1] = (float) ((po->warp - po->speed) / po->warp);
    stars_white[2] = (float) ((po->warp - po->speed) / po->warp);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, stars_white);
    if (fabs(x0 - x1) < 1.0 && fabs(y0 - y1) < 1.0) {
      glBegin(GL_POINTS);
      glVertex2f(x0, y0);
      glEnd();
    } else {
      glBegin(GL_LINES);
      glVertex2f(x0, y0);
      glVertex2f(x1, y1);
      glEnd();
    }
  }
}

static
void updateStars(Stars *po)
{
  int n;

  for (n = 0; n < po->count; n++) {
    if (po->stars[n].z[0] > po->speed ||
       (po->stars[n].z[0] > 0.0 &&
        po->speed < po->warp)) {
      if (pointStar(po, n) == FALSE)
        newStar(po, n, po->maxpos);
    }
    else
      newStar(po, n, po->maxpos);
  }
}

/* create from a fileline */
void Stars::creator(char *l)
{
  new Stars(l);
}

Stars::Stars(char *l)
{
  number = STARS_MAX;
  maxpos = STARS_MAXPOS;
  speed = STARS_SPEED;
  warp = STARS_WARP;

  l = parseName(l, this);
  l = parsePosition(l, this);
  if (l && isdigit((int) *l)) {
    number = atoi(l);
    l = strtok(NULL, SEP);
  }
  if (l && isdigit((int) *l)) {
    speed = (float) atof(l);
    l = strtok(NULL, SEP);
  }
  if (l && isdigit((int) *l)) {
    warp = (float) atof(l);
    l = strtok(NULL, SEP);
  }
  if (l && isdigit((int) *l))
    maxpos = atoi(l);

  nature.movable = VR_NO_ELEM_MOVE;
  nature.bbable = VR_NO_BB;
  nature.selectable = VR_NO_SELECTABLE;
  nature.renderable = VR_SPECIAL_RENDER;
  nature.collision = COL_NEVER;

  initializeObject(this, STARS_TYPE, VR_MOBILE);

  count = number / 2;
  if (pos.x <= 0)
    pos.x = STARS_WIDTH;
  if (pos.y <= 0)
    pos.x = STARS_HEIGHT;
  if (maxpos <= 0)
    maxpos = STARS_MAXPOS;
  if (warp <= speed)
    warp = STARS_WARP;
  move.perm_sec = 1;
  trace(DBG_WMGT, "number=%d speed=%.2f warp=%.2f maxpos=%d",
        number,
        speed,
        warp,
        maxpos);

  stars = (starRec *) calloc(1, number * sizeof(starRec));
  srand((unsigned int) time(NULL));

  for (int n = 0; n < number; n++)
    newStar(this, n, 100);	/* 100 ? */
}

/* system of equations handling permanent motion */
void Stars::changePermanent(float lasting)
{
  moveStars(this);
  updateStars(this);
}

void Stars::render()
{
  glPushMatrix();
  glTranslatef(0, 0, pos.z);
  glRotatef(RADIAN2DEGREE(pos.az), 0, 0, 1);
  glRotatef(RADIAN2DEGREE(pos.ax), 1, 0, 0);

  int n;
  for (n = 0; n < count; n++) {
    if (stars[n].z[0] > speed ||
       (stars[n].z[0] > 0.0 &&
        speed < warp))
      showStar(this, n);
  }
  glPopMatrix();
}

void Stars::quit()
{
  free(stars);
}

void starsInitFuncList(void) { }

#endif /* !VRENGD */
