#ifndef VRENGD

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "list.h"
#include "grid.h"	/* clearGrid, freeGrid */
#include "initobj.h"	/* initGenaralFuncList */
#include "parse.h"	/* parseCfg */
#include "names.h"	/* initObjectsName */
#include "move.h"	/* clearKeyTab */
#include "user.h"	/* USER_TYPE */
#include "link.h"	/* WRL_LINKED */

#include "zv.h"		/* parseGeometry */
#include "gui.h"	/* selectedObjectDeletion */
#include "channel.h"	/* Channel */
#include "netobj.h"	/* getMySsrc */
#include "helpers.h"	/* quittools */


// global variables
World *worlds = NULL;	// World list

// static variables
static User *plocaluser = NULL;
static u_int16 wid = 0;
#include "cfg.h"	// initial builtin configuration


World * allocWorld(void)
{
  World *pworld = (World *) calloc(1, sizeof(World));

  if (!pworld)
    fatal("can't alloc a world");
  if (worlds)
    pworld->next = worlds;
  worlds = pworld;
  return worlds;
}

// backWorld: quitWorld -> quitChannel -> newWorld(back) -> joinChannel(back)
World * backWorld(void)
{
  World *worldcurr;
  World *worldback;
  World *wp;

  if (!(worldcurr = worlds) || !(worldback = worlds->next))
    return NULL;

  quitWorld();		// quit current world first

  wp = worldback;
  while (wp->next != NULL)
    wp = wp->next;
  wp->next = worldcurr;
  worldcurr->next = NULL;
  worlds = worldback;

  if (worldcurr->prop == WRL_LINKED) {	// linked
    newWorld(worlds->url, NULL, VR_OLD);
    setChannelName(worldcurr->chan);
  }
  else {
    trace(DBG_IPMC, "back: quit chan=%s", worldcurr->chan);
    quitChannel(getCurrentChannel());
    newWorld(worlds->url, NULL, VR_OLD);
    trace(DBG_IPMC, "back: join chan=%s", worlds->chan);
    joinChannel(worlds->chan);	// join previous channel
    GuiUpdateWorld(worlds, VR_CURR);
  }
  return worlds;
}

// forwWorld: quitWorld -> quitChannel -> newWorld(back) -> joinChannel(back)
World * forwardWorld(void)
{
  World *worldcurr;
  World *worldforw;
  World *wp;

  if (!(worldcurr = worlds) || !worlds->next)
    return NULL;

  quitWorld();		// quit current world first

  wp = worldcurr;
  while ((worldforw = wp->next)->next != NULL)
    wp = wp->next;
  worldforw->next = worldcurr;
  wp->next  = NULL;
  worlds = worldforw;

  if (worlds->prop == WRL_LINKED) {	// linked
    newWorld(worlds->url, NULL, VR_OLD);
  }
  else {
    trace(DBG_IPMC, "forw: quit chan=%s", worldcurr->chan);
    quitChannel(getCurrentChannel());
    newWorld(worlds->url, NULL, VR_OLD);
    trace(DBG_IPMC, "forw: join chan=%s", worlds->chan);
    joinChannel(worlds->chan);	// join previous channel
    GuiUpdateWorld(worlds, VR_CURR);
  }
  return worlds;
}

// set local world name
void setLocalWorldName(const char *url)
{
  char *p, *q;
  char *tmpurl = strdup(url);

  if (!tmpurl) {
    trace(DBG_FORCE, "setLocalWorldName: can't strdup url=%s", url);
    return;
  }
  p = strrchr(tmpurl, '/') + 1;
  if ((q = strrchr(p, '.')) != NULL)
    *q = '\0';
  strcpy(worlds->name, p);
  free(tmpurl);
  trace(DBG_WMGT, "setLocalWorldName: %s", worlds->name);
}

// get current world name
const char * getCurrentWorldName(void)
{
  return worlds->name;
}

// get current world
World * getCurrentWorld(void)
{
  return worlds;
}

void declareJoinWorldToManager(const char *urlvre, const char *chan, const char *localusername)
{
  // TODO
}

void declareLeaveWorldToManager(const char *wname, const char *chan, const char *localusername)
{
  // TODO
}

// WMgt general initialization
void initWMgt(const char *urlvre, const char *localusername, const char *urlfront, const char *urlback)
{
  clearGrid();
  clearLists();
  initObjectsName();
  initGeneralFuncList();

  worlds->id = 1;	// initial world
  strcpy(worlds->url, urlvre);
  setLocalWorldName(urlvre);
  worlds->ptrgui = GuiAddWorld(worlds, VR_CURR);
  trace(DBG_INIT, "initWMgt: worldname = %s", getCurrentWorldName());

  // plocaluser: global variable
  worlds->plocaluser = User::createLocal(localusername, urlfront, urlback);
  plocaluser = worlds->plocaluser;	// temp

  //
  // initial loading of rendezvous world
  //
  wmgtlockstate = LOCK_TOBUILD;
  if (strlen(urlvre) != 0)
    httpOpen(urlvre, readCfg, NULL, THREAD_NO_BLOCK);
  else {
    trace(DBG_FORCE, "world_initial builtin");
    parseCfgLines(world_initial, sizeof(world_initial));
    wmgtlockstate = LOCK_DOWNLOADED;
  }
  updateObjectIn3D(worlds->plocaluser);
  updateCameraFromObject(worlds->plocaluser);
  declareJoinWorldToManager(urlvre, getCurrentChannelName(), localusername);
}

// WObject destructor
WObject::~WObject()
{
  if (soh) {
    if (soh->object == this)
      soh->object = NULL;
    //else if (soh->id == 1)	// Myself
    //  return;
    else
      trace(DBG_FORCE, "delete: bad link soh->object=%p this=%p name=%s", soh->object, this, this->name.class_name);
  }
}

// end WMgt
void quitWMgt(void)
{
  if (worlds->plocaluser && worlds->plocaluser->name.world_name)
    declareLeaveWorldToManager(worlds->plocaluser->name.world_name,
                               getCurrentChannelName(),
                               worlds->plocaluser->name.instance_name);
  freeGrid();

  ObjectList *pelm = NULL;

  for (pelm = invisiblelist; pelm; ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject)
      delete pelm->pobject;
    pelm = pelm->next;
    free(tmppelm);
  }
  for (pelm = stilllist; pelm; ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject) {
      deleteSolidFromList(pelm->pobject->soh);		// 3D
      pelm->pobject->soh = NULL;			// 3D
      selectedObjectDeletion(pelm->pobject->soh);	// GUI
      delete pelm->pobject;
    }
    pelm = pelm->next;
    free(tmppelm);
  }
  for (pelm = mobilelist; pelm && (pelm->pobject != worlds->plocaluser); ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject) {
      deleteSolidFromList(pelm->pobject->soh);		// 3D
      pelm->pobject->soh = NULL;			// 3D
      selectedObjectDeletion(pelm->pobject->soh);	// GUI
      if (!(pelm->pobject->noh.permanent))
        declareObjDeletion(&(pelm->pobject->noh));
      deleteNetObject((NetObject*) pelm->pobject);	// NET
      delete pelm->pobject;
    }
    pelm = pelm->next;
    free(tmppelm);
  }

  // destroy local user
  if (worlds->plocaluser) {
    delete worlds->plocaluser;
    worlds->plocaluser = NULL;
  }

  // quit all channels
  closeChannel(getCurrentChannel());
  closeChannel(getManagerChannel());

  // quit all tools
  quitalltools();
}

// initialization new world
void newWorld(const char *urlvre, const char *chanstr, boolean isNewWorld)
{
  clearGrid();
  clearLists();
  clearKeyTab(worlds->plocaluser);

  if (isNewWorld) {
    allocWorld();
    worlds->prop = 0;
    worlds->id = ++wid;
  }
  strcpy(worlds->url, urlvre);
  if (chanstr != NULL)
    setChannelName(chanstr);
  setLocalWorldName(urlvre);
  worlds->plocaluser = plocaluser;
  worlds->plocaluser->name.world_name = getCurrentWorldName();

  if (worlds->ptrgui)
    GuiUpdateWorld(worlds, VR_CURR);
  else
    // WARNING! At this step we don't know the channel need by GuiAddWorld
    // May be do an GuiUpdateWorld(worlds, 0) after a joinChannel
    worlds->ptrgui = GuiAddWorld(worlds, VR_CURR);

  // place the local user
#ifdef RANDOM_ORIGIN
  worlds->plocaluser->pos.x = (float) drand48() * 2 -1;
  worlds->plocaluser->pos.y = (float) drand48() * 2 -1;
  worlds->plocaluser->pos.az = (float) drand48() * M_2PI;
#else
  if (!worlds->plocaluser->entry) {
    worlds->plocaluser->pos.x = 0;
    worlds->plocaluser->pos.y = 0;
    worlds->plocaluser->pos.az = 0;
  }
  worlds->plocaluser->entry = 0;
#endif

  worlds->plocaluser->pos.z = worlds->plocaluser->height + 0.15;
  worlds->plocaluser->move.perm_sec = -1;

  updateObjectIn3D(worlds->plocaluser);
  updateBB(worlds->plocaluser);
  mobilelist = addObjectToList(worlds->plocaluser, mobilelist);
  insertObjectIntoGrid(worlds->plocaluser);

  // get the description file
  wmgtlockstate = LOCK_TOBUILD;
  httpOpen(urlvre, readCfg, NULL, THREAD_NO_BLOCK);
  trace(DBG_WMGT, "newWorld: %s downloaded", urlvre);

  GuiUpdateUser(worlds->plocaluser);

  updateCameraFromObject(worlds->plocaluser);

  // declare user to network
  createNetObject((NetObject *) worlds->plocaluser, VR_VOLATILE);
#if 0
  declareObjCreation(&(worlds->plocaluser->noh));
#endif
}

// quit a world
void quitWorld(void)
{
  trace(DBG_WMGT, "quitWorld: %s", worlds->plocaluser->name.world_name);

  declareLeaveWorldToManager(worlds->plocaluser->name.world_name, getCurrentChannelName(), worlds->plocaluser->name.instance_name);

  freeGrid();

  // delete objects
  ObjectList *pelm = NULL;

  for (pelm = invisiblelist; pelm ; ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject) {
      pelm->pobject->quit();
      delete pelm->pobject;
    }
    pelm = pelm->next;
    free(tmppelm);
  }
  invisiblelist = NULL;
  for (pelm = stilllist; pelm ; ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject) {
      deleteSolidFromList(pelm->pobject->soh);		// 3D
      pelm->pobject->soh = NULL;			// 3D
      selectedObjectDeletion(pelm->pobject->soh);	// GUI
      pelm->pobject->quit();
      delete pelm->pobject;
    }
    pelm = pelm->next;
    free(tmppelm);
  }
  stilllist = NULL;
  for (pelm = mobilelist; pelm && pelm->pobject != worlds->plocaluser; ) {
    ObjectList *tmppelm = pelm;
    if (pelm->pobject) {
      deleteSolidFromList(pelm->pobject->soh);		// 3D
      pelm->pobject->soh = NULL;			// 3D
      selectedObjectDeletion(pelm->pobject->soh);	// GUI
      pelm->pobject->quit();
      if (!(pelm->pobject->noh.permanent))
        declareObjDeletion(&(pelm->pobject->noh));
      deleteNetObject((NetObject*) pelm->pobject);	// NET
      delete pelm->pobject;
    }
    pelm = pelm->next;
    free(tmppelm);
  }
  mobilelist = NULL;

  // delete solids
  SolidClose();		// solid_id = 1 (localuser preserved)

  // update GUI
  if (worlds->ptrgui)
    GuiUpdateWorld(worlds, VR_OLD);

  declareObjDeletion(&(worlds->plocaluser->noh));
  deleteNetObject(&(worlds->plocaluser->noh));
  if (worlds->prop == WRL_LINKED)	// linked
    return;
  quittools();
}

#endif // !VRENGD
