#ifndef IFCSERVER_H
#define IFCSERVER_H

// VRENG stuff
#define IFCSERVER_TYPE	43
#define IFCSERVER_NAME	"IfcServer"
#define IFCSERVER_INIT  ifcServerInitFuncList

// Socket constants
#define IFC_SOCK_CLOSED  0
#define IFC_SOCK_RCVOPEN 1
#define IFC_SOCK_CONNECT 2
#define IFC_SOCK_OPEN    3

#define IFC_PING_WAIT    200
#define IFC_PROTO   0xABCD
#define IFC_VERSION 0

// Ports numbers
#define IFC_SERVER_PORT	4151
#define IFC_LOCAL_PORT	4151

// Packet max size
#define IFC_MAX_PACKET		1024
#define IFC_HDR_LEN		20
#define IFC_HDR_DATALEN_IDX	9
#define IFC_HDR_SSRC_IDX	3

// Data sizes
#define IFC_INT8	1
#define IFC_INT16	2
#define IFC_INT32	4

// Message types
#define IFC_MSGT_CTRL	0
#define IFC_MSGT_CLICK	1
#define IFC_MSGT_ADD	2
#define IFC_MSGT_DEL	3
#define IFC_MSGT_POS	4
#define IFC_MSGT_ISEC	5
#define IFC_MSGT_QUERY	6
#define IFC_MSG_TYPES_STRINGS Control Click Add Delete Position Intersect Query

// Message val
// -ctrl
#define IFC_MSGV_REGISTER	0
#define IFC_MSGV_UNREGISTER	1
#define IFC_MSGV_INITIATE	2
#define IFC_MSGV_TERMINATE	3
#define IFC_MSGV_UPDATE		4
#define IFC_MSGV_PING		5
#define IFC_MSG_CTRL_STRINGS Register Unregister Inititate Terminate Update Ping

// -click
#define IFC_MSGV_CLICK  0
#define IFC_MSG_CLICK_STRINGS Click

// - add / del
#define IFC_MSGV_CIRCLE 0
#define IFC_MSGV_LINE   1
#define IFC_MSGV_LOOP   2
#define IFC_MSGV_FILL   3
#define IFC_MSGV_DELALL 4
#define IFC_MSG_DRAW_STRINGS Circle Line Loop Fill DeleteAll

// position
#define IFC_MSGV_ASK	0
#define IFC_MSGV_SET	1
#define IFC_MSGV_UPD	2
#define IFC_MSGV_ERROR	3
#define IFC_MSG_POS_STRINGS Ask Set Update Error

// query
#define IFC_MSGV_QTYPE		0
#define IFC_MSGV_QANS		1
#define IFC_MSG_QUERY_STRINGS TypeQuery QueryAnswer

// Intersection
#define IFC_MSGV_ISECIN		0
#define IFC_MSGV_ISECOUT	1
#define IFC_MSG_ISEC_STRINGS GoingIn GoingOut

#include "ifc.h"

typedef int8_t int8;

// in-out socket holder
typedef struct _ifcSocket {
  // Socket endpoints
  int listenPort;
  int destPort;
  struct sockaddr_in *destAddr;
  // Connection state
  int state;
  // Socket handles
  int recv;
  int send;
} ifcSocket;

// Packet header
// If you update this structure, do not
// forget to update the IFC_HDR_LEN, IFC_HDR_SSRC_IDX and
// IFC_HDR_DATALEN_IDX defines.
// IFC_HDR_LEN is the total length (in 8bit ints, no padding)
// of this structure, and IFC_HDR_DATALEN_IDX is the position
// of the 16bit data length field within the structure.
// IFC_HDR_SSRC_IDX is the same, but to the 32bit app_ssrc field.
typedef struct _ifcHeader {
  int16 proto;    // protocol id
  int8  version;  // protocol version
  int32 app_ssrc; // application's SSRC
  int8  msg_type; // Message type
  int8  msg_val;  // Message id
  int16 data_len;  // Data length
  u_int8 obj_type;  // Sender type
  u_int32 src_id;  // Sender ssrc
  u_int16 port_id;  // Sender port id
  u_int16 obj_id;  // Sender obj id
} ifcHeader;

/**
 * IfcMessage class.
 * This class is used to receive/transmit data between
 * Vreng and the external server.
 */
class IfcMessage {
public:
  // Outgoing constructor
  // WObject *sender should be set to the object sending the data
  // int type is the message type (IFC_MSGT_*)
  // int val  is the message value (subtype) (IFC_MSGV_*)
  IfcMessage(WObject *sender, int type, int val);

  // This constructor is used by IfcServer for special update
  // messages. The additional ssrc int is used to force
  // the message header's ssrc to something else than the current
  // value.
  IfcMessage(WObject *sender, int ssrc, int type, int val);

  // Incoming constructor
  // Rebuilds a message from the data.
  // int8 *data should point to a valid memory area, of lenght
  // int datalen. The data is copied into an internal buffer
  // so the orginal data can be freed once the message is constructed.
  IfcMessage(int8 *data, int datalen);

  // Destructor
  // Frees the internal buffer
  ~IfcMessage();

  // Returns this message's header
  ifcHeader getHeader(void);

  // Checks wether this message is for the object *po
  boolean isForObject(WObject *po);

  // Outgoing message methods
  // Simple data types
  void put8( int val);		// signed 8bit
  void put16(int val);		// signed 16bit
  void put32(int val);		// signed 32bit
  // Object identifier
  void putOID(WObject *po);	// 8bit type, 32bit src id, 16bit port id, 16bit obj id
  // Object position
  void putPos(WObject *po);	// 6*32bit ((x,y,z),(az,ax,0))
  // String
  void putStr(char *str);	// 16bit length, followed by the caracter array

  // Used to transmit the message on a socket.
  // The return value is the binary representation of this message,
  // and the total length of that array is stored in *len
  int8 *toBytes(int *len);

  // Incoming methods
  // Simple data types
  int8  read8();
  int16 read16();
  int32 read32();
  // 2D point
  V3 readPoint2D();		// 2*32bit
  // 3D point
  V3 readPoint3D();		// 3*32bit
  // Message header
  ifcHeader readHeader();	// header struct
 
  // Data availability
  // returns TRUE if there are at least size bytes left
  boolean hasData(int size);	

private:
  // buffer
  int8 *data;
  // header
  ifcHeader header;
  // sender
  WObject *sender;
  // current position within the buffer
  int cursor;
  // length of the buffer
  int maxlen;
 
  // Write the header
  void putHeader();
  // create the buffer and create the header
  void setup(WObject *, int, int, int);
};

/**
 * IfcServer class
 * 
 * This Vreng object class is used to manage
 * communications between Vreng Ifc objects
 * and an external server.
 * There should be only one IfcServer per scene.
 *
 * All the Ifc objects will want to send and
 * receive data from the external server. To avoid
 * having each Ifc object deal with its own sockets,
 * the IfcServer centralizes all the work.
 * 
 * The class keeps a single instance of IfcServer
 * running, and all the methods the Ifc objects
 * use are static and delegate to that singelton.
 */
class IfcServer: public WObject {
public:
  // Controler's hostname
  char host[128];
  // This app's ssrc
  u_int32 ssrc;
  // Controler's listen port
  int serverPort;
  // Local listen port
  int localPort;

  static const WClass wclass;   // class variable
  virtual const WClass* getWClass() {return &wclass;};  //virtual inst. method

  // Constructor
  IfcServer(char *l);
  // Destructor
  virtual ~IfcServer();

  // Method used by Vreng to instanciate this class
  static void (creator)(char *l);

  // Used to scan the sockets for incoming messages
  virtual void render();
 
  // Used to notify the server and close the sockets
  virtual void quit();
  static void stop();

  // Send and receive stuff
  static IfcMessage *getData(WObject *po);
  static int sendData(IfcMessage *msg);
  static int sendCommand(WObject *po, int val);

  // Application control stuff
  static void startApp(Ifc *);
  static void stopApp(Ifc *);

  // Returns the current running instance, NULL if there is none.
  static IfcServer *getServer(void);

private:
  // Sets the current running instance
  static void setServer(IfcServer *server);
  // The singelton instance of this class
  static IfcServer *server;
  // The two-way socket
  ifcSocket *sock;

  // The last message received
  IfcMessage *lastMessage;
  // 'Time' since last ping
  int lastping;

  // Open the sockets
  void start();
  // Send a ping to the server
  void ping();
};

void ifcCopyV3(V3 *, V3);

// Socket functions
ifcSocket *ifcInitSocket(int listenPort, char *destHost, int destPort);
int ifcOpenSocket(ifcSocket *sock);
ifcSocket *ifcCloseSocket(ifcSocket *sock);

// Utilities
int ifcTestSendSocket(ifcSocket *sock);
int isSockOpen(ifcSocket *sock);
int isSockClosed(ifcSocket *sock);

// VRENG initialization function not used
void IFCSERVER_INIT(void);

#endif // IFCSERVER_H
