/*
 *
 *  VSTRING Library supporting structures and functions
 *  (c) Vladi Belperchinov-Shabanski "Cade" <cade@biscom.net> 1998-2000
 *  Distributed under the GPL license, see end of this file for full text!
 *
 */

#ifndef _VSTRLIB_H_
#define _VSTRLIB_H_

#include <assert.h>
#ifndef ASSERT
#define ASSERT assert
#endif

#define RT_ASSERT(expr) \
  ((void)((expr)?0:(__assert_fail(__STRING(expr), \
    __FILE__, __LINE__, __ASSERT_FUNCTION), 0)))

#include "vstring.h"

/***************************************************************************/
/* GLOBALS
/***************************************************************************/

#define VARRAY_BLOCK_SIZE 2048

class VTrie; /* forward */

/***************************************************************************/
/* VARRAY
/***************************************************************************/

class VArray
{
  char**    _data;
  int       _size;
  int       _count;

  unsigned int       _min_len;
  unsigned int       _max_len;

  String    _ret_str;

  void resize( int new_size );

  void q_sort( int lo, int hi );

  public:

  VArray();
  VArray( const VArray& arr );
  VArray( const VTrie& tr );
  ~VArray();

  int count() { return _count; }
  int max_len() { return _max_len; }
  int min_len() { return _min_len; }

  void ins( int n, const char* s );
  void del( int n );
  void set( int n, const char* s );
  void add( const char* s );
  const char* get( int n );
  const char* get2( int n );

  void zap();

  int push( const char* s );
  const char* pop();

  int unshift( const char* s );
  const char* shift();

  int merge( VTrie *tr ); // return new elements count
  int merge( VArray *arr ); // return new elements count

  int fload( const char* fname ); // return 0 for ok
  int fsave( const char* fname ); // return 0 for ok
  int fload( FILE* f ); // return 0 for ok
  int fsave( FILE* f ); // return 0 for ok

  void sort();    // sort
  void reverse(); // reverse elements order
  void shuffle(); // randomize element order

  void split( const char* re, const char* str );
  void join( const char* glue, char* target );
  void join( const char* glue, String& target );

  const char* operator []( int n )
    { return get( n ); }
  const VArray& operator = ( const VArray& arr )
    { zap(); merge( (VArray*)&arr ); return *this; };
  const VArray& operator = ( const VTrie& tr )
    { zap(); merge( (VTrie*)&tr ); return *this; };
};

/***************************************************************************/
/* VTRIE
/***************************************************************************/

struct VTrieNode
{
  VTrieNode() { next = down = NULL; c = 0; data = NULL; }

  VTrieNode *next;
  VTrieNode *down;
  char      c;
  void      *data;
};

class VTrie
{
  VTrieNode *root;
  int _count;
  int _depth;
  int _nodes;

  void zap_node( VTrieNode *node );
  void del_node( VTrieNode *parent, VTrieNode *node, const char* key );
  void key_node( VTrieNode *node,  VArray* arr );
  void print_node( VTrieNode* node );

  String temp_key; /* used for keys() */

  public:

  VTrie();
  VTrie( const VArray& arr );
  VTrie( const VTrie& tr );
  ~VTrie();

  int count() { return _count; }
  int depth() { return _depth; }
  int nodes() { return _nodes; }

  int set( const char* key, const char* data );
  int del( const char* key );
  const char* get( const char* key );
  int get( const char* key, char* data );

  int exist( const char* key ) // return != 0 if key exist (with data)
      { return get( key ) != NULL; }

  void zap(); // delete all key+data pairs
  int keys( VArray* arr ); // return keys count

  int merge( VTrie *tr ); // return new keys count
  int merge( VArray *arr ); // return new keys count

  void print() { print_node( root ); }

  int fload( const char* fname ); // return 0 for ok
  int fsave( const char* fname ); // return 0 for ok
  int fload( FILE* f ); // return 0 for ok
  int fsave( FILE* f ); // return 0 for ok

  const char* operator []( const char* key )
    { return get( key ); }
  const VTrie& operator = ( const VArray& arr )
    { zap(); merge( (VArray*)&arr ); return *this; };
  const VTrie& operator = ( const VTrie& tr )
    { zap(); merge( (VTrie*)&tr ); return *this; };
};

#endif /* _VSTRLIB_H_ */
