#include "vstrlib.h"

/***************************************************************************/
/* VARRAY
/***************************************************************************/

  VArray::VArray()
  {
    _data = NULL;
    _size = 0;
    _count = 0;
    _max_len = 0;
    _min_len = (unsigned int)-1; /* should underflow to max value :) */
    _ret_str = NULL;
  }

  VArray::VArray( const VArray& arr )
  {
    _data = NULL;
    _size = 0;
    _count = 0;
    _max_len = 0;
    _min_len = (unsigned int)-1; /* should underflow to max value :) */
    _ret_str = NULL;
    *this = arr;
  }

  VArray::VArray( const VTrie& tr )
  {
    _data = NULL;
    _size = 0;
    _count = 0;
    _max_len = 0;
    _min_len = (unsigned int)-1; /* should underflow to max value :) */
    _ret_str = NULL;
    *this = tr;
  }

  VArray::~VArray()
  {
    zap();
  }

  void VArray::resize( int new_size )
  {
    RT_ASSERT( new_size >= 0 );
    if ( new_size == 0 )
      {
      delete _data;
      _data = NULL;
      _size = 0;
      _count = 0;
      return;
      }
    new_size  = new_size / VARRAY_BLOCK_SIZE + (new_size % VARRAY_BLOCK_SIZE != 0);
    new_size *= VARRAY_BLOCK_SIZE;
    if ( new_size == _size ) return;
    char** new_data = new (char*)[ new_size ];
    RT_ASSERT( new_data );
    memset( new_data, 0, new_size * sizeof(char*) );
    if ( _data )
      {
      memcpy( new_data, _data,
              (_size < new_size ? _size : new_size) * sizeof(char*) );
      delete _data;
      }
    _size = new_size;
    _data = new_data;
  }

  void VArray::ins( int n, const char* s )
  {
    RT_ASSERT( n >= 0 && n <= _count );
    if ( _count == _size )
      resize( _size + 1 );
    _count++;
    for( int z = n + 1; z < _count; z++ )
      _data[z] = _data[z-1];
    int sl = strlen( s );
    _data[n] = new char[ sl+1 ];
    strncpy( _data[n], s, sl );
    _data[n][sl] = 0;
    if ( sl < _min_len )  _min_len = sl;
    if ( sl > _max_len )  _max_len = sl;
  }

  void VArray::del( int n )
  {
    if ( n < 0 || n >= _count ) return;
    delete _data[n];
    memmove( &_data[0] + n, &_data[0] + n + 1, 
             ( _count - n ) * sizeof(char*) );
    _count--;
    if ( _size - _count > VARRAY_BLOCK_SIZE )
      resize( _count );
  }

  void VArray::set( int n, const char* s )
  {
    RT_ASSERT( n >= 0 && n < _count );
    delete _data[n];
    int sl = strlen( s );
    _data[n] = new char[ sl+1 ];
    strncpy( _data[n], s, sl );
    _data[n][sl] = 0;
  }

  const char* VArray::get( int n )
  {
    if ( n < 0 || n >= _count )
      return NULL;
    else
      return _data[n];
  }

  const char* VArray::get2( int n )
  {
    return _data[n];
  }

  void VArray::add( const char* s )
  {
    ins( _count, s );
  }

  void VArray::zap()
  {
    if ( _count )
      for( int z = 0; z < _count; z++ )
        if ( _data[z] )
          delete _data[z];
    resize( 0 );

    _data = NULL;
    _size = 0;
    _count = 0;
    _max_len = 0;
    _min_len = (unsigned int)-1; /* should underflow to max value :) */
    _ret_str = "";
  }

  int VArray::push( const char* s )
  {
    add( s );
    return _count;
  }

  const char* VArray::pop()
  {
    if ( _count == 0 ) return NULL;
    _ret_str = get( _count - 1 );
    del( _count - 1 );
    return _ret_str.data();
  }

  int VArray::unshift( const char* s )
  {
    ins( 0, s );
    return _count;
  }

  const char* VArray::shift()
  {
    if ( _count == 0 ) return NULL;
    _ret_str = get( 0 );
    del( 0 );
    return _ret_str.data();
  }

  int VArray::merge( VTrie *tr )
  {
    VArray va;
    int cnt = tr->keys( &va );
    for( int z = 0; z < cnt; z++ )
      {
      const char* key = va[z];
      add( key );
      add( tr->get( key ) );
      }
    return _count;
  };

  int VArray::merge( VArray *arr )
  {
    int cnt = arr->count();
    for( int z = 0; z < cnt; z++ )
      add( arr->get( z ) );
    return _count;
  };

  int VArray::fload( const char* fname )
  {
    FILE* f = fopen( fname, "rt" );
    if (!f) return 1;
    int r = fload( f );
    fclose(f);
    return r;
  };

  int VArray::fsave( const char* fname )
  {
    FILE* f = fopen( fname, "wt" );
    if (!f) return 1;
    int r = fsave( f );
    fclose(f);
    return r;
  };

  int VArray::fload( FILE* f )
  {
    char buf[1024];
    String str;
    while( fgets( buf, sizeof(buf)-1, f ) )
      {
      str += buf;
      if ( str_get_ch( str, -1 ) != '\n' ) continue;
      str_trim_right( str, 1 );
      add( str );
      str = "";
      }
    return 0;  
  };

  int VArray::fsave( FILE* f )
  {
    for( int z = 0; z < _count; z++ )
      {
      size_t len = strlen( get(z) );
      if ( fwrite( get(z), 1, len, f ) != len ) return 2;
      if ( fwrite( "\n", 1, 1, f ) != 1 ) return 2;
      }
    return 0;
  };

  void VArray::sort()
  {
    if ( count() > 1 )
      q_sort( 0, count() - 1 );
  };

  void VArray::q_sort( int lo, int hi )
  {
    int m, l, r;
    const char* v;

    m = ( hi + lo ) / 2;
    v = _data[m];
    l = lo;
    r = hi;

    do
      {
      while( (l <= hi) && (strcmp(_data[l],v) < 0) ) l++;
      while( (r >= lo) && (strcmp(v,_data[r]) < 0) ) r--;
      if ( l <= r )
        {
        char *t;
        t = _data[l];
        _data[l] = _data[r];
        _data[r] = t;
        l++;
        r--;
        }
      }
    while( l <= r );

    if ( lo < r ) q_sort( lo, r );
    if ( l < hi ) q_sort( l, hi );
  };

  void VArray::reverse()
  {
    int m = _count / 2;
    for( int z = 0; z < m; z++ )
      {
      char *tmp;
      tmp = _data[z];
      _data[z] = _data[_count-1-z];
      _data[_count-1-z] = tmp;
      }
  };

  void VArray::shuffle() /* Fisher-Yates shuffle */
  {
    int i = _count - 1;
    while( i >= 0 )
      {
      int j = rand() % ( i + 1 );
      char *tmp;
      tmp = _data[i];
      _data[i] = _data[j];
      _data[j] = tmp;
      i--;
      }
  };

  void VArray::split( const char* re, const char* str )
  {
  };

  void VArray::join( const char* glue, char* target )
  {
    target[0] = 0;
    for( int z = 0; z < _count-1; z++ )
      {
      strcat( target, get( z ) );
      strcat( target, glue );
      }
    strcat( target, get( _count-1 ) );
  };

  void VArray::join( const char* glue, String& target )
  {
    target = "";
    for( int z = 0; z < _count-1; z++ )
      {
      target += get( z );
      target += glue;
      }
    target += get( _count-1 );
  };


/***************************************************************************/
/* VTRIE
/***************************************************************************/

  VTrie::VTrie()
  {
    _count = 0;
    _depth = 0;
    _nodes = 0;
    root = new VTrieNode();
    _nodes++;
  }

  VTrie::VTrie( const VArray& arr )
  {
    _count = 0;
    _depth = 0;
    _nodes = 0;
    root = new VTrieNode();
    _nodes++;
    *this = arr;
  };

  VTrie::VTrie( const VTrie& tr )
  {
    _count = 0;
    _depth = 0;
    _nodes = 0;
    root = new VTrieNode();
    _nodes++;
    *this = tr;
  };


  VTrie::~VTrie()
  {
    zap_node( root );
  }

  void VTrie::zap_node( VTrieNode *node )
  {
    if ( node->next ) zap_node( node->next );
    if ( node->down ) zap_node( node->down );
    if ( node->data ) free( node->data );
    delete node;
    _nodes--;
  }

  void VTrie::del_node( VTrieNode *parent, VTrieNode *node, const char* key )
  {
    ASSERT( key[0] );
    if ( key[1] == 0 )
      {
      if ( key[0] == node->c )
        {
        if ( node->data )
          free( node->data );
        node->data = NULL;
        }
      else
        return; /* not found */
      }
    else
      {
      if ( key[0] == node->c )
        {
        if ( node->down )
          del_node( node, node->down, key+1 );
        else
          return; /* not found */
        }
      else
        {
        if ( node->next )
          del_node( node, node->next, key );
        else
          return; /* not found */
        }
      }

      if ( node->data ) return;
      if ( node->down ) return;
      if ( node->next )
        {
        if ( parent->down == node )
          parent->down = node->next;
        else
          parent->next = node->next;
        }
      else
        {
        if ( parent->down == node )
          parent->down = NULL;
        else
          parent->next = NULL;
        }
      delete node;
      _nodes--;
  }

  void VTrie::key_node( VTrieNode *node,  VArray* arr )
  {
    int kl = str_len( temp_key );
    if ( node->c ) str_add_ch( temp_key, node->c );
    if ( node->data ) arr->add( temp_key );
    if ( node->down ) key_node( node->down, arr );
    str_sleft( temp_key, kl );
    if ( node->next ) key_node( node->next, arr );
  };

  int VTrie::set( const char* key, const char* data )
  {
    if ( !key || !key[0] ) return _count;
    int sl = strlen( key );
    if ( sl > _depth )
      _depth = sl;

    VTrieNode *current = root;
    const char* keyp = key;
    while( *keyp )
      {
      VTrieNode *parent = current;
      current = current->down;
      if ( !current )
        {
        current = new VTrieNode();
        parent->down = current;
        current->c = *keyp;
        _nodes++;
        }
      else
        {
        while( current && current->c != *keyp )
          {
          parent = current;
          current = current->next;
          }
        if ( !current )
          {
          current = new VTrieNode();
          parent->next = current;
          current->c = *keyp;
          _nodes++;
          }
        }
      keyp++;
      }
    if ( current->data )
      free( current->data );
    else
      _count++;
    current->data = strdup( data );
    return _count;
  };

  const char* VTrie::get( const char* key )
  {
    if ( !key || !key[0] ) return NULL;

    VTrieNode *current = root;
    const char* keyp = key;
    while( *keyp )
      {
      current = current->down;
      if ( !current )
        {
        return NULL;
        }
      else
        {
        while( current && current->c != *keyp )
          {
          current = current->next;
          }
        if ( !current )
          {
          return NULL;
          }
        }
      keyp++;
      }
    if ( current )
      return (const char*)current->data;
    return NULL;
  };

  int VTrie::get( const char* key, char* data )
  {
    const char* ps = get( key );
    if ( !ps ) return 0;
    strcpy( data, ps );
    return 1;
  };

  int VTrie::del( const char* key )
  {
    if ( !key || !key[0] ) return 0;
    if ( !root->down ) return 0;
    del_node( root, root->down, key );
  };

  void VTrie::zap()
  {
    zap_node( root );
    root = new VTrieNode();
    _nodes++;
  };

  int VTrie::keys( VArray* arr )
  {
    arr->zap();
    temp_key = "";
    key_node( root, arr );
    return arr->count();
  };

  int VTrie::merge( VTrie *tr )
  {
    VArray arr;
    int cnt = tr->keys( &arr );
    for( int z = 0; z < cnt; z++ )
      set( arr[z], tr->get( arr[0] ) );
    return _count;
  };

  int VTrie::merge( VArray *arr )
  {
    while( arr->get( 0 ) )
      {
      if ( arr->get( 1 ) )
        {
        set( arr->get( 0 ), arr->get( 1 ) );
        arr->del( 0 );
        arr->del( 0 );
        }
      else
        {
        set( arr->get( 0 ), "" );
        arr->del( 0 );
        }
      }
    return _count;
  };

  int VTrie::fload( const char* fname )
  {
    FILE* f = fopen( fname, "rt" );
    if (!f) return 1;
    int r = fload( f );
    fclose(f);
    return r;
  };

  int VTrie::fsave( const char* fname )
  {
    FILE* f = fopen( fname, "wt" );
    if (!f) return 1;
    int r = fsave( f );
    fclose(f);
    return r;
  };

  int VTrie::fload( FILE* f )
  {
    VArray arr;
    int r = arr.fload( f );
    if ( r == 0 )
      merge( &arr );
    return r;
  };

  int VTrie::fsave( FILE* f )
  {
    VArray arr;
    arr.merge( this );
    return arr.fsave( f );
  };

  void VTrie::print_node( VTrieNode* node )
  {
    if (!node) return;
    printf( "---------------------------------\n" );
    printf( "this = %p\n", node );
    printf( "key  = %c\n", node->c );
    printf( "next = %p\n", node->next );
    printf( "down = %p\n", node->down );
    printf( "data = %s\n", node->data );
    print_node( node->next );
    print_node( node->down );
  };

/***************************************************************************/
/* EOF
/***************************************************************************/

