/*
			    Print Window

	For querying user about print inputs, uses GL and the GtkGLArea
	Widget to display print preview.
 */

#ifndef PRINTWIN_H
#define PRINTWIN_H


#include <sys/types.h>
#include <GL/gl.h>
#include <gtk/gtk.h>

#ifndef gbool
# define gbool	gboolean
#endif


/*
 *	Print window state flags:
 */
#define PRINT_WIN_FLAG_MOD_ALT		(1 << 1)
#define PRINT_WIN_FLAG_MOD_CTRL		(1 << 2)
#define PRINT_WIN_FLAG_MOD_SHIFT	(1 << 3)

#define PRINT_WIN_FLAG_DRAG_TRANSLATE	(1 << 4)
#define PRINT_WIN_FLAG_DRAG_SCALE	(1 << 5)

/*
 *	Print window parameter flags, used to define which print
 *	parameters are set.
 */
#define PRINT_WIN_PARM_FLAG_PRINT_TO		(1 << 0)
#define PRINT_WIN_PARM_FLAG_PRINT_COMMAND	(1 << 1)
#define PRINT_WIN_PARM_FLAG_PRINT_FILE		(1 << 2)
#define PRINT_WIN_PARM_FLAG_SOURCE_TYPE		(1 << 3)
#define PRINT_WIN_PARM_FLAG_ORIENTATION		(1 << 4)
#define PRINT_WIN_PARM_FLAG_VISUAL		(1 << 5)
#define PRINT_WIN_PARM_FLAG_PPI_X		(1 << 6)
#define PRINT_WIN_PARM_FLAG_PPI_Y		(1 << 7)
#define PRINT_WIN_PARM_FLAG_PAPER_SIZE		(1 << 8)
#define PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_X	(1 << 9)
#define PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_Y	(1 << 10)
#define PRINT_WIN_PARM_FLAG_OUTPUT_WIDTH	(1 << 11)
#define PRINT_WIN_PARM_FLAG_OUTPUT_HEIGHT	(1 << 12)

/*
 *	Print to output codes:
 */
#define PRINT_WIN_PRINT_TO_PRINTER	0
#define PRINT_WIN_PRINT_TO_FILE		1

/*
 *	Visual codes:
 */
#define PRINT_WIN_VISUAL_GREYSCALE	0
#define PRINT_WIN_VISUAL_COLOR		1

/*
 *	Paper size codes:
 */
#define PRINT_WIN_PAPER_SIZE_LETTER	0
#define PRINT_WIN_PAPER_SIZE_LEGAL	1
#define PRINT_WIN_PAPER_SIZE_EXECUTIVE	2
#define PRINT_WIN_PAPER_SIZE_A4		3


/*
 *	Print parameters structure:
 */
typedef struct {

	guint	flags;	/* Any of PRINT_WIN_PARM_FLAG_*. */

	gint	print_to;	/* One of PRINT_WIN_PRINT_TO_*. */
	gchar	*print_command;	/* Print command, can be NULL. */
	gchar	*print_file;	/* Print to file name. */

	gint	source_type;	/* Client defined, but needs to
				 * correspond to a row index on the
				 * print window's source_type_combo.
				 */
	gint	orientation;	/* 0 = Portrait, 1 = Landscape.
				 * When set to landscape, image is
				 * effectivly rotated -90 (270) degrees
				 * and the width and height are swapped
				 * (with respect to the paper).
				 */
	gint	visual;		/* One of PRINT_WIN_VISUAL_*. */
	gdouble ppi_x, ppi_y;	/* Pixels per inch. */
	gint	paper_size;	/* One of PRINT_WIN_PAPER_SIZE_*. */

	gint	output_offset_xp,	/* Output offset (in pixels) from
					 * upper left corner regardless
					 * of orientation.
					 */
		output_offset_yp;
	gint	output_widthp,		/* Output size (in pixels), if the
					 * orientation is set to 1 (landscape)
					 * then the width and height will
					 * be swapped (with respect to the
					 * paper).
					 */
		output_heightp;

} print_win_parms_struct;


/*
 *	Print window structure:
 */
typedef struct {

	gbool initialized;
	gbool map_state;
	gbool preview_realized;

	guint flags;		/* Any of PRINT_WIN_FLAG_*. */

	GdkCursor	*translate_cur,
			*scale_cur,
			*busy_cur;

	GtkAccelGroup	*accelgrp;

	GtkWidget	*toplevel,
			*print_to_printer_radio,
			*print_to_file_radio,
			*print_command_entry,
			*print_file_entry,
			*source_type_combo,	/* Client defined list of
						 * print sources.
						 */
			*print_greyscale_radio,
			*print_color_radio,
			*orient_portrait_radio,
			*orient_landscape_radio,
			*ppi_x_entry, *ppi_y_entry,
			*paper_size_letter_radio,
			*paper_size_legal_radio,
			*paper_size_executive_radio,
			*paper_size_a4_radio,
			*preview_fixed,		/* To size preview_glarea
						 * properly. Note: There
						 * are other widgets inside
						 * this one so size may need
						 * to be a bit bigger.
						 */
			*preview_glarea,
			*preview_menu,		/* Preview right-click menu. */
			*print_btn,
			*cancel_btn;

	/* Last drag positions. */
	gint last_drag_x, last_drag_y;

	/* Offset from upper left corner in units of pixels. */
	gint output_offset_x, output_offset_y;
	/* Size in units of pixels. */
	gint output_width, output_height;

	/* Preview texture. */
	GLuint preview_texture;


	/* Callbacks. */
	gpointer client_data;
	void (*print_cb)(
		gpointer,			/* Print window. */
		gpointer,			/* Client data. */
		print_win_parms_struct *	/* Print parameters. */
	);
	void (*cancel_cb)(
		gpointer,			/* Print window. */
		gpointer			/* Client data. */
	);
	void (*preview_request_cb)(
		gpointer,                       /* Print window. */
		gpointer,                       /* Client data. */
		gint,				/* Source type. */
		gint,				/* One of PRINT_WIN_VISUAL_*. */
		gint, gint			/* Width and height in pixels. */
	);

} print_win_struct;


extern gint PrintWinGetSourceType(
	print_win_struct *pw, const gchar *name
);
extern gint PrintWinGetOrientation(print_win_struct *pw);
extern gint PrintWinGetVisual(print_win_struct *pw);
extern void PrintWinGetPaperSizeFromCode(
	print_win_struct *pw, gint paper_size_code,
	gint *width, gint *height
);
extern void PrintWinGetPaperSize(
	print_win_struct *pw,
	gint *width, gint *height
);
extern void PrintWinGetPToWCoeff(
	print_win_struct *pw,
	gdouble *x_coeff, gdouble *y_coeff
);

extern void PrintWinPreviewSetSize(
	print_win_struct *pw,
	gint paper_wp, gint paper_hp
);
extern void PrintWinPreviewDraw(print_win_struct *pw);
extern gint PrintWinPreviewTextureLoadGreyScale(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
);
extern gint PrintWinPreviewTextureLoadColor(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
);
extern void PrintWinPreviewTextureUnload(print_win_struct *pw);

extern void PrintWinDoTranslate(
	print_win_struct *pw,
	gint dx, gint dy
);
extern void PrintWinDoScale(
	print_win_struct *pw,
	gint dx, gint dy
);

extern print_win_struct *PrintWinNew(gpointer core_ptr);
extern gint PrintWinPreviewEnableContext(print_win_struct *pw);
extern void PrintWinSetCallbacks(
	print_win_struct *pw,
	gpointer client_data,
	void (*print_cb)(
		gpointer, gpointer, print_win_parms_struct *
	),
	void (*cancel_cb)(
		gpointer, gpointer
	),
	void (*preview_request_cb)(
		gpointer, gpointer, gint, gint, gint, gint
	)
);
extern void PrintWinSetSourceTypeNames(
	print_win_struct *pw, gchar **name, gint total_names
);
extern void PrintWinSetBusy(print_win_struct *pw);
extern void PrintWinSetReady(print_win_struct *pw);
extern void PrintWinMap(print_win_struct *pw);
extern void PrintWinMapValues(
	print_win_struct *pw, print_win_parms_struct *parms
);
extern void PrintWinUnmap(print_win_struct *pw);
extern void PrintWinReset(print_win_struct *pw, gbool need_unmap);
extern void PrintWinDelete(print_win_struct *pw);



#endif	/* PRINTWIN_H */
