#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "v3dmh.h"
#include "v3dmodel.h"
#include "v3dfio.h"
#include "v3dmp.h"

#include "guiutils.h"
#include "cdialog.h"

#include "primpalette.h"

#include "editor.h"
#include "editordnd.h"

#include "vmastyles.h"
#include "vma.h"
#include "vmapixmaps.h"


static void VMAPrimPaletteDraw(vma_primitives_palette_struct *pp);
static gint VMAPrimPaletteMatchPosition(
	vma_primitives_palette_struct *pp, gint x, gint y,
	gint *rx1, gint *rx2, gint *ry1, gint *ry2
);
static gint VMAPrimPaletteExposeCB(
	GtkWidget *widget, GdkEventExpose *event, gpointer data
);
static gint VMAPrimPaletteButtonCB(
	GtkWidget *widget, GdkEventButton *event, gpointer data
);
static gint VMAPrimPaletteMotionCB(
	GtkWidget *widget, GdkEventButton *event, gpointer data
);

static gint VMAPrimPaletteScrollIncCB(gpointer data);
static void VMAPrimPalettePressedIncCB(GtkWidget *widget, gpointer data);
static void VMAPrimPaletteReleasedIncCB(GtkWidget *widget, gpointer data);
static gint VMAPrimPaletteScrollDecCB(gpointer data);
static void VMAPrimPalettePressedDecCB(GtkWidget *widget, gpointer data);
static void VMAPrimPaletteReleasedDecCB(GtkWidget *widget, gpointer data);

static void VMAPrimPaletteDNDListDataRequestCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data  
);
static void VMAPrimPaletteDNDListDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

static void VMAPrimPaletteItemLoadFromFile(
	vma_primitives_palette_struct *pp, const char *filename
);
void VMAPrimPaletteItemLoadAll(vma_primitives_palette_struct *pp);
void VMAPrimPaletteItemDelete(vma_primitives_palette_item_struct *item_ptr);
void VMAPrimPaletteItemDeleteAll(vma_primitives_palette_struct *pp);

vma_primitives_palette_struct *VMAPrimPaletteNew(
	void *core_ptr, void *editor_ptr,
	GdkWindow *window,
	gpointer client_data,
	void (*select_cb)(gpointer, gint),
	void (*create_cb)(gpointer, gint, gpointer)
);
void VMAPrimPaletteSensitivity(
	vma_primitives_palette_struct *pp, gbool sensitivity
);
void VMAPrimPaletteDelete(vma_primitives_palette_struct *pp);


/* Scroll interval in milliseconds. */
#define PRIMPALETTE_SCROLL_INT		90


/*
 *	Redraws the primitives palette.
 *
 *	This will also update the adjustment's page_size and 
 *	page_increment as needed.
 */
static void VMAPrimPaletteDraw(vma_primitives_palette_struct *pp)
{
	int i, x_cur, y_cur, start_item;
	int width, height;	/* Of drawing area. */
	int orientation = 0;		/* 0 = horizontal. */
	vma_primitives_palette_item_struct *item_ptr;
	GdkGC *gc_fg, *gc_bg;
	GtkWidget *w;
	GtkDrawingArea *da;
	GdkWindow *window;
	GtkAdjustment *adj;
	GtkStyle *style_ptr;


	if(pp == NULL)
	    return;

	if(!pp->initialized)
	    return;

	/* Get pointer to standard style. */
	style_ptr = styles_list.standard;

	/* Get drawing area widget. */
	da = (GtkDrawingArea *)pp->palette;
	if(da == NULL)
	    return;

	w = GTK_WIDGET(da);
	width = w->allocation.width;
	height = w->allocation.height;


	/* Get window from drawing area. */
	if(GTK_WIDGET_NO_WINDOW(da))
	    return;
	window = GTK_WIDGET(da)->window;
	if(window == NULL)
	    return;

	/* Get scroll adjustments. */
	adj = pp->adj;
	if(adj == NULL)
	    return;

	/* Update adjustment page size as needed. */
	if(orientation)
	{
	    if(adj->page_size != height)
	    {
		adj->page_size = adj->page_increment = height;
/*		gtk_adjustment_changed(adj); */
	    }
	}
	else
	{
	    if(adj->page_size != width)
	    {
		adj->page_size = adj->page_increment = width;
/*              gtk_adjustment_changed(adj); */
	    }
	}

	/* Clear background. */
	if(GTK_WIDGET_SENSITIVE(pp->toplevel))
	    gdk_draw_rectangle(
		(GdkDrawable *)window,
		pp->gc_bg,
		TRUE,
		0, 0, width, height
	    );
	else
	    gdk_draw_rectangle(
		(GdkDrawable *)window,
		pp->gc_insensitive_bg,
		TRUE,
		0, 0, width, height
	    );



	/* Calculating starting positions. */
	if(orientation)
	{
	    start_item = 0;
	    x_cur = 0;
	    y_cur = -(int)adj->value;
	}
	else
	{
	    start_item = 0;
	    x_cur = -(int)adj->value;
	    y_cur = 0;
	}
/*
printf("%i %i %f\n", x_cur, y_cur, adj->page_size);
 */
	/* Itterate through each item. */
	for(i = start_item; i < pp->total_items; i++)
	{
	    item_ptr = pp->item[i];
	    if(item_ptr == NULL)
		continue;

	    /* Clip check. */
	    if(orientation)
	    {
		if((y_cur + item_ptr->height) < 0)
		{
		    /* Skip, increment to next. */
		    y_cur += item_ptr->height;
		    continue;
		}
		else if(y_cur > height)
		{
		    break;
		}
	    }
	    else
	    {
		if((x_cur + item_ptr->width) < 0)
		{
		    /* Skip, increment to next. */
		    x_cur += item_ptr->width;
		    continue;
		}
		else if(x_cur > width)
		{
		    break;
		}
	    }

	    /* This item not selected? */
	    if(i != pp->selected_item)
	    {
		gc_fg = pp->gc_fg;
		if(gc_fg == NULL)
		    continue;
		gc_bg = pp->gc_bg;
		if(gc_bg == NULL)
		    continue;
	    }
	    else
	    {
		gc_fg = pp->gc_selected_fg;
		if(gc_fg == NULL)
		    continue;
		if(GTK_WIDGET_SENSITIVE(pp->toplevel))
		    gc_bg = pp->gc_selected_bg;
		else
		    gc_bg = pp->gc_insensitive_bg;
		if(gc_bg == NULL) 
		    continue;

		gdk_draw_rectangle(
		    (GdkDrawable *)window,
		    gc_bg,
		    TRUE,
		    x_cur, y_cur, item_ptr->width, item_ptr->height
		);
		gdk_draw_rectangle(
		    (GdkDrawable *)window,
		    gc_fg,
		    FALSE,
		    x_cur, y_cur, item_ptr->width, item_ptr->height
		);
	    }

	    /* This item's icon has a mask? */
	    if(item_ptr->mask != NULL)
	    {
		gdk_gc_set_clip_mask(gc_fg, item_ptr->mask);
		gdk_gc_set_clip_origin(gc_fg, x_cur, y_cur);
	    }

	    /* Pixmap available? */
	    if((item_ptr->pixmap != NULL) &&
	       (item_ptr->width > 0) && (item_ptr->height > 0)
	    )
	    {
		gdk_draw_pixmap(
		    (GdkDrawable *)window,
		    gc_fg,
		    (GdkDrawable *)item_ptr->pixmap,
		    0, 0,		/* Src coord. */
		    x_cur, y_cur,	/* Tar coord. */
		    item_ptr->width, item_ptr->height
		);
	    }
	    else
	    {


	    }

	    /* Increment current position to next position. */
	    if(orientation)
		y_cur += item_ptr->height;
	    else
		x_cur += item_ptr->width;

	}

	return;
}

/*
 *	Returns the index number of the matched item from the given
 *	coordinate positions of the palette GtkDrawingArea.
 *
 *	Can return -1 on failed match.
 */
static gint VMAPrimPaletteMatchPosition(
	vma_primitives_palette_struct *pp,
	gint x, gint y,
	gint *rx1, gint *rx2, gint *ry1, gint *ry2
)
{
	gint i, orientation = 0;		/* 0 = horizontal */
	gint bmin, bmax;
	GtkAdjustment *adj;
	vma_primitives_palette_item_struct *item_ptr;


	if(rx1 != NULL)
	    (*rx1) = 0;
	if(rx2 != NULL)
	    (*rx2) = 0; 
	if(ry1 != NULL)
	    (*ry1) = 0; 
	if(ry2 != NULL)
	    (*ry2) = 0; 

	if(pp == NULL)
	    return(-1);

	adj = pp->adj;
	if(adj == NULL)
	    return(-1);

	bmin = -(gint)adj->value;
	bmax = bmin;

	for(i = 0; i < pp->total_items; i++)
	{
	    item_ptr = pp->item[i];
	    if(item_ptr == NULL)
		continue;

	    if(orientation)
		bmax = bmin + item_ptr->height;
	    else
		bmax = bmin + item_ptr->width;

	    /* In bounds? */
	    if(orientation)
	    {
		if((y >= bmin) && (y <= bmax) &&
		   (x >= 0) && (x <= item_ptr->width)
		)
		{
		    if(rx1 != NULL)
			(*rx1) = 0;
		    if(rx2 != NULL)
			(*rx2) = item_ptr->width;
		    if(ry1 != NULL)
			(*ry1) = bmin;
		    if(ry2 != NULL)
			(*ry2) = bmax;
		    return(i);
		}
	    }
	    else
	    {
		if((x >= bmin) && (x <= bmax) &&
		   (y >= 0) && (y <= item_ptr->height)
		)
		{
		    if(rx1 != NULL)
			(*rx1) = bmin;
		    if(rx2 != NULL)
			(*rx2) = bmax;
		    if(ry1 != NULL)
			(*ry1) = 0;
		    if(ry2 != NULL)
			(*ry2) = item_ptr->height;
		    return(i);
		}
	    }

	    bmin = bmax;
	}

	return(-1);
}

/*
 *	Expose event callback.
 */
static gint VMAPrimPaletteExposeCB(
	GtkWidget *widget, GdkEventExpose *event, gpointer data
)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return(FALSE);

	VMAPrimPaletteDraw(pp);

	return(TRUE);
}

/*
 *	Button press event callback.
 */
static gint VMAPrimPaletteButtonCB(
	GtkWidget *widget, GdkEventButton *event, gpointer data
)
{
	int orientation = 0;		/* 0 = horizontal */
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return(FALSE);

	/* Skip handling of buttons if insensitive. */
	if(!GTK_WIDGET_SENSITIVE(pp->toplevel))
	    return(TRUE);

	/* Single click? */
	if(event->type == GDK_BUTTON_PRESS)
	{
	    int bx1, bx2, by1, by2;
	    int matched_item;
	    gbool scrolled = FALSE;

	    /* Do match by given event coordinates and return matched
	     * item index and bounds if available.
	     */
	    matched_item = VMAPrimPaletteMatchPosition(
		pp, (int)event->x, (int)event->y,
		&bx1, &bx2, &by1, &by2
	    );
	    /* Got valid matched item? */
	    if((matched_item >= 0) && (matched_item < pp->total_items))
	    {
		GtkWidget *w = pp->palette;
		GtkAdjustment *adj = pp->adj;
		vma_primitives_palette_item_struct *item_ptr = pp->item[matched_item]; 

		/* Check if item is partially obscured, if it is then
		 * scroll.
		 */
		if((w != NULL) && (adj != NULL))
		{
		    int width = w->allocation.width;
		    int height = w->allocation.height;

		    if(orientation)
		    {
			if(by1 < 0)
			{
			    adj->value -= (adj->page_increment / 2);
			    if(adj->value < adj->lower)
				adj->value = adj->lower;
			    scrolled = TRUE;
			}
			else if(by2 > height)
			{
			    adj->value += (adj->page_increment / 2);
			    if(adj->value > (adj->upper - adj->page_size))
				adj->value = adj->upper - adj->page_size;
			    if(adj->value < adj->lower)
				adj->value = adj->lower;
			    scrolled = TRUE;
			}
		    }
		    else
		    {
			if(bx1 < 0)
			{
			    adj->value -= (adj->page_increment / 2);
			    if(adj->value < adj->lower)
				adj->value = adj->lower;
			    scrolled = TRUE;
			}
			else if(bx2 > width)
			{ 
			    adj->value += (adj->page_increment / 2);
			    if(adj->value > (adj->upper - adj->page_size))
				adj->value = adj->upper - adj->page_size; 
			    if(adj->value < adj->lower)
				adj->value = adj->lower;
			    scrolled = TRUE;
			}
		    }
		}

		/* Update DND icon. */
		if(item_ptr != NULL)
		{
		    GdkPixmap *pixmap = NULL;
		    GdkBitmap *mask = NULL;
		    int icon_code = VMAPixmapsListGetMPCode(
			&vma_pixmaps_list, item_ptr->type
		    );

		    VMAPixmapsListGetValues(
			&vma_pixmaps_list, icon_code,
			&pixmap, &mask, NULL
		    );  
		    if(pixmap != NULL)
		    {
			gint w = 15, h = 15;
			gdk_window_get_size((GdkWindow *)pixmap, &w, &h);
			GUIDNDSetDragIcon(
			    pixmap, mask,
			    (w / 2), (h / 2)
			);
		    }
		}	/* Update DND icon. */

	    }

	    /* Change in selection or must redraw? */
	    if((pp->selected_item != matched_item) || scrolled)
	    {
		/* Update selected item. */
		pp->selected_item = matched_item;

		/* Redraw. */
		VMAPrimPaletteDraw(pp);
	    }

	    /* Call select callback. */
	    if(pp->select_cb != NULL)
	    {
		pp->select_cb(
		    pp->client_data,
		    pp->selected_item
		);
	    }
	}
	/* Double click. */
	else if(event->type == GDK_2BUTTON_PRESS)
	{
	    int i = pp->selected_item;

	    if((pp->create_cb != NULL) &&
	       (i >= 0) && (i < pp->total_items)
	    )
	    {
		vma_primitives_palette_item_struct *item_ptr = pp->item[i];
		if(item_ptr != NULL)
		{
		    pp->create_cb(
			pp->client_data,
			item_ptr->type,
			item_ptr->p
		    );
		}
	    }
	}

	return(TRUE);
}

/*
 *	Pointer motion event callback.
 */
static gint VMAPrimPaletteMotionCB(
	GtkWidget *widget, GdkEventButton *event, gpointer data
)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return(FALSE);

	/* Skip handling of motion if insensitive. */
	if(!GTK_WIDGET_SENSITIVE(pp->toplevel))
	    return(TRUE);

	/* Pointer motion event? */
	if(event->type == GDK_MOTION_NOTIFY)
	{
	    int i, matched_item;

	    /* Do match by given event coordinates and return matched
	     * item index and bounds if available.
	     */
	    matched_item = VMAPrimPaletteMatchPosition(
		pp, (int)event->x, (int)event->y,
		NULL, NULL, NULL, NULL
	    );
	    /* Got valid matched item? */
	    if(matched_item > -1)
	    {
		if(pp->motion_item != matched_item)
		{
		    pp->motion_item = i = matched_item;

		    if((i >= 0) && (i < pp->total_items))
		    {
			vma_primitives_palette_item_struct *item_ptr =
			    pp->item[i];

			/* Update tip. */
			if(item_ptr != NULL)
			    GUISetWidgetTip(pp->palette, item_ptr->description);
			else
			    GUISetWidgetTip(pp->palette, NULL);
			GUIShowTipsNow(pp->palette);
		    }
		}
	    }
	    else
	    {
		/* No match, remove tip. */
		GUISetWidgetTip(pp->palette, NULL);
	    }
	}

	return(TRUE);
}

/*
 *	Scroll increment callback.
 */
static gint VMAPrimPaletteScrollIncCB(gpointer data)
{
	GtkAdjustment *adj;
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if(pp == NULL)
	    return(FALSE);

	adj = pp->adj;
	if(adj == NULL)
	    return(FALSE);

	adj->value += adj->step_increment;
	if(adj->value > (adj->upper - adj->page_size))
	    adj->value = (adj->upper - adj->page_size);
	if(adj->value < adj->lower)
	    adj->value = adj->lower;

	VMAPrimPaletteDraw(pp);

	return(TRUE);
}

/*
 *	Increment pressed callback.
 */
static void VMAPrimPalettePressedIncCB(GtkWidget *widget, gpointer data)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return;

	VMAPrimPaletteScrollIncCB(pp);
	if(pp->scroll_toid == (guint)(-1))
	{
	    pp->scroll_toid = gtk_timeout_add(
		PRIMPALETTE_SCROLL_INT,
		VMAPrimPaletteScrollIncCB,
		(gpointer)pp
	    );
	}

	return;
}

/*
 *	Increment released callback.
 */
static void VMAPrimPaletteReleasedIncCB(GtkWidget *widget, gpointer data)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return;

	if(pp->scroll_toid != (guint)(-1))
	{
	    gtk_timeout_remove(pp->scroll_toid);
	    pp->scroll_toid = (guint)(-1);
	}

	return;
}

/*
 *      Scroll decrement callback.
 */
static gint VMAPrimPaletteScrollDecCB(gpointer data)
{  
	GtkAdjustment *adj;
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if(pp == NULL)
	    return(FALSE);

	adj = pp->adj;
	if(adj == NULL)
	    return(FALSE);

	adj->value -= adj->step_increment;
	if(adj->value < adj->lower)
	    adj->value = adj->lower;

	VMAPrimPaletteDraw(pp);

	return(TRUE);
}       

/*
 *	Decrement pressed callback.
 */
static void VMAPrimPalettePressedDecCB(GtkWidget *widget, gpointer data)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return;

	VMAPrimPaletteScrollDecCB(pp);
	if(pp->scroll_toid == (guint)(-1))
	{
	    pp->scroll_toid = gtk_timeout_add(
		PRIMPALETTE_SCROLL_INT,
		VMAPrimPaletteScrollDecCB,
		(gpointer)pp
	    );
	}

	return;
}

/*
 *	Decrement released callback.
 */
static void VMAPrimPaletteReleasedDecCB(GtkWidget *widget, gpointer data)
{
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL))
	    return;

	if(pp->scroll_toid != (guint)(-1))
	{
	    gtk_timeout_remove(pp->scroll_toid);
	    pp->scroll_toid = (guint)(-1);
	}

	return;
}

/*
 *	DND `data request' callback for the primitive palette.
 *
 *	Given widget should be a GtkDrawingArea.
 */
static void VMAPrimPaletteDNDListDataRequestCB(
	GtkWidget *widget, GdkDragContext *dc, 
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	GtkWidget *w;
	int i, ptype;
	void *p;
	vma_primitives_palette_item_struct *item_ptr;
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)data;
	if((widget == NULL) || (pp == NULL) || (dc == NULL))
	    return;

	/* Get palette widget (a GtkDrawingArea). */
	w = pp->palette;
	if(w == NULL)
	    return;
	/* Wrong widget? */
	if(w != widget)
	    return;

	/* Get selected item on the primitives palette. */            
	i = pp->selected_item;
	if((i >= 0) && (i < pp->total_items))
	    item_ptr = pp->item[i];
	else
	    item_ptr = NULL;

	/* Got valid item? */
	if(item_ptr != NULL)
	{
	    /* Send out data consisting of a command containing the
	     * following arguments:
	     * <core_ptr> <editor_num> <new_ptype>
	     */
	    char *buf;
	    int buf_len;
	    int editor_num = -1;
	    vma_core_struct *core_ptr = (vma_core_struct *)pp->core_ptr;
	    char num_str[80];

	    /* Get new primitive type and referance structure. */
	    ptype = item_ptr->type;
	    p = item_ptr->p;

	    /* Find editor number in core structure. */
	    if(core_ptr != NULL)
	    {
		for(i = 0; i < core_ptr->total_editors; i++)
		{
		    if((void *)core_ptr->editor[i] == (void *)pp->editor_ptr)
		    {
			editor_num = i;
			break;
		    }
		}
	    }

	    /* Allocate buf, enough for the basic arguments and ptype. */
	    buf_len = 24 + 24 + 24 + 24;
	    buf = (char *)malloc(buf_len * sizeof(char));
	    if(buf != NULL)
	    {
		/* Format buf. */
		(*buf) = '\0';
		sprintf(num_str, "%.8x ", (guint)core_ptr);
		strcat(buf, num_str);
		sprintf(num_str, "%i ", editor_num);
		strcat(buf, num_str);
		sprintf(num_str, "%i ", ptype);
		strcat(buf, num_str);
		sprintf(num_str, "%.8x", (guint)p);
		strcat(buf, num_str);

		/* Send out data. */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,		/* 8 bits per character. */
		    buf, strlen(buf)
		);
		data_sent = TRUE;

		/* Free buffer. */
		free(buf);
		buf = NULL;
		buf_len = 0;
	    }
	}

	/* Failed to send out data? */
	if(!data_sent)
	{
	    const char *strptr = "Error";

	    gtk_selection_data_set(  
		selection_data,   
		GDK_SELECTION_TYPE_STRING,
		8,      /* 8 bits per character. */
		strptr, strlen(strptr)
	    );
	    data_sent = TRUE;
	}

	return;
}

/*
 *      DND `data request' callback for the primitive palette.
 * 
 *      Given widget should be a GtkDrawingArea.
 */
static void VMAPrimPaletteDNDListDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	return;
}



/*
 *	Loads any primitives found in the model file specified by 
 *	filename into the given primitives palette as items.
 */
static void VMAPrimPaletteItemLoadFromFile(
	vma_primitives_palette_struct *pp, const char *filename
)
{
	int status, ptype;
	FILE *fp;
	void **mh_item = NULL;
	int total_mh_items = 0;
	v3d_model_struct **model = NULL;
	int total_models = 0;
	mp_comment_struct *mp_comment;
	const char *last_comment;
	GtkAdjustment *adj;


	if((pp == NULL) || (filename == NULL))
	    return;

	adj = pp->adj;
	if(adj == NULL)
	    return;

	if(pp->total_items < 0)
	    pp->total_items = 0;

	/* Open model file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return;

	/* Load models from fp. */
	status = V3DLoadModel(
	    NULL, fp,
	    &mh_item, &total_mh_items,
	    &model, &total_models,
	    NULL, NULL
	);

	/* Close file (don't need it anymore. */
	FClose(fp);
	fp = NULL;
 
	/* Delete the headers items, don't need those. */
	V3DMHListDeleteAll(&mh_item, &total_mh_items);

	/* Transfer the models and delete the model pointer array. */
	if(total_models > 0)
	{
	    int i, n, item_num;
	    v3d_model_struct *model_ptr;
	    void *p;
	    vma_primitives_palette_item_struct *item_ptr;


	    /* Itterate through each loaded model. */
	    for(i = 0; i < total_models; i++)
	    {
		model_ptr = model[i];
		if(model_ptr == NULL)
		    continue;

		/* Reset context pointers. */
		mp_comment = NULL;
		last_comment = NULL;

		/* Fetch all primitives from this model. */
		for(n = 0; n < model_ptr->total_primitives; n++)
		{
		    p = model_ptr->primitive[n];
		    if(p == NULL)
			continue;
		    else
			ptype = (*(int *)p);

		    /* Comment? */
		    if(ptype == V3DMP_TYPE_COMMENT)
		    {
			mp_comment = (mp_comment_struct *)p;

			/* Get first line as last comment. */
			if(mp_comment->total_lines > 0)
			    last_comment = mp_comment->line[0];
			else
			    last_comment = NULL;

			if(last_comment != NULL)
			{
			    /* Seek past spaces and comment characters. */
			    while(ISBLANK(*last_comment))
				last_comment++;
			    while((*last_comment) == V3D_COMMENT_CHAR)
				last_comment++;
			    while(ISBLANK(*last_comment))
				last_comment++;
			}

			/* Keep primitive on model. */
		    }
		    /* All other types. */
		    else
		    {
			/* Transfer this primitive to the newly created
			 * item on the primitives palette.
			 */
			item_num = pp->total_items;
			pp->total_items = item_num + 1;
			pp->item = (vma_primitives_palette_item_struct **)realloc(
			    pp->item,
			    pp->total_items *
				sizeof(vma_primitives_palette_item_struct *)
			);
			if(pp->item == NULL)
			{
			    pp->total_items = 0;
			    item_num = -1;
			    item_ptr = NULL;
			}
			else
			{
			    /* Allocate new item on the primitives palette. */
			    pp->item[item_num] = item_ptr =
				(vma_primitives_palette_item_struct *)calloc(
				    1, sizeof(vma_primitives_palette_item_struct)
				);
			    if(item_ptr != NULL)
			    {
				int icon_num;
				GdkPixmap *pixmap;
				GdkBitmap *mask;

				/* Begin setting up values on the new primitives
				 * palette item.
				 */
				item_ptr->type = ptype;

				icon_num = VMAPixmapsListGetMPCode(
				    &vma_pixmaps_list, item_ptr->type
				);
				VMAPixmapsListGetValues(
				    &vma_pixmaps_list, icon_num,
				    &pixmap, &mask, NULL
				);
				if(pixmap != NULL)
				{
				    gdk_pixmap_ref(pixmap);
				    item_ptr->pixmap = pixmap;
				}
				if(mask != NULL)
				{
				    gdk_bitmap_ref(mask);
				    item_ptr->mask = mask;
				}

				item_ptr->width = 20;
				item_ptr->height = 20;
				adj->upper += (double)item_ptr->width;

				/* Copy last comment as description (if any). */
				if(last_comment != NULL)
				{
				    free(item_ptr->description);
				    item_ptr->description = strdup(last_comment);
				}

				/* Transfer this primitive to the item structure. */
				item_ptr->p = p;

				/* Mark primitive as removed from model. */
				model_ptr->primitive[n] = p = NULL;
				continue;
			    }
			}

		    }
		}	/* Fetch all primitives from this model. */
	    }	/* Itterate through each loaded model. */
	}

	/* Delete all loaded models (if any). */
	V3DModelListDeleteAll(&model, &total_models);

	return;
}

/*
 *	Loads standard primitive items for the primitive palette.
 */
void VMAPrimPaletteItemLoadAll(vma_primitives_palette_struct *pp)
{
	int i, type, icon_num, width, height;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAdjustment *adj;
	const char *description;
	vma_primitives_palette_item_struct *item_ptr;

	if(pp == NULL)
	    return;

	adj = pp->adj;
	if(adj == NULL)
	    return;

	/* Delete any existing items in the primitives palette. */
	VMAPrimPaletteItemDeleteAll(pp);


#define DO_ADD_ITEM	\
{ \
 i = pp->total_items; \
 pp->total_items = i + 1; \
 pp->item = (vma_primitives_palette_item_struct **)realloc( \
  pp->item, \
  pp->total_items * sizeof(vma_primitives_palette_item_struct *) \
 ); \
 if(pp->item == NULL) \
 { \
  pp->total_items = 0; \
  i = -1; \
  item_ptr = NULL; \
 } \
 else \
 { \
  pp->item[i] = item_ptr = (vma_primitives_palette_item_struct *)calloc( \
   1, sizeof(vma_primitives_palette_item_struct) \
  ); \
  if(item_ptr != NULL) \
  { \
   item_ptr->type = type; \
   VMAPixmapsListGetValues( \
    &vma_pixmaps_list, icon_num, \
    &pixmap, &mask, NULL \
   ); \
   if(pixmap != NULL) \
   { \
    gdk_pixmap_ref(pixmap); \
    item_ptr->pixmap = pixmap; \
   } \
   if(mask != NULL) \
   { \
    gdk_bitmap_ref(mask); \
    item_ptr->mask = mask; \
   } \
   item_ptr->width = width; \
   item_ptr->height = height; \
   adj->upper += (double)item_ptr->width; \
   item_ptr->description = strdup(description); \
  } \
 } \
}

	/* Reset upper scroll adjustment. */
	adj->upper = 0.0;

	/* Load default comment primitive. */
	width = 20;
	height = 20;

	type = V3DMP_TYPE_COMMENT;
	icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
	description = "Comment";
	DO_ADD_ITEM


	/* Load global preset primitives from file. */
	if((*dname.preset_primitives_global) != '\0')
	{
	    int i, total;
	    const char *path_ptr;
	    struct stat stat_buf;
	    char **strv, *tmp_path;

	    strv = GetDirEntNames(dname.preset_primitives_global);
	    if(strv != NULL)
	    {
	      for(total = 0; strv[total] != NULL; total++);
	      strv = StringQSort(strv, total);

	      for(i = 0; i < total; i++)
	      {
		path_ptr = (const char *)PrefixPaths(
		    dname.preset_primitives_global, strv[i]
		);
		tmp_path = ((path_ptr == NULL) ? NULL : strdup(path_ptr));

		/* Valid path to load? */
		if((tmp_path == NULL) ?
		    0 : !stat(tmp_path, &stat_buf)
		)
		{
		    /* Regular file? */
		    if(S_ISREG(stat_buf.st_mode))
		    {
			VMAPrimPaletteItemLoadFromFile(
			    pp, tmp_path
			);
		    }
		}

		/* Deallocate paths. */
		free(tmp_path);
		free(strv[i]);
	      }

	      /* Deallocate path array. */
	      free(strv);
	    }
	}

	/* Load local preset primitives from file. */
	if((*dname.preset_primitives_local) != '\0')
	{
	    int i, total;
	    const char *path_ptr;
	    struct stat stat_buf;
	    char **strv, *tmp_path;

	    /* Create local primitives directory as needed. */
	    if(stat(dname.preset_primitives_local, &stat_buf))
	    {
		mode_t mode;

		if(stat(dname.data_local, &stat_buf))
		    mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
		else
		    mode = stat_buf.st_mode;

		rmkdir(dname.preset_primitives_local, mode);
	    }

	    strv = GetDirEntNames(dname.preset_primitives_local);
	    if(strv != NULL)
	    {
	      for(total = 0; strv[total] != NULL; total++);
	      strv = StringQSort(strv, total);

	      for(i = 0; i < total; i++)
	      {
		path_ptr = (const char *)PrefixPaths(
		    dname.preset_primitives_local, strv[i]
		);
		tmp_path = ((path_ptr == NULL) ? NULL : strdup(path_ptr));

		/* Valid path to load? */
		if((tmp_path == NULL) ?
		    0 : !stat(tmp_path, &stat_buf) 
		)
		{
		    /* Regular file? */
		    if(S_ISREG(stat_buf.st_mode))
		    {
			VMAPrimPaletteItemLoadFromFile(
			    pp, tmp_path
			);
		    }
		}

		/* Deallocate paths. */
		free(tmp_path);
		free(strv[i]);
	      }

	      /* Deallocate path array. */
	      free(strv);
	    }
	}



	/* Not enough primitive loaded? Implies none were loaded from file above.
	 * so instead load a default set.
	 */
	if(pp->total_items <= 1)
	{
	    width = 20;
	    height = 20;

	type = V3DMP_TYPE_TRANSLATE;
	icon_num = VMA_PIXMAP_MP_TRANSLATE_20x20;
	description = "Translate";
	DO_ADD_ITEM

	type = V3DMP_TYPE_UNTRANSLATE;
	icon_num = VMA_PIXMAP_MP_TRANSLATE_20x20;
	description = "Untranslate";
	DO_ADD_ITEM

	type = V3DMP_TYPE_ROTATE;
	icon_num = VMA_PIXMAP_MP_ROTATE_20x20;
	description = "Rotate";
	DO_ADD_ITEM

	type = V3DMP_TYPE_UNROTATE;
	icon_num = VMA_PIXMAP_MP_ROTATE_20x20;
	description = "Unrotate";
	DO_ADD_ITEM

	type = V3DMP_TYPE_POINT;
	icon_num = VMA_PIXMAP_MP_POINT_20x20;
	description = "Point";
	DO_ADD_ITEM

	type = V3DMP_TYPE_LINE;
	icon_num = VMA_PIXMAP_MP_LINE_20x20;
	description = "Line";
	DO_ADD_ITEM

	type = V3DMP_TYPE_LINE_STRIP;
	icon_num = VMA_PIXMAP_MP_LINESTRIP_20x20;
	description = "Line Strip";
	DO_ADD_ITEM

	type = V3DMP_TYPE_LINE_LOOP;
	icon_num = VMA_PIXMAP_MP_LINELOOP_20x20;
	description = "Line Loop";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TRIANGLE;
	icon_num = VMA_PIXMAP_MP_TRIANGLE_20x20;
	description = "Triangle";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TRIANGLE_STRIP;
	icon_num = VMA_PIXMAP_MP_TRIANGLESTRIP_20x20;
	description = "Triangle Strip";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TRIANGLE_FAN;
	icon_num = VMA_PIXMAP_MP_TRIANGLEFAN_20x20;
	description = "Triangle Fan";
	DO_ADD_ITEM

	type = V3DMP_TYPE_QUAD;
	icon_num = VMA_PIXMAP_MP_QUAD_20x20;
	description = "Quad";
	DO_ADD_ITEM

	type = V3DMP_TYPE_QUAD_STRIP;
	icon_num = VMA_PIXMAP_MP_QUADSTRIP_20x20;
	description = "Quad Strip";
	DO_ADD_ITEM

	type = V3DMP_TYPE_POLYGON;
	icon_num = VMA_PIXMAP_MP_POLYGON_20x20;
	description = "Polygon";
	DO_ADD_ITEM

	type = V3DMP_TYPE_COLOR;
	icon_num = VMA_PIXMAP_MP_COLOR_20x20;
	description = "Color";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TEXTURE_SELECT;
	icon_num = VMA_PIXMAP_MP_TEXTURE_SELECT_20x20;
	description = "Texture Select";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TEXTURE_ORIENT_XY;
	icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
	description = "Texture Orient XY";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TEXTURE_ORIENT_YZ;
	icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
	description = "Texture Orient YZ";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TEXTURE_ORIENT_XZ;
	icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
	description = "Texture Orient XZ";
	DO_ADD_ITEM

	type = V3DMP_TYPE_TEXTURE_OFF;
	icon_num = VMA_PIXMAP_MP_TEXTURE_OFF_20x20;
	description = "Texture Off";
	DO_ADD_ITEM

	type = V3DMP_TYPE_HEIGHTFIELD_LOAD;
	icon_num = VMA_PIXMAP_MP_HEIGHTFIELD_LOAD_20x20;
	description = "Heightfield";
	DO_ADD_ITEM

	}

#undef DO_ADD_ITEM

	return;
}


/*
 *	Destroys the given primitives palette item.
 */
void VMAPrimPaletteItemDelete(vma_primitives_palette_item_struct *item_ptr)
{
	if(item_ptr == NULL)
	    return;

	/* Unref pixmap and mask on item. */
	if(item_ptr->pixmap != NULL)
	    gdk_pixmap_unref(item_ptr->pixmap);
	if(item_ptr->mask != NULL)
	    gdk_bitmap_unref(item_ptr->mask);

	/* Description. */
	if(item_ptr->description != NULL)
	    free(item_ptr->description);

	/* Referance primitive (asssume unrealized). */
	V3DMPDestroy(item_ptr->p);

	free(item_ptr);
	return;
}


/*
 *	Destroys all items on the given primitives palette.
 */
void VMAPrimPaletteItemDeleteAll(vma_primitives_palette_struct *pp)
{
	int i;
	vma_primitives_palette_item_struct *item_ptr;


	if(pp == NULL)
	    return;

	/* Itterate through each item. */
	for(i = 0; i < pp->total_items; i++)
	{
	    item_ptr = pp->item[i];
	    if(item_ptr == NULL)
		continue;
 
	    VMAPrimPaletteItemDelete(item_ptr);
	}

	free(pp->item);
	pp->item = NULL;
	pp->total_items = 0;
	pp->selected_item = -1;
	pp->motion_item = -1;

	return;
}


/*
 *	Creates a new primitives palette.
 */
vma_primitives_palette_struct *VMAPrimPaletteNew(
	void *core_ptr, void *editor_ptr,
	GdkWindow *window,
	gpointer client_data,
	void (*select_cb)(gpointer, gint),
	void (*create_cb)(gpointer, gint, gpointer)
)
{
	GdkGCValuesMask gcv_mask;
	GdkGCValues gcv;
	GdkColormap *colormap;
	GdkColor *c;
	GtkStyle *style_ptr;
	GtkAdjustment *adj;
	GtkWidget *w, *parent, *parent2;
	GtkTargetEntry dnd_type[1];
	vma_primitives_palette_struct *pp = (vma_primitives_palette_struct *)calloc(
	    1, sizeof(vma_primitives_palette_struct)
	);


	if(window != NULL)
	    colormap = gdk_window_get_colormap(window);
	else
	    colormap = gdk_colormap_get_system();

	/* Reset values. */
	pp->initialized = TRUE;
	pp->core_ptr = core_ptr;
	pp->editor_ptr = editor_ptr;
	pp->scroll_toid = (guint)(-1);

	/* Get pointer to standard style. */
	style_ptr = styles_list.standard;

	/* Set up GC values mask for creating graphics contexts. */
	gcv_mask = (GDK_GC_FOREGROUND | GDK_GC_FONT |
	    GDK_GC_FUNCTION | GDK_GC_FILL |     
	    GDK_GC_LINE_WIDTH | GDK_GC_LINE_STYLE | GDK_GC_CAP_STYLE |
	    GDK_GC_JOIN_STYLE
	);

	/* Create standard foreground graphics context. */
	c = &gcv.foreground;
	c->red = (guint16)(0.0 * 65535.0);
	c->green = (guint16)(0.0 * 65535.0);
	c->blue = (guint16)(0.0 * 65535.0); 
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	gcv.font = (style_ptr == NULL) ? NULL : style_ptr->font;
	gcv.function = GDK_COPY;
	gcv.fill = GDK_SOLID;
	gcv.line_width = 1;
	gcv.line_style = GDK_LINE_SOLID;
	gcv.cap_style = GDK_CAP_NOT_LAST;
	gcv.join_style = GDK_JOIN_MITER;
	pp->gc_fg = gdk_gc_new_with_values(window, &gcv, gcv_mask);
	gdk_colormap_free_colors(colormap, c, 1);

	/* Create standard background graphics context. */
	c = &gcv.foreground;
	c->red = (guint16)(1.0 * 65535.0);
	c->green = (guint16)(1.0 * 65535.0);
	c->blue = (guint16)(1.0 * 65535.0); 
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	gcv.font = (style_ptr == NULL) ? NULL : style_ptr->font;
	gcv.function = GDK_COPY;
	gcv.fill = GDK_SOLID;
	gcv.line_width = 1;
	gcv.line_style = GDK_LINE_SOLID;
	gcv.cap_style = GDK_CAP_NOT_LAST;
	gcv.join_style = GDK_JOIN_MITER;
	pp->gc_bg = gdk_gc_new_with_values(window, &gcv, gcv_mask);
	gdk_colormap_free_colors(colormap, c, 1);

	/* Create selected foreground graphics context. */
	c = &gcv.foreground;
	c->red = (guint16)(1.0 * 65535.0);
	c->green = (guint16)(1.0 * 65535.0);
	c->blue = (guint16)(1.0 * 65535.0);
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	gcv.font = (style_ptr == NULL) ? NULL : style_ptr->font;
	gcv.function = GDK_COPY;
	gcv.fill = GDK_SOLID;
	gcv.line_width = 1;
	gcv.line_style = GDK_LINE_SOLID;
	gcv.cap_style = GDK_CAP_NOT_LAST;
	gcv.join_style = GDK_JOIN_MITER;
	pp->gc_selected_fg = gdk_gc_new_with_values(window, &gcv, gcv_mask);
	gdk_colormap_free_colors(colormap, c, 1);

	/* Create selected background graphics context. */
	c = &gcv.foreground;
	c->red = (guint16)(0.0 * 65535.0);
	c->green = (guint16)(0.0 * 65535.0);
	c->blue = (guint16)(0.61 * 65535.0);
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	gcv.font = (style_ptr == NULL) ? NULL : style_ptr->font;
	gcv.function = GDK_COPY;
	gcv.fill = GDK_SOLID;
	gcv.line_width = 1;
	gcv.line_style = GDK_LINE_SOLID;
	gcv.cap_style = GDK_CAP_NOT_LAST;
	gcv.join_style = GDK_JOIN_MITER;
	pp->gc_selected_bg = gdk_gc_new_with_values(window, &gcv, gcv_mask);
	gdk_colormap_free_colors(colormap, c, 1);

	/* Create insensitive background graphics context. */
	c = &gcv.foreground;
	c->red = (guint16)(0.93 * 65535.0);  
	c->green = (guint16)(0.93 * 65535.0);
	c->blue = (guint16)(0.93 * 65535.0);
	gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);
	gcv.font = (style_ptr == NULL) ? NULL : style_ptr->font;
	gcv.function = GDK_COPY;
	gcv.fill = GDK_SOLID;
	gcv.line_width = 1;
	gcv.line_style = GDK_LINE_SOLID;
	gcv.cap_style = GDK_CAP_NOT_LAST;
	gcv.join_style = GDK_JOIN_MITER;
	pp->gc_insensitive_bg = gdk_gc_new_with_values(window, &gcv, gcv_mask);
	gdk_colormap_free_colors(colormap, c, 1);


	/* Create toplevel hbox. */
	pp->toplevel = parent = w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), 2);

	/* Create frame and drawing area. */
	w = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_widget_show(w);
	parent2 = w;

	pp->palette = w = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(w), 300, 20);
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(VMAPrimPaletteExposeCB),
	    (gpointer)pp
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(VMAPrimPaletteButtonCB),
	    (gpointer)pp
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(VMAPrimPaletteMotionCB),
	    (gpointer)pp
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	dnd_type[0].target = EDITOR_DND_TYPE_PRIMITIVE_CREATE_CMD;
	dnd_type[0].flags = GTK_TARGET_SAME_APP;
	dnd_type[0].info = 12345;		/* Ignored. */
	GUIDNDSetSrc(
	    w,
	    &dnd_type, 1,			/* DND target types. */
	    GDK_ACTION_COPY,			/* Actions. */
	    GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,	/* Buttons. */
	    NULL,
	    VMAPrimPaletteDNDListDataRequestCB,
	    VMAPrimPaletteDNDListDataDeleteCB,
	    NULL,
	    (gpointer)pp
	);
	gtk_widget_show(w);

	/* Create scroll adjustment. */
	pp->adj = adj =(GtkAdjustment *)gtk_adjustment_new(
	    0,
	    0,
	    100,
	    15,		/* Step increment. */
	    50,		/* Page increment. */
	    50		/* Page size. */
	);


	/* Create arrow buttons. */
	w = gtk_button_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(VMAPrimPalettePressedDecCB),
	    (gpointer)pp
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(VMAPrimPaletteReleasedDecCB),
	    (gpointer)pp
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_widget_set_usize(w, 20, 20);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);




	w = gtk_button_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(VMAPrimPalettePressedIncCB),
	    (gpointer)pp
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "released",
	    GTK_SIGNAL_FUNC(VMAPrimPaletteReleasedIncCB),
	    (gpointer)pp
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_widget_set_usize(w, 20, 20);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);




	/* Reset items array. */
	pp->item = NULL;
	pp->selected_item = -1;
	pp->motion_item = -1;
	pp->total_items = 0;

	/* Set up callbacks. */
	pp->client_data = client_data;
	pp->select_cb = select_cb;
	pp->create_cb = create_cb;

	/* Allocate some standard items. */
	VMAPrimPaletteItemLoadAll(pp);

	return(pp);
}

/*
 *	Updates sensitivity.
 */
void VMAPrimPaletteSensitivity(
	vma_primitives_palette_struct *pp, gbool sensitivity
)
{
	GtkWidget *w;

	if(pp == NULL)
	    return;
	if(!pp->initialized)
	    return;

	w = pp->toplevel;
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitivity);

	return;
}

/*
 *	Destroys the given primitives palette and all of its 
 *	resources.
 */
void VMAPrimPaletteDelete(vma_primitives_palette_struct *pp)
{
	GdkGC **gc;
	GtkAdjustment **adj;
	GtkWidget **w;


	if(pp == NULL)
	    return;

	if(pp->initialized)
	{
#define DO_UNREF_GC             \
{ \
 if((*gc) != NULL) \
 { \
  GdkGC *tgc = *gc; \
  (*gc) = NULL; \
  gdk_gc_unref(tgc); \
 } \
}

#define DO_UNREF_ADJ            \
{ \
 if((*adj) != NULL) \
 { \
  GtkAdjustment *tadj = *adj; \
  (*adj) = NULL; \
  gtk_object_unref(GTK_OBJECT(tadj)); \
 } \
}

#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    if(pp->scroll_toid != (guint)(-1))
	    {
		gtk_timeout_remove(pp->scroll_toid);
		pp->scroll_toid = (guint)(-1);
	    }

	    VMAPrimPaletteItemDeleteAll(pp);

	    w = &pp->palette;
	    DO_DESTROY_WIDGET

	    w = &pp->toplevel;
	    DO_DESTROY_WIDGET

	    gc = &pp->gc_fg;
	    DO_UNREF_GC
	    gc = &pp->gc_bg;
	    DO_UNREF_GC
	    gc = &pp->gc_selected_fg;
	    DO_UNREF_GC
	    gc = &pp->gc_selected_bg;
	    DO_UNREF_GC
	    gc = &pp->gc_insensitive_bg;
	    DO_UNREF_GC

	    adj = &pp->adj;
	    DO_UNREF_ADJ

#undef DO_DESTROY_WIDGET
#undef DO_UNREF_GC
#undef DO_UNREF_ADJ
	}

	/* Deallocate structure itself. */
	free(pp);
}
