#include <stdio.h> 
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/disk.h"
#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "csd.h"
#include "fsd.h"

#include "prefwin.h"
#include "prefwincb.h"
#include "prefwinop.h"

#include "vpi.h"
#include "vpiinternal.h"
#include "vpiinternalfio.h"

#include "vmastyles.h"
#include "vma.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


void PrefDestroyItem(gpointer data);
void PrefDrawingAreaDestroyCB(GtkObject *object, gpointer data);

void PrefDestroyCB(GtkObject *widget, gpointer data);
gint PrefCloseCB(GtkWidget *widget, GdkEvent *event, gpointer data);
void PrefCloseMCB(GtkWidget *widget, gpointer data);
void PrefOKCB(GtkWidget *widget, gpointer data);
void PrefApplyCB(GtkWidget *widget, gpointer data);
void PrefSaveCB(GtkWidget *widget, gpointer data);

gint PrefMenuMapCB(GtkWidget *widget, GdkEvent *event, gpointer data);

void PrefSelectColorCB(GtkWidget *widget, gpointer data);
void PrefSelectFontCB(GtkWidget *widget, gpointer data);
void PrefBrowsePathCB(GtkWidget *widget, gpointer data);

void PrefCatagoryCTreeSelectCB(
	GtkCTree *ctree, GtkCTreeNode *branch, gint column, gpointer data
);
void PrefCatagoryCTreeUnselectCB(
	GtkCTree *ctree, GtkCTreeNode *branch, gint column, gpointer data
);
void PrefCatagoryCTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);

void PrefCullFaceCheckCB(GtkWidget *widget, gpointer data);
void PrefBackupEnablePeriodicCheckCB(GtkWidget *widget, gpointer data);

static void PrefPluginsUpdateMenus(
	vma_pref_struct *pref, vma_plugin_struct *plugin_ptr
);
void PrefPluginsListSelectCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
);
void PrefPluginsListUnselectCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
);
void PrefPluginsListColumnClickCB(
	GtkWidget *widget, gint column, gpointer data
);
void PrefPluginsEnableCB(GtkWidget *widget, gpointer data);
void PrefPluginsReloadCB(GtkWidget *widget, gpointer data);
void PrefPluginsConfigureCB(GtkWidget *widget, gpointer data);



#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Deallocates the vma_pref_item_struct structure.
 */
void PrefDestroyItem(gpointer data)
{
	vma_pref_struct *pref;
	vma_pref_item_struct *item_ptr = (vma_pref_item_struct *)data;
	if(item_ptr == NULL)
	    return;

	pref = (vma_pref_struct *)item_ptr->pref_ptr;


	/* Deallocate preferance window catagory ctree branch item data
	 * itself.
	 */
	free(item_ptr);

	return;
}


/*
 *	Drawing area destroy callback.
 *
 *	This is just to deallocate its associated csd_color_struct.
 */
void PrefDrawingAreaDestroyCB(GtkObject *object, gpointer data)
{
	csd_color_struct *color_ptr = (csd_color_struct *)data;
	if(color_ptr == NULL)
	    return;

	/* Deallocate csd_color_struct. */
	g_free(color_ptr);
}

/*
 *	Destroy callback.
 */
void PrefDestroyCB(GtkObject *widget, gpointer data)
{
	return;
}

/*
 *	Close callback.
 */
gint PrefCloseCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	static gbool reenterant = FALSE;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return(TRUE);

	if(!pref->initialized)
	    return(TRUE);

	if(reenterant)
	    return(TRUE);
	else
	    reenterant = TRUE;

	/* Check if currently processing. */
	if(pref->processing)
	{
	    reenterant = FALSE;
	    return(TRUE);
	}   

	/* Unmap preferences window. */
	PrefUnmap(pref);

	reenterant = FALSE;
	return(TRUE);
}


/*
 *      Close callback from menu item.
 */
void PrefCloseMCB(GtkWidget *widget, gpointer data)
{
	PrefCloseCB(widget, NULL, data);
	return;
}

/*
 *      OK callback.
 */
void PrefOKCB(GtkWidget *widget, gpointer data)
{
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	PrefSetBusy(pref);
	PrefDoApply(pref);
	PrefSetReady(pref);

	PrefUnmap(pref);

	return;
}

/*
 *	Apply callback.
 */     
void PrefApplyCB(GtkWidget *widget, gpointer data)
{
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	PrefSetBusy(pref);
	PrefDoApply(pref);
	PrefSetReady(pref);

	return;
}

/*
 *	Save callback.
 */     
void PrefSaveCB(GtkWidget *widget, gpointer data)
{
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;


	return;
}


/*
 *	Menu map callback.
 */
gint PrefMenuMapCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	GtkWidget *w;
	GdkEventButton *button = NULL;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((widget == NULL) || (event == NULL) || (pref == NULL))
	    return(FALSE);

	if(*(gint *)event == GDK_BUTTON_PRESS)
	    button = (GdkEventButton *)event;

	/* Plug-ins clist? */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if(w == widget)
	{
	    /* Map menu? */
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU);
	    if((button != NULL) && (w != NULL))
	    {
		if(button->button == 3)
		{
		    /* Map menu. */
		    gtk_menu_popup(
			GTK_MENU(w),
			NULL, NULL, NULL, NULL,
			button->button, button->time
		    );
		    return(TRUE);  
		}
	    }
	}

	return(TRUE);
}

/*
 *	Select color callback.
 *
 *	When a select color GtkButton is pressed, the color selection
 *	dialog will be mapped to select a color.
 *
 *	If a color is selected the color values will be updated on
 *	the GtkDrawingArea widget.
 *
 *	Input for data is the GtkDrawingArea, not a vma_pref_struct!
 */
void PrefSelectColorCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	GtkWidget *da = (GtkWidget *)data;
	csd_color_struct *start_color, *color_rtn;
	if(da == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Get CSD color structure stored as user data from the
	 * drawing area widget.
	 */
	start_color = gtk_object_get_user_data(GTK_OBJECT(da));
	if(start_color == NULL)
	{
	    reenterant = FALSE;
	    return;
	}

	/* Prompt user to select new color. */
	CSDSetTransientFor(gtk_widget_get_toplevel(da));
	if(CSDGetResponse(
	    "Select Color",
	    "Select", "Cancel",
	    start_color, &color_rtn,
	    NULL, NULL
	))
	{
	    if((color_rtn != NULL) && !GTK_WIDGET_NO_WINDOW(da))
	    {
		/* Got user response, set new color. */
		GdkColormap *colormap = gdk_window_get_colormap(da->window);
		GdkColor c;

		c.red = (guint16)(color_rtn->r * 65535.0);
		c.green = (guint16)(color_rtn->g * 65535.0);
		c.blue = (guint16)(color_rtn->b * 65535.0);
		c.pixel = 0;

		if(colormap != NULL)
		{
		    gdk_colormap_alloc_color(colormap, &c, TRUE, TRUE);
		    gdk_window_set_background(da->window, &c);
		    gdk_window_clear(da->window);
		    gdk_colormap_free_colors(colormap, &c, 1);
		}

		/* Update csd_color_struct on GtkDrawingArea widget. */
		if(start_color != color_rtn)
		    memcpy(start_color, color_rtn, sizeof(csd_color_struct));
	    }
	}
	CSDSetTransientFor(NULL);

	reenterant = FALSE;
}

/*
 *      Select font callback.
 *
 *      When a select font GtkButton is pressed, the font selection
 *      dialog will be mapped to select a font.
 *
 *      If a font name is selected the font name will be updated on
 *      the GtkEntry widget's value.
 *
 *      Input for data is the GtkEntry, not a vma_pref_struct!  
 */
void PrefSelectFontCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	gchar *font_name, *font_name_rtn;
	GtkEntry *entry = (GtkEntry *)data;
	if(entry == NULL)
	    return;

	/* Get font name from entry widget, and make a copy. */
	font_name = gtk_entry_get_text(entry);
	if(font_name != NULL)
	    font_name = g_strdup(font_name);

	/* Query user for new font name. */
	if(FSDGetResponse(
	    "Select Font",
	    "Select", "Cancel",
	    font_name,
	    &font_name_rtn
	))
	{
	    if(font_name_rtn != NULL)
		gtk_entry_set_text(entry, font_name_rtn);
	}

	/* Deallocate font name. */
	g_free(font_name);
	font_name = NULL;

	reenterant = FALSE;
	return;
}

/*
 *	Browse path callback.
 *
 *	Maps the file browser and query's for a new path, starting with
 *	the current path prefix from the GtkEntry (taken as data).
 *
 *      Input for data is a GtkEntry, not a vma_pref_struct!
 */
void PrefBrowsePathCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	gbool status;
	gchar *old_path, *strptr;
	fb_type_struct **ext_type;
	int total_ext_types;
	char **path_rtn;
	int path_total_rtns;
	fb_type_struct *ext_type_rtn;
	GtkEntry *entry = (GtkEntry *)data;
	if(entry == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Record old path, as a coppied string. */
	old_path = gtk_entry_get_text(entry);
	if(old_path != NULL)
	    old_path = strdup(old_path);

	/* Parse old path. */
	if(old_path != NULL)
	{
	    struct stat stat_buf;

	    strptr = strchr(old_path, ' ');
	    if(strptr == NULL)
		strptr = strchr(old_path, '\t');
	    if(strptr != NULL)
		(*strptr) = '\0';

	    /* Old path exists? */
	    if(!stat(old_path, &stat_buf))
	    {
		/* Old path not a directory? */
		if(!S_ISDIR(stat_buf.st_mode))
		{
		    /* Reduce to parent path. */
		    strptr = strrchr(old_path, DIR_DELIMINATOR);
		    if(strptr != NULL)
			(*strptr) = '\0';
		}
	    }
	}

	/* Allocate file extension types list. */
	ext_type = NULL;
	total_ext_types = 0;
	FileBrowserTypeListNew(
	    &ext_type, &total_ext_types,
	    "*.*", "All files"
	);

	/* Map file browser and get response. */
	FileBrowserSetTransientFor(
	    gtk_widget_get_toplevel(GTK_WIDGET(entry))
	);
	status = FileBrowserGetResponse(
	    "Select Path",
	    "Select", "Cancel",
	    old_path,
	    ext_type, total_ext_types,
	    &path_rtn, &path_total_rtns, &ext_type_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Deallocate file extension types list. */
	FileBrowserDeleteTypeList(ext_type, total_ext_types);
	ext_type = NULL;
	total_ext_types = 0;

	/* Got response? */
	if(status)
	{
	    if(path_total_rtns > 0)
	    {
		gchar *new_path = path_rtn[0];

		if(new_path != NULL)
		    gtk_entry_set_text(entry, new_path);
	    }
	}

	/* Deallocate old path. */
	free(old_path);
	old_path = NULL;

	reenterant = FALSE;
	return;
}               


/*
 *	Catagory ctree branch select callback.
 */
void PrefCatagoryCTreeSelectCB(
	GtkCTree *ctree, GtkCTreeNode *branch, gint column, gpointer data
)
{
	static gbool reenterant = FALSE;
	gint row;
	GtkCList *clist;
	vma_pref_item_struct *item_ptr; 
	GtkWidget *panel_parent;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((pref == NULL) || (ctree == NULL))
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	if((gpointer)ctree == (gpointer)pref->catagory_ctree)
	{
	    clist = (GtkCList *)ctree;

	    PrefSetBusy(pref);

	    /* A branch selected previously? */
	    if(pref->selected_branch != NULL)
	    {
		/* Unselect signal will have the values applied, we don't
		 * need to apply values here.
		 */
	    }

	    /* Scroll if row not visibile. */
	    row = gtk_clist_find_row_from_data(
		clist,
		(gpointer)PrefBranchGetData(ctree, branch)
	    );
	    if(row > -1)
	    {
		if(gtk_clist_row_is_visible(clist, row) !=
		    GTK_VISIBILITY_FULL
		)
		    gtk_clist_moveto(
			clist,
			row, 0,         /* Row, column. */
			0.5, 0.0        /* Row, column. */
		    );
	    }

	    /* Newly selected branch valid? */
	    if(branch != NULL)
	    {
		/* Get catagory branch item pointer. */
		item_ptr = PrefBranchGetData(
		    ctree, branch
		);
		if(item_ptr != NULL)
		{
		    /* Get panel parent widget corresponding to the
		     * catagory index.
		     */
		    panel_parent = PrefPanelGetWidget(
			pref, item_ptr->catagory
		    );

		    /* Unmap panel parent. */
		    if(panel_parent != NULL)
			gtk_widget_show(panel_parent);
		}
	    }

	    /* Update selected branch pointer. */
	    pref->selected_branch = branch;

	    PrefSetReady(pref);
	}

	reenterant = FALSE;
	return;
}

/*
 *	Preferences catagory ctree unselect callback.
 */
void PrefCatagoryCTreeUnselectCB(
	GtkCTree *ctree, GtkCTreeNode *branch, gint column, gpointer data
)
{
	static gbool reenterant = FALSE;
	vma_pref_item_struct *item_ptr;
	GtkWidget *panel_parent;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((pref == NULL) || (ctree == NULL))
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	if((gpointer)ctree == (gpointer)pref->catagory_ctree)
	{
	    /* Get catagory branch item pointer. */
	    item_ptr = PrefBranchGetData(
		ctree, branch
	    );
	    if(item_ptr != NULL)
	    {
		/* Get panel parent widget corresponding to the
		 * catagory index.
		 */
		panel_parent = PrefPanelGetWidget(
		    pref, item_ptr->catagory
		);

		/* Unmap panel parent. */
		if(panel_parent != NULL)
		    gtk_widget_hide(panel_parent);
	    }

	    /* Mark selected branch as unselected if this unselected
	     * branch was the selected branch.
	     */
	    if(branch == pref->selected_branch)
		pref->selected_branch = NULL;
	}

	reenterant = FALSE;
	return;
}

/*
 *      Catagory ctree branch expand (and collapse) callback, called
 *      whenever a branch is expanded or collapsed.
 */
void PrefCatagoryCTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
	static gbool reenterant = FALSE;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((pref == NULL) || (ctree == NULL))
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	if((gpointer)ctree == (gpointer)pref->catagory_ctree)
	{
	    /* Update catagory clist column width. */
	    gtk_clist_set_column_width(
		(GtkCList *)ctree,
		0,                      /* Column. */
		gtk_clist_optimal_column_width((GtkCList *)ctree, 0)
	    );

	}

	reenterant = FALSE;
	return;
}


/*
 *	Callback for widget VMA_PREF_PARM_VIEW_CULL_FACES. Updates the
 *	sensitivity of widgets VMA_PREF_PARM_CULL_DIRECTION_CW and
 *	VMA_PREF_PARM_CULL_DIRECTION_CCW.
 */
void PrefCullFaceCheckCB(GtkWidget *widget, gpointer data)
{
	gbool sensitivity;
	GtkWidget *w;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	if(widget == NULL)
	    widget = PrefParmGetWidget(pref, VMA_PREF_PARM_VIEW_CULL_FACES);
	if(widget == NULL)
	    return;

	sensitivity = GTK_TOGGLE_BUTTON(widget)->active;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_CULL_DIRECTION_CW);
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitivity);
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_CULL_DIRECTION_CCW);
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitivity);

	return;
}

/*
 *      Callback for widget VMA_PREF_PARM_BACKUP_PERIODIC. Updates the
 *      sensitivity of widget VMA_PREF_PARM_BACKUP_PERIODIC_INT.
 */
void PrefBackupEnablePeriodicCheckCB(GtkWidget *widget, gpointer data)
{
	gbool sensitivity;
	GtkWidget *w;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;
  
	if(widget == NULL)
	    widget = PrefParmGetWidget(pref, VMA_PREF_PARM_BACKUP_PERIODIC);
	if(widget == NULL)
	    return;

	sensitivity = GTK_TOGGLE_BUTTON(widget)->active;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_BACKUP_PERIODIC_INT);
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitivity);

	return;
}


/*
 *	Plug-ins panel button and menu items update.
 *
 *	The given plugin_ptr should be the current selected plug-in.
 *	If plugin_ptr is NULL then that implies no plug-in is selected.
 */
static void PrefPluginsUpdateMenus(
	vma_pref_struct *pref, vma_plugin_struct *plugin_ptr
)
{
	gbool plugin_loaded = FALSE, plugin_selected = FALSE;
	GtkWidget *w;


	if(pref == NULL)
	    return;

	/* Plug-in loaded? */
	if((plugin_ptr != NULL) ? (plugin_ptr->handle != NULL) : 0)
	    plugin_loaded = TRUE;
	if(plugin_ptr != NULL)
	    plugin_selected = TRUE;


	/* Enable and disable menu items. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_ENABLE);
	if(w != NULL)
	{
	    if(plugin_loaded)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	    gtk_widget_set_sensitive(w, plugin_selected);
	}
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_DISABLE);
	if(w != NULL)
	{
	    if(plugin_loaded)
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_widget_set_sensitive(w, plugin_selected);
	}

	/* Configure. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST_MENU_CONFIGURE);
	if(w != NULL)
	    gtk_widget_set_sensitive(w, plugin_loaded);


	/* Enable and disable buttons. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_ENABLE);
	if(w != NULL)
	{
	    if(plugin_loaded)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	    gtk_widget_set_sensitive(w, plugin_selected);
	}
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DISABLE);
	if(w != NULL)
	{
	    if(plugin_loaded)
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	    gtk_widget_set_sensitive(w, plugin_selected);
	}

	/* Configure. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_CONFIGURE);
	if(w != NULL)
	    gtk_widget_set_sensitive(w, plugin_loaded);


	return;
}

/*
 *	Plug-ins list select callback.
 */
void PrefPluginsListSelectCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
)
{
	gint i;
	GtkCList *clist;
	GtkWidget *w;
	vma_plugin_struct *plugin_ptr;
	vma_core_struct *core_ptr;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	core_ptr = (vma_core_struct *)pref->core_ptr;
	if(core_ptr == NULL)
	    return;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if((w == NULL) ? 0 : GTK_IS_CLIST(w))
	    clist = GTK_CLIST(w);
	else
	    clist = NULL;
	if(clist == NULL)
	    return;

	/* Scroll to row if not visable. */
	if(gtk_clist_row_is_visible(clist, row) !=
	    GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, 0,		/* Row, column. */
		0.5, 0.0	/* Row, column. */
	    );

	/* Get values for newly selected row. */
	plugin_ptr = (vma_plugin_struct *)gtk_clist_get_row_data(clist, row);
	/* Give up if no plug-in selected. */
	if(plugin_ptr == NULL)
	    return;

	/* Double check if plug-in really exists. */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    if(core_ptr->plugin[i] == plugin_ptr)
		break;
	}
	if(i >= core_ptr->total_plugins)
	    return;

	/* Update menus. */
	PrefPluginsUpdateMenus(pref, plugin_ptr);

	/* Update description. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DESCRIPTION);
	if((w != NULL) && (plugin_ptr->description != NULL))
	{
	    if(GTK_IS_TEXT(w))
	    {
		GtkText *text = GTK_TEXT(w);
		GtkStyle *style = styles_list.text_editable;

		gtk_text_freeze(text);
		gtk_text_set_point(text, 0);
		gtk_text_forward_delete(text, gtk_text_get_length(text));
		gtk_text_insert(
		    text,
		    (style == NULL) ? NULL : style->font,
		    NULL, NULL,
		    (const char *)plugin_ptr->description,
		    -1
		);
		gtk_text_set_point(text, 0);
		gtk_text_thaw(text);
	    }
	}

	return;
}

/*
 *      Plug-ins list unselect callback.
 */
void PrefPluginsListUnselectCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
)
{
	GtkWidget *w;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	if(!pref->initialized)                    
	    return;

	/* Update menus. */
	PrefPluginsUpdateMenus(pref, NULL);

	/* Update description. */
	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DESCRIPTION);
	if(w != NULL)
	{
	    if(GTK_IS_TEXT(w))
	    {
		GtkText *text = GTK_TEXT(w);

		gtk_text_freeze(text);
		gtk_text_set_point(text, 0);
		gtk_text_forward_delete(text, gtk_text_get_length(text));
		gtk_text_thaw(text);
	    }
	}

	return;
}

/*
 *	Plug-ins list column click callback.
 */
void PrefPluginsListColumnClickCB(
	GtkWidget *widget, gint column, gpointer data
)
{
	GtkCList *clist;
	GtkWidget *w;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if(pref == NULL)
	    return;

	if(!pref->initialized)
	    return;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if((w == NULL) ? 0 : GTK_IS_CLIST(w))
	    clist = GTK_CLIST(w);
	else
	    clist = NULL;
	if(clist == NULL)
	    return;

	gtk_clist_set_sort_column(clist, column);
	gtk_clist_set_sort_type(clist, GTK_SORT_ASCENDING);
	gtk_clist_sort(clist);

	return;
}

/*
 *	Plug-ins list enable and disable buttons callback.
 */
void PrefPluginsEnableCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	gint i, row, status;
	gbool enable_plugin = FALSE;
	GtkCList *clist;
	GtkWidget *w;
	GList *glist;
	vma_plugin_struct *plugin_ptr;
	vma_core_struct *core_ptr;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((widget == NULL) || (pref == NULL))
	    return;

	if(!pref->initialized || pref->processing)
	    return;

	core_ptr = (vma_core_struct *)pref->core_ptr;
	if(core_ptr == NULL)
	    return;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if((w == NULL) ? 0 : GTK_IS_CLIST(w))
	    clist = GTK_CLIST(w);
	else
	    clist = NULL;
	if(clist == NULL)
	    return;

	/* Get last selected row. */
	row = -1;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    glist = glist->next;
	}

	/* If no row selected, then give up */
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Get values for newly selected row. */
	plugin_ptr = (vma_plugin_struct *)gtk_clist_get_row_data(clist, row);
	/* Give up if no plug-in selected. */
	if(plugin_ptr == NULL)
	    return;

	/* Double check if plug-in really exists. */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    if(core_ptr->plugin[i] == plugin_ptr)
		break;
	}   
	if(i >= core_ptr->total_plugins)
	    return;

	/* Check if plug-in is currently enabled and processing. */
	if((plugin_ptr->handle != NULL) && plugin_ptr->processing)
	{
	    CDialogSetTransientFor(pref->toplevel);
	    status = CDialogGetResponse(
"Plug-in Busy",
"Selected plug-in is currently marked as processing,\n\
please wait a while for the plug-in to finish its\n\
task before disabling it.",
"The selected plug-in is marked to be currently\n\
performing some task or procedure and cannot be\n\
disabled. Please wait a while for the plug-in to\n\
finish its job before disabling it.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}


	/* Set enable_plugin to be opposite of the current plug-in load
	 * state.
	 */
	enable_plugin = !((plugin_ptr->handle == NULL) ? FALSE : TRUE);


	/* Check if reentering. */
	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Mark preferences window as busy. */
	PrefSetBusy(pref);


	/* Enable selected plug-in? */
	if(enable_plugin)
	{
	    /* Update enable and disable button's has default flag. */
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_ENABLE);
	    if(w != NULL)
	    {
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	    }
/*
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DISABLE);
	    if(w != NULL)
		GTK_WIDGET_UNSET_FLAGS(w, GTK_HAS_DEFAULT);
 */

	    /* Enable plug-in. */
	    status = VPIEnable(plugin_ptr);
	    switch(status)
	    {
	      case 0:
		break;

	      case -1:
		CDialogSetTransientFor(pref->toplevel);
		CDialogGetResponse(
"Cannot Enable Plug-In",
"A general error occured while attempting to load\n\
the selected plug-in.",
"Check if the plug-in was compiled and installed\n\
properly, see Help->Plug-Ins for additional help.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		break;

	      case -2:
		CDialogSetTransientFor(pref->toplevel);
		CDialogGetResponse(
"Cannot Enable Plug-In",
"Cannot find the plug-in binary.",
"Make sure that the plug-in is installed properly,\n\
see Help->Plug-Ins for additional help.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		break;

	      case -3:
		CDialogSetTransientFor(pref->toplevel);
		CDialogGetResponse(
"Cannot Enable Plug-In",
"Missing plug-in functions",
"Cannot find one or more standard (required) functions\n\
in the selected plug-in. Make sure that he plug-in was\n\
coded correctly with all standard functions, see\n\
Help->Plug-Ins for additional help.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		break;

	      case -4:
		CDialogSetTransientFor(pref->toplevel);
		CDialogGetResponse(
"Cannot Enable Plug-In",
VPIGetOSLoadError(),
"The operating system returned the above error message,\n\
when attempting to load the selected plug-in (see\n\
Help->Plug-Ins for additional help.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		break;

	      case -5:
		CDialogSetTransientFor(pref->toplevel);
		CDialogGetResponse(
"Cannot Enable Plug-In",
"Plug-ins not supported on this operating system.",
"Your operating system does not support plug-ins, if\n\
you believe otherwise then you should contact the\n\
authors of this program and notify them about this\n\
issue.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		break;
	    }
	}
	else
	{
	    /* Update enable and disable button's has default flag. */
/*
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_ENABLE);
	    if(w != NULL)
		GTK_WIDGET_UNSET_FLAGS(w, GTK_HAS_DEFAULT);
*/
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DISABLE);
	    if(w != NULL)
	    {
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	    }


	    /* Disable plug-in. */
	    status = VPIDisable(plugin_ptr);
	    switch(status)
	    {
	      default:
		break;
	    }
	}

	/* Update menus. */
	PrefPluginsUpdateMenus(pref, plugin_ptr);


	/* Plug-in enabled? */
	while(plugin_ptr->handle != NULL)
	{
	    /* Call plug-in's info function to get information. */
	    if(!VPIDoGetInfo(plugin_ptr))
	    {
		/* Get info returned false, need to disable plug-in. */
		VPIDisable(plugin_ptr);
		break;
	    }

	    /* Update description. */
	    w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_DESCRIPTION);
	    if((w != NULL) && (plugin_ptr->description != NULL))
	    {
		if(GTK_IS_TEXT(w))
		{
		    GtkText *text = GTK_TEXT(w);
		    GtkStyle *style = styles_list.text_editable;

		    gtk_text_freeze(text);
		    gtk_text_set_point(text, 0);
		    gtk_text_forward_delete(text, gtk_text_get_length(text));
		    gtk_text_insert(
			text,
			(style == NULL) ? NULL : style->font,
			NULL, NULL,
			(const char *)plugin_ptr->description,
			-1
		    );
		    gtk_text_set_point(text, 0);
		    gtk_text_thaw(text);
		}
	    }

	    /* Break, since this is one big while() loop. */
	    break;
	}


	/* Update row on plug-ins clist. */
	gtk_clist_freeze(clist);
	VPICListUpdateRow(
	    clist, row, plugin_ptr
	);
	gtk_clist_thaw(clist);


	/* Update related resources. */

	/* Update all editor's render menu. */
	for(i = 0; i < core_ptr->total_editors; i++)
	    EditorRenderMenuRegenerate(core_ptr->editor[i]);



	/* Mark preferences window as ready. */
	PrefSetReady(pref);

	reenterant = FALSE;
	return;
}


/*
 *	Callback to save existing plug-ins configuration and reload
 *	all plug-ins.
 */
void PrefPluginsReloadCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	gint i;
	GtkCList *clist;
	GtkWidget *w;
	vma_plugin_struct *plugin_ptr;
	vma_core_struct *core_ptr;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((widget == NULL) || (pref == NULL))
	    return;

	if(!pref->initialized || pref->processing)
	    return;

	core_ptr = (vma_core_struct *)pref->core_ptr;
	if(core_ptr == NULL)
	    return;

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if((w == NULL) ? 0 : GTK_IS_CLIST(w))
	    clist = GTK_CLIST(w);
	else
	    clist = NULL;
	if(clist == NULL)
	    return;

	/* Make sure no plugins are currently processing. */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    plugin_ptr = core_ptr->plugin[i];
	    if((plugin_ptr == NULL) ? 1 : (plugin_ptr->handle == NULL))
		continue;

	    if(plugin_ptr->processing)
		break;
	}
	if(i < core_ptr->total_plugins)
	{
	    /* One or more plug-ins are busy processing. */
	    CDialogSetTransientFor(pref->toplevel);
	    CDialogGetResponse(
"Plug-ins Busy",
"One or more plug-ins are still busy processing, please\n\
wait for them to finish before trying to reload.",
"There are one or more plug-ins still performing some\n\
task, thus the plug-ins cannot be reloaded untill they\n\
finish. Please wait for them to finish their task before\n\
reloading.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}


	/* Check if reentering. */
	if(reenterant)
	    return;
	else
	    reenterant = TRUE;


	/* Mark preferences window as busy. */
	PrefSetBusy(pref);


	/* Save plug-ins configuration file. */
	VPISaveConfigurationToFile(
	    core_ptr->plugin, core_ptr->total_plugins,
	    fname.plugins  
	);

	/* Unload all plug-ins, calling their shutdown functions as
	 * needed.
	 */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    VPIUnload(core_ptr->plugin[i]);
	    core_ptr->plugin[i] = NULL;
	}
	free(core_ptr->plugin);
	core_ptr->plugin = NULL;
	core_ptr->total_plugins = 0;

	/* Clear plug-ins clist. */
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);


	/* Load plug-ins. */
	if(1)
	{
	    /* Load plug-ins from plug-ins directories. */
	    VPILoadPluginsFromDirectory(
		&core_ptr->plugin, &core_ptr->total_plugins,
		dname.plugins_global,
		TRUE,           /* Load as disabled. */
		TRUE,           /* Is global. */
		core_ptr,
		core_ptr->splash_win
	    );
	    VPILoadPluginsFromDirectory(
		&core_ptr->plugin, &core_ptr->total_plugins,
		dname.plugins_local, 
		TRUE,           /* Load as disabled. */
		FALSE,          /* Is not global. */
		core_ptr,
		core_ptr->splash_win
	    );

	    /* If any plugins were loaded, then we need to set their  
	     * attributes from the plugins configuration file (if it
	     * exists).
	     */
	    VPILoadConfigurationFromFile(
		&core_ptr->plugin, &core_ptr->total_plugins,
		fname.plugins
	    );

	    /* Request information from each loaded plug-in. */
	    for(i = 0; i < core_ptr->total_plugins; i++)
	    {
		plugin_ptr = core_ptr->plugin[i];
		if(plugin_ptr == NULL)
		    continue;

		if(plugin_ptr->handle == NULL)
		    continue;

		if(!VPIDoGetInfo(plugin_ptr))
		{
		    /* Returned false, need to disable plug-in. */
		    VPIDisable(plugin_ptr);
		    continue;
		}
	    }
	}

	/* Update plug-ins clist. */
	gtk_clist_freeze(clist);
	VPICListAppend(clist, core_ptr->plugin, core_ptr->total_plugins);
	gtk_clist_thaw(clist);


	/* Update related resources. */

	/* Update all editor's render menu. */
	for(i = 0; i < core_ptr->total_editors; i++)
	    EditorRenderMenuRegenerate(core_ptr->editor[i]);


	/* Mark preferences window as ready. */
	PrefSetReady(pref);

	reenterant = FALSE;
	return;
}

/*
 *	Configure plug-in callback.
 */
void PrefPluginsConfigureCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterant = FALSE;
	gint i, row, status;
	GList *glist;
	GtkCList *clist;
	GtkWidget *w;
	vma_plugin_struct *plugin_ptr;
	vma_core_struct *core_ptr;
	vma_pref_struct *pref = (vma_pref_struct *)data;
	if((widget == NULL) || (pref == NULL))
	    return;

	if(!pref->initialized || pref->processing)
	    return;

	core_ptr = (vma_core_struct *)pref->core_ptr;
	if(core_ptr == NULL)
	    return;   

	w = PrefParmGetWidget(pref, VMA_PREF_PARM_PLUGINS_LIST);
	if((w == NULL) ? 0 : GTK_IS_CLIST(w))
	    clist = GTK_CLIST(w);
	else
	    clist = NULL;
	if(clist == NULL)
	    return;

	/* Get last selected row. */
	row = -1;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    glist = glist->next;
	}

	/* If no row selected, then give up */
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Get values for newly selected row. */   
	plugin_ptr = (vma_plugin_struct *)gtk_clist_get_row_data(clist, row);
	/* Give up if no plug-in selected. */
	if(plugin_ptr == NULL)
	    return;

	/* Double check if plug-in really exists. */
	for(i = 0; i < core_ptr->total_plugins; i++)
	{
	    if(core_ptr->plugin[i] == plugin_ptr)
		break;
	}
	if(i >= core_ptr->total_plugins)
	    return;

	/* Plug-in needs to be loaded before it can be configured,
	 * check if it is not loaded.
	 */
	if((plugin_ptr->handle == NULL) || (plugin_ptr->configure == NULL))
	{
	    CDialogSetTransientFor(pref->toplevel);
	    status = CDialogGetResponse(
"Configure Failed!",
"Selected plug-in does not have a configure function,\n\
this plug-in cannot be configured.",
"The selected plug-in does not have a configure function,\n\
therefore it cannot be configured. This may be intentional\n\
as the plug-in is not applicatable to be configured or\n\
there is a programming error. If the plug-in was documented\n\
to be configurable, then you should notify the vendor of\n\
this plug-in.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    return;
	}

	/* Check if reentering. */
	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Mark preferences window as busy. */
	PrefSetBusy(pref);

	if(1)
	{
	    vpi_op_configure_struct v;


	    /* Call plug-in's configure function. */
	    v.flags = 0;

	    plugin_ptr->processing = 1;
	    status = plugin_ptr->configure(
		(vpi_id *)plugin_ptr, &v, plugin_ptr->configure_data
	    );
	    plugin_ptr->processing = 0;
	    if(!status)
	    {
		/* Error configuring. */
	    }
	}

	/* Mark preferences window as ready. */
	PrefSetReady(pref);
 
	reenterant = FALSE;
	return;
}
