#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtkgl/gtkglarea.h>

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"

#include "printwin.h"
#include "printwincb.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


void PrintWinSourceTypeTextInsertCB(
        GtkEditable *editable,
        const gchar *text, gint length, gint *position,
        gpointer data
);
void PrintWinPrintFileBrowseCB(void *widget, void *data);
void PrintWinPrintToToggleCB(GtkWidget *widget, gpointer data);
void PrintWinOrientationToggleCB(GtkWidget *widget, gpointer data);
void PrintWinVisualToggleCB(GtkWidget *widget, gpointer data);
void PrintWinPaperSizeToggleCB(GtkWidget *widget, gpointer data);

gint PrintWinEventCB(
	GtkWidget *widget, gpointer event, gpointer data
);
gint PrintWinPreviewEventCB(
	GtkWidget *widget, gpointer event, gpointer data
);
void PrintWinDestroyCB(GtkObject *object, gpointer data);
gint PrintWinCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

void PrintWinPreviewCenterCB(GtkWidget *widget, gpointer data); 
void PrintWinPreviewOriginalSizeCB(GtkWidget *widget, gpointer data);
void PrintWinPreviewSizeToFitCB(GtkWidget *widget, gpointer data);

void PrintWinPrintButtonCB(GtkWidget *widget, gpointer data);
void PrintWinCancelButtonCB(GtkWidget *widget, gpointer data);


#ifndef GDK_BUTTON1
# define GDK_BUTTON1	1
#endif

#ifndef GDK_BUTTON2
# define GDK_BUTTON2	2
#endif

#ifndef GDK_BUTTON3
# define GDK_BUTTON3	3
#endif


/*
 *	Print window source type combo entry insert_text callback.
 */
void PrintWinSourceTypeTextInsertCB(
	GtkEditable *editable,
	const gchar *text, gint length, gint *position,
	gpointer data
)
{
	gchar *name = NULL;
	print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

	/* If given text is valid, then make a copy of it (since the
	 * given text is not null terminated.
	 */
	if((text != NULL) && (length >= 0))
	{
	    name = (gchar *)g_malloc((length + 1) * sizeof(gchar));
	    memcpy(name, text, length * sizeof(gchar));
	    name[length] = '\0';
	}
	if(name == NULL)
	    return;

	if(pw->map_state)
	{
	    /* Call preview request callback. */
	    gint source_type = PrintWinGetSourceType(pw, name);
            gint visual = PrintWinGetVisual(pw);

            if(pw->preview_request_cb != NULL)
            {
		PrintWinSetBusy(pw);
                pw->preview_request_cb(
                    (gpointer)pw,
                    pw->client_data,
		    source_type,
                    visual,
		    pw->output_width, pw->output_height
                );
		PrintWinSetReady(pw);
            }
            PrintWinPreviewDraw(pw);
	}

	/* Deallocate locals. */
	g_free(name);
}

/*
 *	Print to file name browse button callback.
 */
void PrintWinPrintFileBrowseCB(void *widget, void *data)
{
	gbool status;
	GtkWidget *w;
	GtkEntry *entry;
	const gchar *last_path;
	fb_type_struct **ftype, *ftype_rtn;
	gint total_ftypes;
	gchar **path_rtn;
	gint path_total_rtns;
        print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

	w = pw->print_file_entry;
	if(w == NULL)
	    return;

	entry = GTK_ENTRY(w);

	/* Get last path. */
	last_path = gtk_entry_get_text(entry);

	/* Set up file extension types list. */
	ftype = NULL;
	total_ftypes = 0;
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    ".ps", "PostScript files"
	);
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );


	/* Map file browser and query user for response. */
	FileBrowserSetTransientFor(pw->toplevel);
	status = FileBrowserGetResponse(
	    "Select PostScript File", "Select", "Cancel",
	    last_path,
	    ftype, total_ftypes,
	    &path_rtn, &path_total_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	if(status)
	{
	    if(path_total_rtns > 0)
	    {
		const char *new_path = path_rtn[0];
		if(new_path != NULL)
		{
		    gtk_entry_set_text(entry, new_path);
		}
	    }
	}

	/* Deallocate file extension types list. */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
	ftype = NULL;
	total_ftypes = 0;
}

/*
 *	Print window print to printer or print to file radio callback.
 */
void PrintWinPrintToToggleCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *w;
	GtkToggleButton *toggle_button;
	print_win_struct *pw = (print_win_struct *)data;
	if((widget == NULL) || (pw == NULL))
	    return;

	toggle_button = GTK_TOGGLE_BUTTON(widget);

	if(widget == pw->print_to_printer_radio)
	{
	    w = pw->print_command_entry;
	    if(w != NULL)
	    {
		if(toggle_button->active)
		    gtk_widget_set_sensitive(w, TRUE);
		else
		    gtk_widget_set_sensitive(w, FALSE);
	    }
            w = pw->print_file_entry;
            if(w != NULL)
            {
                if(toggle_button->active)   
                    gtk_widget_set_sensitive(w, FALSE);
                else
                    gtk_widget_set_sensitive(w, TRUE);
            }
	}
        else if(widget == pw->print_to_file_radio)
        {
            w = pw->print_command_entry;
            if(w != NULL)
            {
                if(toggle_button->active)
                    gtk_widget_set_sensitive(w, FALSE);
                else
                    gtk_widget_set_sensitive(w, TRUE);
            }
            w = pw->print_file_entry;
            if(w != NULL)
            {
                if(toggle_button->active)
                    gtk_widget_set_sensitive(w, TRUE); 
                else 
                    gtk_widget_set_sensitive(w, FALSE);
            }
        }

	return;
}

/*
 *	Print window orientation radio callback.
 */
void PrintWinOrientationToggleCB(GtkWidget *widget, gpointer data)
{
	gbool need_update_preview = FALSE;
        GtkToggleButton *toggle_button;
        print_win_struct *pw = (print_win_struct *)data;
        if((widget == NULL) || (pw == NULL))
            return;

        toggle_button = GTK_TOGGLE_BUTTON(widget);

	/* Portrait? */
	if(widget == pw->orient_portrait_radio)
	{
	    if(toggle_button->active)
		need_update_preview = TRUE;
	}
	/* Landscape? */
	else if(widget == pw->orient_landscape_radio)
        {
            if(toggle_button->active)
                need_update_preview = TRUE;
        }

	if(need_update_preview)
	{
	    PrintWinPreviewDraw(pw);
	}

	return;
}

/*
 *	Print window visual radio callback.
 */
void PrintWinVisualToggleCB(GtkWidget *widget, gpointer data)
{
        gbool need_update_preview = FALSE;
        GtkToggleButton *toggle_button;
        print_win_struct *pw = (print_win_struct *)data;
        if((widget == NULL) || (pw == NULL))
            return;

        toggle_button = GTK_TOGGLE_BUTTON(widget);

	if((widget == pw->print_greyscale_radio) ||
           (widget == pw->print_color_radio)
	)
	{
	    if(toggle_button->active)
		need_update_preview = TRUE;
	}

        if(need_update_preview && pw->map_state)
        {
	    gint source_type = PrintWinGetSourceType(pw, NULL);
	    gint visual = PrintWinGetVisual(pw);

	    /* Call preview request callback. */
	    if(pw->preview_request_cb != NULL)
	    {
                PrintWinSetBusy(pw);
		pw->preview_request_cb(
                    (gpointer)pw,
                    pw->client_data,
		    source_type,
                    visual,
		    pw->output_width, pw->output_height
                );
                PrintWinSetReady(pw);
	    }

            PrintWinPreviewDraw(pw);
        }

        return;
}

/*
 *	Print window paper size radio callback.
 */
void PrintWinPaperSizeToggleCB(GtkWidget *widget, gpointer data)
{
        gbool need_update_preview = FALSE;
        GtkToggleButton *toggle_button;
        print_win_struct *pw = (print_win_struct *)data;
        if((widget == NULL) || (pw == NULL))
            return;

        toggle_button = GTK_TOGGLE_BUTTON(widget);

	/* Check if this is the right toggle button. */
	if((widget == pw->paper_size_letter_radio) ||
           (widget == pw->paper_size_legal_radio) ||
	   (widget == pw->paper_size_executive_radio) ||
           (widget == pw->paper_size_a4_radio)
	)
	{
	    if(toggle_button->active)
		need_update_preview = TRUE;
	}

        if(need_update_preview)
        {
            gint paper_widthp, paper_heightp;

            PrintWinGetPaperSize(
                pw, &paper_widthp, &paper_heightp
            );
            PrintWinPreviewSetSize(
                pw, paper_widthp, paper_heightp 
            );
            PrintWinPreviewDraw(pw);
        }

	return;
}

/*
 *	Print window (toplevel not preview) event callback.
 */
gint PrintWinEventCB(
        GtkWidget *widget, gpointer event, gpointer data
)
{
        gbool status = FALSE;
        GtkWidget *w;
        GdkEventConfigure *configure;
        GdkEventKey *key;
	gint etype;
	print_win_struct *pw = (print_win_struct *)data;
	if((event == NULL) || (pw == NULL))
	    return(status);

        if(!pw->initialized)
            return(status);

	etype = *(gint *)event;
        switch(etype)
        {
          case GDK_CONFIGURE:
            w = pw->toplevel;
            configure = (GdkEventConfigure *)event;
            if(w == widget)
            {


                status = TRUE;
            }
            break;

          case GDK_KEY_PRESS: case GDK_KEY_RELEASE:
            key = (GdkEventKey *)event;
            switch(key->keyval)
            {
              case GDK_Alt_L: case GDK_Alt_R:
              case GDK_Control_L: case GDK_Control_R:
              case GDK_Shift_L: case GDK_Shift_R:
                /* Pass modifier to preview event management. */
                status = PrintWinPreviewEventCB(
		    pw->preview_glarea, event, data
		);
                break;
            }
            break;

          default:
            /* Unknown event type but for this widget, can't
             * handle it though so need to return FALSE since
             * we did not handle the event.
             */
            break;
	}

	return(status);
}

/*
 *	Print window preview glarea event callback.
 */
gint PrintWinPreviewEventCB(
        GtkWidget *widget, gpointer event, gpointer data
)
{
        gbool status = FALSE;
        GtkWidget *w;
	GtkMenu *menu;
	GtkGLArea *glarea;
	GdkCursor *cur;
        GdkEventConfigure *configure;
        GdkEventKey *key;
        GdkEventButton *button;
        GdkEventMotion *motion;
        gint etype, x, y;
        GdkModifierType mask;
        print_win_struct *pw = (print_win_struct *)data;
	if((event == NULL) || (pw == NULL))
	    return(status);

	if(!pw->initialized)
	    return(status);

	/* Get preview glarea widget. */
	w = pw->preview_glarea;
	if(w == NULL)
	    return(status);

	glarea = GTK_GL_AREA(w);


	/* Handle by event type. */
	etype = *(gint *)event;
        switch(etype)
        {
          case GDK_EXPOSE:
            PrintWinPreviewDraw(pw);
            status = TRUE;
	    break;

          case GDK_CONFIGURE:
            configure = (GdkEventConfigure *)event;
            if(!pw->preview_realized)
                pw->preview_realized = TRUE;
            if(!PrintWinPreviewEnableContext(pw))
                glViewport(
                    0, 0,
                    configure->width, configure->height
                );
            status = TRUE;
            break;

          case GDK_KEY_PRESS: case GDK_KEY_RELEASE:
            key = (GdkEventKey *)event;
	    if((key->keyval == GDK_Alt_L) ||
	       (key->keyval == GDK_Alt_R)
            )
            {
                if(etype == GDK_KEY_PRESS)
                    pw->flags |= PRINT_WIN_FLAG_MOD_ALT;
                else
                    pw->flags &= ~PRINT_WIN_FLAG_MOD_ALT;
            }
            else if((key->keyval == GDK_Control_L) ||
                    (key->keyval == GDK_Control_R)
            )
            {
                if(etype == GDK_KEY_PRESS)
                    pw->flags |= PRINT_WIN_FLAG_MOD_CTRL;
                else
                    pw->flags &= ~PRINT_WIN_FLAG_MOD_CTRL;
            }
            else if((key->keyval == GDK_Shift_L) ||
                    (key->keyval == GDK_Shift_R)
            )
            {
                if(etype == GDK_KEY_PRESS)
                    pw->flags |= PRINT_WIN_FLAG_MOD_SHIFT;
                else
                    pw->flags &= ~PRINT_WIN_FLAG_MOD_SHIFT;
            }
            status = TRUE;
	    break;

          case GDK_BUTTON_PRESS:
            button = (GdkEventButton *)event;
	    cur = NULL;
            switch(button->button)
            {
              case GDK_BUTTON1:
		if(pw->flags & PRINT_WIN_FLAG_MOD_SHIFT)
		{
		    pw->flags |= PRINT_WIN_FLAG_DRAG_SCALE;
		    cur = pw->scale_cur;
		}
		else
		{
		    pw->flags |= PRINT_WIN_FLAG_DRAG_TRANSLATE;
		    cur = pw->translate_cur;
		}
                break;

	      case GDK_BUTTON2:
		pw->flags |= PRINT_WIN_FLAG_DRAG_SCALE;
		cur = pw->scale_cur;
                break;

	      case GDK_BUTTON3:
		menu = (GtkMenu *)pw->preview_menu;
		if(menu != NULL)
		{
		    /* Map menu. */
		    gtk_menu_popup(
			menu,
			NULL, NULL, NULL, NULL,
                        button->button, button->time
                    );
		}
		break;
	    }
	    if(!GTK_WIDGET_NO_WINDOW(w))
		gdk_window_set_cursor(w->window, cur);
	    /* Record coordinates. */
	    pw->last_drag_x = button->x;
	    pw->last_drag_y = button->y;
	    status = TRUE;
	    break;

          case GDK_BUTTON_RELEASE:
            button = event;
	    /* Stop all drags regardless of button number. */
            pw->flags &= ~PRINT_WIN_FLAG_DRAG_TRANSLATE;
            pw->flags &= ~PRINT_WIN_FLAG_DRAG_SCALE;
            if(!GTK_WIDGET_NO_WINDOW(w))
                gdk_window_set_cursor(w->window, NULL);
	    /* Record coordinates. */
            pw->last_drag_x = button->x;
            pw->last_drag_y = button->y;
	    status = TRUE;
            break;

          case GDK_MOTION_NOTIFY:
            motion = (GdkEventMotion *)event;
            if(motion->is_hint)
            {
                if(pw->flags & (PRINT_WIN_FLAG_DRAG_TRANSLATE |
                                PRINT_WIN_FLAG_DRAG_SCALE
                ))
                    gdk_window_get_pointer(
                        motion->window, &x, &y, &mask
                    );
            }
            else
            {
                x = motion->x;
                y = motion->y;
                mask = motion->state;
            }
	    /* Translating? */
	    if(pw->flags & PRINT_WIN_FLAG_DRAG_TRANSLATE)
	    {
		/* Adjust offset of output and redraw. */
		PrintWinDoTranslate(
		    pw,
		    x - pw->last_drag_x,
		    y - pw->last_drag_y
		);
	    }
	    /* Scaling? */
	    else if(pw->flags & PRINT_WIN_FLAG_DRAG_SCALE)
	    {
		/* Adjust size of output and redraw. */
		PrintWinDoScale(
                    pw,    
                    x - pw->last_drag_x,
                    y - pw->last_drag_y
                );
	    }
            /* Record coordinates. */
            pw->last_drag_x = x;
            pw->last_drag_y = y;
	    status = TRUE;
	    break;

          default:
            /* Unknown event type but for this widget, can't
             * handle it though so need to return FALSE since
             * we did not handle the event.
             */
            break;
	}

	return(status);
}

/*
 *	Print window destroy notify callback.
 */
void PrintWinDestroyCB(GtkObject *object, gpointer data)
{
	return;
}

/*
 *	Print window close callback.
 */
gint PrintWinCloseCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return(FALSE);

	/* Call cancel callback. */
	if(pw->cancel_cb != NULL)
	{
	    pw->cancel_cb(
		(gpointer)pw,
		pw->client_data
	    );
	}

	/* Reset values and unmap print window. */
	PrintWinReset(pw, TRUE);

	return(TRUE);
}

/*
 *	Center preview (adjust xy offsets so image is centered) callback.
 */
void PrintWinPreviewCenterCB(GtkWidget *widget, gpointer data)
{
	gint orientation;
	gint x_min, x_max;
	gint y_min, y_max;
	gint pwidth, pheight;
	gint iwidth, iheight;
        print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

	orientation = PrintWinGetOrientation(pw);

	/* Get bounds in pixels. */
	x_min = 0;
	y_min = 0;
	PrintWinGetPaperSize(
	    pw, &x_max, &y_max
	);

	/* Calculate paper size. */
	pwidth = x_max - x_min;
	pheight = y_max - y_min;

	if(orientation)
	{
	    iwidth = pw->output_height;
	    iheight = pw->output_width;
	}
	else
	{
            iwidth = pw->output_width;
            iheight = pw->output_height;
        }

        /* Center by calculating new image output offset. */
	pw->output_offset_x = (pwidth / 2) - (iwidth / 2);
        pw->output_offset_y = (pheight / 2) - (iheight / 2);

	PrintWinPreviewDraw(pw);

	return;
}

/*
 *	Set size back to the original size (1 pixel per pixel) callback.
 */
void PrintWinPreviewOriginalSizeCB(GtkWidget *widget, gpointer data)
{
        print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

        if(1)
        {
            /* Call preview request callback to get the original
             * preview image again at the original size.
             */
            gint source_type = PrintWinGetSourceType(
                pw, NULL
            );
            gint visual = PrintWinGetVisual(pw);

            if(pw->preview_request_cb != NULL)
            {
                PrintWinSetBusy(pw);
                pw->preview_request_cb(
                    (gpointer)pw,
                    pw->client_data,
                    source_type,
                    visual,
                    pw->output_width, pw->output_height
                );
                PrintWinSetReady(pw);
            }
            PrintWinPreviewDraw(pw);
        }

	return;
}

/*
 *	Increase/decrease size and offset for optimal fit callback.
 */
void PrintWinPreviewSizeToFitCB(GtkWidget *widget, gpointer data)
{
        gint orientation;
        gint x_min, x_max;
        gint y_min, y_max;
        gint pwidth, pheight;   
        gint iwidth, iheight;
	gdouble aspect;
        print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

        if(1)
        {
            /* Call preview request callback to get the original
             * preview image again at the original size.
             */
            gint source_type = PrintWinGetSourceType(
                pw, NULL
            );
            gint visual = PrintWinGetVisual(pw);

            if(pw->preview_request_cb != NULL)
            {
                PrintWinSetBusy(pw);
                pw->preview_request_cb(
                    (gpointer)pw,
                    pw->client_data,
                    source_type,
                    visual,
                    pw->output_width, pw->output_height
                );
                PrintWinSetReady(pw);
            }
/* Redraw later.
	    PrintWinPreviewDraw(pw);
 */
	}

        orientation = PrintWinGetOrientation(pw);

        /* Get bounds in pixels. */   
        x_min = 0;
        y_min = 0;
        PrintWinGetPaperSize(
            pw, &x_max, &y_max
        );

        /* Calculate paper size. */
        pwidth = x_max - x_min;
        pheight = y_max - y_min;

        if(orientation)
        {
            iwidth = pw->output_height;
            iheight = pw->output_width;
        }
        else
        {
            iwidth = pw->output_width; 
            iheight = pw->output_height;
        }

	/* Calculate aspect of the output image. */
	aspect = (gdouble)iwidth / (gdouble)iheight;
	if(aspect < 0.0)
	    return;

	/* First try matching image width to paper width. */
	iwidth = pwidth;
	iheight = iwidth / aspect;

	/* If that caused image height to exceed paper height, then
	 * try the other way.
	 */
	if(iheight > pheight)
	{
	    iheight = pheight;
	    iwidth = iheight * aspect;
	}

	/* Update new image output size. */
	if(orientation)
        {
            pw->output_width = iheight;
            pw->output_height = iwidth;
        }
        else
        {
            pw->output_width = iwidth;
            pw->output_height = iheight;
        }

	/* Center by calculating new image output offset. */
        pw->output_offset_x = (pwidth / 2) - (iwidth / 2);
        pw->output_offset_y = (pheight / 2) - (iheight / 2);

        PrintWinPreviewDraw(pw);

	return;
}

/*
 *	Print window print button callback.
 */
void PrintWinPrintButtonCB(GtkWidget *widget, gpointer data)
{
        print_win_struct *pw = (print_win_struct *)data;
        if(pw == NULL)
            return;

        /* Call print callback. */
        if(pw->print_cb != NULL)
        {
	    /* Allocate and set up parms structure. */
	    print_win_parms_struct *parms = (print_win_parms_struct *)calloc(
		1, sizeof(print_win_parms_struct)
	    );
	    if(parms != NULL)
	    {
		GtkWidget *w;


		/* Reset flags, each flag will be set as its value is
		 * set if and only if the widget is found and valid.
		 */
		parms->flags = 0;

		/* Get print to. */
		parms->flags |= PRINT_WIN_PARM_FLAG_PRINT_TO;
		parms->print_to = PRINT_WIN_PRINT_TO_PRINTER;
		w = pw->print_to_file_radio;
                if((w == NULL) ? 0 : GTK_TOGGLE_BUTTON(w)->active)
		    parms->print_to = PRINT_WIN_PRINT_TO_FILE;

		/* Get print command. */
		parms->flags |= PRINT_WIN_PARM_FLAG_PRINT_COMMAND;
		g_free(parms->print_command);
		parms->print_command = NULL;
		w = pw->print_command_entry;
		if(w != NULL)
		{
		    const gchar *strptr = gtk_entry_get_text(GTK_ENTRY(w));
		    if(strptr != NULL)
			parms->print_command = g_strdup(strptr);
		}

                /* Get print file. */
                parms->flags |= PRINT_WIN_PARM_FLAG_PRINT_FILE;
                g_free(parms->print_file);
                parms->print_file = NULL;
                w = pw->print_file_entry;
                if(w != NULL)
                {
                    const gchar *strptr = gtk_entry_get_text(GTK_ENTRY(w));
                    if(strptr != NULL)
                        parms->print_file = g_strdup(strptr);
                }

                /* Get source type. */
                parms->flags |= PRINT_WIN_PARM_FLAG_SOURCE_TYPE;
		parms->source_type = PrintWinGetSourceType(pw, NULL);

		/* Get orientation. */
		parms->flags |= PRINT_WIN_PARM_FLAG_ORIENTATION;
		parms->orientation = PrintWinGetOrientation(pw);

		/* Get visual. */
                parms->flags |= PRINT_WIN_PARM_FLAG_VISUAL;
		parms->visual = PrintWinGetVisual(pw);

                /* Get pixels per inch (PPI). */
                parms->flags |= PRINT_WIN_PARM_FLAG_PPI_X;
		parms->ppi_x = 72.0;
                w = pw->ppi_x_entry;
                if(w != NULL)
                {
                    const gchar *strptr = gtk_entry_get_text(GTK_ENTRY(w));
                    if(strptr != NULL)
                        parms->ppi_x = atof(strptr);
                }
                parms->flags |= PRINT_WIN_PARM_FLAG_PPI_Y;
                parms->ppi_y = 72.0;
                w = pw->ppi_y_entry;
                if(w != NULL)
                {
                    const gchar *strptr = gtk_entry_get_text(GTK_ENTRY(w));
                    if(strptr != NULL)
                        parms->ppi_y = atof(strptr);
  		}

		/* Get paper size. */
		parms->flags |= PRINT_WIN_PARM_FLAG_PAPER_SIZE;
		w = pw->paper_size_a4_radio;
                if((w == NULL) ? 0 : GTK_TOGGLE_BUTTON(w)->active)
                {
                    parms->paper_size = PRINT_WIN_PAPER_SIZE_A4;
                }
		w = pw->paper_size_executive_radio;
                if((w == NULL) ? 0 : GTK_TOGGLE_BUTTON(w)->active)
                {
                    parms->paper_size = PRINT_WIN_PAPER_SIZE_EXECUTIVE;
                }
                w = pw->paper_size_legal_radio;
                if((w == NULL) ? 0 : GTK_TOGGLE_BUTTON(w)->active)
                {
                    parms->paper_size = PRINT_WIN_PAPER_SIZE_LEGAL;
                }
		else
                {
                    parms->paper_size = PRINT_WIN_PAPER_SIZE_LETTER;
                }

		/* Get output offset in pixels. */
		parms->flags |= PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_X;
		parms->output_offset_xp = pw->output_offset_x;
		parms->flags |= PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_Y;
		parms->output_offset_yp = pw->output_offset_y;

		/* Get output size in pixels. */
                parms->flags |= PRINT_WIN_PARM_FLAG_OUTPUT_WIDTH;
                parms->output_widthp = pw->output_width;
                parms->flags |= PRINT_WIN_PARM_FLAG_OUTPUT_HEIGHT;
                parms->output_heightp = pw->output_height;

		/* Call print callback. */
		PrintWinSetBusy(pw);
		pw->print_cb(
		    (gpointer)pw,
		    pw->client_data,
		    parms
		);
		PrintWinSetReady(pw);

		/* Deallocate parms structure. */
		g_free(parms->print_command);
		g_free(parms->print_file);
		g_free(parms);
		parms = NULL;
	    }
        }

        PrintWinReset(pw, TRUE);
}

/*
 *	Print window cancel button callback.
 */
void PrintWinCancelButtonCB(GtkWidget *widget, gpointer data)
{
	PrintWinCloseCB(widget, NULL, data);
}

