#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "v3dmp.h"
#include "v3dmodel.h"

#include "view.h"
#include "editor.h"
#include "editorselect.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


void EditorSelect(gint **s, gint *t, gint n);
void EditorUnselect(gint **s, gint *t, gint n);
void EditorUnselectAll(gint **s, gint *t);
void EditorSelectSingle(gint **s, gint *t, gint n);
gint EditorGetSelected(gint *s, gint t, gint i);

gint EditorSelectPrimitiveRectangular(
	ma_editor_struct *editor,
	gint view_plane,			/* One of VMA_VIEW2D_TYPE_*. */
	gdouble vi0, gdouble vj0,
	gdouble vi1, gdouble vj1,
	gint start_pn,
	gint *vertex_num
);


/*
 *	Add value n to the selection array as needed.
 *
 *	Value n is not checked to be negative.
 */
void EditorSelect(gint **s, gint *t, gint n)
{
	gint i;


	if((s == NULL) || (t == NULL))
	    return;

	if((*t) < 0)
	    (*t) = 0;

	/* Check if value n is already in list. */
/*
	for(i = 0; i < (*t); i++)
	{
	    if((*s)[i] == n)
		return;
	}
 */

	/* Append value n to list. */
	i = (*t);
	(*t) = i + 1;

	(*s) = (gint *)g_realloc(*s, (*t) * sizeof(gint));
	if((*s) == NULL)
	{
	    (*t) = 0;
	    return;
	}

	(*s)[i] = n;
}

/*
 *	Removes value n from the list if it is in there.
 *
 *	List will be reallocated or free()'ed as needed.
 */
void EditorUnselect(gint **s, gint *t, gint n)
{
	gint i;


        if((s == NULL) || (t == NULL))
            return;

        if((*t) < 0)
            (*t) = 0;

	do
	{
            /* Check if value n is in the list. */
            for(i = 0; i < (*t); i++)
            {
		if((*s)[i] == n)
		    break;
	    }
	    if(i >= (*t))
	    {
		/* Value n not in the list. */
		break;
	    }
	    else
	    {
		/* Value n found in list at index i. */

		/* Shift selection. */
	        for(; i < ((*t) - 1); i++)
		    (*s)[i] = (*s)[i + 1];

		/* Reduce number of items in list by 1. */
		(*t) = (*t) - 1;
		if((*t) > 0)
		{
		    (*s) = (gint *)g_realloc(*s, (*t) * sizeof(gint));
		    if((*s) == NULL)
		    {
			(*t) = 0;
			break;
		    }
		}
		else
		{
		    g_free(*s);
		    (*s) = NULL;
		    (*t) = 0;
		    break;
		}
	    }
	} while(1);
}

/*
 *	Unselects all items in list.
 */
void EditorUnselectAll(gint **s, gint *t)
{
	if(s != NULL)
	{
	    g_free(*s);
	    (*s) = NULL;
	}
	if(t != NULL)
	    (*t) = 0;
}

/*
 *	Clears the list and adds value n to it, so it will be the
 *	one and only value in the list.
 */
void EditorSelectSingle(gint **s, gint *t, gint n)
{
	EditorUnselectAll(s, t);
	EditorSelect(s, t, n);

	return;
}

/*
 *	Returns the selected index n from the given array s.
 *
 *	Can return -1 on error.
 */
gint EditorGetSelected(gint *s, gint t, gint i)
{
	if((s == NULL) || (t <= 0))
	    return(-1);
	if((i < 0) || (i >= t))
	    return(-1);
	return(s[i]);
}


/*
 *	Handles a select rectangular area, returns the primitive
 *	number of the editor's selected model or -1 on failed match.
 *
 *	The search will start on the given primitive number start_pn,
 *	pass -1 or 0 to start from beginning.
 *
 *	On a valid match and if vertex_num is not NULL then vertex_num,
 *	will be set to the value of the vertex number on the primitive
 *	that was selected. However this is not a gaurantee even if
 *	on a valid match vertex_num can still be set to -1.
 *
 *	No modifications of the selection numbers on the editor will take
 *	place, this function only does matching.
 */
gint EditorSelectPrimitiveRectangular(
	ma_editor_struct *editor,
	gint view_plane,		/* One of VMA_VIEW2D_TYPE_*. */
	gdouble vi0, gdouble vj0,
	gdouble vi1, gdouble vj1,
	gint start_pn,
	gint *vertex_num
)
{
	gint model_num, pn, vtx_num;
	v3d_model_struct *model_ptr;
	gpointer p;

	mp_point_struct *mp_point;
        mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
        mp_line_loop_struct *mp_line_loop;
        mp_triangle_struct *mp_triangle;
        mp_triangle_strip_struct *mp_triangle_strip;
        mp_triangle_fan_struct *mp_triangle_fan;
        mp_quad_struct *mp_quad;
        mp_quad_strip_struct *mp_quad_strip;
        mp_polygon_struct *mp_polygon;
        mp_texture_orient_xy_struct *mp_texture_xy;
        mp_texture_orient_yz_struct *mp_texture_yz;
        mp_texture_orient_xz_struct *mp_texture_xz;

	mp_vertex_struct *v0, *v1;


        /* Reset vertex number. */
        if(vertex_num != NULL) 
            (*vertex_num) = -1;

	if(editor == NULL)
	    return(-1);

	/* Get selected model. */
	model_num = EditorSelectedModelIndex(editor);
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return(-1);

	if(model_ptr->type != V3D_MODEL_TYPE_STANDARD)
	    return(-1);


/* Checks if v0 is in bounds. */
#define DO_CHECK_VERTEX_IN_BOUNDS	\
switch(view_plane) \
{ \
 case VMA_VIEW2D_TYPE_XY: \
  if(v0 != NULL) \
  { \
   if((v0->x >= vi0) && (v0->x < vi1) && \
      (v0->y >= vj0) && (v0->y < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  break; \
 case VMA_VIEW2D_TYPE_YZ: \
  if(v0 != NULL) \
  { \
   if((v0->y >= vi0) && (v0->y < vi1) && \
      (v0->z >= vj0) && (v0->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  break; \
 case VMA_VIEW2D_TYPE_XZ: \
  if(v0 != NULL) \
  { \
   if((v0->x >= vi0) && (v0->x < vi1) && \
      (v0->z >= vj0) && (v0->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  break; \
}

/* Checks if v0 or v1 are in bounds. */
#define DO_CHECK_VERTEXES_IN_BOUNDS	\
switch(view_plane) \
{ \
 case VMA_VIEW2D_TYPE_XY: \
  if(v0 != NULL) \
  { \
   if((v0->x >= vi0) && (v0->x < vi1) && \
      (v0->y >= vj0) && (v0->y < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  if(v1 != NULL) \
  { \
   if((v1->x >= vi0) && (v1->x < vi1) && \
      (v1->y >= vj0) && (v1->y < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num + 1; \
    return(pn); \
   } \
  } \
  break; \
 case VMA_VIEW2D_TYPE_YZ: \
  if(v0 != NULL) \
  { \
   if((v0->y >= vi0) && (v0->y < vi1) && \
      (v0->z >= vj0) && (v0->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  if(v1 != NULL) \
  { \
   if((v1->y >= vi0) && (v1->y < vi1) && \
      (v1->z >= vj0) && (v1->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num + 1; \
    return(pn); \
   } \
  } \
  break; \
 case VMA_VIEW2D_TYPE_XZ: \
  if(v0 != NULL) \
  { \
   if((v0->x >= vi0) && (v0->x < vi1) && \
      (v0->z >= vj0) && (v0->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num; \
    return(pn); \
   } \
  } \
  if(v1 != NULL) \
  { \
   if((v1->x >= vi0) && (v1->x < vi1) && \
      (v1->z >= vj0) && (v1->z < vj1) \
   ) \
   { \
    if(vertex_num != NULL) \
     (*vertex_num) = vtx_num + 1; \
    return(pn); \
   } \
  } \
  break; \
}

	/* Sanitize starting primitive index. */
	if(start_pn < 0)
	    start_pn = 0;

	/* Go through each primitive on the model starting on primitive
	 * start_pn.
	 */
	for(pn = start_pn; pn < model_ptr->total_primitives; pn++)
	{
	    p = model_ptr->primitive[pn];
	    if(p == NULL)
		continue;

	    switch(*(gint *)p)
	    {
	      case V3DMP_TYPE_POINT:
		mp_point = p;
                for(vtx_num = 0; vtx_num < V3DMP_POINT_NVERTEX; vtx_num++)
                {
                    v0 = &mp_point->v[vtx_num];
		    v1 = NULL;
                    DO_CHECK_VERTEX_IN_BOUNDS
                }
                break;

	      case V3DMP_TYPE_LINE:
		mp_line = p;
		for(vtx_num = 0; vtx_num < (V3DMP_LINE_NVERTEX - 1); vtx_num++)
		{
		    v0 = &mp_line->v[vtx_num];
                    v1 = &mp_line->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
		}
		break;

              case V3DMP_TYPE_LINE_STRIP:
                mp_line_strip = p;
                for(vtx_num = 0; vtx_num < (mp_line_strip->total - 1); vtx_num++)
                {
                    v0 = mp_line_strip->v[vtx_num];
                    v1 = mp_line_strip->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

              case V3DMP_TYPE_LINE_LOOP:
                mp_line_loop = p;
                for(vtx_num = 0; vtx_num < (mp_line_loop->total - 1); vtx_num++)
                {
                    v0 = mp_line_loop->v[vtx_num];
                    v1 = mp_line_loop->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
		/* Need to check last to first line. */
		if(mp_line_loop->total > 2)
		{
		    v0 = mp_line_loop->v[
			mp_line_loop->total - 1
		    ];
		    v1 = mp_line_loop->v[0];
                    DO_CHECK_VERTEXES_IN_BOUNDS
		}
                break;

              case V3DMP_TYPE_TRIANGLE:
                mp_triangle = p;
                for(vtx_num = 0; vtx_num < (V3DMP_TRIANGLE_NVERTEX - 1); vtx_num++)
                {
                    v0 = &mp_triangle->v[vtx_num];
                    v1 = &mp_triangle->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                /* Need to check last to first line. */
                if(V3DMP_TRIANGLE_NVERTEX > 2)
                {
                    v0 = &mp_triangle->v[
                        V3DMP_TRIANGLE_NVERTEX - 1
                    ];
                    v1 = &mp_triangle->v[0];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

              case V3DMP_TYPE_TRIANGLE_STRIP:
                mp_triangle_strip = p;
                for(vtx_num = 0; vtx_num < (mp_triangle_strip->total - 1); vtx_num++)
                {
                    v0 = mp_triangle_strip->v[vtx_num];
                    v1 = mp_triangle_strip->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS 
                }
                break;

              case V3DMP_TYPE_TRIANGLE_FAN:
                mp_triangle_fan = p;
                for(vtx_num = 0; vtx_num < (mp_triangle_fan->total - 1); vtx_num++)
                {
                    v0 = mp_triangle_fan->v[vtx_num];
                    v1 = mp_triangle_fan->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                /* Need to check last to first line. */
                if(mp_triangle_fan->total > 2)
                {
                    v0 = mp_triangle_fan->v[
                        mp_triangle_fan->total - 1
                    ];
                    v1 = mp_triangle_fan->v[0];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

              case V3DMP_TYPE_QUAD:
                mp_quad = p;
                for(vtx_num = 0; vtx_num < (V3DMP_QUAD_NVERTEX - 1); vtx_num++)
                {
                    v0 = &mp_quad->v[vtx_num];
                    v1 = &mp_quad->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                /* Need to check last to first line. */
                if(V3DMP_QUAD_NVERTEX > 2)
                {
                    v0 = &mp_quad->v[
                        V3DMP_QUAD_NVERTEX - 1
                    ];
                    v1 = &mp_quad->v[0];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

              case V3DMP_TYPE_QUAD_STRIP:
                mp_quad_strip = p;
                for(vtx_num = 0; vtx_num < (mp_quad_strip->total - 1); vtx_num++)
                {
                    v0 = mp_quad_strip->v[vtx_num];
                    v1 = mp_quad_strip->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

              case V3DMP_TYPE_POLYGON:
                mp_polygon = p;
                for(vtx_num = 0; vtx_num < (mp_polygon->total - 1); vtx_num++)
                {
                    v0 = mp_polygon->v[vtx_num];
                    v1 = mp_polygon->v[vtx_num + 1];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                /* Need to check last to first line. */
                if(mp_polygon->total > 2)
                {
                    v0 = mp_polygon->v[
                        mp_polygon->total - 1
                    ];
                    v1 = mp_polygon->v[0];
                    DO_CHECK_VERTEXES_IN_BOUNDS
                }
                break;

	      case V3DMP_TYPE_TEXTURE_ORIENT_XY:
		mp_texture_xy = p;
		if(view_plane == VMA_VIEW2D_TYPE_XY)
		{
		    double	tvi0 = mp_texture_xy->x,
				tvi1 = mp_texture_xy->x + mp_texture_xy->dx;
                    double      tvj0 = mp_texture_xy->y,
                                tvj1 = mp_texture_xy->y + mp_texture_xy->dy;

		    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
		    )
			return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
                    )
                        return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
                    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
		}
		break;

             case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
                mp_texture_yz = p;
                if(view_plane == VMA_VIEW2D_TYPE_YZ)
                {
                    double      tvi0 = mp_texture_yz->y,
				/* Flip y. */
                                tvi1 = mp_texture_yz->y - mp_texture_yz->dy;
                    double      tvj0 = mp_texture_yz->z,
                                tvj1 = mp_texture_yz->z + mp_texture_yz->dz;

                    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
                    )
                        return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
                    )
                        return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
                    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
                }
                break;

             case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
                mp_texture_xz = p;
                if(view_plane == VMA_VIEW2D_TYPE_XZ)
                {
                    double      tvi0 = mp_texture_xz->x,
                                tvi1 = mp_texture_xz->x + mp_texture_xz->dx;
                    double      tvj0 = mp_texture_xz->z,
                                tvj1 = mp_texture_xz->z + mp_texture_xz->dz;

                    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
                    )
                        return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj0 >= vj0) && (tvj0 < vj1)
                    )
                        return(pn);
                    if((tvi1 >= vi0) && (tvi1 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
                    if((tvi0 >= vi0) && (tvi0 < vi1) &&
                       (tvj1 >= vj0) && (tvj1 < vj1)
                    )
                        return(pn);
                }
                break;


	    }
	}


#undef DO_CHECK_VERTEX_IN_BOUNDS
#undef DO_CHECK_VERTEXES_IN_BOUNDS

	return(-1);
}
