/*
                               Image Viewer Widget
 */

#ifndef IMGVIEW_H
#define IMGVIEW_H

#include <sys/types.h>
#include <gtk/gtk.h>

#ifndef gbool
# define gbool 	guchar
#endif

/*
 *	Widget names:
 */
#define IMGVIEW_WNAME_VIEW		"imgview-view"
#define IMGVIEW_WNAME_BUTTON_ZOOM_IN	"imgview-button-zoom-in"
#define IMGVIEW_WNAME_BUTTON_ZOOM_OUT	"imgview-button-zoom-out"
#define IMGVIEW_WNAME_BUTTON_ZOOM_ONETOONE	"imgview-button-zoom-onetoone"
#define IMGVIEW_WNAME_BUTTON_ZOOM_TOFIT	"imgview-button-zoom-tofit"

/*
 *	Image viewer drag modes:
 */
#define IMGVIEW_DRAG_MODE_NONE			0
#define IMGVIEW_DRAG_MODE_TRANSLATE		1
#define IMGVIEW_DRAG_MODE_ZOOM			2
#define IMGVIEW_DRAG_MODE_ZOOM_RECTANGLE	3
#define IMGVIEW_DRAG_MODE_CROP_RECTANGLE	4

/*
 *	Keyboard modifier flags:
 */
#define IMGVIEW_MODIFIER_ALT		(1 << 1)
#define IMGVIEW_MODIFIER_CTRL		(1 << 2)
#define IMGVIEW_MODIFIER_SHIFT		(1 << 3)

/*
 *	Image data format codes:
 */
#define IMGVIEW_FORMAT_GREYSCALE8	10	/* 1 bpp. */
#define IMGVIEW_FORMAT_GREYSCALE16	11	/* 2 bpp. */
#define IMGVIEW_FORMAT_GREYSCALE32	12	/* 4 bpp. */
#define IMGVIEW_FORMAT_GREYSCALEA32	13	/* 4 bpp. */
#define IMGVIEW_FORMAT_RGB		20	/* 3 bpp. */
#define IMGVIEW_FORMAT_RGBA		21	/* 4 bpp. */

/*
 *	Image viewer image structure:
 */
typedef struct {

	gint visual_depth;	/* Bits per pixel of associated GdkDrawable. */

	gint depth;	/* Bits per pixel (does not indicate allocation). */
	gint bpp;	/* Allocated bytes per pixel. */
	gint bpl;	/* Allocated bytes per line. */
	gint	width,	/* Size in pixels. */
		height;
	guint8 *mem;	/* Image data buffer, allocated to the size of
			 * bpl * height bytes.
			 */

} imgview_image_struct;

/*
 *	Image viewer structure:
 */
typedef struct {

	gbool initialized;
	gbool map_state;

	/* If this is a stand alone viewer window, then this will be
	 * set to true, indicating toplevel is a GtkWindow. Otherwise
	 * toplevel will be a GtkVBox.
	 */
	gbool toplevel_is_window;

	/* Specifies if geometry, zoom, zoom rectangular, etc values
	 * should be drawn (including the geometry on the info_label).
	 */
	gbool show_values;

	/* Specifies if a resized version of the image should be drawn
	 * on to the WM icon when ImgViewLoad() is called.
	 */
	gbool show_image_on_wm_icon;

	/* Quality hint, a value from 0 to 2. Where 0 is poor/fastest and
	 * 2 is best/slowest.
	 */
	gint quality;

	/* Specifies the flags for the alpha channel. */
#define IMGVIEW_ALPHA_DEFINED	(1 << 0)	/* Hint that alpha channel
						 * is not all 0xff bytes.
						 */
#define IMGVIEW_ALPHA_THRESHOLD	(1 << 1)	/* Use thresholding instead
						 * of blended alpha.
						 */
#define IMGVIEW_ALPHA_INVERTED	(1 << 2)	/* Alpha values are inverted,
						 * ie 0xff means fully
						 * transparent. Otherwise
						 * 0x00 means fully transparent.
						 */
	guint alpha_channel_flags;

	/* Used if flag IMGVIEW_ALPHA_THRESHOLD is set. */
	guint8 alpha_threshold;

	/* Specifies the current pointer drag mode, one of 
	 * IMGVIEW_DRAG_MODE_*.
	 */
	gint drag_mode;

        /* Keyboard modifiers that affect the drag mode, any of
         * IMGVIEW_MODIFIER_*.
         */
        guint modifiers;

	/* Last drag positions (in window coordinates). */
	gint	drag_last_x,
		drag_last_y;

	/* Last drag mode zoom rectangle (in window coordinates). */
	gint	drag_zoom_rectangle_start_x,
		drag_zoom_rectangle_start_y,
		drag_zoom_rectangle_cur_x,
		drag_zoom_rectangle_cur_y;

	/* Last time motion event was handled (in milliseconds). */
	guint last_motion_time;

	/* Crop information. */
#define IMGVIEW_CROP_ALLOWED	(1 << 0)
#define IMGVIEW_CROP_DEFINED	(1 << 1)
	guint crop_flags;
	/* Crop rectangle coordinates, relative to original image. */
	gint	crop_rectangle_start_x,
		crop_rectangle_start_y,
		crop_rectangle_cur_x,
		crop_rectangle_cur_y;
	gpointer crop_dialog;

	/* Graphic context for view selection cursory. */
	GdkGC *view_selection_gc;
	/* Font used to draw text on the view. */
	GdkFont *view_font;
	/* View background color in RGBA format. */
	guint32 view_bg_pixel;


	GdkCursor	*busy_cur,
			*translate_cur,
			*zoom_cur,
			*zoom_rectangle_cur,
			*crop_cur;

	GdkFont		*font;		/* Default font. */

	GtkAccelGroup	*accelgrp;

	GtkWidget	*toplevel,
			*main_vbox;	/* NULL if toplevel_is_window is FALSE. */

	/* Window manager icon pixmap/mask pair. */
	GdkPixmap *wm_icon_pixmap;
	GdkBitmap *wm_icon_mask;

	/* Indicates if tool bar is mapped or not. */
        gbool toolbar_map_state;

	/* Toolbar toplevel, a GtkContainer. */
	GtkWidget *toolbar_toplevel;
	/* Toolbar widgets. */
	GtkWidget	*info_label,	/* Really a GtkDrawingArea. */
			*zoom_in_btn,
			*zoom_out_btn,
			*zoom_tofit_btn,
			*zoom_onetoone_btn;

        /* Indicates if status bar is mapped or not. */
        gbool statusbar_map_state;

        /* Status bar toplevel, a GtkContainer. */
        GtkWidget *statusbar_toplevel;
        /* Status bar widgets. */
	GtkWidget *progress_bar;

	/* Right-click menu. */
	GtkWidget	*menu,
			*zoom_in_mi,
			*zoom_out_mi,
			*zoom_tofit_mi,
			*zoom_onetoone_mi,
			*show_toolbar_micheck,
			*show_values_micheck,
			*show_statusbar_micheck,
			*quality_submenu_mi;

	/* Quality submenu. */
	GtkWidget	*quality_menu,
			*quality_poor_mi,
                        *quality_optimal_mi,
                        *quality_best_mi;

	/* View translate and zoom values. */
	/* Translation to upper left corner from upper left corner in
	 * window coordinates.
	 */
	GtkAdjustment *view_x_adj, *view_y_adj;

	/* Zoom, original image geometry values are to be divided by
	 * this value. Ie if the zoom was 2.0, then orig_x / 2.0.
	 */
	gdouble view_zoom;

	/* Zoom timeout id. */
	guint view_zoom_toid;

	/* View drawing area. */
	GtkWidget *view_da;

	/* View image buffer. */
	imgview_image_struct *view_img;

	/* Original image. */
	imgview_image_struct *orig_img;

	/* Callbacks. */
	/* Image changed callback (iv, changed_image, client_data).
	 * Specifies when an image has just been modified in some way.
	 */
	void (*changed_cb)(gpointer, imgview_image_struct *, gpointer);
	gpointer changed_data;

} imgview_struct;


/* Image IO. */
extern imgview_image_struct *ImgViewImageNew(
        gint width, gint height, gint depth
);
extern void ImgViewImageClear(
        imgview_image_struct *img, guint32 v
);
extern void ImgViewImageSendRectangle(
        imgview_image_struct *image, GdkDrawable *d, GdkGC *gc, gint quality,
        const GdkRectangle *rect
);
extern void ImgViewImageSend(
        imgview_image_struct *image, GdkDrawable *d, GdkGC *gc, gint quality
);
extern void ImgViewImageDelete(imgview_image_struct *image);

/* Image viewer utilities. */
extern GtkAccelGroup *ImgViewGetAccelGroup(imgview_struct *iv);
extern gbool ImgViewToplevelIsWindow(imgview_struct *iv);
extern GtkWidget *ImgViewGetToplevelWidget(imgview_struct *iv);
extern GtkDrawingArea *ImgViewGetViewWidget(imgview_struct *iv);
extern GtkMenu *ImgViewGetMenuWidget(imgview_struct *iv);
extern gbool ImgViewIsLoaded(imgview_struct *iv);
extern imgview_image_struct *ImgViewGetImage(imgview_struct *iv);
extern guint8 *ImgViewGetImageData(
	imgview_struct *iv,
	gint *width, gint *height, gint *bpl, gint *bpp,
	gint *format
);

/* Image viewer front ends. */
extern void ImgViewClear(imgview_struct *iv);
extern gint ImgViewLoad(
	imgview_struct *iv,
	gint width, gint height,
	gint bytes_per_line,	/* Can be 0 to auto calculate. */
	gint format,		/* One of IMGVIEW_FORMAT_*. */
	const guint8 *data
);
extern gint ImgViewLoadToFit(
        imgview_struct *iv,
        gint width, gint height,
        gint bytes_per_line,    /* Can be 0 to auto calculate. */
        gint format,            /* One of IMGVIEW_FORMAT_*. */
        const guint8 *data
);

extern imgview_struct *ImgViewNew(
	gbool show_toolbar,
	gbool show_values,
	gbool show_statusbar,
	gbool show_image_on_wm_icon,	/* Only if toplevel_is_window is TRUE. */
	gint quality,		/* 0 to 2 (2 being best/slowest). */
	gbool toplevel_is_window,
	GtkWidget **toplevel_rtn
);
extern void ImgViewSetChangedCB(
	imgview_struct *iv,
	void (*changed_cb)(gpointer, imgview_image_struct *, gpointer),
	gpointer data
);
extern void ImgViewReset(imgview_struct *iv, gbool need_unmap);
extern void ImgViewDraw(imgview_struct *iv);
extern void ImgViewUpdateMenus(imgview_struct *iv);

extern void ImgViewShowToolBar(imgview_struct *iv, gbool show);
extern void ImgViewShowStatusBar(imgview_struct *iv, gbool show);

extern void ImgViewSetViewBG(
	imgview_struct *iv,
	GdkColor *c		/* 5 colors. */
);

extern void ImgViewZoomIn(imgview_struct *iv);
extern void ImgViewZoomOut(imgview_struct *iv);
extern void ImgViewZoomOneToOne(imgview_struct *iv);
extern void ImgViewZoomToFit(imgview_struct *iv);

extern void ImgViewAllowCrop(imgview_struct *iv, gbool allow_crop);
extern void ImgViewSetBusy(imgview_struct *iv, gbool is_busy);
extern void ImgViewProgressUpdate(
	imgview_struct *iv, gdouble position, gbool allow_gtk_iteration
);
extern void ImgViewMap(imgview_struct *iv);
extern void ImgViewUnmap(imgview_struct *iv);
extern void ImgViewDelete(imgview_struct *iv);



#endif	/* IMGVOEW_H */
