/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_PAIRS
#define VISU_PAIRS

#include "visu_data.h"
#include "coreTools/toolColor.h"

G_BEGIN_DECLS

/**
 * PAIRS_MIN
 *
 * Flag used to define the minimum length to draw pair. This is useful with
 * the getPairsDistance() and the getPairsDistance() methods.
 */
#define PAIRS_MIN 0
/**
 * PAIRS_MAX
 *
 * Flag used to define the maximum length to draw pair. This is useful with
 * the getPairsDistance() and the getPairsDistance() methods.
 */
#define PAIRS_MAX 1

/**
 * pairsData_struct:
 * @minMax: storage for the length bounds for drawn pairs ;
 * @drawn: a boolean to say if the pair is drawn or not ;
 * @printLength: a boolean to say if length of pairs are drawn near
 * them ;
 * @properties: some key:value association for this link.
 *
 * This structure is used to describe a link between two elements. A
 * link is drawn only its length is between the minimum and the
 * maximum value stored in the minMax array.
 */
struct pairsData_struct
{
  float minMax[2];
  gboolean drawn;
  gboolean printLength;

  GHashTable *properties;
};
/**
 * VisuPairData:
 * 
 * A common way to name #pairsData_struct objects.
 */
typedef struct pairsData_struct VisuPairData;

/**
 * VisuPair_struct:
 * 
 * An opaque structure to store informations about links between two elements.
 */
struct VisuPair_struct;
/**
 * VisuPair:
 * 
 * A common way to name #pairsData_struct objects.
 */
typedef struct VisuPair_struct VisuPair;

/**
 * VisuPairDistribution_struct:
 * @histo: an array containing the distribution ;
 * @nValues: the size of the array ;
 * @stepValue: the step increase in distance at each value ;
 * @nNodesEle1: the number of nodes used during the computation ;
 * @nNodesEle2: idem for #VisuElement 2.
 *
 * This structure stores for a given pair, the distance distribution
 * on a given range [0;@nValues * @stepValue[.
 */
struct VisuPairDistribution_struct
{
  int *histo;
  int nValues;
  float stepValue;
  int nNodesEle1, nNodesEle2;
};
/**
 * VisuPairDistribution:
 *
 * A convenient name for #VisuPairDistribution_struct objects.
 */
typedef struct VisuPairDistribution_struct VisuPairDistribution;


/**
 * initEndOpenGlPairsFunc:
 *
 * Prototype of functions called at the beginning and
 * the end of opengl pairs drawing. Such methods are useful
 * to change some OpenGL variables such as lighting or
 * blending...
 */
typedef void (*initEndOpenGlPairsFunc)();
/**
 * startEndPairsFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object.
 *
 * Prototype of functions called at the beginning and
 * the end of drawing of each pairs types. @ele1 and @ele2
 * arguments are the two elements between the pair defined by @data is drawn.
 * This is useful to set some OpenGL definition specific to each pair, such
 * as the color for example.
 */
typedef void (*startEndPairsFunc)(VisuElement *ele1, VisuElement *ele2, VisuPairData *data);
/**
 * pairDefinitionFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object ;
 * @view: a #OpenGLView object, giving some constants describing
 *        the OpenGL scene ;
 * @x1: a floating point value ;
 * @y1: a floating point value ;
 * @z1: a floating point value ;
 * @x2: a floating point value ;
 * @y2: a floating point value ;
 * @z2: a floating point value ;
 * @d2: a floating point value ;
 * @alpha: a floating point value.
 *
 * Prototype of function to draw a pair. Such function are called each time a pair
 * is drawn between the two points (@x1, @y1, @z1) and (@x2, @y2, @z2). The @d2 argument
 * is the square distance between the two points. The @alpha argument
 * is a proposed alpha colour from the main program, its value is in [0;1].
 */
typedef void (*pairDefinitionFunc)(VisuElement *ele1, VisuElement *ele2,
				   VisuPairData *data, OpenGLView *view,
				   double x1, double y1, double z1,
				   double x2, double y2, double z2,
				   float d2, float alpha);
/**
 * foreachPairsFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object ;
 * @userData: some user defined data.
 *
 * Prototype of functions called with the foreach method apply to each
 * pairs.
 */
typedef void (*foreachPairsFunc)(VisuElement *ele1, VisuElement *ele2,
				 VisuPairData *data, gpointer userData);

/**
 * PairsExtension_struct:
 *
 * Structure to store pairs extensions. All fields are private and should not
 * be accessed directly.
 */
struct PairsExtension_struct;
/**
 * PairsExtension:
 * 
 * Common naming for #PairsExtension_struct objects.
 */
typedef struct PairsExtension_struct PairsExtension;

/**
 * visuPairExtensionNew:
 * @name: name of the extension (must be non null) ;
 * @printName: a string to label the method that can be safely translated ;
 * @description: a brief description of the extension (can be null) ;
 * @sensitive: a boolean 0 or 1 ;
 * @init: a initEndOpenGlPairsFunc() method or NULL ;
 * @stop: a initEndOpenGlPairsFunc() method or NULL ;
 * @start: a startEndPairsFunc() method or NULL ;
 * @end: a startEndPairsFunc() method or NULL ;
 * @draw: a pairDefinitionFunc() method (not NULL).
 *
 * This creates a new pairs extension. Such an extension describes how to draw
 * links (called pairs) between elements. The @sensitive argument is to inform if
 * pairs must be redrawn when the OpenGL engine sends the OpenGLFacetteChanged signal.
 *
 * Returns: the new PairsExtension or null if something wrong happens.
 */
PairsExtension* visuPairExtensionNew(const char* name, const char* printName,
				     const char* description, gboolean sensitive,
				     initEndOpenGlPairsFunc init,
				     initEndOpenGlPairsFunc stop,
				     startEndPairsFunc start,
				     startEndPairsFunc end,
				     pairDefinitionFunc draw);
/**
 * visuPairExtensionFree:
 * @extension: the extension to delete.
 *
 * Free all the allocated attributes of the specified method.
 */
void visuPairExtensionFree(PairsExtension* extension);
/**
 * visuPairExtensionAdd:
 * @extension: an extension.
 *
 * A method used by user to registered a new extension.
 */
void visuPairExtensionAdd(PairsExtension *extension);
/**
 * visuPairExtensionSet:
 * @extension: a #PairsExtension object.
 *
 * Choose the method used to draw pairs. If necessary, visuPairBuild() are called.
 *
 * Returns: TRUE if the OpenGLAskForReDraw signal show be emitted or not.
 */
gboolean visuPairExtensionSet(PairsExtension *extension);
/**
 * visuPairExtensionGet_allMethods:
 * 
 * Useful to know all #PairsExtension.
 *
 * Returns: a list of all the known #PairsExtension. This list should
 *          be considered read-only.
 */
GList* visuPairExtensionGet_allMethods();
/**
 * visuPairExtensionGet_current:
 *
 * If some process need to know the current #PairsExtension. Such extension has
 * been set with setPairsMethod().
 *
 * Returns: the current #PairsExtension, NULL if none has been set.
 */
PairsExtension* visuPairExtensionGet_current();



/**
 * visuPairBuild:
 * @dataObj: the #VisuData object to create pairs for.
 *
 * This methods recreates the OpenGL list of the #OpenGLExtension associated
 * to the pairs.
 * Thus it is the heart of the pairs drawing. When called, if a
 * valid #PairsExtension has been set with setPairsMethod(), it takes all the
 * nodes and compute all the distances between two different nodes. If this distance
 * is compatible with the distance of drawn pairs for the two elements it calls
 * the pairDefinitionFunc() for these two nodes. WARNING! this method is very expensive
 * in computing cost.
 *
 * Returns: TRUE if the OpenGLAskForReDraw signal show be emitted or not.
 */
gboolean visuPairBuild(VisuData *dataObj);

/**
 * visuPairSet_status:
 * @onOff : a boolean 0 or 1.
 * 
 * Turn on or off the pairs.
 *
 * Returns: TRUE if visuPairBuild() should be called and then the 'OpenGLAskForReDraw'
 *          signal be emitted.
 */
gboolean visuPairSet_status(gboolean onOff);
/**
 * visuPairSet_drawn:
 * @data: a #PairsData object ;
 * @drawn: a boolean.
 *
 * A pair can or cannot be drawn, use this method to tune it.
 *
 * Returns: TRUE if visuPairBuild() should be called.
 */
gboolean visuPairSet_drawn(VisuPairData *data, gboolean drawn);
/**
 * visuPairGet_drawn:
 * @data: a #PairsData object ;
 *
 * A pair can or cannot be drawn, use this method to retrieve its state.
 *
 * Returns: TRUE if pairs can be drawn.
 */
gboolean visuPairGet_drawn(VisuPairData *data);
/**
 * visuPairSet_distance:
 * @val: a floating point value ;
 * @data: a #PairsData object ;
 * @minOrMax: #PAIRS_MAX or #PAIRS_MIN.
 *
 * Set the minimum or the maximum length for the given pair.
 *
 * Returns: TRUE if visuPairBuild() should be called or not.
 */
gboolean visuPairSet_distance(VisuPairData *data, float val, int minOrMax);
/**
 * visuPairSet_color:
 * @data: a #PairsData object ;
 * @destColor: a #Color object.
 *
 * Set the color of the given pair.
 *
 * Returns: TRUE if visuPairBuild() should be called or not.
 */
gboolean visuPairSet_color(VisuPairData *data, Color* destColor);
Color* visuPairGet_color(VisuPairData *data);
/**
 * visuPairSet_printLength:
 * @data: a #PairsData object ;
 * @status: TRUE to print length near pairs.
 *
 * Set the attribute that controls if the length of pairs are drawn near pairs.
 *
 * Returns: TRUE if visuPairBuild() should be called or not.
 */
gboolean visuPairSet_printLength(VisuPairData *data, gboolean status);
/**
 * visuPairSet_outOfDate:
 *
 * Use this method to change the internal flag that pairs should be
 * rebuilt. It is useful when an extension of pairs has one of its
 * parameters that has changed.
 */
void visuPairSet_outOfDate();
/**
 * visuPairGet_status:
 * 
 * Get the status of pairs, drawn or not.
 *
 * Returns: TRUE if pairs are drawn.
 */
gboolean visuPairGet_status();
/**
 * visuPairSet_linkProperty:
 * @data: a #PairsData object ;
 * @key: a static string ;
 * @value: a pointer to some allocated data.
 *
 * Each pair can store some informations that can be retrieve by a string key.
 * This method is used to registered a new value associated to the key @key.
 */
void visuPairSet_linkProperty(VisuPairData *data, const gchar* key, gpointer value);
/**
 * visuPairSet_property:
 * @pair: a #Pairs object ;
 * @key: a static string ;
 * @value: a pointer to some allocated data ;
 * @freeFunc: a destroying method (can be NULL).
 *
 * Each element/element can have associated data.
 */
void visuPairSet_property(VisuPair *pair, const gchar* key,
			  gpointer value, GDestroyNotify freeFunc);
/**
 * visuPairGet_property:
 * @pair: a #VisuPair object ;
 * @key: a string.
 *
 * Retrieve the property associated to the @key or NULL if none exist.
 *
 * Returns: the associated data.
 */
gpointer visuPairGet_property(VisuPair *pair, const gchar* key);
/**
 * visuPairGet_distance:
 * @data: a #PairsData object ;
 * @minOrMax: #PAIRS_MIN or #PAIRS_MAX.
 *
 * A pair between @ele1 and @ele2 is drawn only if its length is between
 * a minimum and a maximum value. This method can get these values.
 *
 * Returns: the minimum or the maximum value for the pair between @ele1 and @ele2.
 */
float visuPairGet_distance(VisuPairData *data, int minOrMax);
/**
 * visuPairGet_printLength:
 * @data: a #PairsData object.
 * 
 * Get the print length parameter of a pair. This parameter is used to tell if
 * length should be drawn near pairs of this kind.
 *
 * Returns: TRUE if length are printed.
 */
gboolean visuPairGet_printLength(VisuPairData *data);
/**
 * visuPairGet_linkProperty:
 * @data: a #PairsData object ;
 * @key: a static string.
 *
 * Each pair can store some informations that can be retrieve by a string key.
 * This method is used to retrieve a stored value associated to the key @key.
 *
 * Returns: a found value, or NULL if nothing is associated to the @key.
 *          If something is returned it should not be freed.
 */
gpointer visuPairGet_linkProperty(VisuPairData *data, const gchar* key);
/**
 * visuPairGet_pair:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object.
 *
 * The object #VisuPair is used to characterized links between two elements.
 *
 * Returns: the #VisuPair object associated to the given two elements. If
 *          none exists it is created. The returned value should not be freed.
 */
VisuPair* visuPairGet_pair(VisuElement *ele1, VisuElement *ele2);
/**
 * visuPairGet_distanceDistribution:
 * @pair: a #VisuPair ;
 * @dataObj: a #VisuData ;
 * @step: a float for the distance mesh (negative value to use
 * built-in default) ;
 * @max: a float for the maximum scanning value (negative value to use
 * built-in default).
 * 
 * This will compute the distnace distribution of nodes for the given
 * @pair.
 *
 * Returns: a structure defining the distance distribution. This
 * structure is private and should not be freed.
 */
VisuPairDistribution* visuPairGet_distanceDistribution(VisuPair *pair,
						       VisuData *dataObj,
						       float step, float max);

/**
 * visuPairGet_links:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object.
 *
 * There can be one or several links between elements, retrieve them
 * with this routine.
 *
 * Returns: a list of #VisuPairData. The list is owned by V_Sim and
 * should not be freed.
 */
GList* visuPairGet_links(VisuElement *ele1, VisuElement *ele2);
/**
 * visuPairGet_link:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @minMax: the two min and max distances.
 *
 * A link between two elements is characterized by its boundary distances.
 *
 * Returns: the #VisuPairData object associated to the given two
 *          elements and distances. If none exists it is created. The
 *          returned value should not be freed.
 */
VisuPairData* visuPairGet_link(VisuElement *ele1, VisuElement *ele2, float minMax[2]);
/**
 * visuPairGet_linkFromId:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @pos: the position in the list of links.
 *
 * A link can also be retrieved by its position.
 *
 * Returns: the #VisuPairData object associated to the given two
 *          elements and distances. If none exists NULL is returned.
 */
VisuPairData* visuPairGet_linkFromId(VisuElement *ele1, VisuElement *ele2, guint pos);
/**
 * visuPairRemove_link:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a link object.
 *
 * Delete the given link.
 *
 * Returns: TRUE if the link exists and has been successfully removed.
 */
gboolean visuPairRemove_link(VisuElement *ele1, VisuElement *ele2, VisuPairData *data);
/**
 * visuPairForeach:
 * @whatToDo: a foreachPairsFunc() method ;
 * @userData: some user defined data.
 *
 * The way #PairsData are stored in V_Sim is private and could changed between version.
 * This method is used to apply some method each pairs.
 */
void visuPairForeach(foreachPairsFunc whatToDo, gpointer userData);
/**
 * visuPairRead_linkFromTokens:
 * @tokens: array of tokens resulting from a call to g_strsplit() with " " as separator ;
 * @index: IN, the position of the beginning in @tokens ; OUT, one token
 *            after the last read ;
 * @data: a pointer to return an allocated link object ;
 * @position: the number of the line of the config
 *            file which the @line argument is taken from ;
 * @error: a location to store a possible reading error.
 *
 * This routine is used to read the resource file. Given @tokens, it
 * associate a link object by reading the two elements and the two
 * distances that characterised this link.
 *
 * Returns: TRUE if succeed.
 */
gboolean visuPairRead_linkFromTokens(gchar **tokens, int *index, VisuPairData **data,
				     int position, GError **error);


/****************/
/* Private area */
/****************/

/**
 * PairsExtension_struct:
 * @name: an internal name ;
 * @printName: the UTF-8 name, used for the GUI ;
 * @description: an UTF-8 short description of the pairs method.
 * @sensitiveToFacette: if TRUE, the OpenGL list of the pairs is recompute
 *                      each time the number of facette changes ;
 * @initOpenGl: method called sometime ;
 * @stopOpenGl: method called sometime ;
 * @beginDrawingPairs: method called a startup of drawing pairs ;
 * @endDrawingPairs: method called a ending of drawing pairs ;
 * @drawPairs: the drawing method for each pair.
 *
 * Structure to store pairs extensions. All fields are private and should not
 * be accessed directly. This structure will not be public in near future, do not use it.
 */
struct PairsExtension_struct
{
  /* Some variable to describe this OpenGL extension.
     The attribute name is mandatory since it is
     used to identify the method. */
  char* name;
  char* printName;
  char* description;

  gboolean sensitiveToFacette;

  initEndOpenGlPairsFunc initOpenGl;
  initEndOpenGlPairsFunc stopOpenGl;
  startEndPairsFunc beginDrawingPairs;
  startEndPairsFunc endDrawingPairs;
  pairDefinitionFunc drawPairs;
};


/**
 * initPairsModule:
 *
 * Initialise all the variable of this part. It calls all
 * the elements in %listInitPairsFunc (that stores the
 * init function of the pairs extensions). If these elements return
 * valid PairsExtension, they are registered through a call
 * to registerPairsExtension().
 * This method is called by the main program at the initialisation stage and
 * should not be called in others circonstempses.
 *
 * Returns: 1 if everything goes allright during the initialisation.
 */
int initPairsModule();

G_END_DECLS

#endif
