/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <math.h>

#include "gtkAtomic.h"

#include <renderingMethods/renderingAtomic.h>
#include "panelElements.h"
#include <visu_gtk.h>
#include <renderingBackend/visu_windowInterface.h>

/**
 * SECTION:gtkAtomic
 * @short_description: The gtk interface elements that are specific to
 * the atomic rendering method.
 *
 * <para>This part gathers all the routine specific to the widgets
 * related to the atomic rendering method. For the method itself,
 * there is no specific widgets. For the #VisuElement part, one can
 * tune the shape, the radius and the elipsoid orientation (when
 * selected). There is no specific #GtkFileChooser widget.</para>
 */

#define SPIN_ATOMIC_RADIUS_UPPER (gdouble)999.
#define SPIN_ATOMIC_RADIUS_LOWER (gdouble)0.001
#define SPIN_ATOMIC_RADIUS_STEP  (gdouble)0.05
#define LABEL_RADIUS      _("Radius:")

/* the spin button to control the radius. */
static GtkWidget *spinRadius;
static GtkWidget *entryShape;
static GtkWidget *labelRadius;
static GtkWidget *spinRatio;
static GtkWidget *spinPhi;
static GtkWidget *spinTheta;

enum
  {
    paramRadius,
    paramShape,
    paramRatio,
    paramPhi,
    paramTheta
  };

/* Pointer on a list to all selected elements. */
GList *currentListElement;

/* Callbacks */
static void paramChanged(GtkSpinButton* button, gpointer data);
static void shapeChanged(GtkComboBox *box, gpointer data);

/* Local routines. */


/***************/
/* Public part */
/***************/

/* Initialise the specific area in the element panel
   for the atomic rendering method. */
void panelElementAtomicInit_gtkPanel()
{
  panelElements_setInterfaceMethods(visuRenderingClassGet_methodByName
				    (VISU_RENDERING_ATOMIC),
				    panelElementAtomicCallback_onElementChange,
				    panelElementAtomicCreate_gtkInterface);
  currentListElement = (GList*)0;
}

/* Initialise the gtk methods associated with
   the atomic rendering method. */
void panelElementAtomicInit_gtkMain()
{
  visuGtkSet_renderingSpecificMethods(visuRenderingClassGet_methodByName
				      (VISU_RENDERING_ATOMIC),
				      (VisuGtkSetFilesFunc)0);
}



/* Create the gtk widgets (a hbox with a spin with
   positive values) and return it. */
GtkWidget* panelElementAtomicCreate_gtkInterface()
{
  GtkWidget* hbox, *vbox;
  GtkWidget* label;
  GtkObject *adj;
  GtkWidget *comboShape;
  const char **names;
  int i;

  DBG_fprintf(stderr, "GTK Atomic : create the gtk interface.\n");

  vbox = gtk_vbox_new(FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

  labelRadius = gtk_label_new("");
  gtk_label_set_text(GTK_LABEL(labelRadius), LABEL_RADIUS);
  gtk_box_pack_start(GTK_BOX(hbox), labelRadius, FALSE, FALSE, 1);

  adj = gtk_adjustment_new(1.0, SPIN_ATOMIC_RADIUS_LOWER,
			   SPIN_ATOMIC_RADIUS_UPPER, SPIN_ATOMIC_RADIUS_STEP,
			   0.1, 0.);
  spinRadius = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0.01, 3);
  gtk_box_pack_start(GTK_BOX(hbox), spinRadius, FALSE,FALSE, 3);

  label = gtk_label_new(_("Shape: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 1);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);

  comboShape = gtk_combo_box_new_text();
  names = renderingAtomicGet_allShapesI18n();
  if (names)
    for (i = 0; names[i]; i++)
      gtk_combo_box_append_text(GTK_COMBO_BOX(comboShape), names[i]);
  else
    gtk_combo_box_append_text(GTK_COMBO_BOX(comboShape),
			      (char*)renderingAtomicGet_shapeNameDefault());
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboShape), 0);
  entryShape = comboShape;
  /* set callback for the combo button. */
  g_signal_connect(G_OBJECT(entryShape), "changed",
		   G_CALLBACK(shapeChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), comboShape, FALSE, FALSE, 3);

  /* set callback for the spin button. */
  g_signal_connect((gpointer)spinRadius, "value-changed",
		   G_CALLBACK(paramChanged), GINT_TO_POINTER(paramRadius));

  /* Set widgets for the elipsoid parameters. */
  label = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(label), _("<i>Parameters for elipsoid shape</i>"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(label), 10, 0);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Ratio: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  spinRatio = gtk_spin_button_new_with_range(1., 10., 0.1);
  gtk_box_pack_start(GTK_BOX(hbox), spinRatio, FALSE, FALSE, 0);
  label = gtk_label_new(_("Phi: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  spinPhi = gtk_spin_button_new_with_range(-180., 180., 1.);
  gtk_box_pack_start(GTK_BOX(hbox), spinPhi, FALSE, FALSE, 0);
  label = gtk_label_new(_("Theta: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  spinTheta = gtk_spin_button_new_with_range(-180., 180., 1.);
  gtk_box_pack_start(GTK_BOX(hbox), spinTheta, FALSE, FALSE, 0);

  g_signal_connect((gpointer)spinRatio, "value-changed",
		   G_CALLBACK(paramChanged), GINT_TO_POINTER(paramRatio));
  g_signal_connect((gpointer)spinPhi, "value-changed",
		   G_CALLBACK(paramChanged), GINT_TO_POINTER(paramPhi));
  g_signal_connect((gpointer)spinTheta, "value-changed",
		   G_CALLBACK(paramChanged), GINT_TO_POINTER(paramTheta));
  

  gtk_widget_show_all(vbox);
  return vbox;
}

/* This function is called whenever an element is changed. */
void panelElementAtomicCallback_onElementChange(GList *eleList)
{
  float radius, ratio, phi, theta;
  int shape;
  VisuElement *ele;

  g_return_if_fail(eleList);

  if (currentListElement)
    g_list_free(currentListElement);
  currentListElement = g_list_copy(eleList);
  
  /* If the list has only one element, we continue
     and update the values on the widgets. */
  if (g_list_next(eleList))
    return;

  ele = (VisuElement*)eleList->data;
  /* Change the radius for the new element. */
  radius = renderingAtomicGet_radius(ele);
  if (radius >= 0.)
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinRadius), radius);
  else
    {
      g_warning("Can't find a value for radius of element '%s'.\n", ele->name);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinRadius),
				renderingAtomicGet_radiusDefault());
    }
  /* Change the shape for the new element. */
  shape = renderingAtomicGet_shape(ele);
  if (shape >= 0)
    gtk_combo_box_set_active(GTK_COMBO_BOX(entryShape), shape);
  else
    {
      g_warning("Can't find the shape of element '%s'.\n", ele->name);
      gtk_combo_box_set_active(GTK_COMBO_BOX(entryShape), renderingAtomicGet_shapeDefault());
    }
  /* Change the elipsoid parameters. */
  ratio = renderingAtomicGet_elipsoidRatio(ele);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinRatio), ratio);
  phi = renderingAtomicGet_elipsoidPhi(ele);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinPhi), phi);
  theta = renderingAtomicGet_elipsoidTheta(ele);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTheta), theta);
}




/****************/
/* Private part */
/****************/
static void shapeChanged(GtkComboBox *box, gpointer data _U_)
{
  int shape;
  GList *tmpLst;
  gboolean refresh;
  VisuElement *ele;
  int id;
  visuRenderingCreate_elementFunc createElement;

  g_return_if_fail(currentListElement);

  shape = (int)gtk_combo_box_get_active(box);

  createElement = visuRenderingClassGet_currentElementFunc();
  refresh = FALSE;
  tmpLst = currentListElement;
  while (tmpLst)
    {
      ele = (VisuElement*)tmpLst->data;
      if (renderingAtomicSet_shape(ele, shape))
	{
	  id = createElement(toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS)), ele);
	  ele->openGLIdentifier = id;
	  refresh = TRUE;
	}
      tmpLst = g_list_next(tmpLst);
    }
  if (refresh)
    g_idle_add(visuObjectRedraw, (gpointer)0);
}

static void paramChanged(GtkSpinButton* button, gpointer data)
{
  int param, id;
  float value;
  VisuElement *ele;
  VisuData *dataObj;
  GList *tmpLst;
  gboolean refresh, res;
  visuRenderingCreate_elementFunc createElement;

  g_return_if_fail(currentListElement);

  dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));

  param = GPOINTER_TO_INT(data);
  value = gtk_spin_button_get_value(button);

  createElement = visuRenderingClassGet_currentElementFunc();
  refresh = FALSE;
  tmpLst = currentListElement;
  while (tmpLst)
    {
      ele = (VisuElement*)tmpLst->data;
      switch (param)
	{
	case paramRadius:
	  res = renderingAtomicSet_radius(ele, value);
	  break;
	case paramRatio:
	  res = renderingAtomicSet_elipsoidRatio(ele, value);
	  break;
	case paramPhi:
	  res = renderingAtomicSet_elipsoidPhi(ele, value);
	  break;
	case paramTheta:
	  res = renderingAtomicSet_elipsoidTheta(ele, value);
	  break;
	default:
	  res = FALSE;
	}
      if (res)
	{
	  id = createElement(dataObj, ele);
	  ele->openGLIdentifier = id;
	  refresh = TRUE;
	}
      tmpLst = g_list_next(tmpLst);
    }
  if (refresh)
    g_idle_add(visuObjectRedraw, (gpointer)0);
}
