/* Simple converter between RGB / HLS.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "color.h"
#include <math.h>

static void rgb_to_hls (gdouble r, gdouble g, gdouble b,
			gdouble *h, gdouble *l, gdouble *s);

static gdouble hls_to_rgb_1 (gdouble n1, gdouble n2, gdouble hue);

static void hls_to_rgb (gdouble h, gdouble l, gdouble s,
			gdouble *r, gdouble *g, gdouble *b);

static void load_rgb_database (const gchar *rgb_color_database);

static GHashTable *rgb_hash_table = NULL;

inline static UptColor *
_upt_color_new (void)
{
  UptColor *color;

  color = g_new0 (UptColor, 1);
  color->free = free;

  return color;
}

UptColor *
upt_color_by_rgb (red, green, blue)
     gdouble red, green, blue;
{
  UptColor *color;

  color = _upt_color_new ();
  color->space = COLORSPACE_RGB;
  color->data.rgb.red = red;
  color->data.rgb.green = green;
  color->data.rgb.blue = blue;

  return color;
}

UptColor *
upt_color_by_hls (hue, lightness, saturation)
     gdouble hue, lightness, saturation;
{
  UptColor *color;

  color = _upt_color_new ();
  color->space = COLORSPACE_HLS;
  color->data.hls.hue = hue;
  color->data.hls.lightness = lightness;
  color->data.hls.saturation = saturation;

  return color;
}

UptColor *
upt_color_by_name (name)
     const gchar *name;
{
  UptColor *color;

  if (!rgb_hash_table)
    load_rgb_database (RGBColorDatabase);
  color = g_hash_table_lookup (rgb_hash_table, name);

  return color;
}

static void
load_rgb_database (rgb_color_database)
     const gchar *rgb_color_database;
{
  UptColor *color;
  FILE *database;
  gchar name[64], text[256];
  gint red, green, blue, count;

  database = fopen (rgb_color_database, "r");
  if (!database)
    return;

  rgb_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
  while (fgets (text, sizeof(text), database))
    {
      count = sscanf (text, "%d %d %d %[^\n]\n", &red, &green, &blue,
		      name);
      if (count != 4)
	continue;
      color = upt_color_by_rgb (red / (gdouble)0xff, green / (gdouble)0xff,
				blue / (gdouble)0xff);
      g_hash_table_insert (rgb_hash_table, g_strdup (name), color);
    }
  fclose (database);
}

void
upt_color_to_hls (color)
     UptColor *color;
{
  if (color->space == COLORSPACE_HLS)
    return;

  color->space = COLORSPACE_HLS;
  rgb_to_hls (color->data.rgb.red,
	      color->data.rgb.green,
	      color->data.rgb.blue,
	      &color->data.hls.hue,
	      &color->data.hls.lightness,
	      &color->data.hls.saturation);
}

void
upt_color_to_rgb (color)
     UptColor *color;
{
  if (color->space == COLORSPACE_RGB)
    return;

  color->space = COLORSPACE_RGB;
  hls_to_rgb (color->data.hls.hue,
	      color->data.hls.lightness,
	      color->data.hls.saturation,
	      &color->data.rgb.red,
	      &color->data.rgb.green,
	      &color->data.rgb.blue);
}

ggi_pixel
upt_color_map (visual, color)
     ggi_visual_t visual;
     UptColor *color;
{
  ggi_color map_color;
  ggi_pixel pixel;

  upt_color_to_rgb (color);
  map_color.r = color->data.rgb.red * 0xffff;
  map_color.g = color->data.rgb.green * 0xffff;
  map_color.b = color->data.rgb.blue * 0xffff;
  pixel = ggiMapColor (visual, &map_color);

  return pixel;
}

UptColor *
upt_color_unmap (visual, pixel)
     ggi_visual_t visual;
     ggi_pixel pixel;
{
  UptColor *color;
  ggi_color map_color;

  ggiUnmapPixel (visual, pixel, &map_color);
  color = upt_color_by_rgb (map_color.r / (gdouble)0xffff,
			    map_color.g / (gdouble)0xffff,
			    map_color.b / (gdouble)0xffff);

  return color;
}

static void
rgb_to_hls (r, g, b, h, l, s)
     gdouble r, g, b, *h, *l, *s;
{
  gdouble max = MAX(MAX(r, g), b), min = MIN(MIN(r, g), b),
    delta = max - min;

  *l = (max + min) / 2;
  if (!delta)
    {
      *s = 0;
      return;
    }
  if (*l <= 0.5)
    *s = delta  / (max + min);
  else
    *s = delta / (2 - max - min);
  if (r == max)
    *h = (g - b) / delta;
  else if (g == max)
    *h = 2 + (b - r) / delta;
  else if (b == max)
    *h = 4 + (r - g) / delta;
  *h *= 60;
  if (*h < 0)
    *h += 360;
}

static gdouble
hls_to_rgb_1 (n1, n2, hue)
     gdouble n1, n2, hue;
{
  gdouble value;

  hue = fmod (hue, 360.0);
  if (hue < 0)
    hue += 360;

  if (hue < 60)
    value = n1 + (n2 - n1) * hue / 60;
  else if (hue >= 60 && hue < 180)
    value = n2;
  else if (hue >= 180 && hue < 240)
    value = n1 + (n2 - n1) * (240 - hue) / 60;
  else
    value = n1;

  return value;
}

static void
hls_to_rgb (h, l, s, r, g, b)
     gdouble h, l, s, *r, *g, *b;
{
  gdouble m1, m2;

  if (l <= 0.5)
    m2 = l * (1 + s);
  else
    m2 = l + s - l * s;
  m1 = 2 * l - m2;

  if (!s && !h)
    {
      *r = *g = *b = l;
      return;
    }

  *r = hls_to_rgb_1 (m1, m2, h + 120);
  *g = hls_to_rgb_1 (m1, m2, h);
  *b = hls_to_rgb_1 (m1, m2, h - 120);
}
