/*
   Uptime Daemon
   Copyright (C) 1998 Matthew Trent <root@piguy.dyn.ml.org>
   Changes by Johnny Teveen <j.tevessen@line.org>
   Changes by Andreas Muck <andi@koala.rhein-neckar.de>
   Changes by John Campbell <jcampbel@lynn.ci-n.com>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 */

/** User configurable options **/

/* Where to write records to */
#define DEFAULT_UPTIME_RECORD_FILE "/etc/uptime.record"

/* How often to check (in seconds) */
int UPTIME_UPDATE_INTERVAL = 120;

/* file to write PID to */
#define DEFAULT_PID_FILE "/var/run/"PACKAGE".pid"

/* File to write HTML stats to */
char STATS_FILE[128] = "/home/httpd/ud.html";

/* Template file for HTML customization */
char INPUT_FILE[128] = "/etc/template.ud";

/* You may change this */
#define HTML_BODY "BGCOLOR=#000000 TEXT=#FFFFFF ALINK=#8E0000 VLINK=#FFA600"

/* End user serviceable part */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>
#include "ud.h"

static struct record ut_record;

/* These variables and defines added by John Campbell <jcampbel@lynn.ci-n.com> 
   to support changing the PID file and the record file with command-line 
   switches. */
#define UPTIME_RECORD_FILE (uptime_record_file)?(uptime_record_file):(DEFAULT_UPTIME_RECORD_FILE)
#define PID_FILE (pid_file)?(pid_file):(DEFAULT_PID_FILE)

char *pid_file = NULL;
char *uptime_record_file = NULL;

static void
write_pid (void)
{
  pid_t pid;
  FILE *ud_pid;

  pid = getpid ();
  if ((ud_pid = fopen (PID_FILE, "w")) == NULL)
    {
      fprintf (stderr, "Can't write %s\n", PID_FILE);
      exit (EXIT_FAILURE);
    }
  else
    {
      fprintf (ud_pid, "%d\n", (int) pid);
      (void) fclose (ud_pid);
    }
}

static void
check_pid (void)
{
  FILE *ud_pid;
  pid_t pid;

  if ((ud_pid = fopen (PID_FILE, "r")) != NULL)
    {
      fscanf (ud_pid, "%d\n", &pid);
      (void) fclose (ud_pid);
      if (!kill (pid, 0))
	{			/* If process 'pid' is alive & running */
	  fprintf (stderr, "I'm already running!\n", PID_FILE);
	  exit (EXIT_FAILURE);
	}
    }
}

static float
get_uptime (void)
{
  static const char proc_uptime_path[] = "/proc/uptime";
  FILE *proc_uptime;
  float uptime;
  int err = 0;

  if ((proc_uptime = fopen (proc_uptime_path, "r")) == NULL)
    {
      fprintf (stderr, "Can't open `%s'.\n", proc_uptime_path);
      err = 1;
    }
  else
    {
      if (1 != fscanf (proc_uptime, "%f", &uptime))
	{
	  fprintf (stderr, "Error parsing `%s'.\n", proc_uptime_path);
	  err = 1;
	}
      (void) fclose (proc_uptime);
    }
  if (0 != err)
    exit (EXIT_FAILURE);
  return (uptime);
}

static void
get_kernel_version (char *kernel_version, int IsCurrent)
{
  struct utsname temp_uname;
  struct timeval time_ended_struct;
  struct timezone tz;
  static char time_ended[40];

  if (uname (&temp_uname) != 0)
    {
      perror ("Cannot get uname or time");	/* print an error but don't abort */
      kernel_version[0] = '\0';
    }
  else
    {
      sprintf (kernel_version, "running ");
      strcat (kernel_version, temp_uname.sysname);
      strcat (kernel_version, " ");
      strcat (kernel_version, temp_uname.release);
    }
  if (gettimeofday (&time_ended_struct, &tz) != 0)
    {
      perror ("Cannot get time");
    }
  else if (IsCurrent == 0)
    {
      sprintf (time_ended, ", ended %s", ctime (&time_ended_struct.tv_sec));
      time_ended[32] = '\0';
      strcat (kernel_version, time_ended);
    }
}

static void
write_record (const struct record *rec)
{
  FILE *uptime_record;

  /* NEW: also save the ud version to the record file.
     Only the first 3 chars are compared, so 0.5, 0.5.1 and 0.5.103
     are the same. Need to change this if you ever get the version
     number > 10 :-) */

  if ((uptime_record = fopen (UPTIME_RECORD_FILE, "w")) == NULL)
    {
      fprintf (stderr, "Can't write record.\n");
      exit (EXIT_FAILURE);
    }
  fprintf (uptime_record, "%s\n%f\n%s\n%f\n%s\n%f\n%s\n",
	   VERSION,
	   rec->le[0].uptime, rec->le[0].kern_vers,
	   rec->le[1].uptime, rec->le[1].kern_vers,
	   rec->le[2].uptime, rec->le[2].kern_vers);
  (void) fclose (uptime_record);
}

static void
create_record_file (void)
{
  /* Well, seems like GNU indent is broken */
  struct record nullrec =
  {
    {
      {0}
    }
  };
  printf ("Creating `%s'...\n", UPTIME_RECORD_FILE);
  write_record (&nullrec);
  printf ("Created. Please restart uptime daemon.\n");
}

static void
get_record (void)
{

  char line_buffer[3][LINELEN];
  int i, j;
  FILE *uptime_record;

  if ((uptime_record = fopen (UPTIME_RECORD_FILE, "r")) == NULL)
    {
      fprintf (stderr,
	       "\n"
	       "Can't open record, attempting to create %s...\n",
	       UPTIME_RECORD_FILE);
      create_record_file ();
      exit (EXIT_FAILURE);
    }

  if (fgets (line_buffer[0], LINELEN, uptime_record) == NULL)
    {
      fprintf (stderr, "Error reading %s\n", UPTIME_RECORD_FILE);
      exit (EXIT_FAILURE);
    }
  else
    {
      /* cut off trailing newline - see fgets(3) */
      line_buffer[0][strlen (line_buffer[0]) - 1] = '\0';

      if (strncmp (VERSION, line_buffer[0], 4) != 0)
	{
	  fprintf (stderr, "\nError: version of %s (%s)\n       does not match the version of " PACKAGE " (%s).\n", UPTIME_RECORD_FILE, line_buffer[0], VERSION);
	  fprintf (stderr, "See ChangeLog for details.\n");
	  if (0 != remove (PID_FILE))
	    {
//            fprintf (stderr, "Removing %s failed: ", PID_FILE);
	      //            perror ("");
	    }
	  exit (EXIT_FAILURE);
	}
    }

  for (i = 0; i < 3; ++i)	/* 3 records */
    {
      for (j = 0; j < 2; ++j)	/* 3 lines/record */
	{
	  if (fgets (line_buffer[j], LINELEN, uptime_record) == NULL)
	    {
	      fprintf (stderr, "Error parsing %s\n", UPTIME_RECORD_FILE);
	      exit (EXIT_FAILURE);
	    }
	}

      /* first line is the uptime */
      sscanf (line_buffer[0], "%f", &ut_record.le[i].uptime);

      /* second line is the kernel version */
      /* cut off trailing newline - see fgets(3) */
      line_buffer[1][strlen (line_buffer[1]) - 1] = '\0';
      strncpy (ut_record.le[i].kern_vers, line_buffer[1], VERSLEN);

    }
  fclose (uptime_record);
}

static void
fork_prog (void)
{
  pid_t pid;
  pid = fork ();
  if (pid < 0)
    {
      perror ("fork()");
      exit (EXIT_FAILURE);
    }
  else if (pid > 0)
    {
      printf ("(pid %d)\n", (int) pid);
      exit (EXIT_SUCCESS);
    }
}

static void
print_uptime (float time_in_sec, char *kernel_version)
{
  int days, hours, min, sec;

  days = time_in_sec / (60 * 60 * 24);
  hours = time_in_sec / (60 * 60) - (days * 24);
  min = time_in_sec / 60 - ((days * 24 + hours) * 60);
  sec = time_in_sec - ((((days * 24 + hours) * 60) + min) * 60);
  if (days > 0)
    {
      printf ("%d day(s), %02d:%02d:%02d %s\n", days, hours,
	      min, sec, kernel_version);
    }
  else
    {
      printf ("%02d:%02d:%02d %s\n", hours, min, sec,
	      kernel_version);
    }
}

/* Modified the help to reflect the new switches I added -- jcampbel */
static void
print_help (char **argv)
{
  printf ("Usage: %s [options]\n"
	  "\n"
	  "-d          display current uptime records\n"
	  "-s          toggle usage of HTML stats file (default: use)\n"
	  "-o          use original HTML output format\n"
	  "-w          write HTML stats and exit\n"
	  "-of <file>  use <file> for HTML output (default: %s)\n"
	  "-if <file>  use <file> for HTML template (default: %s)\n"
	  "-in <value> use <value> seconds for the stats update interval (default: %d)\n"
	  "-p <file>   use <file> for recording PID (default: %s)\n"
	  "-r <file>   use <file> for storing records (default: %s)\n"
	  "--version   output version information and exit\n"
	  "--help      get help (duh)\n",
	  *argv, STATS_FILE, INPUT_FILE, UPTIME_UPDATE_INTERVAL, DEFAULT_PID_FILE, DEFAULT_UPTIME_RECORD_FILE);
}

static void
print_version (char **argv)
{
  printf ("Uptime daemon " VERSION
	  " by Matthew Trent <root@piguy.dyn.ml.org>\n"
	  "Additional programming by many cool people, see AUTHORS\n");
}

static RETSIGTYPE
sigproc (int unusedInt)
{
  int erg = EXIT_SUCCESS;
  if (0 != remove (PID_FILE))
    {
      fprintf (stderr, "Removing %s failed: ", PID_FILE);
      perror ("");
      erg = EXIT_FAILURE;
    }
  exit (erg);
}

/* TRISTAN changed this bit:- new functions and replacement
   write_stats_file() */
/* FormatTime:- formats the float 'uptime' into a string
   representation of days, hours, minutes, seconds and puts the string
   into output */
/* Added version (andi) */
void
FormatTime (char *output, float uptime, char *kernel_version)
{
  int days, hours, min, sec;

  days = uptime / (60 * 60 * 24);
  hours = uptime / (60 * 60) - (days * 24);
  min = uptime / 60 - ((days * 24 + hours) * 60);
  sec = uptime - ((((days * 24 + hours) * 60) + min) * 60);

  if (days > 0)
    {
      sprintf (output,
	       "%d &nbsp;days, %02d:%02d:%02d %s",
	       days,
	       hours,
	       min,
	       sec,
	       kernel_version);
    }
  else
    {
      sprintf (output,
	       "%02d:%02d:%02d %s",
	       hours,
	       min,
	       sec,
	       kernel_version);
    }
}

/* GetValueForString: find out what value 'input' should return, and
   put it in 'output' */
void
GetValueForString (char *input, char *output, float uptime)
{
  float timestats[4];
  char kernel_version[4][VERSLEN];
  char tempstring[128 + VERSLEN];
  char hostname[128];
  long value;
  int i;

  timestats[0] = uptime;
  get_kernel_version (kernel_version[0], 1);
  get_kernel_version (kernel_version[1], 0);

  for (i = 0; i < 3; ++i)
    {
      timestats[i + 1] = ut_record.le[i].uptime;
      strncpy (kernel_version[i + 1], ut_record.le[i].kern_vers,
	       VERSLEN);
    }

  if (0 == strcmp (input, "time0"))
    {
      FormatTime (tempstring, timestats[0], kernel_version[0]);
    }
  else if (0 == strcmp (input, "time1"))
    {
      FormatTime (tempstring, timestats[1], kernel_version[1]);
    }
  else if (0 == strcmp (input, "time2"))
    {
      FormatTime (tempstring, timestats[2], kernel_version[2]);
    }
  else if (0 == strcmp (input, "time3"))
    {
      FormatTime (tempstring, timestats[3], kernel_version[3]);
    }
  else if (0 == strcmp (input, "hostname"))
    {
      if ((gethostname (hostname, sizeof (hostname)) == -1))
	{
	  sprintf (hostname, "%s", "unknown");
	}
      sprintf (tempstring, "%s", hostname);
    }
  else if (0 == strcmp (input, "interval"))
    {
      sprintf (tempstring, "%d", UPTIME_UPDATE_INTERVAL);
    }
  else if (0 == strcmp (input, "version"))
    {
      sprintf (tempstring, "%s", VERSION);
    }
  else if (0 == strcmp (input, "currenttime"))
    {
      value = time (0);
      sprintf (tempstring, "%s", ctime (&value));
    }
  else if (0 == strcmp (input, "url"))
    {
      sprintf (tempstring, "%s", "http://www.purelinux.ml.org/ud");
    }
  else if (0 == strcmp (input, "author"))
    {
      sprintf (tempstring, "%s", "Matthew Trent");
    }
  else if (0 == strcmp (input, "authoremail"))
    {
      sprintf (tempstring, "%s", "root@piguy.dyn.ml.org");
    }
  else if (0 == strcmp (input, "extraauthors"))
    {
      sprintf (tempstring, "%s", "Johnny Teveben and Tristan Rowley");
    }
  else
    {
      sprintf (tempstring, "%s", "UNKNOWN");
    }

  sprintf (output, "%s", tempstring);

}

/* Replacement write_stats_file to allow the user to customise the
   HTML output easily */
static void
write_stats_file (float uptime)
{
  FILE *fin, *fout;
  int inchar, count;
  char *stringpointer;
  char inputstring[128];
  char outputstring[128 + VERSLEN];

  if ((fin = fopen (INPUT_FILE, "r")) == NULL)
    {
      perror ("Reading template file (use -o to use built in template)");
      exit (EXIT_FAILURE);
    }
  else
    {

      if ((fout = fopen (STATS_FILE, "w")) == NULL)
	{
	  perror ("Writing stats file (use -s for no HTML output");
	  exit (EXIT_FAILURE);
	}
      else
	{
	  inchar = getc (fin);
	  do
	    {
	      if (inchar == '$')
		{
		  stringpointer = inputstring;
		  count = 0;

		  inchar = getc (fin);
		  do
		    {
		      *stringpointer = (char) inchar;
		      stringpointer++;
		      count++;
		      inchar = getc (fin);
		    }
		  while (inchar != '$');

		  inputstring[count] = '\0';

		  GetValueForString (inputstring, outputstring, uptime);

//          printf ("%s", outputstring);
		  fputs (outputstring, fout);
		}
	      else
		{
//          printf ("%c", inchar);
		  fputc (inchar, fout);
		}

	      inchar = getc (fin);
	    }
	  while (inchar != EOF);
	  fclose (fout);
	}
      fclose (fin);
    }
}

/* TRISTAN end changed bit -- notice that the old write_stats_file (below) has been renamed to old_write_stats_file (to allow the old method to be retained) */

static void
old_write_stats_file (float uptime)
{
  float time[4];
  FILE *stats;
  char hostname[128];

  time[0] = uptime;
  time[1] = ut_record.le[0].uptime;
  time[2] = ut_record.le[1].uptime;
  time[3] = ut_record.le[2].uptime;

  if ((gethostname (hostname, sizeof (hostname)) == -1))	/* success? FIXED? */
    {
      sprintf (hostname, "%s", "unknown");
    }

  if ((stats = fopen (STATS_FILE, "w")) == NULL)
    {
      perror ("Writing stats file");
      sigproc (0);
    }
  else
    {
      int days, hours, min, sec, i;
      fprintf (stats,
	       "<!DOCTYPE HTML PUBLIC \"-//W3C/DTD HTML 3.2 Final//EN\">\n"
	       "<HTML><HEAD>\n"
	       "  <TITLE>Uptime daemon stats for host %s</TITLE>\n"
	       "  <META NAME=creator CONTENT=\"" PACKAGE " " VERSION "\">\n"
	       "  <META HTTP-EQUIV=Refresh CONTENT=%d>\n"
	       "</HEAD>\n"
	       "<BODY " HTML_BODY ">\n"
	       "<H1>Uptime daemon stats for host -&lt; %s &gt;-</H1>\n"
	       "\n",
	       hostname,
	       UPTIME_UPDATE_INTERVAL,
	       hostname);

      for (i = 0; i < 4; ++i)
	{
	  if (0 == i)
	    {
	      fprintf (stats,
		       "<P><EM>Current Uptime: </EM></P>\n"
		       "<UL TYPE=SQUARE>\n");
	    }
	  else if (1 == i)
	    {
	      fprintf (stats,
		       "</UL>\n"
		       "\n"
		       "<P><EM>Uptime Records:</EM></P>\n"
		       "\n"
		       "<UL TYPE=SQUARE>\n");
	    }
	  days = time[i] / (60 * 60 * 24);
	  hours = time[i] / (60 * 60) - (days * 24);
	  min = time[i] / 60 - ((days * 24 + hours) * 60);
	  sec = time[i] - ((((days * 24 + hours) * 60) + min) * 60);
	  if (days > 0)
	    {
	      fprintf (stats,
		       "  <LI>up %d&nbsp;days, %02d:%02d:%02d\n"
		       "  </LI>\n"
		       "\n",
		       days, hours, min, sec);
	    }
	  else
	    {
	      fprintf (stats,
		       "  <LI>up %02d:%02d:%02d\n"
		       "  </LI>\n"
		       "\n",
		       hours, min, sec);
	    }
	}
      fprintf (stats,
	       "</UL>\n"
	       "\n"
	       "<HR>\n"
	       "This page updated every %d&nbsp;seconds.\n"
	       "<BR><A HREF=\"http://purelinux.ml.org/ud/\"\n"
	       "TITLE=\"UD homepage\">Uptime daemon</A>\n"
	       "by <A HREF=\"mailto:root@piguy.dyn.ml.org?subject=" PACKAGE "-" VERSION "\"\n"
	       "TITLE=\"Author\">Matthew Trent</A>\n"
	       "</BODY></HTML>\n",
	       UPTIME_UPDATE_INTERVAL);
      (void) fclose (stats);
    }
}

int
main (int argc, char **argv)
{
  int i;
  char hostname[128], activefield = 0;
  char kern_vers[VERSLEN];
  int write_stats = 2;		/* TRISTAN changed this bit - mode 2 for new custom HTML file output, mode 1 for the old style, and 0 for none */
  struct logentry actle;

  signal (SIGINT, sigproc);
  signal (SIGHUP, sigproc);
  signal (SIGQUIT, sigproc);
  signal (SIGTERM, sigproc);

  for (i = 1; i < argc; ++i)
    {
      if (0 == strcmp (argv[i], "-d"))
	{
	  get_record ();
	  actle.uptime = get_uptime ();
	  gethostname (hostname, sizeof (hostname));
	  printf ("- Uptime for %s -\n", hostname);
	  get_kernel_version (kern_vers, 1);
	  printf ("Now  : ");
	  print_uptime (actle.uptime, kern_vers);
	  printf ("One  : ");
	  print_uptime (ut_record.le[0].uptime, ut_record.le[0].kern_vers);
	  printf ("Two  : ");
	  print_uptime (ut_record.le[1].uptime, ut_record.le[1].kern_vers);
	  printf ("Three: ");
	  print_uptime (ut_record.le[2].uptime, ut_record.le[2].kern_vers);
	  return (EXIT_SUCCESS);
	}
      else if (0 == strcmp (argv[i], "-s"))
	{
	  write_stats = 0;
	}
/* TRISTAN added this bit: checking for the new command-line options */
      else if (0 == strcmp (argv[i], "-o"))
	{
	  write_stats = 1;
	}
      else if (0 == strcmp (argv[i], "-of"))
	{
	  if (argc <= 2)
	    {
	      fprintf (stdout, "Error: option requires argument\n");
	      return (EXIT_FAILURE);

	    }
	  strcpy (STATS_FILE, argv[++i]);
	}
      else if (0 == strcmp (argv[i], "-if"))
	{
	  if (argc <= 2)
	    {
	      fprintf (stdout, "Error: option requires argument\n");
	      return (EXIT_FAILURE);

	    }
	  strcpy (INPUT_FILE, argv[++i]);
	}
      else if (0 == strcmp (argv[i], "-in"))
	{
	  if (argc <= 2)
	    {
	      fprintf (stdout, "Error: option requires argument\n");
	      return (EXIT_FAILURE);

	    }
	  UPTIME_UPDATE_INTERVAL = atoi (argv[++i]);
	}
/* TRISTAN end added bit */
      else if (0 == strcmp (argv[i], "-w"))
	{
	  actle.uptime = get_uptime ();
	  get_record ();
	  if (write_stats == 1)
	    {
	      old_write_stats_file (actle.uptime);
	    }
	  else if (write_stats == 2)
	    {
	      write_stats_file (actle.uptime);
	    }
	  return (EXIT_SUCCESS);
	}
      /* Added these to handle -p pid_file and -r record_file -- jcampbel */
      else if (0 == strcmp (argv[i], "-p"))
	{
	  if (argc <= 2)
	    {
	      fprintf (stdout, "Error: option requires argument\n");
	      return (EXIT_FAILURE);

	    }
	  pid_file = argv[++i];
	}
      else if (0 == strcmp (argv[i], "-r"))
	{
	  if (argc <= 2)
	    {
	      fprintf (stdout, "Error: option requires argument\n");
	      return (EXIT_FAILURE);

	    }
	  uptime_record_file = argv[++i];
	}

      else if (0 == strcmp (argv[i], "--version"))
	{
	  print_version (argv);
	  return (EXIT_SUCCESS);
	}
      else if (0 == (strcmp (argv[i], "--?") &&
		     strcmp (argv[i], "--help") &&
		     strcmp (argv[i], "-h")))
	{
	  print_help (argv);
	  return (EXIT_SUCCESS);
	}
      else
	{
	  printf ("\n"
		  "Error: unknown option \"%s\", %s --help for help\n",
		  argv[i], argv[0]);
	  return (EXIT_FAILURE);
	}
    }
  check_pid ();
  get_record ();
  fork_prog ();
  write_pid ();
  get_record ();
//  get_kernel_version (actle.kern_vers, 0);
  while (1)
    {
      actle.uptime = get_uptime ();
      get_kernel_version (actle.kern_vers, 0);
      if (actle.uptime > ut_record.le[0].uptime)
	{
	  if (activefield != 1)
	    { 
	      if (activefield != 2)
	      ut_record.le[2] = ut_record.le[1];
	      ut_record.le[1] = ut_record.le[0];
	      activefield = 1;
	    }
	  ut_record.le[0] = actle;
	  write_record (&ut_record);
	}
      else if (actle.uptime > ut_record.le[1].uptime)
	{
	  if (activefield != 2)
	    {
	      ut_record.le[2] = ut_record.le[1];
	      activefield = 2;
	    }
	  ut_record.le[1] = actle;
	  write_record (&ut_record);
	}
      else if (actle.uptime > ut_record.le[2].uptime)
	{
	  ut_record.le[2] = actle;
	  activefield = 3;
	  write_record (&ut_record);
	}
/* TRISTAN changed this bit:- checking the HTML output options */
      if (write_stats)
	{
	  if (write_stats == 1)
	    {
	      old_write_stats_file (actle.uptime);
	    }
	  else if (write_stats == 2)
	    {
	      write_stats_file (actle.uptime);
	    }
	}
/* TRISTAN end changed bit */
      sleep (UPTIME_UPDATE_INTERVAL);
    }
  return (EXIT_SUCCESS);
}
