
// Copyright (c) 1995-2003 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com

//---------------------------------------------------------------------------
// 
// $Id: AccessVariable.hh,v 1.2 2003/12/08 19:24:39 dmartin Exp $
// 
//---------------------------------------------------------------------------

#ifndef ACCESSVARIABLE_HH
#define ACCESSVARIABLE_HH

#include "tyvis/vhdl.hh"
#include "tyvis/ObjectBase.hh"
#include "tyvis/ArrayInfo.hh"
#include "tyvis/Types.hh"
#include <string.h>
#include <iostream>

using std::cout;

class AccessVarBase : public VHDLData{
public:
  const string &getName() const { return name; }

protected:
  string name;
  VarType variableType;  // normal or access type

  virtual int length() const = 0;

  AccessVarBase() {}
  AccessVarBase( const string &newname ) :
    name(newname),
    variableType( NORMAL ){}

  AccessVarBase( const string &newname, VarType initType ) :
    name(newname), variableType( initType ) {}

  AccessVarBase(const AccessVarBase&);

  virtual VHDLData::UniversalType getUniversalKind() const{
    return ACCESS_VARIABLE;
  }
  
  static const string &getAccessVariableType(){
    static const string accessVariableType = "AccessVariable";
    return accessVariableType;
  }

  const string &getDataType() const {
    return getAccessVariableType();
  }

  void serialize( SerializedInstance * ) const {
    abort();
  }
};

// Temporary Solution:  AccessVariable must be derived from VarBase or 
// ObjectBase.
class AccessVariable : public AccessVarBase {
public:
  AccessVariable();
  AccessVariable( const string &initName );
  AccessVariable( const AccessVariable & );
  ~AccessVariable(){ }
  AccessVariable &operator=(const AccessVariable & );
  AccessVariable &operator=( const string &newVal );

  void print(ostream& os = cout) const;
  int getSize() const { return sizeof(*this); }
  int length() const { return val.size(); }
  int getCurrentPos(){ return curPos; }
  void setCurrentPos( int newCurPos ){ curPos = newCurPos; }

  int getIntValue() const { abort(); return 0; }
  LONG getInt64Value() const { abort(); return 0; }
  double getDoubleValue() const { abort(); return 0; }

  void setVal( const string &newVal ){
    curPos = 0;
    val = newVal;
  }
  
  const string &getVal(){
    return val;
  }

  bool operator==( const RValue &compareTo ) const {
    return ( dynamic_cast<const AccessVariable &>( compareTo ).curPos == curPos &&
	     dynamic_cast<const AccessVariable &>( compareTo ).val == val );
  }

  bool operator!=( const RValue &compareTo ) const {
    return !( dynamic_cast<const AccessVariable &>( compareTo ).curPos == curPos &&
	      dynamic_cast<const AccessVariable &>( compareTo ).val == val );
  }

  bool operator>( const RValue &compareTo ) const{
    return val > dynamic_cast<const AccessVariable &>( compareTo ).val;
  }

  bool operator>=( const RValue &compareTo ) const {
    return val >= dynamic_cast<const AccessVariable &>( compareTo ).val;
  }

  bool operator<( const RValue &compareTo ) const {
    return val < dynamic_cast<const AccessVariable &>( compareTo ).val;
  }

  bool operator<=( const RValue &compareTo ) const {
    return val <= dynamic_cast<const AccessVariable &>( compareTo ).val;
  }

  inline void printstr(ostream& os) {
    os << val;
  }
  
  /**
     Eats whitespace starting at currentPos.  Returns true if we're still
     inside the string at the end, false otherwhise.
  */
  bool eatwhite(){
    for( int i = curPos; i < length(); i++ ){
      while( val[i] == ' ' || val[i] == '\t' ){
	curPos++;
      }
    }
    return curPos < length();
  }

private:
  int curPos; // pointer into val indicating, e.g., the current read position
  string val;
};

#endif
