//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

#ifndef SOURCEBASE_HH
#define SOURCEBASE_HH

// The SourceBase forms the base class for the nodes of the source tree.
// The nodes may be one of SourceData or SourceInfo.  This class privides
// virtual functions for these two derived classes.

#ifndef ASSERT
#ifdef DEVELOPER_ASSERTIONS
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#define ASSERT( x ) assert( x )
#else
#define ASSERT( x )
#endif
#endif

#include <iostream.h>

#include "VHDLType.hh"
#include "VHDLData.hh"
class SourceData;
class VHDLKernelBase;

// Type for up and down type conversion functions, and resolution function
// ids.  These functions are accessed from a global function array.  This
// is necessary for static elaboration.  We must be able to write these
// function "references" to a file and read it back before execution.
// This can not be done if write pointers to file.
typedef int TypeConversionFnId_t;
typedef int ResolutionFnId_t;
typedef int SourceId_t;

#define ANONYMOUS_PROCESS_ID ((SourceId_t) -2)

const int DEFAULT_ID = -1;

typedef VHDLType* (*ResolutionFnPtr)(VHDLKernelBase*, int, VHDLType**);
typedef VHDLType* (*TypeConversionFnPtr)(VHDLKernelBase*, VHDLType*);

class SourceBase {
public:
  SourceBase() : parent(NULL) {};
  virtual ~SourceBase() {};

  enum SourceKind { SOURCE_BASE, SOURCE_DATA, SOURCE_INFO, SOURCE_TYPE_CONVERT};

  virtual SourceKind get_kind(void) const { return SOURCE_BASE; };

  // Methods to act on parent of this node.
  void setParent(SourceBase*);
  SourceBase *getParent() const { return parent; };

  virtual SourceBase& operator[](const int) { 
    cerr << "SourceBase::operator[] called." << endl;
    abort();
    return *this;		// Just to keep CC from shouting.
  };
  virtual VHDLType* getData() const { return NULL; };
  virtual void setData(VHDLType *);

  virtual void updateData(const VHDLType&) { return; };
  virtual void updateData(const VHDLData&) { return; };

  virtual SourceData* getDriver(const SourceId_t) const { return NULL; };

  //While getDriver(id) returns the driver with SourceId_t == id,
  //getDriveratIndex(id) returns the driver at childTable[id]
  virtual SourceData* getDriveratIndex(int) const { return NULL; };

  virtual SourceId_t getSourceId() const { return DEFAULT_ID; };
  virtual void setSourceId(SourceId_t) { abort(); };

  virtual int addChild(SourceBase *) { return 0; };
  virtual int addChild(VHDLType *, SourceId_t) { return 0; };
  virtual int getNumChildren() const { return 0; };

  //The Following function returns the number of connected drivers
  //At the time applying the resolution function
  virtual int getNumActiveChildren() const { return 0; }

  virtual bool _is_child_present(SourceId_t) { return false; }

  //This functions returns if the driver is active in the current simulation
  //cycle
  virtual bool _is_driver_connected() { return true; }

  // This function is needed to handle default initial values for signals
  // in the system. This function is overloaded in SourceData to do the right
  // job. This virtual function is here just to be handy.
  virtual bool _is_anonymous_driver() { return false; }
  
  // Methods to deal with up and down type conversion functions and
  // resolution functions.
  virtual void setUpTypeConversionFnId(TypeConversionFnId_t) { return; };
  virtual void setDownTypeConversionFnId(TypeConversionFnId_t) { return; };
  virtual void setResolutionFnId(ResolutionFnId_t) { return; };
  virtual TypeConversionFnId_t getUpTypeConversionFnId(void) { return DEFAULT_ID; };
  virtual TypeConversionFnId_t getDownTypeConversionFnId(void) { return DEFAULT_ID; };
  virtual ResolutionFnId_t getResolutionFnId(void) { return DEFAULT_ID; };

//   virtual VHDLType* resolve(void) = 0;
  virtual VHDLType* resolve(VHDLKernelBase*, SigType typ = EXPLICIT);

  virtual void copyTable(const SourceBase*);
  virtual void addToTable(SourceData*);

  // Table size is equivalent to one since this is a SourceData object and
  // it has one driver.  This will be used to get the driver using
  // getDriver(n) which will return the driver in this node.
  virtual int getTableSize(void) const { return 1; };

//   virtual void print(ostream&) const = 0;
  virtual void print(ostream&) const { return; };

protected:
  // A pointer to the parent of this node.  Required for type conversion
  // and resolution functions.
  SourceBase * parent;
};

#endif
