#include "ArrayType.hh"
#include "VectorBase.hh"
#include "ScalarType.hh"
#include "EnumerationType.hh"
#include "Types.hh"
#include "standard.hh"

ArrayInfo* ArrayType::rangeInfo = NULL;

ArrayType::ArrayType(ObjectBase::ObjectType, const TypeInfo&) : VHDLType()
{
  object = NULL;
  isCompositeResolvedType = false;
}

ArrayType::ArrayType(bool alias) : VHDLType(alias) { 
  object = NULL;
  isCompositeResolvedType = false;
}

ArrayType::ArrayType(bool alias, ObjectBase::ObjectType, int, 
		     ArrayDirn_t, int, const ArrayType&, const ArrayInfo&) :
  VHDLType(alias) { 
  object = NULL;
  isCompositeResolvedType = false;
}

ArrayType::~ArrayType() { 
  if((is_alias == false)&& (object != NULL)) {
    delete object;
    object = NULL;
  }
}
  
void 
ArrayType::setRange(ObjectBase::ObjectType, ArrayInfo*, int, const TypeInfo&) {
  cerr << "This should NEVER be called" << endl;
  abort();
}

// Note: The method "get_kind()" MUST NOT be overloaded by a derived type.
// This is to ensure that the type is recognized correctly.
Type 
ArrayType::get_kind() const {
  return ARRAY_TYPE;
}

char*
ArrayType::getString() const { 
  return object->getString();
}

void 
ArrayType::print(ostream& os) const {
  object->print(os);
}

VHDLType& 
ArrayType::operator[](const int pos) const {
  return (*object)[pos];
}

// VHDLType& 
// ArrayType::operator[](const ScalarType& pos) const {
//   UniversalInteger val;
//   val = pos.object->readVal();
//   return (*object)[int(val)];
// }

VHDLType& 
ArrayType::operator =(const VHDLType& val) {
  if (object == NULL) {
    object = (VectorBase *) val.clone();
  }
  
  *object = (*((ArrayType&) val).object);
  return (*this);
}

VHDLType& 
ArrayType::assignVal(const VHDLType& val) {
  if(val.is_alias == true) {
    ASSERT(val.get_kind() == ARRAY_TYPE);
    if ( object != NULL ){
      delete object;
    }
    object = ((ArrayType&)val).object;
    is_alias = true;
    //code added so that the val's object's is_alias is set to true and
    //is_alias for (this) is set to false so that (this) will delete the
    //Vector object newed by val or someone else and val will be silent
    is_alias = true;
    object->is_alias = true;
  } else {
    *this = val;
  }
  return (*this);
}

ArrayType& 
ArrayType::operator =(const ArrayType& val) {
  if (object == NULL) {
    object = (VectorBase *) val.object->clone();
  }
  
  *(this->object) = *(val.object);
  is_alias = val.is_alias;
  return (*this);
}

bool
ArrayType::operator==(const VHDLType& val) const {
  ASSERT(val.get_kind() == ARRAY_TYPE);
  int lhs_length = this->object->bounds.length();
  int rhs_length = ((ArrayType&)val).object->bounds.length();

  if(lhs_length != rhs_length) {
    return false;
  }
  for(int i = 0; i < lhs_length; i++) {
    if(!(this->get_element(i) == val.get_element(i))) {
      return false;
    }
  }
  return true;
}

VHDLType& 
ArrayType::get_element(const int index) const {
  return object->get_element(index);
}

VHDLType& 
ArrayType::get_array() {
  return *object;
}

int 
ArrayType::get_number_of_elements() const {
  return object->numElems;
}

ArrayInfo* 
ArrayType::get_bounds() const {
  return object->get_bounds();
}

ObjectBase* 
ArrayType::getObject() const {
  return NULL;
}

ObjectBase::ObjectType
ArrayType::getKind() const {
  return object->getKind();
}

VHDLType*
ArrayType::clone() const {
  VHDLType *retval = new ArrayType(is_alias);
  ((ArrayType *) retval)->object = (VectorBase *) object->clone();

  return retval;
}

int 
ArrayType::left() const {
  return object->left();
}

int 
ArrayType::right() const {
  return object->right();
}

ArrayDirn_t 
ArrayType::dirn() const {
  return object->dirn();
}

int 
ArrayType::length() const { 
  return object->length();
}

void
ArrayType::setResolutionFunctionId(int resolutionFnId) {
  object->setResolutionFunctionId(resolutionFnId);
}

void
ArrayType::setTypeConversionFunctionId(int typeConversionFnId) {
  object->setTypeConversionFunctionId(typeConversionFnId);
}

void
ArrayType::setElaborationInfo(const VHDLType &obj_info) {
  ASSERT (_is_signal() == true);
  ASSERT (obj_info.get_kind() == ARRAY_TYPE);
  for (int index = 0; index < object->numElems; index++) {
    this->get_element(index).setElaborationInfo(((ArrayType&)obj_info).get_element(index));
  }
}

void
ArrayType::setNumAttributes(int noAttribs) {
  ASSERT (_is_signal() == true);
  for (int index = 0; index < object->numElems; index++) {
    this->get_element(index).setNumAttributes(noAttribs);
  }
}

void
ArrayType::setAttrib(AttribType typ, VHDLType& attr) {
  ASSERT (_is_signal() == true);
  ASSERT (attr.get_kind() == ARRAY_TYPE);
  for (int index = 0; index < object->numElems; index++) {
    this->get_element(index).setAttrib(typ, ((ArrayType&)attr).get_element(index));
  }
}

int
ArrayType::savantwrite(AccessVariable <char*> &line) const {
  for(int counter = 0; (counter < length()); counter++) {
    this->get_element(counter).savantwrite(line);
  }

  return NORMAL_RETURN;
}

int
ArrayType::savantwrite(SavantlineType &line) const {
  for(int counter = 0; (counter < length()); counter++) {
    this->get_element(counter).savantwrite(line);
  }

  return NORMAL_RETURN;
}  

int
ArrayType::savantread(AccessVariable <char*> &line)  {
  for(int counter = 0; (counter < length()); counter++) {
    this->get_element(counter).savantread(line);
  }

  return NORMAL_RETURN;
}

int
ArrayType::savantread(SavantlineType &line)  {
  for(int counter = 0; (counter < length()); counter++) {
    this->get_element(counter).savantread(line);
  }

  return NORMAL_RETURN;
}

ArrayType*
ArrayType::getNewArray(const ArrayInfo& newBounds, const ArrayInfo& actualBounds) const {
  ArrayType *newArray = new ArrayType(true); // This is an alias.
  newArray->object = object->getNewArray(newBounds, actualBounds);
  return newArray;
}

void
ArrayType::setParentCompositeType(VHDLType* ptr) {
  object->setParentCompositeType(ptr);
  isCompositeResolvedType = true;
}

void
ArrayType::setCompositeResolvedSignal(bool val) {
  ASSERT(this->getKind() == ObjectBase::SIGNAL);
  object->setCompositeResolvedSignal(val);
  isCompositeResolvedType = val;
}

void
ArrayType::dump_connectivity_info(ofstream& fileStream) {
  switch(get_element(left()).get_kind()) {
  case INTEGER_TYPE:
  case ENUMERATION_TYPE:
  case REAL_TYPE:
  case PHYSICAL_TYPE:
    ((ScalarType&)get_element(left())).dump_connectivity_info(fileStream);
    break;
  case ARRAY_TYPE:
    ((ArrayType&)get_element(left())).dump_connectivity_info(fileStream);
    break;
  }
}

void
ArrayType::updateEffVal(const VHDLType* ptr) {
  ASSERT(this->getKind() == ObjectBase::SIGNAL);
  ASSERT(ptr->get_kind() == ARRAY_TYPE);
  object->updateEffVal(((ArrayType*)ptr)->object);
}

//The TYPE's resolve is called only for composite resolved signals
//This resolve goes down to first sub-element of the VHDLType and
//calls the sub-elements resolve, but which actually does the resolution
//for the whole composite type
// The above given reason was the original reason for writing resolve
// in the ArrayTypes. But when TypeConversion came into picture, we needed
// resolve to be written for ArrayTypes even if the ArrayType was not a
// composite resolved signal. In case of not a composite resolved type,
// resolve method just goes down to the sub elements and resolves the
// individual elements and then updates once the resolved value is obtained.
VHDLType*
ArrayType::resolve(VHDLKernelBase* processPtr) {
  int index ;
  VHDLType *retval = NULL, *element = NULL;
  
  ASSERT(this->getKind() == ObjectBase::SIGNAL);
  
  if( _is_composite_resolved_type() ){
    if (object != NULL) {
      return object->resolve(processPtr);
    } else {
      return NULL;
    }
  }
  else {
    for ( index = 0 ; index < object->get_number_of_elements(); index++){
      // if the subelements are not composite resolved types. then the value
      // is update in their down most scalar types itself. So we dont' need
      // to udpate the value physically. Otherwise, we need to update it
      // physically. that is why this assignment.

      element = &(object->get_element(index));
      retval = element->resolve(processPtr);
      
      if ( element->_is_composite_resolved_type() ){
	*element = *retval;
      } 
    }
    return NULL;
  }
}

bool
ArrayType::_is_signal() const {
  return object->_is_signal();
}

EnumerationType
savantEqual(const ArrayType& lhs, const ArrayType& rhs) {
  int index;
  for(index = 0; index < lhs.length(); index++) {
    if(savantEqual(lhs.get_element(index), rhs.get_element(index)) == ENUMERATION_FALSE) {
      return ENUMERATION_FALSE;
    }
  }

  return ENUMERATION_TRUE;
}  

EnumerationType
savantNotEqual(const ArrayType& lhs, const ArrayType& rhs) {
  return savantNot(savantEqual(lhs, rhs));
}

EnumerationType
savantLessThan(const ArrayType& lhs, const ArrayType &rhs) {
 bool is_lhs_null = lhs.object->bounds.is_null_range();
 bool is_rhs_null = rhs.object->bounds.is_null_range();

 register int counter = 0;
 
 if (is_lhs_null && is_rhs_null)  { 
   return ENUMERATION_FALSE;
 } 
 else  { 
   if (is_lhs_null && !is_rhs_null)  { 
     return ENUMERATION_TRUE;
   }
   else if (!is_lhs_null && is_rhs_null)  { 
     return ENUMERATION_FALSE;
   }
 }
 
 register int lhs_counter   = lhs.object->bounds.left();
 register int rhs_counter   = rhs.object->bounds.left();
 register int lhs_increment = lhs.object->bounds.dirn();
 register int rhs_increment = rhs.object->bounds.dirn();
 register int max_counter   = 0;
 
 if ((max_counter = lhs.get_number_of_elements()) > rhs.get_number_of_elements())  {
   max_counter = rhs.get_number_of_elements();
 }
 
 for(counter = 0; (counter < max_counter); counter++)  { 
   if (ENUMERATION_TRUE == savantLessThan(lhs[lhs_counter], rhs[rhs_counter]))  {
     return ENUMERATION_TRUE;
   }
   else  
     if (ENUMERATION_TRUE == savantGreaterThan(lhs[lhs_counter ], rhs[rhs_counter ]))  {
       return ENUMERATION_FALSE;
     }
   lhs_counter += lhs_increment;
   rhs_counter += rhs_increment;
 }

 if (max_counter == lhs.get_number_of_elements()) {
   return ENUMERATION_TRUE;
 } else {
   return ENUMERATION_FALSE;
 }
}

EnumerationType
savantLessThanOrEqual(const ArrayType& lhs, const ArrayType& rhs)  {
  return savantOr(savantLessThan(lhs, rhs), savantEqual(lhs, rhs));
}

EnumerationType
savantGreaterThan(const ArrayType& lhs, const ArrayType& rhs)  {
  return savantNot(savantLessThanOrEqual(lhs, rhs));
}

EnumerationType
savantGreaterThanOrEqual(const ArrayType& lhs, const ArrayType &rhs) {
  return savantOr(savantGreaterThan(lhs, rhs), savantEqual(lhs, rhs));
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantAnd(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are ANDed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantAnd(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantNand(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are NANDed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantNand(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantOr(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are ORed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantOr(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantNor(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are NORed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantNor(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantXnor(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are XNORed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantXnor(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantXor(const ArrayType& lhs, const ArrayType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int rhs_length = rhs.object->bounds.length();

  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are XORed. Bailing out" << endl;
    abort();
  }
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    EnumerationType& enumerationRhs = (EnumerationType&) rhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationRhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantXor(enumerationLhs, enumerationRhs);
  }
  return *retval;
}

ArrayType&
savantNot( const ArrayType& lhs) {
  int lhs_length = lhs.object->bounds.length();
  ArrayType* retval = (ArrayType*)lhs.clone();
  for(int i =0; i < lhs_length; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    enumerationPtr = savantNot(enumerationLhs);
  }
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantSLL(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSRL(lhs, newRhs);
  }
  
  ArrayType* retval = (ArrayType*)lhs.clone();
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }
  
  // Initialize the newwly allocated return value...
  for(i = 0; (i < lhs_length); i++) {
    EnumerationType& enumerationPtr = (EnumerationType &) retval->get_element(i);
    enumerationPtr = enumerationPtr.LEFT(enumerationPtr.range);
  }
  
  if (number_of_shifts > lhs_length) {
    return *retval;
  }
  
  for(i = number_of_shifts; (i < lhs_length); i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i - number_of_shifts);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantSRL(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSLL(lhs, newRhs);
  }
  
  ArrayType* retval = (ArrayType*)lhs.clone();
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }
  
  // Initialize the newwly allocated return value...
  for(i = 0; (i < lhs_length); i++) {
    EnumerationType& enumerationPtr = (EnumerationType &) retval->get_element(i);
    enumerationPtr = enumerationPtr.LEFT(enumerationPtr.range);
  }
  
  if (number_of_shifts > lhs_length) {
    return *retval;
  }
  
  for(i = lhs_length; (i >= number_of_shifts); i--) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i - number_of_shifts);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return *retval;
}

ArrayType&
savantSLA(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSRL(lhs, newRhs);
  }
  
  ArrayType* retval = (ArrayType*)lhs.clone();
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }

  EnumerationType last_element = (EnumerationType&) lhs.get_element(lhs_length - 1);
  
  for(i = number_of_shifts; (i < lhs_length); i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i - number_of_shifts);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }

  for(i = lhs_length - number_of_shifts; (i < lhs_length); i++) {
    retval->get_element(i) = last_element;
  }
      
  return *retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType&
savantSRA(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSLL(lhs, newRhs);
  }
  
  ArrayType* retval = (ArrayType*)lhs.clone();
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }
  
  EnumerationType first_value = (EnumerationType&) lhs.get_element(0);
  
  if (number_of_shifts > lhs_length) {
    return *retval;
  }
  
  for(i = lhs_length; (i >= number_of_shifts); i--) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(i - number_of_shifts);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }

  for(i = 0; (i < lhs_length); i++) {
    retval->get_element(i) = first_value;
  }
  
  return *retval;
}

ArrayType&
savantROR(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0, src_pos = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSRA(lhs, newRhs);
  }

  ArrayType* retval = (ArrayType*)lhs.clone();
  number_of_shifts   = number_of_shifts % lhs_length;
  
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }

  // Copy tail of lhs to head of return value
  for(i = 0, src_pos = lhs_length - number_of_shifts; (src_pos < lhs_length); src_pos++, src_pos++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }

  // Copy head of lhs to tail of return value
  for(src_pos = 0, i = number_of_shifts; (i < lhs_length); src_pos++, i++)  {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return *retval;
}

ArrayType&
savantROL(const ArrayType& lhs, const IntegerType& rhs) {
  int lhs_length = lhs.object->bounds.length();
  int number_of_shifts = ((UniversalInteger &) rhs.getVHDLData()).val;
  int i = 0, src_pos = 0;
  
  if (number_of_shifts < 0) {
    IntegerType newRhs = savantAbs(rhs);
    return savantSLA(lhs, newRhs);
  }
  
  ArrayType* retval = (ArrayType*)lhs.clone();
  number_of_shifts   = number_of_shifts % lhs_length;
  
  if ((number_of_shifts == 0) || (lhs_length == 0)) {
    return *retval;
  }
  
  // Copy head of lhs to tail of return value
  for(src_pos = 0, i = lhs_length - number_of_shifts; (src_pos < number_of_shifts); i++, src_pos++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  // Copy tail of lhs to head of return value
  for(i = 0, src_pos = lhs_length - number_of_shifts; (src_pos < lhs_length); src_pos++, i++)  {
    EnumerationType& enumerationPtr = (EnumerationType&)retval->get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType&) lhs.get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return *retval;
}

SignalBase*
ArrayType::locateSig(int sigId) {
  return object->locateSig(sigId);
}

SignalBase*
ArrayType::findSigInBlock(int sigId, int srcId){
  return object->findSigInBlock(sigId, srcId);
}

bool
ArrayType::_is_composite_resolved_type() const{
  return isCompositeResolvedType;
}

//Array type attributes
IntegerType 
ArrayType::LEFT(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1))) {  
    return IntegerType(ObjectBase::VARIABLE, 
		       (int)(aInfo.get_rangeInfo()[dimension-1]).get_left());
  }
  else {
    cerr << "The parameter of `LEFT attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, (int)0);
  }
}

IntegerType 
ArrayType::RIGHT(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1)))  {
    return IntegerType(ObjectBase::VARIABLE, 
		       (int)(aInfo.get_rangeInfo()[dimension-1]).get_right());
  }
  else {
    cerr << "The parameter of `RIGHT attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, (int)0);
  }
}

IntegerType 
ArrayType::HIGH(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1))) {
    int right, left;
    right = (aInfo.get_rangeInfo()[dimension-1]).get_right();
    left  = (aInfo.get_rangeInfo()[dimension-1]).get_left();
    if (right > left) {
      return IntegerType(ObjectBase::VARIABLE, right);
    }
    else {
      return IntegerType(ObjectBase::VARIABLE, left);
    }
  }
  else {
    cerr << "The parameter of `HIGH attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, (int)0);
  }
}

IntegerType 
ArrayType::LOW(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1))) {
    int right, left;
    right = (aInfo.get_rangeInfo()[dimension-1]).get_right();
    left  = (aInfo.get_rangeInfo()[dimension-1]).get_left();
    if (right > left) {
      return IntegerType(ObjectBase::VARIABLE, left);
    }
    else {
      return IntegerType(ObjectBase::VARIABLE, right);
    }
  }
  else {
    cerr << "The parameter of `LOW attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, (int)0);
  }
}

IntegerType 
ArrayType::LENGTH(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1)))  {
    int right, left, length;
    right = (aInfo.get_rangeInfo()[dimension-1]).get_right();
    left  = (aInfo.get_rangeInfo()[dimension-1]).get_left();
    length = abs(right - left) + 1;
    return IntegerType(ObjectBase::VARIABLE, length);
  }
  else {
    cerr << "The parameter of `LENGTH attribute not within range ";abort();
    return IntegerType(ObjectBase::VARIABLE, (int)0);
  }
}

EnumerationType
ArrayType::ASCENDING(IntegerType& dimen, const arrayTypeInfo& aInfo){
  UniversalInteger dimension = (UniversalInteger&) dimen.getVHDLData();
  if ((aInfo.get_rangeInfo() != NULL) &&
      (dimension <= (aInfo.get_dimensions() + 1)))  {
    if ((aInfo.get_rangeInfo()[dimension-1]).get_direction() == to){
      return SAVANT_BOOLEAN_TRUE;
    }
    else {
      return SAVANT_BOOLEAN_FALSE;
    }
  }
  else {
    cerr << "The parameter of `ASCENDING attribute not within range ";abort();
    return SAVANT_BOOLEAN_FALSE;
  }
}

