// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "LogRBF.h"

namespace Torch {

LogRBF::LogRBF(int n_inputs_, int n_outputs_, EMTrainer* kmeans_trainer)
{
  n_inputs = n_inputs_;
  n_outputs = n_outputs_;
  initial_kmeans_trainer = kmeans_trainer;
}

int LogRBF::numberOfParams()
{
  return(2*n_inputs*n_outputs);
}

void LogRBF::init()
{
  GradientMachine::init();
  reset();
}

void LogRBF::reset()
{
  if (initial_kmeans_trainer) {
    initial_kmeans_trainer->machine->reset();
    initial_kmeans_trainer->train(NULL);
    // transform variances into gamma
    real* kmeans_params = 
      (real*)initial_kmeans_trainer->distribution->params->ptr;
    real* kmeans_var = kmeans_params + n_outputs + n_inputs;
    real* kmeans_mu = kmeans_params + n_outputs;
    real* mu = (real*)params->ptr;
    real* gamma = (real*)params->ptr + n_inputs;
    for (int i=0;i<n_outputs;i++) {
      for (int j=0;j<n_inputs;j++,gamma++,kmeans_var++,kmeans_mu++,mu++) {
        *gamma = 1./sqrt(*kmeans_var);
        *mu = *kmeans_mu;
      }
      kmeans_var += n_inputs;
      gamma += n_inputs;
      kmeans_mu += n_inputs;
      mu += n_inputs;
    }
  } else {
    // think more about this...
    real *params_ = (real *)params->ptr;
    real bound = 1./sqrt((real)n_inputs);

    for(int i = 0; i < n_params; i++)
      params_[i] = bounded_uniform(-bound, bound);
  }
}

void LogRBF::forward(List *inputs)
{
  real *ptr_params = (real *)params->ptr;
  real *ptr_outputs = (real *)outputs->ptr;
  real *mu = ptr_params;
  real *gamma = mu + n_inputs;

  for(int s = 0; s < n_outputs; s++)
  {
    real out = 0;
    List *inputs_ = inputs;
    while(inputs_)
    {
      real *x = (real *)inputs_->ptr;
      for(int j = 0; j < inputs_->n; j++) {
        real diff = (*x++ - *mu++) * *gamma++;
        out += diff*diff;
      }
      inputs_ = inputs_->next;
    }
    out *= -0.5;
    *ptr_outputs++ = out;
    mu += n_inputs;
    gamma += n_inputs;
  }
}

void LogRBF::backward(List *inputs, real *alpha)
{
  real *beta_ptr = beta;
  for(int k = 0; k < n_inputs; k++)
    *beta_ptr++ = 0;

  real *alpha_ptr = alpha;
  real *params_ptr = (real *)params->ptr;
  real *der_params_ptr = (real *)der_params->ptr;

  real *mu = params_ptr;
  real *gamma = mu + n_inputs;
  real *der_mu = der_params_ptr;
  real *der_gamma = der_mu + n_inputs;

  for(int j = 0; j < n_outputs; j++, alpha_ptr++)
  {
    List *inputs_ = inputs;
    beta_ptr = beta;
    while(inputs_)
    {
      real *x = (real *)inputs_->ptr;
      for(int k = 0; k < inputs_->n; k++) {
        real diff = (*x++ - *mu++);
        real g2 = (*gamma * *gamma);
        real dd = *alpha_ptr * 2. * diff * g2;
        *der_mu++ = dd;
        *beta_ptr++ += dd;
        *der_gamma++ = *alpha_ptr * 2. * diff*diff * *gamma++;
      }
      inputs_ = inputs_->next;
    }
    mu += n_inputs;
    gamma += n_inputs;
    der_mu += n_inputs;
    der_gamma += n_inputs;
  }
}

LogRBF::~LogRBF()
{

}

}

