// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include "EditDistance.h"

namespace Torch {

EditDistance::EditDistance()
{
  obtained = NULL;
  desired = NULL;
  obt_size = 0;
  des_size = 0;
  insert_cost = 1;
  delete_cost = 1;
  subst_cost = 1;
  reset();
}

void EditDistance::setCosts(int i_cost, int d_cost, int s_cost)
{
  insert_cost = i_cost;
  delete_cost = d_cost;
  subst_cost = s_cost;
}

void EditDistance::reset()
{
  accuracy = 0;
  n_insert=0;
  n_delete=0;
  n_subst=0;
  n_seq=0;
}

void EditDistance::distance(int* obtained_, int obt_size_, int* desired_, int des_size_)
{
  obtained = obtained_;
  obt_size = obt_size_;
  desired = desired_;
  des_size = des_size_;

  n_insert = 0;
  n_delete = 0;
  n_subst = 0;
  int subst;
  int **d = (int**)xalloc((des_size+1)*sizeof(int*));
  int **d_ins = (int**)xalloc((des_size+1)*sizeof(int*));
  int **d_del = (int**)xalloc((des_size+1)*sizeof(int*));
  int **d_sub = (int**)xalloc((des_size+1)*sizeof(int*));
  for (int i=0;i<des_size+1;i++) {
    d[i] = (int*)xalloc((obt_size+1)*sizeof(int));
    d_ins[i] = (int*)xalloc((obt_size+1)*sizeof(int));
    d_del[i] = (int*)xalloc((obt_size+1)*sizeof(int));
    d_sub[i] = (int*)xalloc((obt_size+1)*sizeof(int));
    for (int j=0;j<obt_size+1;j++) {
      d[i][j] = 0;
      d_ins[i][j] = 0;
      d_del[i][j] = 0;
      d_sub[i][j] = 0;
    }
  }
  for (int i=0;i<des_size;i++) {
    d[i+1][0] = d[i][0] + delete_cost;
    d_del[i+1][0] = d[i+1][0];
  }
  for (int i=0;i<obt_size;i++) {
    d[0][i+1] = d[0][i] + insert_cost;
    d_ins[0][i+1] = d[0][i+1];
  }
  for (int i=0;i<des_size;i++) {
    for (int j=0;j<obt_size;j++) {
      if (desired[i] == obtained[j]) {
        subst = 0;
      } else {
        subst = subst_cost;
      }
      int s_cost = d[i][j]+subst;
      int d_cost = d[i][j+1]+delete_cost;
      int i_cost = d[i+1][j]+insert_cost;
      if (s_cost <= d_cost && s_cost <= i_cost) {
        d[i+1][j+1] = s_cost;
        d_sub[i+1][j+1] = d_sub[i][j]+subst;
        d_del[i+1][j+1] = d_del[i][j];
        d_ins[i+1][j+1] = d_ins[i][j];
      } else if (d_cost <= i_cost && d_cost <= s_cost) {
        d[i+1][j+1] = d_cost;
        d_del[i+1][j+1] = d_del[i][j+1]+delete_cost;
        d_sub[i+1][j+1] = d_sub[i][j+1];
        d_ins[i+1][j+1] = d_ins[i][j+1];
      } else {
        d[i+1][j+1] = i_cost;
        d_ins[i+1][j+1] = d_ins[i+1][j]+insert_cost;
        d_del[i+1][j+1] = d_del[i+1][j];
        d_sub[i+1][j+1] = d_sub[i+1][j];
      }
    }
  }
  n_subst = d_sub[des_size][obt_size] / subst_cost;
  n_delete = d_del[des_size][obt_size] / delete_cost;
  n_insert = d_ins[des_size][obt_size] / insert_cost;
  n_seq = des_size;
  //dist = d[des_size][obt_size];
  accuracy = (n_seq - n_delete - n_subst - n_insert) * 100. / (real)n_seq;

  for (int i=0;i<des_size+1;i++) {
    free(d[i]);
    free(d_ins[i]);
    free(d_del[i]);
    free(d_sub[i]);
  }
  free(d);
  free(d_ins);
  free(d_del);
  free(d_sub);
}

void EditDistance::add(EditDistance* d)
{
  n_insert += d->n_insert;
  n_delete += d->n_delete;
  n_subst += d->n_subst;
  n_seq += d->n_seq;
  accuracy = (n_seq - n_delete - n_subst - n_insert) * 100. / (real)n_seq;
}

void EditDistance::print(FILE *f)
{
  fprintf(f,"total %d insert %d delete %d subst %d N %d\n",
    n_insert+n_delete+n_subst,n_insert,n_delete,n_subst,n_seq);
  fflush(f);
}

void EditDistance::printRatio(FILE *f)
{
  fprintf(f,"accuracy %5.2f insert %5.2f delete %5.2f subst %5.2f N %d\n",  
    accuracy,n_insert*100./n_seq,n_delete*100./n_seq,n_subst*100./n_seq,n_seq);
  fflush(f);
}

EditDistance::~EditDistance() 
{
}

}

