/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2009 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include "CovMat_local.h"
#include "CovMat.h"


void
PrintCovMatGnuPlot(const double **mat, const int dim, char *outfile)
{
    FILE           *fp = NULL;
    int             i, j;

    fp = myfopen(outfile, "w");
/*     for (i = 0; i < dim; ++i) */
/*     { */
/*         for (j = 0; j < dim; ++j) */
/*             fprintf(fp, "% 11.8f ", mat[i][j]); */
/*  */
/*         fprintf(fp, "\n"); */
/*     } */
/*     fprintf(fp, "\n"); */

    for (i = 0; i < dim; ++i)
    {
        for (j = 0; j < dim; ++j)
            fprintf(fp, "%4d %4d % 14.8f\n", i, j, mat[i][j]);

        fprintf(fp, "\n");
    }
    fprintf(fp, "\n");

    fclose(fp);
}


void
SetupCovWeighting(CoordsArray *cdsA)
{
    int             i;
    const int       vlen = cdsA->vlen;

    /* set up matrices and initialize to identity for full covariance matrix weighting */
    if (cdsA->CovMat == NULL)
        cdsA->CovMat = MatInit(vlen, vlen);

    if (cdsA->WtMat == NULL)
        cdsA->WtMat = MatInit(vlen, vlen);

    if (cdsA->AxCovMat == NULL)
        cdsA->AxCovMat = MatInit(3, 3);

    for (i = 0; i < vlen; ++i)
        cdsA->CovMat[i][i] = cdsA->WtMat[i][i] = 1.0;

    if (cdsA->tmpvecK == NULL)
        cdsA->tmpvecK = malloc(vlen * sizeof(double));

    if (cdsA->tmpmatKK1 == NULL)
        cdsA->tmpmatKK1 = MatInit(vlen, vlen);

    if (cdsA->tmpmatKK2 == NULL)
        cdsA->tmpmatKK2 = MatInit(vlen, vlen);

    if (cdsA->tmpmatK3a == NULL)
        cdsA->tmpmatK3a = MatInit(vlen, 3);

    if (cdsA->tmpmatK3b == NULL)
        cdsA->tmpmatK3b = MatInit(vlen, 3);

    if (cdsA->tmpmat3K == NULL)
        cdsA->tmpmat3K = MatInit(3, vlen);
}


/* returns 1 if all variances are about zero (< DBL_EPSILON) */
int
CheckZeroVariances(CoordsArray *cdsA)
{
    Algorithm      *algo = cdsA->algo;
    int             i, zeroflag = 1;

    if (algo->varweight != 0)
    {
        for (i = 0; i < cdsA->vlen; ++i)
            if (cdsA->var[i] > DBL_EPSILON)
                zeroflag = 0;
    }
    else if (algo->covweight != 0)
    {
        for (i = 0; i < cdsA->vlen; ++i)
            if (cdsA->CovMat[i][i] > DBL_EPSILON)
                zeroflag = 0;
    }

    return(zeroflag);

/*     if (zeroflag == 1) */
/*     { */
/*         double var = cdsA->stats->wRMSD_from_mean * cdsA->stats->wRMSD_from_mean; */
/*  */
/*      if (algo->varweight != 0) */
/*      { */
/*          memsetd(cdsA->var, var, cdsA->vlen); */
/*      } */
/*      else if (algo->covweight != 0) */
/*      { */
/*          for (i = 0; i < cdsA->vlen; ++i) */
/*              cdsA->CovMat[i][i] = var; */
/*      } */
/*     } */
}


void
CalcBfactC(CoordsArray *cdsA)
{
    int         i, j;
    double      trBS, occsum;

    for (i = 0; i < cdsA->cnum; ++i)
    {
        trBS = occsum = 0.0;
        for (j = 0; j < cdsA->vlen; ++j)
        {
            if (cdsA->coords[i]->o[j] > 0)
            {
                occsum += 1.0;
                trBS += cdsA->coords[i]->prvar[j] / cdsA->var[j];
                /*printf("trBS[%d] = % f\n", j, cdsA->coords[i]->prvar[j] / cdsA->var[j]);*/
            }
        }

        cdsA->coords[i]->bfact_c = occsum / trBS;
        /*printf("bfact_c[%d] = % f\n", i, cdsA->coords[i]->bfact_c);*/
    }
}


/* Weighting by dimensional, axial Xi covariance matrix, here diagonal. */
/* Holding the superposition constant, calculates the covariance
   matrices and corresponding weight matrices, looping till 
   convergence. */
void
CalcCovariances(CoordsArray *cdsA)
{
    Algorithm      *algo = cdsA->algo;

    if (algo->dimweight != 0)
    {
        if (algo->verbose == 1)
        {
            printf("  Begin Dimensional Cov Calc\n");
            fflush(NULL);
        }

        if (algo->varweight != 0)
            mlMatNormCovMats(cdsA, 1 * algo->precision, 1);
        else if (algo->covweight != 0)
            mlMatNormCovMats2(cdsA, 1 * algo->precision, 1);
        else if (algo->leastsquares != 0)
            mlMatNormCovMatsConstVar(cdsA, 1 * algo->precision, 1);

        if (algo->verbose == 1)
        {
            printf("  End Dimensional Cov Calc\n");
            fflush(NULL);
        }
    }
    else /* only atomic row-wise weighting (spherical variances) */
    {
        if (algo->varweight != 0 || algo->leastsquares != 0)
        {
            if (algo->alignment == 1)
                VarianceCoordsOcc(cdsA);
            else
                VarianceCoords(cdsA);
        }
        else if (algo->covweight != 0)
            CalcCovMat(cdsA);
    }
}


void
mlMatNormCovMats(CoordsArray *cdsA, const double precision, const int iterate)
{
    int             counter;
    double        **rotmat = cdsA->tmpmat3a;
    double        **lastmat = cdsA->tmpmat3b;
    double         *evals = cdsA->avecoords->evals;

    if (cdsA->algo->noave == 0)
        AveCoords(cdsA);

    counter = 0;
    do
    {
        Mat3Cpy(lastmat, (const double **) cdsA->AxCovMat);

        VarianceCoordsFullAx(cdsA);
        CalcWts(cdsA);
/*         if (cdsA->algo->noave == 0) */
/*             AveCoords(cdsA); */
        CalcAxCovMat(cdsA);

        if (cdsA->algo->verbose == 1)
        {
            printf(" Axes Cov Mat: %3d %e\n", counter,
                   MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3));
            Mat3Print(cdsA->AxCovMat);
            fflush(NULL);
        }

        if (iterate == 0 || cdsA->algo->abort == 1)
            break;
        if (counter > 50)
        {
/*          printf("\n WARNING04: Axial covariance matrix estimation failed to converge in mlMatNormCovMats(), round %d\n    ", */
/*                     cdsA->algo->rounds); */
            putchar(':');
            fflush(NULL);
            break;
        }
        else
            ++counter;
    } while(MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3) > precision);

/*     printf("\n before:"); */
/*     Mat3Print(cdsA->AxCovMat); */
    PrincAxesAxCovMat(cdsA, rotmat, evals);
    RotateCoordsArrayIp(cdsA, (const double **) rotmat);
    CalcAxCovMat(cdsA);
/*     printf("\n after:"); */
/*     Mat3Print(cdsA->AxCovMat); */
}


/* Use full atomic row-wise covariance matrix with correlations. */
/* Aligns a set of coordinates with the principle axes of its
   dimensional covariance matrix. Iiterate till convergence. */
void
mlMatNormCovMats2(CoordsArray *cdsA, const double precision, const int iterate)
{
    int             counter;
    double        **rotmat = MatInit(3,3)/* cdsA->tmpmat3a */;
    double        **lastmat = MatInit(3,3)/* cdsA->tmpmat3b */;
    double         *evals = cdsA->avecoords->evals;

    if (cdsA->algo->noave == 0)
        AveCoords(cdsA);

    counter = 0;
    do
    {
        Mat3Cpy(lastmat, (const double **) cdsA->AxCovMat);

        CalcCovMatFullAx(cdsA);
/*         CalcCovMat(cdsA); */

/*         int i, j; */
/*      for (i = 0; i < cdsA->vlen; ++i) */
/*          for (j = 0; j < i; ++j) */
/*              cdsA->CovMat[i][j] = cdsA->CovMat[j][i] = 0.0; */

/*         VarianceCoordsFullAx(cdsA); */
/*         int i; */
/*         for (i = 0; i < cdsA->vlen; ++i) */
/*             printf("\n%3d %f %f %e", i, cdsA->var[i], cdsA->CovMat[i][i], cdsA->CovMat[i][i] - cdsA->var[i]); */
/*         for (i = 0; i < cdsA->vlen; ++i) */
/*             cdsA->var[i] = cdsA->CovMat[i][i]; */
        CalcWts(cdsA);
        /* InvSymEigenOp(cdsA->WtMat, (const double **) cdsA->CovMat, cdsA->vlen, cdsA->tmpvecK, cdsA->tmpmatKK1, DBL_MIN); */

/*         if (cdsA->algo->noave == 0) */
/*             AveCoords(cdsA); */

/* MatPrint(cdsA->WtMat, cdsA->vlen); */

        CalcAxCovMat2(cdsA);

/*         if (counter % 5 == 3) */
/*         { */
/*             PrincAxesAxCovMat(cdsA, rotmat, evals); */
/*             RotateCoordsArrayIp(cdsA, (const double **) rotmat); */
/*         } */

/*      PrincAxesAxCovMat(cdsA, rotmat, evals); */
/*      RotateCoordsArrayIp(cdsA, (const double **) rotmat); */
/*      CalcAxCovMat2(cdsA); */

/*      printf("\n\n *******************************************************************:"); */
/*      printf("\n 2:"); */
/*      Mat3Print(cdsA->AxCovMat); */
/*  */
/*         for (i = 0; i < cdsA->vlen; ++i) */
/*             printf("\n%3d %e", i, cdsA->WtMat[i][i]); */
/*  */
/*         cdsA->algo->varweight = 1; */
/*         cdsA->algo->covweight = 0; */
/*  */
/*         VarianceCoordsFullAx(cdsA); */
/*         CalcWts(cdsA); */
/*  */
/*         for (i = 0; i < cdsA->vlen; ++i) */
/*             printf("\n%3d %e", i, cdsA->w[i]); */
/*  */
/*         CalcAxCovMat(cdsA); */
/*      printf("\n 1:"); */
/*      Mat3Print(cdsA->AxCovMat); */
/*      fflush(NULL); */
/*  */
/*         cdsA->algo->varweight = 0; */
/*         cdsA->algo->covweight = 1; */
/*  */
        printf(" Axes Cov Mat: %3d %e\n", counter,
               MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3));
        fflush(NULL);

        if (cdsA->algo->verbose == 1)
        {
            printf(" Axes Cov Mat: %3d %e\n", counter,
                   MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3));
            Mat3Print(cdsA->AxCovMat);
            fflush(NULL);
        }

        if (iterate == 0 || cdsA->algo->abort == 1)
            break;
        else if (counter > 50)
        {
/*          printf("\n WARNING04: Axial covariance matrix estimation failed to converge in mlMatNormCovMats(), round %d\n    ", */
/*                     cdsA->algo->rounds); */
            putchar(':');
            fflush(NULL);
            break;
        }
        else
            ++counter;
    } while(MatDiff((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3) > precision);

/*     printf("\n before:"); */
/*     Mat3Print(cdsA->AxCovMat); */
    PrincAxesAxCovMat(cdsA, rotmat, evals);
    RotateCoordsArrayIp(cdsA, (const double **) rotmat);
    CalcAxCovMat2(cdsA);
    printf(" after:\n");
    Mat3Print(cdsA->AxCovMat);

    free(rotmat);
    free(lastmat);
}


void
mlMatNormCovMatsConstVar(CoordsArray *cdsA, const double precision, const int iterate)
{
    int             counter;
    double        **rotmat = cdsA->tmpmat3a;
    double        **lastmat = cdsA->tmpmat3b;
    double         *evals = cdsA->avecoords->evals;
    double          var;

    if (cdsA->algo->noave == 0)
        AveCoords(cdsA);

    counter = 0;
    do
    {
        Mat3Cpy(lastmat, (const double **) cdsA->AxCovMat);

        var = VarianceCoordsFullAx(cdsA);
        memsetd(cdsA->var, var, cdsA->vlen);
        CalcWts(cdsA);
        CalcAxCovMat(cdsA);

        if (cdsA->algo->verbose == 1)
        {
            printf(" Axes Cov Mat: %3d %e\n", counter,
                   MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3));
            Mat3Print(cdsA->AxCovMat);
            fflush(NULL);
        }

        if (iterate == 0 || cdsA->algo->abort == 1)
            break;
        if (counter > 50)
        {
/*          printf("\n WARNING04: Axial covariance matrix estimation failed to converge in mlMatNormCovMats(), round %d\n    ", */
/*                     cdsA->algo->rounds); */
            putchar(':');
            fflush(NULL);
            break;
        }
        else
            ++counter;
    } while(MatFrobNorm((const double **) cdsA->AxCovMat, (const double **) lastmat, 3, 3) > precision);

/*     printf("\n before:"); */
/*     Mat3Print(cdsA->AxCovMat); */
    PrincAxesAxCovMat(cdsA, rotmat, evals);
    RotateCoordsArrayIp(cdsA, (const double **) rotmat);
    CalcAxCovMat(cdsA);
/*     printf("\n after:"); */
/*     Mat3Print(cdsA->AxCovMat); */
}


void
MVCovMat(CoordsArray *cdsA)
{
    int             i, j;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double    idf = 1.0 / (double)(cnum * vlen);
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi;
    double        **MVCovMat = NULL;

    if (cdsA->MVCovMat == NULL)
        MVCovMat = cdsA->MVCovMat = MatInit(3, 3);
    else
        MVCovMat = cdsA->MVCovMat;

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            MVCovMat[i][j] = 0.0;

    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < vlen; ++j)
        {
            coordsi = (Coords *) coords[i];
            MVCovMat[0][0] += mysquare(coordsi->residual_x[j]);
            MVCovMat[1][1] += mysquare(coordsi->residual_y[j]);
            MVCovMat[2][2] += mysquare(coordsi->residual_z[j]);
            MVCovMat[0][1] += coordsi->residual_x[j] * coordsi->residual_y[j];
            MVCovMat[0][2] += coordsi->residual_x[j] * coordsi->residual_z[j];
            MVCovMat[1][2] += coordsi->residual_y[j] * coordsi->residual_z[j];
        }
    }

    MVCovMat[1][0] = MVCovMat[0][1];
    MVCovMat[2][0] = MVCovMat[0][2];
    MVCovMat[2][1] = MVCovMat[1][2];

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            MVCovMat[i][j] *= idf;
}


/* Calculate the axial or dimensional covariance matrix, assuming a diagonal
   (no covariances) atomic, row-wise covariance matrix.
   We can use the normalized weights instead of the inverse of the variances
   here, since we re-normalize the trace to 3 at the end anyway. 
   (The inverse of the variances and the weights are proportional up to
   a constant). */
void
CalcAxCovMat(CoordsArray *cdsA)
{
    int             i, j;
    double          wtj;
    double          trace;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi;
    double          tmpx, tmpy, tmpz;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **AxCovMat = cdsA->AxCovMat;
    const double   *wts = cdsA->w;

    if (cdsA->AxCovMat == NULL)
    {
        printf("\nERROR652\n");
        exit(EXIT_FAILURE);
    }

    memset(&AxCovMat[0][0], 0, 9 * sizeof(double));

    for (i = 0; i < cnum; ++i)
    {
        coordsi = (Coords *) coords[i];

        for (j = 0; j < vlen; ++j)
        {            
            wtj = wts[j];

            tmpx = coordsi->x[j] - avex[j];
            tmpy = coordsi->y[j] - avey[j];
            tmpz = coordsi->z[j] - avez[j];

            AxCovMat[0][0] += tmpx * tmpx * wtj;
            AxCovMat[1][1] += tmpy * tmpy * wtj;
            AxCovMat[2][2] += tmpz * tmpz * wtj;
            AxCovMat[0][1] += tmpx * tmpy * wtj;
            AxCovMat[0][2] += tmpx * tmpz * wtj;
            AxCovMat[1][2] += tmpy * tmpz * wtj;
        }
    }

    AxCovMat[1][0] = AxCovMat[0][1];
    AxCovMat[2][0] = AxCovMat[0][2];
    AxCovMat[2][1] = AxCovMat[1][2];

    trace = (AxCovMat[0][0] + AxCovMat[1][1] + AxCovMat[2][2]) / 3.0;

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            AxCovMat[i][j] /= trace;

    /* Mat3Print(AxCovMat); */
}


void
CalcAxCovMat1(CoordsArray *cdsA)
{
    int             i, j, k, m;
    double          trace;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double   *wt = (const double *) cdsA->w;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **AxCovMat = cdsA->AxCovMat;
    double        **tmpCoords = cdsA->tmpmatK3a;

    if (AxCovMat == NULL)
    {
        printf("\n  ERROR652 \n");
        exit(EXIT_FAILURE);
    }

    if (tmpCoords == NULL)
        tmpCoords = cdsA->tmpmatK3a = MatInit(vlen, 3);

    memset(&AxCovMat[0][0], 0, 9 * sizeof(double));

    for (m = 0; m < cnum; ++m)
    {
        coordsi = (Coords *) coords[m];

        for (i = 0; i < vlen; ++i)
        {
            tmpCoords[i][0] = coordsi->x[i] - avex[i];
            tmpCoords[i][1] = coordsi->y[i] - avey[i];
            tmpCoords[i][2] = coordsi->z[i] - avez[i];
        }

        /* (i x k)(k x j) = (i x j) */
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                for (k = 0; k < vlen; ++k)
                    AxCovMat[i][j] += tmpCoords[k][i] * wt[k] * tmpCoords[k][j];
    }

/*     AxCovMat[1][0] = AxCovMat[0][1]; */
/*     AxCovMat[2][0] = AxCovMat[0][2]; */
/*     AxCovMat[2][1] = AxCovMat[1][2]; */

    trace = (AxCovMat[0][0] + AxCovMat[1][1] + AxCovMat[2][2]) / 3.0;

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            AxCovMat[i][j] /= trace;

    /* Mat3Print(AxCovMat); */
}


/* Weighted by a full atomic, row-wise covariance matrix (non-zero correlations) */
void
CalcAxCovMat2(CoordsArray *cdsA)
{
    int             i, j, k, m;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    double          trace;
    double         *tmpCoordsi = NULL;
    const Coords   *coordsm;
    const double  **WtMat = (const double **) cdsA->WtMat;
    double        **AxCovMat = cdsA->AxCovMat;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **tmpCoords = cdsA->tmpmatK3a;
    double        **tmpMat = cdsA->tmpmat3K;

    if (tmpCoords == NULL || tmpMat == NULL || WtMat == NULL || AxCovMat == NULL)
    {
        printf("\nERROR655\n");
        exit(EXIT_FAILURE);
    }

    memset(&AxCovMat[0][0], 0, 9 * sizeof(double));

    for (m = 0; m < cnum; ++m)
    {
        coordsm = (Coords *) coords[m];

        for (i = 0; i < vlen; ++i)
        {
            tmpCoordsi = tmpCoords[i];
            tmpCoordsi[0] = coordsm->x[i] - avex[i];
            tmpCoordsi[1] = coordsm->y[i] - avey[i];
            tmpCoordsi[2] = coordsm->z[i] - avez[i];
        }

        /* (i x k)(k x j) = (i x j) */
        for (i = 0; i < 3; ++i)
        {
            for (j = 0; j < vlen; ++j)
            {
                tmpMat[i][j] = 0.0;
                for (k = 0; k < vlen; ++k)
                    tmpMat[i][j] += tmpCoords[k][i] * WtMat[k][j];
            }
        }

        for (i = 0; i < 3; ++i)
            for (j = i; j < 3; ++j)
                for (k = 0; k < vlen; ++k)
                    AxCovMat[i][j] += tmpMat[i][k] * tmpCoords[k][j];
    }

    trace = (AxCovMat[0][0] + AxCovMat[1][1] + AxCovMat[2][2]) / 3.0;

    for (i = 0; i < 3; ++i)
        for (j = i; j < 3; ++j)
            AxCovMat[j][i] = AxCovMat[i][j] = AxCovMat[i][j] / trace;

    /* Mat3Print(AxCovMat); */
}


void
PrincAxesAxCovMat(CoordsArray *cdsA, double **rotmat, double *evals)
{
    int             i;
    double        **AxCovMat = cdsA->AxCovMat;

    /* eigenvals smallest to largest */
    eigensym((const double **) AxCovMat, evals, rotmat, 3);

    /* Make sure that the rotation is proper, i.e. no inversion.
       If there is an inversion, invert it. 
       This is OK because, in terms of the eigen decomp, the sign
       of the elements of an eigenvector is arbitrary, a result
       of the more general property that eigenvectors are only
       defined up to an arbitrary multiplicative constant. */
    if (Mat3Det((const double **) rotmat) < 0)
    {
        for (i = 0; i < 3; ++i)
            rotmat[i][0] *= -1.0;
    }

    for (i = 0; i < 3; ++i)
    {
        cdsA->axesvar[i] = evals[i];
        cdsA->axesw[i] = 1.0 / evals[i];
    }

    /* Mat3TransposeIp(rotmat); */

    if (VerifyRotMat(rotmat, cdsA->algo->precision) == 0)
        Mat3Print(rotmat);
}


void
CalcCovMat(CoordsArray *cdsA)
{
    double          newx1, newy1, newz1, newx2, newy2, newz2;
    double          covsum;
    double         *coordskx, *coordsky, *coordskz;
    int             i, j, k;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double    normalize = 1.0 / (3.0 * cnum);
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsk;
    double        **CovMat = cdsA->CovMat;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    const double    axesx = cdsA->axesw[0],
                    axesy = cdsA->axesw[1],
                    axesz = cdsA->axesw[2];

    if (cdsA->CovMat == NULL)
    {
        printf("\nERROR654\n");
        exit(EXIT_FAILURE);
    }

    /* calculate covariance matrix of atoms across structures,
       based upon current superposition, put in CovMat */
    for (i = 0; i < vlen; ++i)
    {
        for (j = 0; j <= i; ++j)
        {
            covsum = 0.0;
            for (k = 0; k < cnum; ++k)
            {
                coordsk = coords[k];
                coordskx = coordsk->x;
                coordsky = coordsk->y;
                coordskz = coordsk->z;

                newx1 = coordskx[i] - avex[i];
                newy1 = coordsky[i] - avey[i];
                newz1 = coordskz[i] - avez[i];

                newx2 = coordskx[j] - avex[j];
                newy2 = coordsky[j] - avey[j];
                newz2 = coordskz[j] - avez[j];

                #ifdef FP_FAST_FMA
                covsum += fma(newx1, newx2*axesx, fma(newy1, newy2*axesy, newz1 * newz2 * axesz));
                #else
                covsum += (newx1 * newx2 * axesx +
                           newy1 * newy2 * axesy +
                           newz1 * newz2 * axesz);
                #endif
            }

            CovMat[i][j] = CovMat[j][i] = covsum * normalize; /* sample variance, ML biased not n-1 definition */
        }
    }

    for (i = 0; i < vlen; ++i)
        cdsA->var[i] = CovMat[i][i];
}


void
CalcCovMatFullAx(CoordsArray *cdsA)
{
    int             i, j, k, m;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsm;
    double        **CovMat = cdsA->CovMat;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **ErrMat = cdsA->tmpmatK3a;
    double        **TmpMat = cdsA->tmpmatK3b;
    double        **WtMat = MatInit(3, 3);

    if (cdsA->CovMat == NULL)
    {
        printf("\nERROR654\n");
        exit(EXIT_FAILURE);
    }

    InvSymEigenOp(WtMat, (const double **) cdsA->AxCovMat, 3, cdsA->tmpmat3c[0], cdsA->tmpmat3d, DBL_MIN);

    memset(&CovMat[0][0], 0, vlen * vlen * sizeof(double));

    for (m = 0; m < cnum; ++m)
    {
        for (i = 0; i < vlen; ++i)
        {
            coordsm = coords[m];
            ErrMat[i][0] = coordsm->x[i] - avex[i];
            ErrMat[i][1] = coordsm->y[i] - avey[i];
            ErrMat[i][2] = coordsm->z[i] - avez[i];
        }

        for (i = 0; i < vlen; ++i)
        {
            for (j = 0; j < 3; ++j)
            {
                TmpMat[i][j] = 0.0;
                for (k = 0; k < 3; ++k)
                    TmpMat[i][j] += ErrMat[i][k] * WtMat[k][j];
            }
        }

        for (i = 0; i < vlen; ++i)
            for (j = i; j < vlen; ++j)
                for (k = 0; k < 3; ++k)
                    CovMat[i][j] += TmpMat[i][k] * ErrMat[j][k];
    }

    for (i = 0; i < vlen; ++i)
        for (j = i; j < vlen; ++j)
            CovMat[j][i] = CovMat[i][j] = CovMat[i][j] / (3.0 * cnum);

    MatDestroy(&WtMat);
}


/* Same as CalcCovMat() but weights by the occupancies */
void
CalcCovMatOcc(CoordsArray *cdsA)
{
    double          newx1, newy1, newz1, newx2, newy2, newz2;
    double          covsum;
    double         *coordskx, *coordsky, *coordskz;
    int             i, j, k;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsk;
    double        **CovMat = cdsA->CovMat;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    const double    axesx = cdsA->axesw[0],
                    axesy = cdsA->axesw[1],
                    axesz = cdsA->axesw[2];
    double         *occ, osum;

    if (cdsA->CovMat == NULL)
    {
        printf("\nERROR653\n");
        exit(EXIT_FAILURE);
    }

    /* calculate covariance matrix of atoms across structures,
       based upon current superposition, put in CovMat */
    for (i = 0; i < vlen; ++i)
    {
        for (j = 0; j <= i; ++j)
        {
            covsum = osum = 0.0;
            for (k = 0; k < cnum; ++k)
            {
                coordsk = coords[k];
                occ = coordsk->o;
                coordskx = coordsk->x;
                coordsky = coordsk->y;
                coordskz = coordsk->z;

                newx1 = coordskx[i] - avex[i];
                newy1 = coordsky[i] - avey[i];
                newz1 = coordskz[i] - avez[i];

                newx2 = coordskx[j] - avex[j];
                newy2 = coordsky[j] - avey[j];
                newz2 = coordskz[j] - avez[j];

                covsum += occ[i] * occ[j] * 
                          (newx1 * newx2 * axesx +
                           newy1 * newy2 * axesy +
                           newz1 * newz2 * axesz);

                osum += occ[i] * occ[j];
            }

            if (osum > 0.0)
                CovMat[i][j] = CovMat[j][i] = covsum / osum; /* sample variance, ML biased not n-1 definition */
            else
                CovMat[i][j] = CovMat[j][i] = 0.0;
        }
    }
}


void
CalcStructCovMat(CoordsArray *cdsA)
{
    double          invdf, cov_sum;
    int             i, j, k;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi = NULL, *coordsj = NULL;
    double        **SCovMat = NULL;

    invdf = 1.0 / (double) vlen; /* ML, biased */

    if (cdsA->SCovMat == NULL)
        cdsA->SCovMat = MatInit(cnum, cnum);
    else
        memset(&cdsA->SCovMat[0][0], 0, cnum * cnum * sizeof(double));

    SCovMat = cdsA->SCovMat;

    /* calculate covariance matrix of structures across atoms, put in SCovMat */
    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < cnum; ++j)
        {
            cov_sum = 0.0;
            for (k = 0; k < vlen; ++k)
            {
                coordsi = coords[i];
                coordsj = coords[j];
                cov_sum += coordsi->x[k] * coordsj->x[k];
                cov_sum += coordsi->y[k] * coordsj->y[k];
                cov_sum += coordsi->z[k] * coordsj->z[k];
            }

            SCovMat[i][j] = cov_sum * invdf; /* sample variance, ML biased not n-1 definition */
        }
    }
}


void
CalcFullCovMat(CoordsArray *cdsA)
{
    double          newx1, newy1, newz1, newx2, newy2, newz2;
    double          invdf;
    double        **FullCovMat = cdsA->FullCovMat;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    const int       vlen = cdsA->vlen, cnum = cdsA->cnum;
    int             i, j, k, m, n, p, q0, q1, q2;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsk;

    invdf = 1.0 / (double) cnum; /* ML, biased */

    AveCoords(cdsA);

    if (FullCovMat == NULL)
        FullCovMat = cdsA->FullCovMat = MatInit(3 * cdsA->vlen, 3 * cdsA->vlen);

    for (i = 0; i < 3 * cdsA->vlen; ++i)
        for (j = 0; j < 3 * cdsA->vlen; ++j)
            FullCovMat[i][j] = 0.0;

    /* calculate covariance matrix based upon current superposition, put in FullCovMat */
    for (m = i = 0; i < vlen; ++i, m += 3)
    {
        for (n = j = 0; j <= i; ++j, n += 3)
        {
            for (k = 0; k < cnum; ++k)
            {
                coordsk = (Coords *) coords[k];
                newx1 = coordsk->x[i] - avex[i];
                newy1 = coordsk->y[i] - avey[i];
                newz1 = coordsk->z[i] - avez[i];

                newx2 = coordsk->x[j] - avex[j];
                newy2 = coordsk->y[j] - avey[j];
                newz2 = coordsk->z[j] - avez[j];

                q0 = n+0;
                q1 = n+1;
                q2 = n+2;

                p = m+0;
                FullCovMat[p][q0] += newx1 * newx2;
                FullCovMat[p][q1] += newx1 * newy2;
                FullCovMat[p][q2] += newx1 * newz2;

                p = m+1;
                FullCovMat[p][q0] += newy1 * newx2;
                FullCovMat[p][q1] += newy1 * newy2;
                FullCovMat[p][q2] += newy1 * newz2;

                p = m+2;
                FullCovMat[p][q0] += newz1 * newx2;
                FullCovMat[p][q1] += newz1 * newy2;
                FullCovMat[p][q2] += newz1 * newz2;
            }
        }
    }

    for (i = 0; i < 3 * vlen; ++i)
        for (j = 0; j <= i; ++j)
            FullCovMat[i][j] *= invdf;

    for (i = 0; i < 3 * vlen; ++i)
        for (j = 0; j < i; ++j)
            FullCovMat[j][i] = FullCovMat[i][j];
}


/* calculate covariance matrix weighted coords
   \Sigma^-1 * \CoordsMat */
void
CalcCovCoords(Coords *coords, const double **covmat)
{
    int             i, k;
    double         *covx = coords->covx,
                   *covy = coords->covy,
                   *covz = coords->covz;
    const double   *x = (const double *) coords->x,
                   *y = (const double *) coords->y,
                   *z = (const double *) coords->z;
    double          covmatik;

    for (i = 0; i < coords->vlen; ++i)
    {
        covx[i] = covy[i] = covz[i] = 0.0;

        for (k = 0; k < coords->vlen; ++k)
        {
            covmatik = covmat[i][k];

            covx[i] += (covmatik * x[k]);
            covy[i] += (covmatik * y[k]);
            covz[i] += (covmatik * z[k]);
        }
    }
}


int
CovInvWeightLAPACK(CoordsArray *cdsA)
{
    int                      i, j;
    long int                 info = 0;
    char                     upper = 'U';
    long int                 vlen = (long int) cdsA->vlen;

    /* LAPACK dpotrf and dpotri compute the inverse using cholesky decomposition */
    DPOTRF(&upper, &vlen, &cdsA->CovMat[0][0], &vlen, &info);

/*     long int        lwork = 1000; */
/*     long int       *ipiv = malloc(vlen * sizeof(long int)); */
/*     double         *work = (double *) malloc(lwork * sizeof(double)); */
/*  */
/*     DSYTRF(&upper, &vlen, &cdsA->CovMat[0][0], &vlen, ipiv, work, &lwork, &info); */

    if (info == 0)
        DPOTRI(&upper, &vlen, &cdsA->CovMat[0][0], &vlen, &info);
        /* DSYTRI(&upper, &vlen, &cdsA->CovMat[0][0], &vlen, ipiv, work, &info); */
    else if (info > 0)
    {
        fprintf(stderr, " \n\n ERROR111: LAPACK dpotrf Choleski decomposition choked; \n");
        fprintf(stderr, "           covariance matrix is singular; \n");
        fprintf(stderr, "           leading minor of order %ld is not positive definite \n\n", info);
        return((int) info);
    }

    /* copy lower to upper to WtMat*/
    for (i = 0; i < cdsA->vlen; ++i)
        for (j = 0; j <= i; ++j)
            cdsA->WtMat[j][i] = cdsA->WtMat[i][j] = cdsA->CovMat[i][j];

    /* MatPrint(cdsA->WtMat, cdsA->vlen); */
    /* SCREAMS("LAPACK"); */
    return((int) info);
}


/* Normalize the covariance matrix to form the correlation matrix 
   by dividing each element by the square root of the product of the
   corresponding diagonal elements.
   This makes a pearson correlation matrix.
   The diagonal elements are always equal to 1, while
   the off-diagonals range from -1 to 1.
*/
void
CovMat2CorMat(double **CovMat, const int size)
{
    int             i, j;

    for (i = 0; i < size; ++i)
        for (j = 0; j < i; ++j)
            CovMat[i][j] = CovMat[j][i] = CovMat[i][j] / sqrt(CovMat[i][i] * CovMat[j][j]);

    for (i = 0; i < size; ++i)
        CovMat[i][i] = 1.0;
}


void
CorMat2CovMat(double **CovMat, const double *vars, const int size)
{
    int             i, j;

    for (i = 0; i < size; ++i)
        for (j = 0; j < i; ++j)
            CovMat[i][j] = CovMat[j][i] = CovMat[i][j] * sqrt(vars[i] * vars[j]);

    for (i = 0; i < size; ++i)
        CovMat[i][i] = vars[i];
}


/* Normalizes a covariance matrix by dividing every cell by the 
   average variance */
double
NormalizeCovMat(double **mat, const int size)
{
    int             i, j;
    double          normalize;

    normalize = 0.0;
    for (i = 0; i < size; ++i)
        normalize += mat[i][i];

    normalize = size / normalize;

/*     normalize = 0.0; */
/*     for (i = 0; i < size; ++i) */
/*         for (j = 0; j < size; ++j) */
/*             normalize += mat[i][j]; */
/*     normalize = (double) size / normalize; */

    for (i = 0; i < size; ++i)
        for (j = 0; j < size; ++j)
            mat[i][j] *= normalize;

    /* fprintf(stderr, "\n     Mat[%3d][%3d] = %12.5f", size/2, size/2, mat[size/2][size/2]); */
/*     fprintf(stderr, "\n     norm = %12.5f", normalize); */
/*     fflush(NULL); */
    return(normalize);
}


void
PrintCovMat(CoordsArray *cdsA)
{
    int             i, j;
    const double  **CovMat = (const double **) cdsA->CovMat;

    for (i = 0; i < cdsA->vlen; ++i)
    {
        printf("\n");
        for (j = 0; j < cdsA->vlen; ++j)
            printf("%8.3f ", CovMat[i][j]);
    }
    printf("\n");
}
