/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009, 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef TOUCHPADMANAGER_H
#define TOUCHPADMANAGER_H

/**
 * @file
 *
 * Provide a touchpad management class.
 */

#include <QtDBus/QDBusContext>
#include <QtDBus/QDBusVariant>
#include <QtCore/QVariant>


namespace synaptiks {
    class Touchpad;
    class KeyboardMonitor;
    class MouseDevicesMonitor;

    class TouchpadManagerPrivate;

    /**
     * @brief Manage the touchpad.
     *
     * This class controls, whether the touchpad is on or off.  It monitors
     * the keyboard (using KeyboardMonitor) and mouse devices (using
     * MouseDevicesMonitor) and switches the touchpad on or off accordingly.
     *
     * \section usage Usage
     *
     * Simple create the touchpad
     *
     * \section dbus D-Bus
     *
     * All members marked as @c Q_SCRIPTABLE are exported on D-Bus on the
     * interface @c org.kde.TouchpadManager.
     */
    class TouchpadManager: public QObject, protected QDBusContext {
        Q_OBJECT
        Q_CLASSINFO("D-Bus Interface", "org.kde.TouchpadManager")

        /**
         * @brief Whether mouse device monitoring is enabled.
         *
         * @sa setMonitorMouseDevices(bool)
         * @sa monitorMouseDevices()
         */
        Q_PROPERTY(bool monitorMouseDevices
                   READ monitorMouseDevices WRITE setMonitorMouseDevices
                   DESIGNABLE false)

        /**
         * @brief Whether keyboard monitoring is enabled.
         *
         * @sa setMonitorKeyboard(bool)
         * @sa monitorKeyboard()
         */
        Q_PROPERTY(bool monitorKeyboard
                   READ monitorKeyboard WRITE setMonitorKeyboard
                   DESIGNABLE false)

        /**
         * @brief A list of ignored mouse devices
         *
         * @sa setIgnoredMouseDevices(const QStringList&)
         * @sa ignoredMouseDevices()
         */
        Q_PROPERTY(QStringList ignoredMouseDevices
                   READ ignoredMouseDevices
                   WRITE setIgnoredMouseDevices
                   DESIGNABLE false)

    public:
        /**
         * Create a new TouchpadManager for the given @p touchpad.
         *
         * The new instance will become the parent object of the @p touchpad
         * and the @p mouseDevicesMonitor.  Do @e not use the same touchpad
         * object with multiple managers!
         *
         * @param touchpad the synaptiks::Touchpad to manage
         * @param mouseDevicesMonitor used for keyboard monitoring
         * @param parent same as for QObject(QObject*)
         */
        TouchpadManager(Touchpad *touchpad,
                        MouseDevicesMonitor *mouseDevicesMonitor,
                        QObject *parent = 0);

        /**
         * Destroy this object.
         */
        ~TouchpadManager();

    public Q_SLOTS:
        /**
         * @brief Switch the touchpad according to the given parameter.
         *
         * @p reason tells this class, why the touchpad was switched.  This
         * class makes a difference between @c "interactive" and all other
         * strings.  If @p reason is @c "interactive", the touchpad is @em
         * always switched on or off, depending on the parameter @p on.  If
         * the touchpad is switched off by @c "interactive" @p reason, no
         * other value for @p reason will switch the touchpad on again.
         * Switching the touchpad off interactively thus overrides
         * everything else.
         *
         * All other values for @p "reason" are considered ”automatic“
         * reasons.  No such reason will switch the touchpad on, if it was
         * switched off by @p reason @c "interactive".  Moreover, no such
         * reason will switch the touchpad on, if there is still a automatic
         * reason remaining.  For instance, if the user stops typing with
         * keyboard monitoring enabled, while the touchpad is still switched
         * off by @p reason @c "mouse" (a mouse device is plugged), the
         * touchpad will @em not be switched on.  Only if the mouse device
         * is unpluggged, and the touchpad switched on by @p reason @c
         * "mouse", the touchpad will be switched again.
         *
         * Sometimes you might to pass additional context data to slots (for
         * instance, the name of a plugged mouse device).  For such
         * situations this method provides the @p closure parameter, which
         * is passed unchanged to
         * touchpadSwitched(bool, const QString&, const QVariant&).
         *
         * @param on if @c true, try to switch the touchpad on, otherwise
         *        switch it off
         * @param reason the reason for switching the touchpad
         * @param closure closure data for slots
         */
        void setTouchpadOn(bool on, const QString &reason=QString("unknown"),
                           const QVariant &closure=QVariant());

        /**
         * @brief Switch the touchpad on or off (DBus overload)
         *
         * This is an overloaded slot with a D-Bus specific signature.  See
         * setTouchpadOn(bool, const QString&, const QVariant&) for
         * documentation.
         *
         * If called over D-Bus, this method may return the D-Bus error @c
         * org.kde.TouchpadError, if the underlying Touchpad object
         * throws a QXDeviceError.
         *
         * @param on if @c true, try to switch the touchpad on, otherwise
         *        switch it off
         * @param reason the reason for switching the touchpad
         * @param closure closure data for slots
         * @sa setTouchpadOn(bool, const QString&, const QVariant&)
         */
        Q_SCRIPTABLE void setTouchpadOn(bool on, const QString &reason,
                                        const QDBusVariant &closure);

        /**
         * @brief Is the touchpad on or not?
         *
         * This method is mainly intented for DBus clients, it is not
         * recommended to use it from C++ code.
         *
         * This method is distinguished from @c touchpad()->isOn() by its
         * error reporting behaviour.  Unlike @c touchpad()->isOn() it does
         * not throw any exceptions, but emits touchpadError(const QString&)
         * in case of error.  If called from D-Bus, a @c
         * org.kde.TouchpadError is reported to the caller.
         *
         * @return @c true, if the touchpad is on, @c false otherwise
         */
        Q_SCRIPTABLE bool isTouchpadOn();

        /**
         * @brief Enable or disable mouse device monitoring.
         *
         * If mouse device monitoring is enabled, the touchpad will
         * automatically be switched off, if any mouse is plugged, and on
         * again, if no mouse is plugged anymore.  This is done by calling
         * setTouchpadOn(bool, const QString&, const QVariant&) with @p
         * reason @c "mouse".  The @p closure will be set to the product
         * name of the (un-)plugged mouse.  Thus, touchpad switches caused
         * by mouse devices emit
         * touchpadSwitched(bool, const QString&, const QVariant&)
         * with @p reason @c "mouse" and the product name as @p closure.
         *
         * @param enabled if @c true, monitor mouse devices, otherwise stop
         * monitoring.
         * @sa monitorMouseDevices()
         * @sa setIgnoredMouseDevices(const QStringList&)
         */
        Q_SCRIPTABLE void setMonitorMouseDevices(bool enabled);

        /**
         * @brief Is mouse device monitoring enabled?
         *
         * @return @c true, if it is enabled, @c false otherwise
         * @sa setMonitorMouseDevices(bool)
         * @sa mouseDevicesMonitor()
         */
        Q_SCRIPTABLE bool monitorMouseDevices() const;

        /**
         * @brief Set the mouse device, which should be ignored by the
         * monitor.
         *
         * @p devices must contain device IDs as used by
         * synaptiks::MouseDevicesMonitor.  Do not attempt to construct such
         * IDs manually, only rely on synaptiks::MouseDevicesMonitor.
         *
         * Some computers have virtual mouse devices, which do not
         * correspond to any physical device.  Such devices should not cause
         * changes to the touchpad state.  Therefore, this class will ignore
         * all given @p devices.  If such a device is plugged or unplugged,
         * nothing will happen.
         *
         * @param devices a list of devices to ignore (may be empty)
         * @sa ignoredMouseDevices()
         * @sa setMonitorMouseDevices(bool)
         */
        Q_SCRIPTABLE void setIgnoredMouseDevices(const QStringList &devices);

        /**
         * @brief Return the list of ignored mouse devices.
         *
         * @return the list of ignored mouse devices as HAL udis (may be
         *         empty)
         * @sa monitorMouseDevices()
         * @sa setIgnoredMouseDevices(const QStringList&)
         */
        Q_SCRIPTABLE QStringList ignoredMouseDevices() const;

        /**
         * @brief Enable or disable keyboard monitoring.
         *
         * If keyboard monitoring is enabled, the touchpad will
         * automatically be switched off, while the keyboard is active.
         * This is done by calling setTouchpadOn(bool, const QString&, const
         * QVariant&) with @p reason @c "keyboard".  The @p closure
         * parameter is unused.  Thus, touchpad switches caused by keyboard
         * events emit
         * touchpadSwitched(bool, const QString&, const QVariant&) with @p
         * reason @c "keyboard".
         *
         * @param enabled if @c true, monitor keyboard, otherwise stop
         * monitoring.
         * @sa monitorKeyboard()
         */
        Q_SCRIPTABLE void setMonitorKeyboard(bool enabled);

        /**
         * @brief Is mouse device monitoring enabled?
         *
         * @return @c true, if it is enabled, @c false otherwise
         * @sa setMonitorMouseDevices(bool)
         * @sa keyboardMonitor()
         */
        Q_SCRIPTABLE bool monitorKeyboard() const;

    public:
        /**
         * @brief Return the touchpad managed by this class.
         *
         * You can use this object to configure touchpad settings like
         * scrolling speed, tap buttons and the like.
         *
         * @return the touchpad object
         */
        Touchpad *touchpad() const;

        /**
         * @brief Return the KeyboardMonitor used by this class
         *
         * @return the keyboard monitor, or a null pointer, if keyboard
         *         monitoring is disabled
         * @sa monitorKeyboard()
         */
        KeyboardMonitor *keyboardMonitor() const;

        /**
         * @brief Return the MouseDevicesMonitor used by this class
         *
         * @return a MouseDevicesMonitor instance
         * @sa monitorMouseDevices()
         */
        MouseDevicesMonitor *mouseDevicesMonitor() const;

    Q_SIGNALS:
        /**
         * @brief Emitted, if the touchpad is switched.
         *
         * @p closure and @p reason are passed unchanged from the call to
         * setTouchpadOn(bool, const QString&, const QVariant&)
         *
         * @param on @c true, if the touchpad is now on, otherwise the
         *           touchpad is now off
         * @param reason the reason for the switch
         * @param closure some closure data (e.g. the name of the mouse
         *        device)
         * @sa setTouchpadOn(bool, const QString&, const QVariant&)
         */
        void touchpadSwitched(bool on, const QString &reason,
                              const QVariant &closure);

        /**
         * @brief Emitted, if the touchpad is switched (DBus overload)
         *
         * This is an overloaded signal with a D-Bus specific signature. See
         * touchpadSwitched(bool, const QString&, const QVariant&) for
         * documentation.
         *
         * @param on @c true, if the touchpad is now on, otherwise the
         *           touchpad is now off
         * @param reason the reason for the switch
         * @param closure some closure data (e.g. the name of the mouse
         *        device)
         * @sa touchpadSwitched(bool, const QString&, const QVariant&)
         */
        Q_SCRIPTABLE void touchpadSwitched(
            bool on, const QString &reason, const QDBusVariant &closure);

        /**
         * @brief Emitted, if a touchpad error occurs.
         *
         * @param message a human readable error message
         */
        Q_SCRIPTABLE void touchpadError(const QString &message);

    private:
        Q_DISABLE_COPY(TouchpadManager)
        Q_DECLARE_PRIVATE(TouchpadManager)

        TouchpadManagerPrivate * const d_ptr;

        Q_PRIVATE_SLOT(d_func(),
                       void _k_registerMouseDevice(const QString &))
        Q_PRIVATE_SLOT(d_func(),
                       void _k_unregisterMouseDevice(const QString &))
        Q_PRIVATE_SLOT(d_func(),
                       void _k_keyboardActivity())
    };
}

#endif /* TOUCHPADMANAGER_H */
