/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2009 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef KEYBOARDMONITOR_H
#define KEYBOARDMONITOR_H

/**
 * @file
 *
 * Provide a class to monitor the keyboard state on X11
 */

#include <QtCore/QObject>


namespace synaptiks {

    class KeyboardMonitorPrivate;

    /**
     * @brief Monitor the keyboard state and emit signals, if the keyboard
     * becomes active or inactive.
     *
     * This class polls the keyboard state periodically (see #pollDelay).
     * Whenever the keyboard state changes, the corresponding signals are
     * emitted (see typingStarted() and typingStopped()).  However, to
     * minimize "false positives" (if the user is just making a short break
     * during typing), this class waits a configurable amount of time (see
     * #idleTime), before emitting typingStopped().
     *
     * @bug
     *
     * Polling is neither nice nor elegant, however the required XRecord
     * extension is unfortunately broken as of Xorg server 1.6.
     */
    class KeyboardMonitor: public QObject {
        Q_OBJECT
        Q_ENUMS(IgnoreKeys)

        /**
         * @brief Which keys to ignore.
         *
         * The default is to ignore no keys (IgnoreKeys::IgnoreNoKeys).
         *
         * @sa setIgnoreKeys(IgnoreKeys)
         * @sa ignoreKeys()
         */
        Q_PROPERTY(IgnoreKeys ignoreKeys READ ignoreKeys
                   WRITE setIgnoreKeys)
        /**
         * @brief The time in ms between the end of the keyboard activity
         * and emitting typingStopped().
         *
         * typingStopped() is not immediately emitted, when the keyboard
         * becomes inactive.  The user could just be making a short break.
         * Instead, the class waits for the given amount of milliseconds,
         * before emitting typingStopped().
         *
         * The default is 2000 ms, which is 2 s.
         *
         * @sa setIdleTime(int)
         * @sa idleTime()
         */
        Q_PROPERTY(int idleTime READ idleTime WRITE setIdleTime)

        /**
         * @brief The time in ms between to consecutive keyboard checks.
         *
         * The polling delay is the time span between two consecutive
         * queries of the keyboard state.  The default is 200 ms, which is
         * 0.2 s.  This means, the keyboard will be checked every 200 ms.
         *
         * @sa setPollDelay(int)
         * @sa pollDelay()
         */
        Q_PROPERTY(int pollDelay READ pollDelay WRITE setPollDelay)

        /**
         * @brief Is the keyboard active?
         *
         * @sa isKeyboardActive()
         */
        Q_PROPERTY(bool keyboardActive READ isKeyboardActive)

    public:
        /**
         * @brief Which keys to ignore when checking the keyboard state
         */
        enum IgnoreKeys {
            /** No keys are ignored, all keys change the keyboard state. */
            IgnoreNoKeys = 0x0,
            /** Ignore modifier keys like Shift or Ctrl.
             *
             * @note This does @em not include keys pressed in combinations
             * with modifiers (e.g. Ctrl+A) */
            IgnoreModifierKeys = 0x1,
            /** Ignore keys pressing in combination with modifiers
             * (e.g. Ctrl+A).
             *
             * @note This causes uppercase letters (e.g. Shift+A) to be
             * ignored, too. */
            IgnoreModifierCombos = 0x2
        };

        /**
         * @brief Create a new instance
         *
         * @param parent same as for QObject(QObject *)
         */
        KeyboardMonitor(QObject *parent=0);

        /**
         * Destroy this instance.
         */
        virtual ~KeyboardMonitor();

    public Q_SLOTS:

        /**
         * @brief Start monitoring the keyboard state.
         */
        void start();

        /**
         * @brief Stop monitoring the keyboard state.
         */
        void stop();

        /**
         * @brief Which keys to ignore?
         *
         * @return the keys to ignore
         * @sa ignoreKeys
         */
        IgnoreKeys ignoreKeys() const;

        /**
         * @brief Ignore the given keys.
         *
         * @param keys which keys to ignore
         * @sa ignoreKeys
         */
        void setIgnoreKeys(IgnoreKeys keys);

        /**
         * @brief Get the idle time.
         *
         * @return the idle time in milliseconds
         * @sa idleTime
         */
        int idleTime() const;

        /**
         * @brief Set the idle time.
         *
         * @param msec the idle time in milliseconds
         * @sa idleTime
         */
        void setIdleTime(int msec);

        /**
         * @brief Get the polling delay.
         *
         * @return the polling delay in milliseconds.
         * @sa pollDelay
         */
        int pollDelay() const;

        /**
         * @brief Set the polling delay.
         *
         * @param msec the polling delay in milliseconds
         * @sa pollDelay
         */
        void setPollDelay(int msec);

        /**
         * @brief Is the keyboard currently active?
         *
         * @return @c true, if the keyboard is active right now, @c false
         *         otherwise.
         * @sa keyboardActive
         */
        bool isKeyboardActive() const;

    Q_SIGNALS:
        /**
         * @brief Emitted, if the keyboard became active, e.g. the user
         * started typing.
         */
        void typingStarted();

        /**
         * @brief Emitted, if the keyboard became inactive, e.g. the user
         * stopped typing.
         *
         * @sa setIdleTime()
         */
        void typingStopped();

    private:
        Q_DISABLE_COPY(KeyboardMonitor)
        Q_DECLARE_PRIVATE(KeyboardMonitor)

        Q_PRIVATE_SLOT(d_func(), void _k_checkKeyboardActivity())

        KeyboardMonitorPrivate * const d_ptr;
    };
}

#endif /* KEYBOARDMONITOR_H */
