/*
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 */
#ifndef __SWAMI_OBJECT_H__
#define __SWAMI_OBJECT_H__

#include <glib.h>
#include <instpatch.h>

#include "swamidll.h"
#include "SwamiConfig.h"
#include "SwamiWavetbl.h"

/* FIXME */
// #include <glib-object.h>
#include "GObjSup.h"

#define SWAMI_OK	0
#define SWAMI_FAIL	1

#define SWAMI_UNDO_ENTER(swami_obj)   if ((swami_obj)->undo->running) return;

typedef struct _SwamiObject SwamiObject;
typedef struct _SwamiObjectClass SwamiObjectClass;

typedef struct _SwamiUndo SwamiUndo;
typedef struct _SwamiUndoEntry SwamiUndoEntry;
typedef struct _SwamiUndoItem SwamiUndoItem;
typedef struct _SwamiUndoTypeInfo SwamiUndoTypeInfo;

#define SWAMI_TYPE_OBJECT   (swami_object_get_type ())
#define SWAMI_OBJECT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj), SWAMI_TYPE_OBJECT, SwamiObject))
#define SWAMI_OBJECT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST ((klass), SWAMI_TYPE_OBJECT, SwamiObjectClass))
#define SWAMI_IS_OBJECT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), SWAMI_TYPE_OBJECT))
#define SWAMI_IS_OBJECT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE ((klass), SWAMI_TYPE_OBJECT))

#define SWAMI_ITEM_TYPE(item) (swami_item_get_type (item->type))
#define SWAMI_TYPE_SFITEM        (swami_item_get_type (0))
#define SWAMI_TYPE_SFONT         (swami_item_get_type (IPITEM_SFONT))
#define SWAMI_TYPE_SFPRESET      (swami_item_get_type (IPITEM_PRESET))
#define SWAMI_TYPE_SFINST        (swami_item_get_type (IPITEM_INST))
#define SWAMI_TYPE_SFSAMPLE      (swami_item_get_type (IPITEM_SAMPLE))
#define SWAMI_TYPE_SFZONE        (swami_item_get_type (IPITEM_ZONE))
#define SWAMI_TYPE_SFVBANK       (swami_item_get_type (IPITEM_VBANK))
#define SWAMI_TYPE_SFVBANK_MAP   (swami_item_get_type (IPITEM_VBANK_MAP))

struct _SwamiObject
{
  GObject object;

  /*< private >*/

  IPItem *patches;		/* private - list of patch file objects */
  GSList *objects;		/* list of registered objects */
  SwamiUndo *undo;		/* undo system vars */
};

struct _SwamiObjectClass
{
  GObjectClass parent_class;

  void (*item_add)(SwamiObject *swami, IPItem *item);
  void (*item_remove)(SwamiObject *swami, IPItem *item);
  void (*item_prop_change)(SwamiObject *swami, IPItem *item,
			   char *property_name);
  void (*zone_gen_change)(SwamiObject *swami, IPZone *zone, int gen_id);

  void (*undo_entry)(SwamiObject *swami, GNode *entry);
  void (*undo)(SwamiObject *swami);
  void (*redo)(SwamiObject *swami, GNode *item);
  void (*group_start)(SwamiObject *swami);
  void (*group_end)(SwamiObject *swami);
};

struct _SwamiUndoEntry
{
  gchar *descr;	   /* description of this action, or NULL */
  GList *items;    /* list of SwamiUndoItems (prepend) */
};

struct _SwamiUndoItem
{
  guint16 type;			/* item type enumeration */
  gpointer state;		/* item state data */
};

/* Swami Undo structure */
struct _SwamiUndo
{
  GNode *root;	    /* points to the root (NULL) entry of the tree */
  GNode *curpos;    /* current position in do tree (SwamiUndoEntry GNode) */
  GList *groups;    /* sub group SwamiUndoItem ptrs (prepend) */
  gboolean running; /* is undo system currently running (stops recursion) */
};

/**
 * SwamiUndoRestoreFunc:
 * @swami: Swami object
 * @item: Undo item to restore the state of (undo)
 *
 * Function type used to restore an undo type's state.
 */
typedef void (*SwamiUndoRestoreFunc)(SwamiObject *swami,
				     const SwamiUndoItem *item);
/**
 * SwamiUndoRestateFunc:
 * @swami: Swami object
 * @undo: Existing undo item to re-state
 * @redo: New undo item to initialize with the redo state
 *
 * Function type used to convert undo state data to redo state
 */
typedef void (*SwamiUndoRestateFunc)(SwamiObject *swami,
				     const SwamiUndoItem *undo,
				     SwamiUndoItem *redo);
/**
 * SwamiUndoDependFunc:
 * @swami: Swami object
 * @item1: First undo item (must be the undo type associated with the
 *   #SwamiUndoTypeInfo where this function pointer resides).
 * @item2: Second undo item
 *
 * Function type used to check dependency between SwamiUndoItems
 * Not used yet (FIXME).
 *
 * Returns: TRUE if item1 is dependent on item2 (can't undo item1 without
 *   undoing item2 as well). FALSE if not dependent.
 */
typedef gboolean (*SwamiUndoDependFunc)(SwamiObject *swami,
					const SwamiUndoItem *item1,
					const SwamiUndoItem *item2);
/**
 * SwamiUndoDescribeFunc:
 * @swami: Swami object
 * @item: Undo item to describe
 *
 * Function type used to describe the action of a SwamiUndoItem.
 * Not used yet (FIXME).
 *
 * Returns: String description which should be freed when done with.
 */
typedef char * (*SwamiUndoDescribeFunc)(SwamiObject *swami,
					const SwamiUndoItem *item);
/**
 * SwamiUndoFreeFunc:
 * @swami: Swami object
 * @item: Undo item to free state data of
 *
 * Function type used to free state data
 */
typedef void (*SwamiUndoFreeFunc)(SwamiObject *swami, SwamiUndoItem *item);

/* parameters for undo types */
struct _SwamiUndoTypeInfo
{
  gchar *descr;			/* description of type */
  SwamiUndoRestoreFunc restore; /* restore state function */
  SwamiUndoRestateFunc restate;	/* restate function (convert undo -> redo) */
  SwamiUndoFreeFunc free;	/* free state data function */
};

SWAMI_API void swami_init (void);
SWAMI_API GType swami_object_get_type (void);
SWAMI_API GType swami_item_get_type (IPItemType itemtype);

SWAMI_API SwamiObject *swami_object_new (void);

SWAMI_API void swami_register_object (SwamiObject *swami, GObject *object);
SWAMI_API GObject *swami_register_object_new (SwamiObject *swami,
					      const char *type_name);
SWAMI_API GList *swami_lookup_objects_by_type (GObject *obj,
					       const char *type_name);
SWAMI_API GObject *swami_get_object_by_type (GObject *obj,
					     const char *type_name);

SWAMI_API IPItem *swami_patch_load (SwamiObject *swami, const char *filename);
SWAMI_API int swami_patch_save (SwamiObject *swami, IPItem *item,
				const char *filename);
SWAMI_API IPItem *swami_get_patch_list (SwamiObject *swami);

/**
 * swami_item_add:
 * @swami: A Swami object
 * @parent: New parent of item to add
 * @item: Item to add
 *
 * Append an item in the patch item tree of the @swami object.
 */
#define swami_item_add(swami, parent, item)  \
    swami_item_insert (swami, parent, item, -1)

SWAMI_API void swami_item_insert (SwamiObject *swami, IPItem *parent,
				  IPItem *item, int pos);
SWAMI_API void swami_item_insert_before (SwamiObject *swami, IPItem *parent,
					 IPItem *item, IPItem *sibling);
SWAMI_API void swami_item_remove (SwamiObject *swami, IPItem *item);
SWAMI_API IPItem *swami_item_new (SwamiObject *swami, IPItemType type,
				  IPItem *parent,
				  const char *first_property_name, ...);
SWAMI_API IPItem *swami_item_new_valist (SwamiObject *swami, IPItemType type,
					 IPItem *parent,
					 const char *first_property_name,
					 va_list prop_args);
SWAMI_API void swami_zone_set_gen (SwamiObject *swami, IPZone *zone,
				   guint16 genid, IPGenAmount amt);
SWAMI_API void swami_zone_unset_gen (SwamiObject *swami, IPZone *zone,
				     guint16 genid);

/**
 * swami_zone_get_gen:
 * @swami: A Swami object
 * @zone: #IPZone to get generator from
 * @genid: ID of generator to fetch
 * @out_amt: #IPGenAmount to return value in
 *
 * Get a generator from a SoundFont zone.
 *
 * Returns: TRUE if value is explicitly set, FALSE if default value (unset).
 */
#define swami_zone_get_gen(swami, zone, genid, out_amt) \
    instp_zone_get_gen (zone, genid, out_amt)

SWAMI_API GList *swami_item_get_zone_references (SwamiObject *swami,
						 IPItem *item);
SWAMI_API char *swami_item_get_formatted_name (SwamiObject *swami,
					       IPItem *item);

SWAMI_API void swami_item_set_valist (SwamiObject *swami, IPItem *item,
				      const char *first_property_name,
				      va_list var_args);
SWAMI_API void swami_item_set (SwamiObject *swami, IPItem *item,
			       const char *first_property_name, ...);
SWAMI_API void swami_item_get_valist (SwamiObject *swami, IPItem *item,
				      const char *first_property_name,
				      va_list var_args);
SWAMI_API void swami_item_get (SwamiObject *swami, IPItem *item,
			       const char *first_property_name, ...);

SWAMI_API void swami_item_set_property (SwamiObject *swami, IPItem *item,
					const char *property_name,
					GValue *value);
SWAMI_API void swami_item_set_int (SwamiObject *swami, IPItem *item,
				   const char *property_name, int value);
SWAMI_API void swami_item_set_float (SwamiObject *swami, IPItem *item,
				     const char *property_name, float value);
SWAMI_API void swami_item_set_boolean (SwamiObject *swami, IPItem *item,
				       const char *property_name,
				       gboolean value);
SWAMI_API void swami_item_set_string (SwamiObject *swami, IPItem *item,
				      const char *property_name,
				      const char *value);
SWAMI_API void swami_item_set_pointer (SwamiObject *swami, IPItem *item,
				       const char *property_name,
				       const gpointer value);

SWAMI_API gboolean swami_item_get_property (SwamiObject *swami,
					    const IPItem *item,
					    const char *property_name,
					    GValue *out_value);
SWAMI_API int swami_item_get_int (SwamiObject *swami, const IPItem *item,
				  const char *property_name);
SWAMI_API float swami_item_get_float (SwamiObject *swami, const IPItem *item,
				      const char *property_name);
SWAMI_API gboolean swami_item_get_boolean (SwamiObject *swami,
					   const IPItem *item,
					   const char *property_name);
SWAMI_API char *swami_item_get_string (SwamiObject *swami, const IPItem *item,
				       const char *property_name);
SWAMI_API gpointer swami_item_get_pointer (SwamiObject *swami,
					   const IPItem *item,
					   const char *property_name);

SWAMI_API void swami_undo_system_init (void);
SWAMI_API void swami_undo_group_start (SwamiObject *swami, const char *descr);
SWAMI_API void swami_undo_group_end (SwamiObject *swami);
SWAMI_API void swami_undo_item_add (SwamiObject *swami, int type,
				    gpointer state);
SWAMI_API void swami_undo_retract (SwamiObject *swami);
SWAMI_API void swami_undo_retract_all (SwamiObject *swami);
SWAMI_API void swami_undo (SwamiObject *swami);
SWAMI_API void swami_redo (SwamiObject *swami, GNode *node);
SWAMI_API void swami_undo_jump (SwamiObject *swami, GNode *node);

#endif
