/*
 * Copyright (C) 2005  Stig Venaas <venaas@uninett.no>
 * $Id: ssmping.c,v 1.22 2005/11/29 16:27:26 sv Exp $
 *
 * Contributions:
 * Solaris support by Alexander Gall <gall@switch.ch>
 * Initial Windows support by Nick Lamb <njl@ecs.soton.ac.uk>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */

#include "ssmping.h"

extern int optind;
extern char *optarg;

#ifndef MCAST_JOIN_SOURCE_GROUP
#ifdef WIN32 /* Only useful on Vista */
#define MCAST_JOIN_SOURCE_GROUP         45
#endif
#ifdef linux
#define MCAST_JOIN_SOURCE_GROUP         46
#endif
struct group_source_req {
  uint32_t                gsr_interface; /* interface index */
  struct sockaddr_storage gsr_group;     /* group address */
  struct sockaddr_storage gsr_source;    /* source address */
};
#endif

void joinchannel(int s, struct sockaddr *src, struct sockaddr *grp, uint32_t intface, struct sockaddr *ifaddr) {
    int level;
    socklen_t addrlen;
#ifdef WIN32
    struct ip_mreq_source imsr;
#endif
    struct group_source_req gsreq;
    
    if (src->sa_family != grp->sa_family) {
	fprintf(stderr, "joinchannel failed, source and group must be of same address family\n");
	exit(1);
    }

    if (ifaddr && ifaddr->sa_family != grp->sa_family) {
	fprintf(stderr, "joinchannel failed, group and interface must be of same address family\n");
	exit(1);
    }
    
    switch (grp->sa_family) {
    case AF_INET:
	addrlen = sizeof(struct sockaddr_in);
	level = IPPROTO_IP;
	break;
    case AF_INET6:
	addrlen = sizeof(struct sockaddr_in6);
	level = IPPROTO_IPV6;
	break;
    default:
	fprintf(stderr, "joinchannel failed, unsupported address family\n");
	exit(1);
    }

    memset(&gsreq, 0, sizeof(gsreq));
    memcpy(&gsreq.gsr_source, src, addrlen);
    memcpy(&gsreq.gsr_group, grp, addrlen);
    gsreq.gsr_interface = intface;
#ifndef WIN32
    if (setsockopt(s, level, MCAST_JOIN_SOURCE_GROUP, (char *)&gsreq, sizeof(gsreq)) < 0)
	errx("Failed to join multicast channel");
#else
    if (setsockopt(s, level, MCAST_JOIN_SOURCE_GROUP, (char *)&gsreq, sizeof(gsreq)) >= 0)
	return;
    if (level != IPPROTO_IP)
	errx("Failed to join multicast channel");

    /* For Windows XP the above setsockopt fails, below works for IPv4
     * While for Windows Vista the above should work
     */
    
    memset(&imsr, 0, sizeof(imsr));
    imsr.imr_sourceaddr = ((struct sockaddr_in *)src)->sin_addr;
    imsr.imr_multiaddr = ((struct sockaddr_in *)grp)->sin_addr;
    imsr.imr_interface = ((struct sockaddr_in *)ifaddr)->sin_addr;

    if (setsockopt(s, level, IP_ADD_SOURCE_MEMBERSHIP, (char *)&imsr, sizeof(imsr)) < 0)
	errx("Failed to join multicast channel");
#endif
}

int main(int argc, char **argv) {
    int family, count, us, ms;
    char *pingee, *group, source[INET6_ADDRSTRLEN];
    uint32_t intface;
    struct sockaddr_storage name, ucaddr, mcaddr, grpaddr;
    size_t namelen;
#ifdef WIN32
    WORD wVersionRequested;
    WSADATA wsaData;

    wVersionRequested = MAKEWORD( 2, 0 );
    WSAStartup(wVersionRequested, &wsaData);
    /* lots of complicated Win32 error checking expected - sigh - */
#endif

    parseargs(argc, argv, SSMMODE, &pingee, &family, &intface, &count, &group);

    us = name2addrsock(pingee, &family, &ucaddr);
    if (us < 0)
	errx("Failed to create any usable socket");

    ms = name2addrsock(group, &family, &mcaddr);
    if (ms < 0)
	errx("Failed to create any usable socket");

    prep_sock(family, us);
    prep_sock(family, ms);

    findsrc((struct sockaddr *)&name, (struct sockaddr *)&ucaddr);
    namelen = SOCKADDR_SIZE(name);
    
    setport((struct sockaddr *)&name, 0);

    if (bind(us, (struct sockaddr *)&name, namelen) < 0)
	errx("bind");

    if (connect(us, (struct sockaddr *)&ucaddr, namelen) < 0)
	errx("connect");

    if (getsockname(us, (struct sockaddr *)&name, &namelen) == -1)
	errx("getsockname");
    
    grpaddr = name;
    setaddr(&grpaddr, group ? &mcaddr : NULL, "ff3e::4321:1234", "232.43.211.234");
#ifdef WIN32
    {
	struct sockaddr_storage any = name;
	setaddr(&any, NULL, "::", "0.0.0.0");
	if (bind(ms, (struct sockaddr *)&any, namelen) < 0)
	    errx("bind [ANY]");
    }
#else    
    if (bind(ms, (struct sockaddr *)&grpaddr, namelen) < 0)
	errx("bind [multicast]");
#endif
     /* using name to specify interface is wrong, only problem for old API */
    joinchannel(ms, (struct sockaddr *)&ucaddr,
		(struct sockaddr *)&grpaddr, intface, (struct sockaddr *)&name);
    strcpy(source, addr2string((struct sockaddr *)&ucaddr, namelen));
    printf("ssmping joined (S,G) = (%s,%s)\n", source,
	   addr2string((struct sockaddr *)&grpaddr, namelen));
    printf("pinging S from %s\n", addr2string((struct sockaddr *)&name, namelen));

    return doit(count, us, ms, &ucaddr, &grpaddr, source);
}    
