# This module encapsulated a standard set of Logging functions that Spong
# programs use as. This module uses Logger objects for handling the actual
# logging procedures.
#
# methods
#  error_context_init() - Initialize error output context
#  debug_context_init() - Initialize debug output context
#  error()              - output an error entry
#  debug()              - output a debug entry

package Spong::Log;


use strict;
use Carp;
use IO::Socket;

use Spong::Logger;

require Exporter;
use vars qw(@ISA @EXPORT_OK $VERSION);
@ISA = qw(Exporter);
@EXPORT_OK = qw(error debug);
$VERSION = 0.01;

my($stdout, $stderr, $syslog, $ident, $facility, $logopt, $priority);
my($debuglevel);
my($filename);
my($errlog);

# Call the set context functions to initialize default values
&set_debug_context();
&set_error_context();

# Setup intercept routines for die and warn in order to log their infomation
$main::SIG{'__DIE__'} = sub { if ( defined $^S && ! $^S && defined $errlog)
                            { $errlog->log("die(): " . $_[0]); } };
$main::SIG{'__WARN__'} = sub { if ( defined $^S && ! $^S && defined $errlog)
                            { $errlog->log("warn(): " . $_[0]); } };

# This function is used to set the debug logging context

sub set_debug_context {
   my( %arg ) = @_;

   $debuglevel = defined $arg{debuglevel} ? $arg{debuglevel} : 0;
}

# This function used to set the error logging context

sub set_error_context {
   my( %arg ) = @_;

   $stdout   = defined $arg{stdout} ? $arg{stdout} : 0;
   $stderr   = defined $arg{stderr} ? $arg{stderr} : 1;
   $syslog   = defined $arg{syslog} ? $arg{syslog} : 0;
   $ident    = defined $arg{ident} ? $arg{ident} : "";
   $facility = defined $arg{facility} ? $arg{facility} : 'user';
   $logopt   = defined $arg{logopt} ? $arg{logopt} : "";
   $priority = defined $arg{priority} ? $arg{priority} : 'info';
   $filename = $arg{filename} if defined $arg{filename};

}

# Function create a logger object with the current error context
sub open_error_log {

   $errlog = new Spong::Logger( stdout => $stdout,
                                stderr => $stderr,
                                syslog => $syslog,
                                logopt => $logopt,
                                ident  => $ident,
                                facility => $facility,
                                priority => $priority,
                                filename => $filename,
                              );

   if (! defined $errlog ) {
      croak "Could not create error logger: $!";
      return 1;
   }

}

# Function closes the error log by undefineding the error logger object
sub close_error_log {
   undef $errlog;
}

sub log {
   my( $message ) = @_;

   # if the error logger is not defined, create it.
   if ( ! defined $errlog ) { &open_error_log(); }
   
   $errlog->log("$message");
}

sub error {
   my( $message ) = @_;

   &log("Error: $message");
}

sub debug {
   my ($message,$level) = @_;
   $level = 1 if (! defined $level);  # Default to level 1
   print STDERR scalar localtime, " ", $message, "\n" if $debuglevel >= $level;
}

1;  # I'm included perl code, I need this line

