-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

---------------------------------------------------------------------------------
-- Package for generating an XML version of the summary file.                  --
---------------------------------------------------------------------------------

with PathFormatter;
with SPARK_XML;

package body XMLSummary
--# own State is Last_Summary_Section,
--#              Schema,
--#              Schema_State,
--#              Tag_Depths,
--#              Tag_IDs;
is

   type Tags is (
                 T_Pogs,
                 T_Type,
                 T_Types,
                 T_Directory,
                 T_Produced,
                 T_Results,
                 T_File,
                 T_Error,
                 T_Vc,
                 T_Summary,
                 T_Subprograms,
                 T_Subprograms_In_Error,
                 T_Vcs,
                 T_Tools,
                 T_Assert_Post,
                 T_Check,
                 T_Inheritance,
                 T_Precond,
                 T_Refinement,
                 T_Runtime,
                 T_Totals,
                 T_Percentages,
                 T_Examiner,
                 T_Simplifier,
                 T_Simplifier_User_Rule,
                 T_Victor,
                 T_Checker,
                 T_Review,
                 T_False,
                 T_Contra,
                 T_User,
                 T_Undetermined,
                 T_Undischarged,
                 T_Undet_Section);

   type Tag_Depth_Array is array (Tags) of SPARK_XML.Tag_Depth;

   type Tag_ID_Array is array (Tags) of SPARK_XML.Tag_ID;

   -- Package State
   Tag_Depths   : Tag_Depth_Array;
   Tag_IDs      : Tag_ID_Array;
   Schema       : SPARK_XML.Schema_Record;
   Schema_State : SPARK_XML.Schema_State_Record;

   Last_Summary_Section : Tags;

   --------------------
   -- Error handling --
   --------------------

   procedure Handle_Error (Status : in SPARK_XML.Schema_Status)
   --# global in Schema;
   --#        in Schema_State;
   --# derives null from Schema,
   --#                   Schema_State,
   --#                   Status;
   is
      --# hide Handle_Error;
   begin
      if SPARK_XML.Is_Error (Error => Status) then
         SPARK_XML.Print_Schema_Error (Error => Status);
         SPARK_XML.Print_Working_State (Schema       => Schema,
                                        Schema_State => Schema_State);
      end if;
   end Handle_Error;

   procedure Handle_Schema_Error (Success : in Boolean;
                                  Msg     : in String)
   --# derives null from Msg,
   --#                   Success;
   is
      --# hide Handle_Schema_Error;
   begin
      if not Success then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Msg, 0);
         SPARK_XML.Print_Working_State (Schema       => Schema,
                                        Schema_State => Schema_State);
      end if;
   end Handle_Schema_Error;

   procedure Init
   --# global out Last_Summary_Section;
   --#        out Schema;
   --#        out Schema_State;
   --#        out Tag_Depths;
   --#        out Tag_IDs;
   --# derives Last_Summary_Section,
   --#         Schema,
   --#         Schema_State,
   --#         Tag_Depths,
   --#         Tag_IDs              from ;
   is

      -- Load the tag definitions in to the Schema
      procedure Load_Tags
      --# global in out Schema;
      --#           out Tag_IDs;
      --# derives Schema,
      --#         Tag_IDs from Schema;
      is
         type X_String_Array_Type is array (Tags) of E_Strings.T;

         Tag_Strings : X_String_Array_Type;
         Tmp_Tag     : SPARK_XML.Tag_ID;
      begin
         Tag_Strings :=
           X_String_Array_Type'
           (T_Pogs                 => SPARK_XML.X_Str (Str => "pogs"),
            T_Type                 => SPARK_XML.X_Str (Str => "type"),
            T_Types                => SPARK_XML.X_Str (Str => "types"),
            T_Directory            => SPARK_XML.X_Str (Str => "directory"),
            T_Produced             => SPARK_XML.X_Str (Str => "produced"),
            T_Results              => SPARK_XML.X_Str (Str => "results"),
            T_File                 => SPARK_XML.X_Str (Str => "file"),
            T_Error                => SPARK_XML.X_Str (Str => "error"),
            T_Vc                   => SPARK_XML.X_Str (Str => "vc"),
            T_Summary              => SPARK_XML.X_Str (Str => "summary"),
            T_Subprograms          => SPARK_XML.X_Str (Str => "subprograms"),
            T_Subprograms_In_Error => SPARK_XML.X_Str (Str => "subprograms_in_error"),
            T_Vcs                  => SPARK_XML.X_Str (Str => "vcs"),
            T_Tools                => SPARK_XML.X_Str (Str => "tools"),
            T_Assert_Post          => SPARK_XML.X_Str (Str => "assert_post"),
            T_Check                => SPARK_XML.X_Str (Str => "check"),
            T_Inheritance          => SPARK_XML.X_Str (Str => "inheritance"),
            T_Precond              => SPARK_XML.X_Str (Str => "precond"),
            T_Refinement           => SPARK_XML.X_Str (Str => "refinement"),
            T_Runtime              => SPARK_XML.X_Str (Str => "runtime"),
            T_Totals               => SPARK_XML.X_Str (Str => "totals"),
            T_Percentages          => SPARK_XML.X_Str (Str => "percentages"),
            T_Examiner             => SPARK_XML.X_Str (Str => "examiner"),
            T_Simplifier           => SPARK_XML.X_Str (Str => "simplifier"),
            T_Simplifier_User_Rule => SPARK_XML.X_Str (Str => "simplifier_user_rule"),
            T_Victor               => SPARK_XML.X_Str (Str => "victor"),
            T_Checker              => SPARK_XML.X_Str (Str => "checker"),
            T_Review               => SPARK_XML.X_Str (Str => "review"),
            T_False                => SPARK_XML.X_Str (Str => "false"),
            T_Contra               => SPARK_XML.X_Str (Str => "contradiction"),
            T_User                 => SPARK_XML.X_Str (Str => "user"),
            T_Undetermined         => SPARK_XML.X_Str (Str => "undetermined"),
            T_Undischarged         => SPARK_XML.X_Str (Str => "undischarged"),
            T_Undet_Section        => SPARK_XML.X_Str (Str => "undetermined_summary"));

         Tag_IDs := Tag_ID_Array'(others => SPARK_XML.Null_Tag);

         for I in Tags loop
            SPARK_XML.Add_Tag (Schema => Schema,
                               Name   => Tag_Strings (I),
                               ID     => Tmp_Tag);
            Tag_IDs (I) := Tmp_Tag;
            if SPARK_XML.Is_Null_Tag (TID => Tmp_Tag) then
               -- Run out of room, so generate an error.
               Handle_Schema_Error (Success => False,
                                    Msg     => "Failure adding tag");
               exit;
            end if;
         end loop;
      end Load_Tags;

      -- Load the attribute definitions
      procedure Load_Attributes
      --# global in     Tag_IDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     Tag_IDs;
      is
         type Attribute is record
            ID   : SPARK_XML.Tag_ID;
            Name : E_Strings.T;
            Typ  : SPARK_XML.Attribute_Type;
            Req  : Boolean;
         end record;

         -- Set Total_Attributes to the total number of attributes to be added
         -- to tags in the Schema.
         Total_Attributes : constant Integer := 26;

         type Attribute_Index is range 1 .. Total_Attributes;
         type Attribute_List is array (Attribute_Index) of Attribute;

         Tmp_Attrib  : SPARK_XML.Attribute_ID;
         Tmp_Success : Boolean;
         Attributes  : Attribute_List;
      begin
         -----------------------------------|TAG                  |Attribute name |Type                |Required?|
         ---------------------------------------------------------|---------------|--------------------|---------|
         Attributes :=
           Attribute_List'
           (Attribute'(Tag_IDs (T_Pogs), SPARK_XML.X_Str (Str => "report_name"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Pogs), SPARK_XML.X_Str (Str => "version"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Pogs), SPARK_XML.X_Str (Str => "licensee"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Pogs), SPARK_XML.X_Str (Str => "ignoredates"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_File), SPARK_XML.X_Str (Str => "name"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_File), SPARK_XML.X_Str (Str => "procedure"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_File), SPARK_XML.X_Str (Str => "generated"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_File), SPARK_XML.X_Str (Str => "simplified"), SPARK_XML.At_String, False),
            Attribute'(Tag_IDs (T_File), SPARK_XML.X_Str (Str => "checked"), SPARK_XML.At_String, False),
            Attribute'(Tag_IDs (T_Vc), SPARK_XML.X_Str (Str => "vcnumber"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Vc), SPARK_XML.X_Str (Str => "from"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Vc), SPARK_XML.X_Str (Str => "to"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Vc), SPARK_XML.X_Str (Str => "topointtype"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Vc), SPARK_XML.X_Str (Str => "status"), SPARK_XML.At_String, True),
            Attribute'(Tag_IDs (T_Subprograms), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Subprograms_In_Error), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Tools), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Assert_Post), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Check), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Inheritance), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Precond), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Refinement), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Runtime), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Undet_Section), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Totals), SPARK_XML.X_Str (Str => "total"), SPARK_XML.At_Integer, True),
            Attribute'(Tag_IDs (T_Error), SPARK_XML.X_Str (Str => "filetype"), SPARK_XML.At_String, True));

         for I in Attribute_Index loop
            --# accept F, 10, Tmp_Attrib, "Tmp_Attrib unused here";
            SPARK_XML.Add_Attribute_To_Tag
              (Schema,
               Attributes (I).ID,
               Attributes (I).Name,
               Attributes (I).Typ,
               Attributes (I).Req,
               Tmp_Attrib,
               Tmp_Success);
            --# end accept;
            Handle_Schema_Error (Success => Tmp_Success,
                                 Msg     => "Failure adding attribute to tag");
            exit when not Tmp_Success;
         end loop;

         --# accept F, 33, Tmp_Attrib, "Tmp_Attrib unused here";
      end Load_Attributes;

      -- Load the hierarchy definitions
      procedure Build_Hierarchy
      --# global in     Tag_IDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     Tag_IDs;
      is
         type Tag_Rel is record
            Parent   : SPARK_XML.Tag_ID;
            Child    : SPARK_XML.Tag_ID;
            Required : Boolean;
         end record;

         Max_Relations : constant Integer := 106;

         subtype Rel_Index is Integer range 1 .. Max_Relations;
         type Rel_Array is array (Rel_Index) of Tag_Rel;

         Relations   : Rel_Array;
         Tmp_Success : Boolean;
      begin

         Relations :=
           Rel_Array'
           (
            -- Base tag
            Tag_Rel'(SPARK_XML.Null_Tag, Tag_IDs (T_Pogs), False),
            -- Pogs
            Tag_Rel'(Tag_IDs (T_Pogs), Tag_IDs (T_Types), False),
            Tag_Rel'(Tag_IDs (T_Pogs), Tag_IDs (T_Directory), False),
            Tag_Rel'(Tag_IDs (T_Pogs), Tag_IDs (T_Produced), False),
            Tag_Rel'(Tag_IDs (T_Pogs), Tag_IDs (T_Results), False),
            -- Types
            Tag_Rel'(Tag_IDs (T_Types), Tag_IDs (T_Type), False),
            -- Results
            Tag_Rel'(Tag_IDs (T_Results), Tag_IDs (T_File), False),
            Tag_Rel'(Tag_IDs (T_Results), Tag_IDs (T_Summary), False),
            --File
            Tag_Rel'(Tag_IDs (T_File), Tag_IDs (T_Vc), False),
            Tag_Rel'(Tag_IDs (T_File), Tag_IDs (T_Error), False),
            -- Summary
            Tag_Rel'(Tag_IDs (T_Summary), Tag_IDs (T_Subprograms), False),
            Tag_Rel'(Tag_IDs (T_Summary), Tag_IDs (T_Subprograms_In_Error), False),
            Tag_Rel'(Tag_IDs (T_Summary), Tag_IDs (T_Vcs), False),
            --Subprograms
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Contra), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_User), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Subprograms), Tag_IDs (T_Undetermined), False),
            --Subprograms in error
            Tag_Rel'(Tag_IDs (T_Subprograms_In_Error), Tag_IDs (T_Undetermined), False),
            --VCs
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Assert_Post), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Precond), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Check), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Runtime), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Refinement), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Inheritance), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Undet_Section), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Totals), False),
            Tag_Rel'(Tag_IDs (T_Vcs), Tag_IDs (T_Percentages), False),
            -- Assert_post
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Assert_Post), Tag_IDs (T_Undetermined), False),
            -- Precondition
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Precond), Tag_IDs (T_Undetermined), False),
            -- Check
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Check), Tag_IDs (T_Undetermined), False),
            -- Runtime
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Runtime), Tag_IDs (T_Undetermined), False),
            -- Refinement
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Refinement), Tag_IDs (T_Undetermined), False),
            -- Inheritance
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Inheritance), Tag_IDs (T_Undetermined), False),
            -- Undetermined_summary
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Undet_Section), Tag_IDs (T_Undetermined), False),
            -- Totals
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Totals), Tag_IDs (T_Undetermined), False),
            -- Percentages
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Examiner), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Simplifier), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Simplifier_User_Rule), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Victor), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Checker), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Review), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_False), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Undischarged), False),
            Tag_Rel'(Tag_IDs (T_Percentages), Tag_IDs (T_Undetermined), False));

         for I in Rel_Index loop
            SPARK_XML.Add_Child_Tag (Schema, Relations (I).Parent, Relations (I).Child, Relations (I).Required, Tmp_Success);
            Handle_Schema_Error (Success => Tmp_Success,
                                 Msg     => "Failure loading tag hierarchy");
            exit when not Tmp_Success;
         end loop;
      end Build_Hierarchy;

   begin
      Last_Summary_Section := T_Assert_Post;

      -- Zero the tag depths.
      Tag_Depths := Tag_Depth_Array'(others => 0);

      SPARK_XML.Init_Schema (Schema => Schema);
      SPARK_XML.Init_Schema_State (Schema_State => Schema_State);

      -- Load the tags
      Load_Tags;
      Load_Attributes;
      Build_Hierarchy;

   end Init;

   --
   -- Generalised procedures for starting simple container tags, and closing contianer tags.
   --

   procedure Start_Simple_Container (Tag_Ident : in Tags;
                                     Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Ident,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Ident,
   --#                                Tag_IDs;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID
        (Schema       => Schema,
         Schema_State => Schema_State,
         TID          => Tag_IDs (Tag_Ident),
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      Tag_Depths (Tag_Ident) := Depth;

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag
   end Start_Simple_Container;

   procedure End_Container (Tag_Ident : in Tags;
                            Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths,
   --#                                Tag_Ident &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths,
   --#                                Tag_Ident;
   is
      XML  : E_Strings.T;
      Stat : SPARK_XML.Schema_Status;
   begin
      SPARK_XML.Close_Tag
        (Schema       => Schema,
         Schema_State => Schema_State,
         Depth        => Tag_Depths (Tag_Ident),
         XML          => XML,
         Status       => Stat);

      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);
   end End_Container;

   procedure End_Tag (Depth  : in SPARK_XML.Tag_Depth;
                      Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Depth &
   --#         SPARK_IO.File_Sys from *,
   --#                                Depth,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State;
   is
      XML  : E_Strings.T;
      Stat : SPARK_XML.Schema_Status;
   begin
      SPARK_XML.Close_Tag (Schema       => Schema,
                           Schema_State => Schema_State,
                           Depth        => Depth,
                           XML          => XML,
                           Status       => Stat);

      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);
   end End_Tag;

   procedure Start_Pogs
     (Report_Name  : in E_Strings.T;
      Pogs_Version : in E_Strings.T;
      Licensee     : in E_Strings.T;
      Ignore_Dates : in Boolean;
      Report       : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Ignore_Dates,
   --#                                Licensee,
   --#                                Pogs_Version,
   --#                                Report_Name,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Ignore_Dates,
   --#                                Licensee,
   --#                                Pogs_Version,
   --#                                Report,
   --#                                Report_Name,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (T_Pogs),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "report_name"),
         Value        => Report_Name,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "version"),
         Value        => Pogs_Version,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "licensee"),
         Value        => Licensee,
         Status       => Stat);
      Handle_Error (Status => Stat);

      if Ignore_Dates then
         SPARK_XML.Add_Attribute_Str
           (Schema       => Schema,
            Schema_State => Schema_State,
            Name         => SPARK_XML.X_Str (Str => "ignoredates"),
            Value        => SPARK_XML.X_Str (Str => "true"),
            Status       => Stat);
      else
         SPARK_XML.Add_Attribute_Str
           (Schema       => Schema,
            Schema_State => Schema_State,
            Name         => SPARK_XML.X_Str (Str => "ignoredates"),
            Value        => SPARK_XML.X_Str (Str => "false"),
            Status       => Stat);
      end if;

      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);

      Handle_Error (Status => Stat);

      Tag_Depths (T_Pogs) := Depth;

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag
   end Start_Pogs;

   procedure End_Pogs (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Pogs,
                     Report    => Report);
   end End_Pogs;

   procedure Start_Types (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
   begin
      Start_Simple_Container (Tag_Ident => T_Types,
                              Report    => Report);
   end Start_Types;

   procedure End_Types (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Types,
                     Report    => Report);
   end End_Types;

   procedure Start_Results (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
   begin
      Start_Simple_Container (Tag_Ident => T_Results,
                              Report    => Report);
   end Start_Results;

   procedure End_Results (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Results,
                     Report    => Report);
   end End_Results;

   procedure Start_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
   begin
      Start_Simple_Container (Tag_Ident => T_Summary,
                              Report    => Report);
   end Start_Summary;

   procedure End_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Summary,
                     Report    => Report);
   end End_Summary;

   procedure Start_Subprogram_Summary (Total_Subprogs : in Natural;
                                       Report         : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Total_Subprogs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Total_Subprogs;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin

      SPARK_XML.Init_Opening_Tag_By_ID
        (Schema       => Schema,
         Schema_State => Schema_State,
         TID          => Tag_IDs (T_Subprograms),
         Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      SPARK_XML.Add_Attribute_Int
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "total"),
         Value        => Total_Subprogs,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      Tag_Depths (T_Subprograms) := Depth;

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag
   end Start_Subprogram_Summary;

   procedure End_Subprogram_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Subprograms,
                     Report    => Report);
   end End_Subprogram_Summary;

   procedure Start_Subprograms_In_Error_Summary (Total_Subprogs_In_Error : in Natural;
                                                 Report                  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Total_Subprogs_In_Error &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Total_Subprogs_In_Error;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID
        (Schema       => Schema,
         Schema_State => Schema_State,
         TID          => Tag_IDs (T_Subprograms_In_Error),
         Status       => Stat);

      Handle_Error (Status => Stat);

      -- Add the attributes

      SPARK_XML.Add_Attribute_Int
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "total"),
         Value        => Total_Subprogs_In_Error,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      Tag_Depths (T_Subprograms_In_Error) := Depth;

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag
   end Start_Subprograms_In_Error_Summary;

   procedure End_Subprograms_In_Error_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Subprograms_In_Error,
                     Report    => Report);
   end End_Subprograms_In_Error_Summary;

   procedure Start_VC_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
   begin
      Start_Simple_Container (Tag_Ident => T_Vcs,
                              Report    => Report);
   end Start_VC_Summary;

   procedure End_VC_Summary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_Vcs,
                     Report    => Report);
   end End_VC_Summary;

   ---------------------------------
   -- Container tags with attributes
   ---------------------------------

   -- For optional attributes, if the EmptyString is supplied then the attribute
   -- will not be added to the tag.

   procedure Start_File
     (Name       : in E_Strings.T;
      Proc       : in E_Strings.T;
      Generated  : in E_Strings.T;
      Simplified : in E_Strings.T;
      Checked    : in E_Strings.T;
      Report     : in SPARK_IO.File_Type)
   --# global in     CommandLine.Data;
   --#        in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --# derives Schema_State,
   --#         Tag_Depths        from *,
   --#                                Checked,
   --#                                CommandLine.Data,
   --#                                Generated,
   --#                                Name,
   --#                                Proc,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Simplified,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Checked,
   --#                                CommandLine.Data,
   --#                                Generated,
   --#                                Name,
   --#                                Proc,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Simplified,
   --#                                Tag_IDs;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (T_File),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      -- quick fix to avoid a lot of work in SPARK_XML
      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "name"),
         Value        => PathFormatter.Format (Name),
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "procedure"),
         Value        => Proc,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "generated"),
         Value        => Generated,
         Status       => Stat);
      Handle_Error (Status => Stat);

      if not E_Strings.Is_Empty (E_Str => Simplified) then
         SPARK_XML.Add_Attribute_Str
           (Schema       => Schema,
            Schema_State => Schema_State,
            Name         => SPARK_XML.X_Str (Str => "simplified"),
            Value        => Simplified,
            Status       => Stat);
         Handle_Error (Status => Stat);
      end if;

      if not E_Strings.Is_Empty (E_Str => Checked) then
         SPARK_XML.Add_Attribute_Str
           (Schema       => Schema,
            Schema_State => Schema_State,
            Name         => SPARK_XML.X_Str (Str => "checked"),
            Value        => Checked,
            Status       => Stat);
         Handle_Error (Status => Stat);
      end if;

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      Tag_Depths (T_File) := Depth;

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag
   end Start_File;

   procedure End_File (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => T_File,
                     Report    => Report);
   end End_File;

   procedure File_Error (Message : in E_Strings.T;
                         F_Type  : in File_Type;
                         Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                F_Type,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                F_Type,
   --#                                Message,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (T_Error),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      case F_Type is
         when VCG_File =>
            XML := E_Strings.Copy_String (Str => "vcg");
         when SIV_File =>
            XML := E_Strings.Copy_String (Str => "siv");
         when VCTR_File =>
            XML := E_Strings.Copy_String (Str => "vct");
         when PLG_File =>
            XML := E_Strings.Copy_String (Str => "plg");
         when REV_File =>
            XML := E_Strings.Copy_String (Str => "rev");
      end case;

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "filetype"),
         Value        => XML,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag

      E_Strings.Put_String (File  => Report,
                            E_Str => Message);  -- Output the error message

      End_Tag (Depth  => Depth,
               Report => Report); -- Close the tag.
   end File_Error;

   procedure VC
     (VC_Number     : in E_Strings.T;
      From_Point    : in E_Strings.T;
      To_Point_Type : in E_Strings.T;
      To_Point      : in E_Strings.T;
      Status        : in VC_Status;
      Report        : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                From_Point,
   --#                                Schema,
   --#                                Status,
   --#                                Tag_IDs,
   --#                                To_Point,
   --#                                To_Point_Type,
   --#                                VC_Number &
   --#         SPARK_IO.File_Sys from *,
   --#                                From_Point,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Status,
   --#                                Tag_IDs,
   --#                                To_Point,
   --#                                To_Point_Type,
   --#                                VC_Number;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (T_Vc),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "vcnumber"),
         Value        => VC_Number,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "from"),
         Value        => From_Point,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "to"),
         Value        => To_Point,
         Status       => Stat);
      Handle_Error (Status => Stat);
      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "topointtype"),
         Value        => To_Point_Type,
         Status       => Stat);
      Handle_Error (Status => Stat);

      case Status is
         when VCG =>
            XML := E_Strings.Copy_String (Str => "vcg");
         when SIV =>
            XML := E_Strings.Copy_String (Str => "siv");
         when VCTR =>
            XML := E_Strings.Copy_String (Str => "vct");
         when PLG =>
            XML := E_Strings.Copy_String (Str => "plg");
         when REV =>
            XML := E_Strings.Copy_String (Str => "rev");
         when WRONG =>
            XML := E_Strings.Copy_String (Str => "false");
         when CONTRA =>
            XML := E_Strings.Copy_String (Str => "contradiction");
         when USER =>
            XML := E_Strings.Copy_String (Str => "user");
         when TODO =>
            XML := E_Strings.Copy_String (Str => "todo");
      end case;

      SPARK_XML.Add_Attribute_Str
        (Schema       => Schema,
         Schema_State => Schema_State,
         Name         => SPARK_XML.X_Str (Str => "status"),
         Value        => XML,
         Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag

      End_Tag (Depth  => Depth,
               Report => Report); -- Close the tag.
   end VC;

   procedure Start_Summary_Section (Section : in Summary_Section;
                                    Total   : in Integer;
                                    Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Tag_Depths;
   --#           out Last_Summary_Section;
   --# derives Last_Summary_Section from Section &
   --#         Schema_State,
   --#         Tag_Depths           from *,
   --#                                   Schema,
   --#                                   Schema_State,
   --#                                   Section,
   --#                                   Tag_IDs,
   --#                                   Total &
   --#         SPARK_IO.File_Sys    from *,
   --#                                   Report,
   --#                                   Schema,
   --#                                   Schema_State,
   --#                                   Section,
   --#                                   Tag_IDs,
   --#                                   Total;
   is
      XML     : E_Strings.T;  -- XML accumulator and temp variable.
      Stat    : SPARK_XML.Schema_Status;
      Tmp_Tag : Tags;
   begin
      case Section is
         when SS_Totals =>
            Tmp_Tag := T_Totals;
         when SS_Percentages =>
            Tmp_Tag := T_Percentages;
         when SS_Assert_Post =>
            Tmp_Tag := T_Assert_Post;
         when SS_Check =>
            Tmp_Tag := T_Check;
         when SS_Inheritance =>
            Tmp_Tag := T_Inheritance;
         when SS_Precondition =>
            Tmp_Tag := T_Precond;
         when SS_Refinement =>
            Tmp_Tag := T_Refinement;
         when SS_Runtime =>
            Tmp_Tag := T_Runtime;
         when SS_Undetermined =>
            Tmp_Tag := T_Undet_Section;
      end case;

      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (Tmp_Tag),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      -- Add the attributes

      if not (Tmp_Tag = T_Percentages) then  -- Percentages don't need a total.
         SPARK_XML.Add_Attribute_Int
           (Schema       => Schema,
            Schema_State => Schema_State,
            Name         => SPARK_XML.X_Str (Str => "total"),
            Value        => Total,
            Status       => Stat);
         Handle_Error (Status => Stat);
      end if;

      SPARK_XML.Output_Opening_Tag
        (Schema       => Schema,
         Schema_State => Schema_State,
         XML          => XML,
         Depth        => Tag_Depths (Tmp_Tag),
         Status       => Stat);
      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the opening tag

      Last_Summary_Section := Tmp_Tag;
   end Start_Summary_Section;

   -- Summary sections cannot enclose each other, so calls to this
   -- procedure will just close the last opened section.
   procedure End_Summary_Section (Report : in SPARK_IO.File_Type)
   --# global in     Last_Summary_Section;
   --#        in     Schema;
   --#        in     Tag_Depths;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Last_Summary_Section,
   --#                                Tag_Depths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Last_Summary_Section,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_Depths;
   is
   begin
      End_Container (Tag_Ident => Last_Summary_Section,
                     Report    => Report);
   end End_Summary_Section;

   -- Simple non-container tags.
   -- These either have just attributes or values

   procedure Text_Tag (Text   : in E_Strings.T;
                       T_Tag  : in Tags;
                       Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Schema,
   --#                                Tag_IDs,
   --#                                T_Tag &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Text,
   --#                                T_Tag;
   is
      XML   : E_Strings.T;  -- XML accumulator and temp variable.
      Stat  : SPARK_XML.Schema_Status;
      Depth : SPARK_XML.Tag_Depth;
   begin
      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (T_Tag),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML);  -- Output the directory name

      E_Strings.Put_String (File  => Report,
                            E_Str => Text);  -- Output the directory name

      End_Tag (Depth  => Depth,
               Report => Report);
   end Text_Tag;

   procedure Directory (Directory_Name : in E_Strings.T;
                        Report         : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Directory_Name,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is

   begin
      Text_Tag (Text   => Directory_Name,
                T_Tag  => T_Directory,
                Report => Report);
   end Directory;

   procedure Produced (Produced_Date : in E_Strings.T;
                       Report        : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Produced_Date,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs;
   is

   begin
      Text_Tag (Text   => Produced_Date,
                T_Tag  => T_Produced,
                Report => Report);
   end Produced;

   procedure Summary_Item_Str (Item   : in Item_Type;
                               Value  : in E_Strings.T;
                               Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Item,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Value;
   is
      type ID_Lookup_Array is array (Item_Type) of Tags;

      ID_Table : constant ID_Lookup_Array :=
        ID_Lookup_Array'
        (IT_Examiner             => T_Examiner,
         IT_Simplifier           => T_Simplifier,
         IT_Simplifier_User_Rule => T_Simplifier_User_Rule,
         IT_Victor               => T_Victor,
         IT_Checker              => T_Checker,
         IT_Review               => T_Review,
         IT_False                => T_False,
         IT_Contra               => T_Contra,
         IT_User                 => T_User,
         IT_Undischarged         => T_Undischarged,
         IT_Undetermined         => T_Undetermined);

      XML     : E_Strings.T;  -- XML accumulator and temp variable.
      Stat    : SPARK_XML.Schema_Status;
      Depth   : SPARK_XML.Tag_Depth;
      Tmp_Tag : Tags;
   begin
      Tmp_Tag := ID_Table (Item);

      SPARK_XML.Init_Opening_Tag_By_ID (Schema       => Schema,
                                        Schema_State => Schema_State,
                                        TID          => Tag_IDs (Tmp_Tag),
                                        Status       => Stat);
      Handle_Error (Status => Stat);

      SPARK_XML.Output_Opening_Tag (Schema       => Schema,
                                    Schema_State => Schema_State,
                                    XML          => XML,
                                    Depth        => Depth,
                                    Status       => Stat);
      Handle_Error (Status => Stat);

      E_Strings.Put_String (File  => Report,
                            E_Str => XML); -- Output the opening tag

      E_Strings.Put_String (File  => Report,
                            E_Str => Value);  -- Output the value

      End_Tag (Depth  => Depth,
               Report => Report);
   end Summary_Item_Str;

   procedure Summary_Item_Int (Item   : in Item_Type;
                               Value  : in Integer;
                               Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Item,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Value;
   is
      Val_String : E_Strings.T;
   begin
      -- Convert the integer to a string.
      E_Strings.Put_Int_To_String (Dest     => Val_String,
                                   Item     => Value,
                                   Start_Pt => 1,
                                   Base     => 10);

      Val_String := E_Strings.Trim (E_Str => Val_String);  -- Remove spaces.

      Summary_Item_Str (Item   => Item,
                        Value  => Val_String,
                        Report => Report);
   end Summary_Item_Int;

   procedure Analysis_Type (Typ    : in E_Strings.T;
                            Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     Tag_IDs;
   --#        in out Schema_State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Schema_State      from *,
   --#                                Schema,
   --#                                Tag_IDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                Schema_State,
   --#                                Tag_IDs,
   --#                                Typ;
   is
   begin
      Text_Tag (Text   => Typ,
                T_Tag  => T_Type,
                Report => Report);
   end Analysis_Type;

end XMLSummary;
