-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with VCG;
with Error_Types;

separate (Sem)
procedure CompUnit (Top_Node : in STree.SyntaxNode;
                    Do_VCG   : in Boolean) is

   type Hidden_Class is (Not_Hidden, Handler_Hidden, All_Hidden);

   -------------------------------------------------------

   --# inherit ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors;
   package Stack
   --# own State;
   is

      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push (X : in Boolean);
      --# global in out State;
      --# derives State from *,
      --#                    X;

      procedure Pop;
      --# global in out State;
      --# derives State from *;

      -- Note: this is not a proper function as it has a hidden
      --       side effect if a system error is raised
      function Top return Boolean;
      --# global in State;

   end Stack;

   ------------------------------------------------------------------------

   --------------- Procedure variables -----------------------------

   Semantic_Error_In_Subprogram_Body  : Boolean;
   Data_Flow_Error_In_Subprogram_Body : Boolean;
   Unused_Data_Flow_Error_Flag        : Boolean;
   Stmt_Node, Last_Node, Next_Node    : STree.SyntaxNode;
   NodeType                           : SP_Symbols.SP_Symbol;
   ErrStatus                          : ErrorHandler.Error_Level;
   ErrNum_On_Success                  : Error_Types.ErrNumRange;
   Current_Scope                      : Dictionary.Scopes;
   TheHeap                            : Heap.HeapRecord;
   NodeTable                          : RefList.HashTable;
   GlobalComponentData                : ComponentManager.ComponentData;
   Unused                             : Dictionary.Symbol;

   -------------- Package bodies ------------------------------

   package body Stack is separate;

   ------------------------------------------------------------
   ---------------- Embedded subprograms ----------------------
   ------------------------------------------------------------

   procedure Create_Implicit_String_Subtype
     (String_Length      : in     Maths.Value;
      Location           : in     Dictionary.Location;
      The_String_Subtype :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    String_Length &
   --#         The_String_Subtype    from ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length;
   is
      String_Subtype_Str       : E_Strings.T;
      String_Subtype_Lex_Str   : LexTokenManager.Lex_String;
      Index_Constraint         : Dictionary.Symbol;
      The_String_Subtype_Local : Dictionary.Symbol;
   begin
      -- First create an index subtype of the form positive__n
      Create_Implicit_Positive_Subtype
        (String_Length    => String_Length,
         Location         => Location,
         Index_Constraint => Index_Constraint);
      -- Create an ExaminerString of the form "String__n" where n is the string length
      String_Subtype_Str := E_Strings.Copy_String (Str => "String__");
      -- The value of "n" will not exceed a size that can be printed within an ExaminerString
      -- so the conversion will not truncate here.
      E_Strings.Append_Examiner_String (E_Str1 => String_Subtype_Str,
                                        E_Str2 => Maths.ValueToString (String_Length));
      -- Insert this name into the string table; either we add it an get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      -- Insert this name into the string table; either we add it and get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      LexTokenManager.Insert_Examiner_String (Str     => String_Subtype_Str,
                                              Lex_Str => String_Subtype_Lex_Str);
      -- Look up type in Dictionary in case it has previously been added
      -- Note that we put these implicit subtype in Standard (where Positive and String themselves live)
      -- and that we declare them in proof rather than Ada context
      The_String_Subtype_Local :=
        Dictionary.LookupItem
        (Name              => String_Subtype_Lex_Str,
         Scope             => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
         Context           => Dictionary.ProofContext,
         Full_Package_Name => False);
      if The_String_Subtype_Local = Dictionary.NullSymbol then
         -- not previously added, so we add a new subtype here
         Dictionary.AddArraySubtype
           (Name            => String_Subtype_Lex_Str,
            Parent          => Dictionary.GetPredefinedStringType,
            ParentReference => Location,
            Comp_Unit       => ContextManager.Ops.Current_Unit,
            Declaration     => Location,
            Scope           => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
            Context         => Dictionary.ProofContext,
            Static          => CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83,
            ArraySubtype    => The_String_Subtype_Local);
         Dictionary.AddArrayIndex
           (TheArrayType       => The_String_Subtype_Local,
            TheIndexType       => Index_Constraint,
            Comp_Unit          => ContextManager.Ops.Current_Unit,
            IndexTypeReference => Location);

      end if;
      The_String_Subtype := The_String_Subtype_Local;
   end Create_Implicit_String_Subtype;

   -----------------------------------------------------------------------------------

   ----------------------------------------------------------------------
   -- Find_Previous_Package
   --
   -- Given a initial_declarative_item_rep node, we sometimes need to find
   -- if there's a embedded package declaration in the tree rooted at that
   -- node.  This function returns the LexString of that package if it
   -- can be found, or NullString if it can't.
   --
   -- This function is used by wf_renaming_declararation and
   -- wf_use_type_declaration, both of which need to locate such
   -- packages, so this function appears here so it can be called
   -- by both subunits.
   ----------------------------------------------------------------------

   function Find_Previous_Package (Node : STree.SyntaxNode) return LexTokenManager.Lex_String
   --# global in STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.initial_declarative_item_rep;
   is
      Last_Node, Next_Node : STree.SyntaxNode;
      Pack_Ident           : LexTokenManager.Lex_String;
   begin
      -- Phase 1 - search down the tree rooted at Node for a
      -- basic_declarative_item node or a package_declaration node
      Last_Node := Child_Node (Current_Node => Node);
      loop
         -- ASSUME Last_Node = initial_declarative_item_rep OR basic_declarative_item OR
         --                    package_declaration OR generic_package_instantiation
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Last_Node) = SP_Symbols.initial_declarative_item_rep
              or else Syntax_Node_Type (Node => Last_Node) = SP_Symbols.basic_declarative_item
              or else Syntax_Node_Type (Node => Last_Node) = SP_Symbols.package_declaration
              or else Syntax_Node_Type (Node => Last_Node) = SP_Symbols.generic_package_instantiation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Last_Node = initial_declarative_item_rep OR basic_declarative_item OR " &
              "package_declaration OR generic_package_instantiation in Find_Previous_Package");
         exit when Syntax_Node_Type (Node => Last_Node) /= SP_Symbols.initial_declarative_item_rep;
         --# assert Syntax_Node_Type (Last_Node, STree.Table) = SP_Symbols.initial_declarative_item_rep;
         Next_Node := Next_Sibling (Current_Node => Last_Node);
         -- ASSUME Next_Node = basic_declarative_item OR package_declaration OR renaming_declaration OR
         --                    use_type_clause OR proof_renaming_declaration OR apragma
         if Syntax_Node_Type (Node => Next_Node) = SP_Symbols.basic_declarative_item
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.package_declaration then
            -- ASSUME Next_Node = basic_declarative_item OR package_declaration
            -- If there's a basic_declarative_item or a package_declaration to the right
            -- of Last_Node, then set Last_Node to that node and exit.
            Last_Node := Next_Node;
         elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.renaming_declaration
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.use_type_clause
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.proof_renaming_declaration
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.apragma then
            -- ASSUME Next_Node = renaming_declaration OR use_type_clause OR
            --                    proof_renaming_declaration OR apragma
            -- No?  Then go down the tree and try again
            Last_Node := Child_Node (Current_Node => Last_Node);
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = basic_declarative_item OR package_declaration OR renaming_declaration OR " &
                 "use_type_clause OR proof_renaming_declaration OR apragma in Find_Previous_Package");
         end if;
      end loop;
      --# assert Syntax_Node_Type (Last_Node, STree.Table) = SP_Symbols.basic_declarative_item or
      --#   Syntax_Node_Type (Last_Node, STree.Table) = SP_Symbols.package_declaration or
      --#   Syntax_Node_Type (Last_Node, STree.Table) = SP_Symbols.generic_package_instantiation;
      -- Phase 2
      -- Last_Node should be a basic_declarative_item or a package_declaration
      case Syntax_Node_Type (Node => Last_Node) is
         when SP_Symbols.basic_declarative_item =>
            -- ASSUME Last_Node = basic_declarative_item
            -- No previous package, so return NullString
            Pack_Ident := LexTokenManager.Null_String;
         when SP_Symbols.package_declaration =>
            -- ASSUME Last_Node = package_declaration
            -- If this package has an inherit clause, then skip over it
            Last_Node := Child_Node (Current_Node => Last_Node);
            -- ASSUME Last_Node = inherit_clause OR package_specification
            if Syntax_Node_Type (Node => Last_Node) = SP_Symbols.inherit_clause then
               -- ASSUME Last_Node = inherit_clause
               Last_Node := Next_Sibling (Current_Node => Last_Node);
            elsif Syntax_Node_Type (Node => Last_Node) /= SP_Symbols.package_specification then
               Last_Node := STree.NullNode;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Last_Node = inherit_clause OR package_specification in Find_Previous_Package");
            end if;
            -- ASSUME Last_Node = package_specification
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Last_Node) = SP_Symbols.package_specification,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Last_Node = package_specification in Find_Previous_Package");
            -- Find and return the package's identifier node
            Last_Node := Last_Child_Of (Start_Node => Last_Node);
            -- ASSUME Last_Node = identifier
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Last_Node) = SP_Symbols.identifier,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Last_Node = identifier in Find_Previous_Package");
            Pack_Ident := Node_Lex_String (Node => Last_Node);
         when SP_Symbols.generic_package_instantiation =>
            -- ASSUME Last_Node = generic_package_instantiation
            Pack_Ident := LexTokenManager.Null_String;
         when others =>
            Pack_Ident := LexTokenManager.Null_String; -- to avoid flow error
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Last_Node = basic_declarative_item OR package_declaration OR " &
                 "generic_package_instantiation in Find_Previous_Package");
      end case;
      return Pack_Ident;
   end Find_Previous_Package;

   ----------------------------------------------------------------------

   -- Returns whether a subprogram_implementation node has a fully hidden body,
   -- a hidden handler part, or no hiding at all.  This code depends on the grammar
   -- in SPARK.LLA section 6.3
   function Body_Hidden_Class (Node : STree.SyntaxNode) return Hidden_Class
   --# global in STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subprogram_implementation;
   is
      Result          : Hidden_Class;
      Pragma_Rep_Node : STree.SyntaxNode;
      End_Node        : STree.SyntaxNode;
   begin
      Pragma_Rep_Node := Child_Node (Current_Node => Node);
      -- ASSUME Pragma_Rep_Node = pragma_rep
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Pragma_Rep_Node) = SP_Symbols.pragma_rep,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Pragma_Rep_Node = pragma_rep in Body_Hidden_Class");
      -- if the Pragma_Rep is immediately followed by hidden part, then the whole
      -- body must be hidden.  If the PragmaRep is followed by something else, then
      -- we _might_ have a hidden handler part at End_Node.
      End_Node := Next_Sibling (Current_Node => Pragma_Rep_Node);
      -- ASSUME End_Node = declarative_part OR sequence_of_statements OR code_insertion OR hidden_part
      if Syntax_Node_Type (Node => End_Node) = SP_Symbols.hidden_part then
         -- ASSUME End_Node = hidden_part
         Result := All_Hidden;
      elsif Syntax_Node_Type (Node => End_Node) = SP_Symbols.declarative_part
        or else Syntax_Node_Type (Node => End_Node) = SP_Symbols.sequence_of_statements
        or else Syntax_Node_Type (Node => End_Node) = SP_Symbols.code_insertion then
         -- ASSUME End_Node = declarative_part OR sequence_of_statements OR code_insertion
         End_Node := Last_Sibling_Of (Start_Node => End_Node);
         -- ASSUME End_Node = designator OR hidden_part
         if Syntax_Node_Type (Node => End_Node) = SP_Symbols.hidden_part then
            -- ASSUME End_Node = hidden_part
            Result := Handler_Hidden;
         elsif Syntax_Node_Type (Node => End_Node) = SP_Symbols.designator then
            -- ASSUME End_Node = designator
            Result := Not_Hidden;
         else
            Result := Not_Hidden;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect End_Node = designator OR hidden_part in Body_Hidden_Class");
         end if;
      else
         Result := Not_Hidden;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect End_Node =  declarative_part OR sequence_of_statements OR code_insertion OR " &
              "hidden_part in Body_Hidden_Class");
      end if;
      return Result;
   end Body_Hidden_Class;

   ----------------------------------------------------------------------

   -- Given a (sub)type S with range S'First .. S'Last
   -- and a base type   T with range T'First .. T'Last
   --
   -- First, checks that T'First and T'Last are well-defined, or raises sem error 793.
   -- Then, if S is real and unconstrained, then no further checks.
   -- Then, checks that (S'First >= T'First and S'Last <= T'Last) or raises sem error 794.
   --
   -- Used to check legality of derived numeric type declarations and base type assertions
   procedure Check_Subtype_Against_Basetype_Bounds
     (Base_Type_Sym  : in     Dictionary.Symbol;
      Subtype_First  : in     LexTokenManager.Lex_String;
      Subtype_Last   : in     LexTokenManager.Lex_String;
      Ident_Node_Pos : in     LexTokenManager.Token_Position;
      Range_Node_Pos : in     LexTokenManager.Token_Position;
      Errors         : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Base_Type_Sym,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Errors,
   --#                                         Ident_Node_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Range_Node_Pos,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Subtype_First,
   --#                                         Subtype_Last &
   --#         Errors                     from *,
   --#                                         Base_Type_Sym,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Subtype_First,
   --#                                         Subtype_Last;
   is
      Base_Type_First, Base_Type_Last : LexTokenManager.Lex_String;

      Type_First_Val, Type_Last_Val, Base_Type_First_Val, Base_Type_Last_Val, Comp_Val : Maths.Value;

      Bounds_OK : Boolean;

      Maths_Error1, Maths_Error2 : Maths.ErrorCode;
   begin
      -- check that there are defined bounds for the base type, and also that
      -- the range of the type fits within the range of the specified base type
      if not Errors and then Dictionary.IsTypeMark (Base_Type_Sym) then
         Base_Type_First := Dictionary.GetScalarAttributeValue (False, LexTokenManager.First_Token, Base_Type_Sym);
         Base_Type_Last  := Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, Base_Type_Sym);

         if Base_Type_First = LexTokenManager.Null_String or Base_Type_Last = LexTokenManager.Null_String then

            -- we require that the base type have defined bounds,
            ErrorHandler.Semantic_Error
              (Err_Num   => 793,
               Reference => ErrorHandler.No_Reference,
               Position  => Ident_Node_Pos,
               Id_Str    => LexTokenManager.Null_String);
            Errors := True;
         elsif Subtype_First = LexTokenManager.Null_String or Subtype_Last = LexTokenManager.Null_String then
            -- no check possible with unconstrained ranges
            null;
         else
            -- check that the range of the base type is at least that of the type
            Type_First_Val      := Maths.ValueRep (Subtype_First);
            Type_Last_Val       := Maths.ValueRep (Subtype_Last);
            Base_Type_First_Val := Maths.ValueRep (Base_Type_First);
            Base_Type_Last_Val  := Maths.ValueRep (Base_Type_Last);
            Maths.Lesser (Type_First_Val, Base_Type_First_Val, Comp_Val, Maths_Error1);
            Bounds_OK := (Comp_Val = Maths.FalseValue);
            Maths.Greater (Type_Last_Val, Base_Type_Last_Val, Comp_Val, Maths_Error2);
            Bounds_OK := Bounds_OK and then Comp_Val = Maths.FalseValue;
            if not Bounds_OK and then Maths_Error1 = Maths.NoError and then Maths_Error2 = Maths.NoError then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 794,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Range_Node_Pos,
                  Id_Str    => LexTokenManager.Null_String);
               Errors := True;
            end if;
         end if;
      end if;
   end Check_Subtype_Against_Basetype_Bounds;

   ----------------------------------------------------------------------

   procedure Wf_External_Interface
     (Pragma_Node : in     STree.SyntaxNode;
      Entity_Sym  : in     Dictionary.Symbol;
      Error_Found :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Pragma_Node, STree.Table) = SP_Symbols.apragma;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------------

   procedure Wf_Elaborate_Body (Pragma_Node : in STree.SyntaxNode;
                                Pack_Sym    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         Pack_Sym,
   --#                                         Pragma_Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pack_Sym,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Pack_Sym,
   --#                                         Pragma_Node;
   --# pre Syntax_Node_Type (Pragma_Node, STree.Table) = SP_Symbols.apragma;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------------

   procedure Add_Child
     (Root_Id_Node : in     STree.SyntaxNode;
      Is_Private   : in     Boolean;
      Scope        : in     Dictionary.Scopes;
      Child_Sym    :    out Dictionary.Symbol;
      Child_Str    :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Child_Str                  from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Root_Id_Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Child_Sym,
   --#         Dictionary.Dict            from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Is_Private,
   --#                                         LexTokenManager.State,
   --#                                         Root_Id_Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Is_Private,
   --#                                         LexTokenManager.State,
   --#                                         Root_Id_Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Private,
   --#                                         LexTokenManager.State,
   --#                                         Root_Id_Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Is_Private,
   --#                                         LexTokenManager.State,
   --#                                         Root_Id_Node,
   --#                                         Scope;
   --# pre Syntax_Node_Type (Root_Id_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
   is
      Curr_Node  : STree.SyntaxNode;
      Curr_Sym   : Dictionary.Symbol;
      Parent_Sym : Dictionary.Symbol := Dictionary.NullSymbol;
      Child_Sort : Dictionary.PackageSort;
   begin
      Curr_Node := Root_Id_Node;
      Child_Str := Node_Lex_String (Node => Curr_Node);
      Curr_Sym  :=
        Dictionary.LookupItem (Name              => Child_Str,
                               Scope             => Scope,
                               Context           => Dictionary.ProofContext,
                               Full_Package_Name => False);
      while Syntax_Node_Type (Node => Next_Sibling (Current_Node => Parent_Node (Current_Node => Curr_Node))) =
        SP_Symbols.identifier loop
         -- to handle multiple prefixes
         if Curr_Sym = Dictionary.NullSymbol then
            -- not declared or visible
            Parent_Sym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 140,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Curr_Node),
               Id_Str    => Child_Str);
            exit;
         end if;

         if not Dictionary.IsPackage (Curr_Sym) then
            -- can't be a parent
            Curr_Sym   := Dictionary.NullSymbol;
            Parent_Sym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 18,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Curr_Node),
               Id_Str    => Child_Str);
            exit;
         end if;

         -- Child_Str (Curr_Sym) is visible and its a package
         STree.Set_Node_Lex_String (Sym  => Curr_Sym,
                                    Node => Curr_Node);
         Parent_Sym := Curr_Sym;
         Curr_Node  := Next_Sibling (Current_Node => Parent_Node (Current_Node => Curr_Node));
         --# assert STree.Table = STree.Table~ and
         --#   Syntax_Node_Type (Root_Id_Node, STree.Table) = SP_Symbols.identifier and
         --#   Syntax_Node_Type (Curr_Node, STree.Table) = SP_Symbols.identifier;
         Child_Str := Node_Lex_String (Node => Curr_Node);
         Curr_Sym  :=
           Dictionary.LookupSelectedItem
           (Prefix   => Curr_Sym,
            Selector => Child_Str,
            Scope    => Scope,
            Context  => Dictionary.ProofContext);
      end loop;

      if Curr_Sym /= Dictionary.NullSymbol then
         -- child already declared
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Curr_Node),
            Id_Str    => Child_Str);
      elsif Parent_Sym /= Dictionary.NullSymbol then
         -- check that Child_Str has not been declared as a body stub
         Curr_Sym :=
           Dictionary.LookupImmediateScope
           (Name    => Child_Str,
            Scope   => Dictionary.LocalScope (Parent_Sym),
            Context => Dictionary.ProgramContext);
         if Curr_Sym /= Dictionary.NullSymbol and then Dictionary.HasBodyStub (Curr_Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Curr_Node),
               Id_Str    => Child_Str);
         elsif Curr_Sym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => Curr_Sym,
                                       Node => Curr_Node);
         end if;

         -- check that Child_Str not inherited by parent of private child:
         Curr_Sym :=
           Dictionary.LookupItem
           (Name              => Child_Str,
            Scope             => Dictionary.VisibleScope (Parent_Sym),
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False);
         if Is_Private and then Curr_Sym /= Dictionary.NullSymbol then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Curr_Node),
               Id_Str    => Child_Str);
         elsif Curr_Sym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => Curr_Sym,
                                       Node => Curr_Node);
         end if;
      end if;

      if Parent_Sym /= Dictionary.NullSymbol then
         if Is_Private then
            Child_Sort := Dictionary.PrivateChild;
         else
            Child_Sort := Dictionary.Public;
         end if;
         Dictionary.AddChildPackage
           (TheParent     => Parent_Sym,
            Sort          => Child_Sort,
            Name          => Child_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Root_Id_Node),
                                                  End_Position   => Node_Position (Node => Root_Id_Node)),
            Scope         => Scope,
            ThePackage    => Child_Sym);
      else
         -- Parent is not valid (i.e. undeclared or not a package) so we can't do any more.
         -- Signal failure back to caller with null child symbol
         Child_Sym := Dictionary.NullSymbol;
         Child_Str := LexTokenManager.Null_String;
      end if;
   end Add_Child;

   -----------------------------------------------------------------------------------------

   -- This procedure has essentially the functionality of WalkExpression
   -- but is used to walk expressions which the language rules require to be a
   -- "name".  In addition to the things returned by WalkExpression, a flag is
   -- returned indicating whether the expression was indeed a name.
   -- If the expression is not a name the expression is not traversed at all
   -- and Result is set to the Unknown_Type_Record
   --
   -- After the name node has been found it is traversed by WalkExpression and a subset
   -- of the checks usually done by wf_primary are acarried out on the result.  More
   -- checks are done here than are necessary for this application (e.g. getting bounds
   -- of a type mark) but they have been left in to make Walk_Name more widely applicable;
   -- it could be moved to Wellformed if ever needed elsewhere.

   procedure Walk_Name
     (Exp_Node       : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData;
      Result         :    out Exp_Record;
      Is_A_Name      :    out Boolean;
      Ref_Var_Param  : in     SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Is_A_Name                  from Exp_Node,
   --#                                         STree.Table &
   --#         Result                     from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.expression;
   --# post STree.Table = STree.Table~;
   is
      Node, Next_Node : STree.SyntaxNode;
      Sym             : Dictionary.Symbol;
   begin
      -- preset result for all non-name cases advance to relation node;
      Result := Unknown_Type_Record;
      Node   := Child_Node (Current_Node => Exp_Node);
      -- ASSUME Node = relation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SP_Symbols.relation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = relation in Walk_Name");
      Next_Node := Next_Sibling (Current_Node => Node);
      -- ASSUME Next_Node = expression_rep1 OR expression_rep2 OR expression_rep3 OR
      --                    expression_rep4 OR expression_rep5 OR NULL
      if Next_Node = STree.NullNode then
         -- ASSUME Next_Node = NULL
         -- advance to simple_expression;
         Node := Child_Node (Current_Node => Node);
         -- ASSUME Node = simple_expression
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SP_Symbols.simple_expression,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = simple_expression in Walk_Name");
         Next_Node := Next_Sibling (Current_Node => Node);
         -- ASSUME Next_Node = relational_operator OR inside OR outside OR NULL
         if Next_Node = STree.NullNode then
            -- ASSUME Next_Node = NULL
            -- advance to simple_expression_opt;
            Node := Child_Node (Current_Node => Node);
            -- ASSUME Node = simple_expression OR simple_expression_opt
            if Syntax_Node_Type (Node => Node) = SP_Symbols.simple_expression_opt then
               -- Node = simple_expression_opt
               -- advance to term;
               Node := Child_Node (Current_Node => Node);
               -- ASSUME Node = unary_adding_operator OR term
               if Syntax_Node_Type (Node => Node) = SP_Symbols.term then
                  -- ASSUME Node = term
                  -- advance to factor;
                  Node := Child_Node (Current_Node => Node);
                  -- ASSUME Node = term OR factor
                  if Syntax_Node_Type (Node => Node) = SP_Symbols.factor then
                     -- ASSUME Node = factor
                     -- advance to primary;
                     Node := Child_Node (Current_Node => Node);
                     -- ASSUME Node = primary OR RWabs OR RWnot
                     if Syntax_Node_Type (Node => Node) = SP_Symbols.primary then
                        -- ASSUME Node = primary
                        Next_Node := Next_Sibling (Current_Node => Node);
                        -- ASSUME Next_Node = double_star OR NULL
                        if Next_Node = STree.NullNode then
                           -- ASSUME Next_Node = NULL
                           -- advance to name;
                           Node := Child_Node (Current_Node => Node);
                           -- ASSUME Node = numeric_literal OR character_literal OR string_literal OR name OR
                           --               qualified_expression OR expression OR attribute
                           if Syntax_Node_Type (Node => Node) = SP_Symbols.name then
                              -- ASSUME Node = name
                              Is_A_Name := True;
                              Walk_Expression_P.Walk_Expression
                                (Exp_Node                => Node,
                                 Scope                   => Scope,
                                 Type_Context            => Dictionary.GetUnknownTypeMark,
                                 Context_Requires_Static => False,
                                 Ref_Var                 => Ref_Var_Param,
                                 Result                  => Result,
                                 Component_Data          => Component_Data,
                                 The_Heap                => TheHeap);
                              -- now perform some checks on the result akin to those
                              -- of wf_primary
                              Sym := Result.Other_Symbol;
                              case Result.Sort is
                                 when Is_Unknown =>
                                    Result := Unknown_Type_Record;
                                 when Type_Result =>
                                    null;
                                 when Is_Package =>
                                    Result := Unknown_Type_Record;
                                    ErrorHandler.Semantic_Error
                                      (Err_Num   => 5,
                                       Reference => ErrorHandler.No_Reference,
                                       Position  => Node_Position (Node => Node),
                                       Id_Str    => Dictionary.GetSimpleName (Sym));
                                 when Is_Function =>
                                    ErrorHandler.Semantic_Error
                                      (Err_Num   => 3,
                                       Reference => ErrorHandler.No_Reference,
                                       Position  => Node_Position (Node => Node),
                                       Id_Str    => Dictionary.GetSimpleName (Sym));
                                    Result.Is_Static   := False;
                                    Result.Is_Constant := False;
                                    Result.Is_ARange   := False;
                                 when Is_Object =>
                                    Result.Is_ARange := False;
                                    if Dictionary.IsVariableOrSubcomponent (Sym) then
                                       SeqAlgebra.AddMember (TheHeap, Ref_Var_Param, Natural (Dictionary.SymbolRef (Sym)));
                                    end if;
                                 when Is_Type_Mark =>
                                    Result.Is_Static   := Dictionary.IsStatic (Sym, Scope);
                                    Result.Is_Constant := True;
                                    if Dictionary.IsScalarTypeMark (Sym, Scope) then -- we can get bounds of range and so on
                                       Result.Is_ARange := True;
                                       -- get upper and lower bounds of type from dictionary
                                       Result.Value     :=
                                         Maths.ValueRep
                                         (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                                              LexTokenManager.First_Token, Sym));
                                       Result.Range_RHS :=
                                         Maths.ValueRep
                                         (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                                              LexTokenManager.Last_Token, Sym));
                                    else -- not scalar so we don't do bounds and its not a range
                                       Result.Is_ARange := False;
                                    end if;
                                 when Is_Parameter_Name =>
                                    null; -- should never occur
                              end case;
                              Result.Param_Count := 0;
                              Result.Param_List  := Lists.Null_List;
                           elsif Syntax_Node_Type (Node => Node) = SP_Symbols.numeric_literal
                             or else Syntax_Node_Type (Node => Node) = SP_Symbols.character_literal
                             or else Syntax_Node_Type (Node => Node) = SP_Symbols.string_literal
                             or else Syntax_Node_Type (Node => Node) = SP_Symbols.qualified_expression
                             or else Syntax_Node_Type (Node => Node) = SP_Symbols.expression
                             or else Syntax_Node_Type (Node => Node) = SP_Symbols.attribute then
                              -- ASSUME Node = numeric_literal OR character_literal OR string_literal OR
                              --               qualified_expression OR expression OR attribute
                              Is_A_Name := False;
                           else
                              Is_A_Name := False;
                              SystemErrors.Fatal_Error
                                (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                                 Msg     => "Expect Node = numeric_literal OR character_literal OR string_literal OR " &
                                   "name OR qualified_expression OR expression OR attribute in Walk_Name");
                           end if;
                        elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.double_star then
                           -- ASSUME Next_Node = double_star
                           Is_A_Name := False;
                        else
                           Is_A_Name := False;
                           SystemErrors.Fatal_Error
                             (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                              Msg     => "Expect Next_Node = double_star OR NULL in Walk_Name");
                        end if;
                     elsif Syntax_Node_Type (Node => Node) = SP_Symbols.RWabs
                       or else Syntax_Node_Type (Node => Node) = SP_Symbols.RWnot then
                        -- ASSUME Node = RWabs OR RWnot
                        Is_A_Name := False;
                     else
                        Is_A_Name := False;
                        SystemErrors.Fatal_Error
                          (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                           Msg     => "Expect Node = primary OR RWabs OR RWnot in Walk_Name");
                     end if;
                  elsif Syntax_Node_Type (Node => Node) = SP_Symbols.term then
                     -- ASSUME Node = term
                     Is_A_Name := False;
                  else
                     Is_A_Name := False;
                     SystemErrors.Fatal_Error
                       (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                        Msg     => "Expect Node = term OR factor in Walk_Name");
                  end if;
               elsif Syntax_Node_Type (Node => Node) = SP_Symbols.unary_adding_operator then
                  -- ASSUME Node = unary_adding_operator
                  Is_A_Name := False;
               else
                  Is_A_Name := False;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Node = unary_adding_operator OR term in Walk_Name");
               end if;
            elsif Syntax_Node_Type (Node => Node) = SP_Symbols.simple_expression then
               -- ASSUME Node = simple_expression
               Is_A_Name := False;
            else
               Is_A_Name := False;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Node = simple_expression OR simple_expression_opt in Walk_Name");
            end if;
         elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.relational_operator
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.inside
           or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.outside then
            -- ASSUME Next_Node = relational_operator OR inside OR outside
            Is_A_Name := False;
         else
            Is_A_Name := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = relational_operator OR inside OR outside OR NULL in Walk_Name");
         end if;
      elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.expression_rep1
        or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.expression_rep2
        or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.expression_rep3
        or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.expression_rep4
        or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.expression_rep5 then
         -- ASSUME Next_Node = expression_rep1 OR expression_rep2 OR expression_rep3 OR
         --                    expression_rep4 OR expression_rep5
         Is_A_Name := False;
      else
         Is_A_Name := False;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Node = expression_rep1 OR expression_rep2 OR expression_rep3 OR " &
              "expression_rep4 OR expression_rep5 OR NULL in Walk_Name");
      end if;
   end Walk_Name;

   ---------------------------------------------------------------------

   procedure CreateInterruptStreamVariables
     (For_PO         : in Dictionary.Symbol;
      Error_Node_Pos : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         For_PO,
   --#                                         LexTokenManager.State &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         For_PO,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckCeilingPriority
     (Sym                : in Dictionary.Symbol;
      Scope              : in Dictionary.Scopes;
      Check_List         : in Dictionary.Iterator;
      Priority_Lex_Value : in LexTokenManager.Lex_String;
      Error_Node_Pos     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Check_List,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Priority_Lex_Value,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Sym;
      is separate;

   ---------------------------------------------------------------------

   function Unexpected_Initialization (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return -- not an initialized own variable
        ((Dictionary.IsOwnVariable (Sym) and then not Dictionary.OwnVariableIsInitialized (Sym))
         or else
           -- not an initialized constituent
           (Dictionary.IsConstituent (Sym) and then not Dictionary.OwnVariableIsInitialized (Dictionary.GetSubject (Sym))))
        and then
        -- not moded (this last limb to suppress error 333 for stream vars
        Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.DefaultMode;
   end Unexpected_Initialization;

   ------------------------------------------------------------------------

   procedure Wf_Pragma (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.apragma;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Priority_Pragma (Node  : in STree.SyntaxNode;
                                 Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.priority_pragma;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Renaming_Declaration (Node  : in STree.SyntaxNode;
                                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.renaming_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Use_Type_Clause (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.use_type_clause;
      is separate;

   ------------------------------------------------------------------

   procedure Wf_Proof_Renaming_Declaration (Node  : in STree.SyntaxNode;
                                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.proof_renaming_declaration;
      is separate;

   --------------------------------------------------------------------

   procedure CheckNamedAssociation
     (The_Formals               : in Dictionary.Symbol;
      Scope                     : in Dictionary.Scopes;
      Named_Argument_Assoc_Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Named_Argument_Assoc_Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Formals;
   --# pre Syntax_Node_Type (Named_Argument_Assoc_Node, STree.Table) = SP_Symbols.named_argument_association;
      is separate;

   ------------------------------------------------------------------------

   function FindActualNode
     (For_Formal                : Dictionary.Symbol;
      Named_Argument_Assoc_Node : STree.SyntaxNode)
     return                      STree.SyntaxNode
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
   --# pre Syntax_Node_Type (Named_Argument_Assoc_Node, STree.Table) = SP_Symbols.named_argument_association;
   --# return Node => (Syntax_Node_Type (Node, STree.Table) = SP_Symbols.expression or Node = STree.NullNode);
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Justification_Statement
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Component_Data,
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.justification_statement;
   --# post STree.Table = STree.Table~;
      is separate;

   ------------------------------------------------------------------------

   procedure WalkStatements
     (Seq_Node       : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Table          : in out RefList.HashTable;
      Component_Data :    out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Seq_Node,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap &
   --#         Component_Data             from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Seq_Node,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Seq_Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Seq_Node, STree.Table) = SP_Symbols.sequence_of_statements;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Subtype_Declaration (Node  : in STree.SyntaxNode;
                                     Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subtype_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------

   procedure CheckTaskModifierConsistency
     (The_Own_Var_Type : in     Dictionary.Symbol;
      The_Var_Type     : in     Dictionary.Symbol;
      Modifier_Is_Task : in     Boolean;
      Error_Node       : in     STree.SyntaxNode;
      Consistent       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Modifier_Is_Task,
   --#                                         The_Own_Var_Type,
   --#                                         The_Var_Type &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node,
   --#                                         LexTokenManager.State,
   --#                                         Modifier_Is_Task,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Own_Var_Type,
   --#                                         The_Var_Type;
   --# pre Syntax_Node_Type (Error_Node, STree.Table) = SP_Symbols.identifier;
      is separate;

   --------------------------------------------------------------------

   procedure CheckProtectedModifierConsistency
     (The_Type              : in     Dictionary.Symbol;
      Modifier_Is_Protected : in     Boolean;
      Error_Node            : in     STree.SyntaxNode;
      Consistent            :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Modifier_Is_Protected,
   --#                                         The_Type &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node,
   --#                                         LexTokenManager.State,
   --#                                         Modifier_Is_Protected,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Type;
   --# pre Syntax_Node_Type (Error_Node, STree.Table) = SP_Symbols.identifier;
      is separate;

   -----------------------------------------------------------------------------
   -- wf_variable_declaration
   --
   -- Note: Enclosing_Unit_Scope represents the scope of the enclosing program
   -- unit, whilst Declaration_Scope represents the scope of the current
   -- declaration. These are normally the same. The exception is the special
   -- case of a declaration within a protected type, where they may differ.
   -- For example, the declaration could be in the private part of the PT but
   -- the PT itself could be in the visible part of the enclosing unit.
   -----------------------------------------------------------------------------

   procedure Wf_Variable_Declaration
     (Node                 : in STree.SyntaxNode;
      Enclosing_Unit_Scope : in Dictionary.Scopes;
      Declaration_Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.variable_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Constant_Declaration (Node          : in STree.SyntaxNode;
                                      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out Aggregate_Stack.State;
   --#           out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.constant_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Inherit_Clause
     (Node     : in STree.SyntaxNode;
      Comp_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.inherit_clause;
   --# post STree.Table = STree.Table~;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Context_Clause
     (Node     : in STree.SyntaxNode;
      Comp_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.context_clause;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   -- procedure to check whether a properly-defined subprogram or stub creates
   -- overloading by re-using the name of a potentially inheritable root
   -- operation

   procedure CheckNoOverloadingFromTaggedOps
     (Spec_Node     : in STree.SyntaxNode;
      Subprog_Sym   : in Dictionary.Symbol;
      Scope         : in Dictionary.Scopes;
      Abstraction   : in Dictionary.Abstractions;
      Is_Overriding : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Spec_Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Overriding,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Spec_Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym;
   --# pre Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.procedure_specification or
   --#   Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.function_specification or
   --#   Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.entry_specification;
      is separate;

   ----------------------------------------------------------------------------

   -- function used by wf_subprogram_body and wf_body_stub

   function Has_Parameter_Global_Or_Return_Of_Local_Private_Type (Subprog_Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   -- returns true if the subprogram has a parameter or global which is of a private type
   -- but which is not private when viewd from the subprogram
   is
      Result   : Boolean := False;
      It       : Dictionary.Iterator;
      Type_Sym : Dictionary.Symbol;

      function Is_Private_But_Not_Private_Here (Type_Sym    : Dictionary.Symbol;
                                                Subprog_Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.TypeIsPrivate (Type_Sym)
           and then not Dictionary.IsPrivateType (Type_Sym, Dictionary.LocalScope (Subprog_Sym));
      end Is_Private_But_Not_Private_Here;

   begin -- Has_Parameter_Global_Or_Return_Of_Local_Private_Type

      -- iterate through parameters
      It := Dictionary.FirstSubprogramParameter (Subprog_Sym);
      while It /= Dictionary.NullIterator loop
         Type_Sym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
         Result   := Is_Private_But_Not_Private_Here (Type_Sym    => Type_Sym,
                                                      Subprog_Sym => Subprog_Sym);
         exit when Result;

         It := Dictionary.NextSymbol (It);
      end loop;

      if not Result then -- no parameters were private, so check globals
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Subprog_Sym);
         while It /= Dictionary.NullIterator loop
            Type_Sym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
            Result   := Is_Private_But_Not_Private_Here (Type_Sym    => Type_Sym,
                                                         Subprog_Sym => Subprog_Sym);
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
      end if;

      -- no parameters or globals were private, so check for function return type.
      if not Result then
         if Dictionary.IsFunction (Subprog_Sym) then
            Type_Sym := Dictionary.GetType (Subprog_Sym);
            Result   := Is_Private_But_Not_Private_Here (Type_Sym    => Type_Sym,
                                                         Subprog_Sym => Subprog_Sym);
         end if;
      end if;

      return Result;
   end Has_Parameter_Global_Or_Return_Of_Local_Private_Type;

   ----------------------------------------------------------------------------

   procedure Wf_Subprogram_Body
     (Node           : in     STree.SyntaxNode;
      Scope          : in out Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData;
      Next_Node      :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subprogram_body or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_body;
   --# post STree.Table = STree.Table~;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Entry_Body
     (Node           : in     STree.SyntaxNode;
      Scope          : in out Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData;
      Next_Node      :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.entry_body;
   --# post (Syntax_Node_Type (Next_Node, STree.Table) = SP_Symbols.subprogram_implementation or
   --#         Next_Node = STree.NullNode) and
   --#   STree.Table = STree.Table~;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Proof_Function_Declaration
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      The_Heap      : in out Heap.HeapRecord;
      Func_Sym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         Func_Sym                   from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.proof_function_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------

   procedure Wf_Body_Stub
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.body_stub;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------

   procedure Wf_Subunit (Node  : in     STree.SyntaxNode;
                         Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subunit;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------

   procedure CheckEmbedBodies (Comp_Sym : in Dictionary.Symbol;
                               Node_Pos : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         SPARK_IO.File_Sys;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckSuspendsListAccountedFor
     (Proc_Or_Task : in Dictionary.Symbol;
      Node_Pos     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         Proc_Or_Task,
   --#                                         SPARK_IO.File_Sys;
      is separate;

   ---------------------------------------------------------------------

   procedure Up_Wf_Subprogram_Body (Node  : in     STree.SyntaxNode;
                                    Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subprogram_body or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_body;
      is separate;

   ---------------------------------------------------------------------

   -- this subprogram used to a subunit of wf_package_declaration.wf_visible; it has
   -- been moved up to this level to make it available to wf_protected_type.wf_protected_operation

   procedure Wf_Subprogram_Declaration
     (Node           : in     STree.SyntaxNode;
      Current_Scope  : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData;
      Subprog_Sym    :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.subprogram_declaration or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.not_overriding_subprogram_declaration or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.entry_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Known_Discriminant_Part
     (Node               : in STree.SyntaxNode;
      Protected_Type_Sym : in Dictionary.Symbol;
      Scope              : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Protected_Type_Sym,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Protected_Type_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.known_discriminant_part;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------

   procedure Wf_Array_Type_Definition
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      Ident_Node   : in     STree.SyntaxNode;
      Dec_Loc      : in     LexTokenManager.Token_Position;
      Is_Generic   : in     Boolean;
      Errors_Found :    out Boolean;
      The_Array    :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Errors_Found,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         The_Array                  from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.array_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------

   procedure Wf_Full_Type_Declaration
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.full_type_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------------------------------------------

   procedure Wf_Basic_Declarative_Item
     (Node           : in     STree.SyntaxNode;
      Current_Scope  : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.basic_declarative_item;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------

   procedure CheckAnnouncedTypesDeclared
     (Pack_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes;
      Node_Pos : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         Pack_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys;
      is separate;

   --------------------------------------------------------------------

   procedure Wf_Generic_Actual_Part
     (Generic_Node      : in     STree.SyntaxNode;
      Generic_Sym       : in     Dictionary.Symbol;
      Instantiation_Sym : in     Dictionary.Symbol;
      Scope             : in     Dictionary.Scopes;
      Error_Found       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Node,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Generic_Node,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Error_Found                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Generic_Node,
   --#                                         Generic_Sym,
   --#                                         GlobalComponentData,
   --#                                         Instantiation_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Generic_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------

   procedure Wf_Generic_Subprogram_Instantiation (Node  : in STree.SyntaxNode;
                                                  Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_subprogram_instantiation;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   -- Factored out of wf_package_declaration to allow reuse in generic package declaration

   procedure Wf_Package_Specification
     (Node          : in STree.SyntaxNode;
      Ident_Str     : in LexTokenManager.Lex_String;
      Pack_Sym      : in Dictionary.Symbol;
      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         Ident_Str,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.package_specification;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Package_Declaration (Node          : in STree.SyntaxNode;
                                     Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.package_declaration or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.private_package_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   -- PNA temporary reduced annotation for phase 1 generics cfr 1340

   procedure Wf_Generic_Package_Instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_package_instantiation;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Generic_Declaration (Node          : in STree.SyntaxNode;
                                     Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_declaration or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.private_generic_package_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure CheckPackageNeedsBody (Node_Pos : in LexTokenManager.Token_Position;
                                    Pack_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         Pack_Sym,
   --#                                         SPARK_IO.File_Sys;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Package_Body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.package_body;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure Up_Wf_Package_Body (Node  : in     STree.SyntaxNode;
                                 Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.package_body;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Package_Initialization (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.package_initialization;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Protected_Body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Next_Node,
   --#         Scope,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.protected_body;
   --# post STree.Table = STree.Table~ and
   --#   (Syntax_Node_Type (Next_Node, STree.Table) = SP_Symbols.protected_operation_item or
   --#      Next_Node = STree.NullNode);
      is separate;

   ------------------------------------------------------------------------

   procedure Up_Wf_Protected_Body (Node  : in     STree.SyntaxNode;
                                   Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.protected_body;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Task_Body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.task_body;
   --# post STree.Table = STree.Table~ and
   --#   (Syntax_Node_Type (Next_Node, STree.Table) = SP_Symbols.subprogram_implementation or Next_Node = STree.NullNode);
      is separate;

   ------------------------------------------------------------------------

   procedure Up_Wf_Task_Body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.task_body;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Machine_Code_Insertion (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.code_insertion;
      is separate;

begin -- CompUnit
   Next_Node := Top_Node;
   Stack.Init;
   Heap.Initialize (TheHeap);
   RefList.Init (NodeTable);
   Aggregate_Stack.Init;
   ComponentManager.Initialise (GlobalComponentData);

   Current_Scope := Dictionary.GlobalScope;

   while Next_Node /= STree.NullNode
   --# assert STree.Table = STree.Table~;
   loop
      Last_Node := Next_Node;
      NodeType  := Syntax_Node_Type (Node => Last_Node);
      case NodeType is
         when SP_Symbols.package_declaration | SP_Symbols.private_package_declaration =>
            -- ASSUME Last_Node = package_declaration OR private_package_declaration
            ErrorHandler.Start_Unit;
            Wf_Package_Declaration (Node          => Last_Node,
                                    Current_Scope => Current_Scope);
            ErrorHandler.End_Unit;
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.generic_declaration | SP_Symbols.private_generic_package_declaration =>
            -- ASSUME Last_Node = generic_declaration OR private_generic_package_declaration
            ErrorHandler.Start_Unit;
            Wf_Generic_Declaration (Node          => Last_Node,
                                    Current_Scope => Current_Scope);
            ErrorHandler.End_Unit;
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.generic_package_instantiation =>
            -- ASSUME Last_Node = generic_package_instantiation
            Wf_Generic_Package_Instantiation (Node  => Last_Node,
                                              Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.generic_subprogram_instantiation =>
            -- ASSUME Last_Node = generic_subprogram_instantiation
            Wf_Generic_Subprogram_Instantiation (Node  => Last_Node,
                                                 Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.package_body =>
            -- ASSUME Last_Node = package_body
            ErrorHandler.Start_Unit;
            Wf_Package_Body (Node  => Last_Node,
                             Scope => Current_Scope);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SP_Symbols.subprogram_body | SP_Symbols.not_overriding_subprogram_body =>
            -- ASSUME Last_Node = subprogram_body OR not_overriding_subprogram_body
            ErrorHandler.Error_Reset;
            ErrorHandler.Start_Unit;
            Wf_Subprogram_Body
              (Node           => Last_Node,
               Scope          => Current_Scope,
               Component_Data => GlobalComponentData,
               Next_Node      => Next_Node);
            ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
            if ErrStatus = ErrorHandler.No_Error then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;
         when SP_Symbols.entry_body =>
            -- ASSUME Last_Node = entry_body
            ErrorHandler.Error_Reset;
            ErrorHandler.Start_Unit;
            Wf_Entry_Body (Node           => Last_Node,
                           Scope          => Current_Scope,
                           Component_Data => GlobalComponentData,
                           Next_Node      => Next_Node);
            ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
            if ErrStatus = ErrorHandler.No_Error then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;
         when SP_Symbols.task_body =>
            -- ASSUME Last_Node = task_body
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.Ravenscar_Selected then
               --# end accept;
               ErrorHandler.Error_Reset;
               ErrorHandler.Start_Unit;
               Wf_Task_Body (Node      => Last_Node,
                             Scope     => Current_Scope,
                             Next_Node => Next_Node);
               ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
               if ErrStatus = ErrorHandler.No_Error then
                  Stack.Push (False);
               else
                  Stack.Push (True);
               end if;
            else -- can't use task except in Ravenscar
               ErrorHandler.Semantic_Error
                 (Err_Num   => 850,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Last_Node),
                  Id_Str    => LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               Next_Node := STree.NullNode;
            end if;
         when SP_Symbols.protected_body =>
            -- ASSUME Last_Node = protected_body
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.Ravenscar_Selected then
               --# end accept;
               ErrorHandler.Start_Unit;
               Wf_Protected_Body (Node      => Last_Node,
                                  Scope     => Current_Scope,
                                  Next_Node => Next_Node);
            else -- can't use protected type except in Ravenscar
               ErrorHandler.Semantic_Error
                 (Err_Num   => 850,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Last_Node),
                  Id_Str    => LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               Next_Node := STree.NullNode;
            end if;
         when SP_Symbols.body_stub =>
            -- ASSUME Last_Node = body_stub
            Wf_Body_Stub (Node           => Last_Node,
                          Scope          => Current_Scope,
                          Component_Data => GlobalComponentData);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.subunit =>
            -- ASSUME Last_Node = subunit
            Wf_Subunit (Node  => Last_Node,
                        Scope => Current_Scope);
            Next_Node := Child_Node (Current_Node => Last_Node);
            -- ASSUME Next_Node = parent_unit_name
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Next_Node) = SP_Symbols.parent_unit_name,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = parent_unit_name in CompUnit");
         when SP_Symbols.proof_function_declaration =>
            -- ASSUME Last_Node = proof_function_declaration
            --# accept Flow, 10, Unused, "Expected ineffective assignment to Unused";
            Wf_Proof_Function_Declaration
              (Node          => Last_Node,
               Current_Scope => Current_Scope,
               The_Heap      => TheHeap,
               Func_Sym      => Unused);
            --# end accept;
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.basic_declarative_item =>
            -- ASSUME Last_Node = basic_declarative_item
            Wf_Basic_Declarative_Item (Node           => Last_Node,
                                       Current_Scope  => Current_Scope,
                                       Component_Data => GlobalComponentData);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.renaming_declaration =>
            -- ASSUME Last_Node = renaming_declaration
            Wf_Renaming_Declaration (Node  => Last_Node,
                                     Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.proof_renaming_declaration =>
            -- ASSUME Last_Node = proof_renaming_declaration
            Wf_Proof_Renaming_Declaration (Node  => Last_Node,
                                           Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.use_type_clause =>
            -- ASSUME Last_Node = use_type_clause
            Wf_Use_Type_Clause (Node => Last_Node);
            Next_Node := STree.NullNode; -- nothing below here used
         when SP_Symbols.sequence_of_statements =>
            -- ASSUME Last_Node = sequence_of_statements
            Heap.ReportUsage (TheHeap);
            Heap.Initialize (TheHeap);
            RefList.Init (NodeTable);
            WalkStatements (Seq_Node       => Last_Node,
                            Scope          => Current_Scope,
                            Table          => NodeTable,
                            Component_Data => GlobalComponentData);
            ComponentManager.ReportUsage (GlobalComponentData);
            Next_Node := STree.NullNode;
         when SP_Symbols.package_initialization =>
            -- ASSUME Last_Node = package_initialization
            Wf_Package_Initialization (Node  => Last_Node,
                                       Scope => Current_Scope);
            ErrorHandler.Error_Reset;
            Next_Node := Child_Node (Last_Node);
            -- ASSUME Next_Node = sequence_of_statements OR hidden_part
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Next_Node) = SP_Symbols.sequence_of_statements
                 or else Syntax_Node_Type (Node => Next_Node) = SP_Symbols.hidden_part,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Node = sequence_of_statements OR hidden_part in CompUnit");
         when SP_Symbols.apragma =>
            -- ASSUME Last_Node = apragma
            Wf_Pragma (Node  => Last_Node,
                       Scope => Current_Scope);
            Next_Node := STree.NullNode;
         when SP_Symbols.code_insertion =>
            -- ASSUME Last_Node = code_insertion
            Wf_Machine_Code_Insertion (Node  => Last_Node,
                                       Scope => Current_Scope);
            Next_Node := STree.NullNode;
            -- following series of cases are places where we prune tree walk
            -- because there is nothing below node of use to us
         when SP_Symbols.with_clause           |
           SP_Symbols.code_statement        | -- probably can't reach this one any more
           SP_Symbols.context_clause_rep    |
           SP_Symbols.refinement_definition =>
            -- ASSUME Last_Node = with_clause OR code_statement OR context_clause_rep OR refinement_definition
            Next_Node := STree.NullNode;
            -- this tests for whether down loop end has been reached at a terminal
         when others =>
            if NodeType in SP_Symbols.SP_Non_Terminal then
               Next_Node := Child_Node (Current_Node => Last_Node);
            else
               Next_Node := STree.NullNode;
            end if;
      end case;

      if Next_Node = STree.NullNode then ------up loop----------
         loop
            --# assert STree.Table = STree.Table~;
            Next_Node := Next_Sibling (Current_Node => Last_Node);
            exit when Next_Node /= STree.NullNode;

            Next_Node := Parent_Node (Current_Node => Last_Node);
            exit when Next_Node = STree.NullNode;

            Last_Node := Next_Node;
            case Syntax_Node_Type (Node => Last_Node) is
               when SP_Symbols.package_body =>
                  -- ASSUME Last_Node = package_body
                  Up_Wf_Package_Body (Node  => Last_Node,
                                      Scope => Current_Scope);
                  ErrorHandler.End_Unit;
               when SP_Symbols.protected_body =>
                  -- ASSUME Last_Node = protected_body
                  Up_Wf_Protected_Body (Node  => Last_Node,
                                        Scope => Current_Scope);
                  ErrorHandler.End_Unit;
               when SP_Symbols.task_body =>
                  -- ASSUME Last_Node = task_body
                  Up_Wf_Task_Body (Node  => Last_Node,
                                   Scope => Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SP_Symbols.entry_body =>
                  -- ASSUME Last_Node = entry_body
                  Current_Scope := Dictionary.GetEnclosingScope (Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SP_Symbols.package_initialization =>
                  -- ASSUME Last_Node = package_initialization
                  Stmt_Node := Child_Node (Current_Node => Last_Node);
                  ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
                  if Syntax_Node_Type (Node => Stmt_Node) /= SP_Symbols.hidden_part
                    and then ErrStatus = ErrorHandler.No_Error then
                     --# accept Flow, 10, Unused_Data_Flow_Error_Flag,
                     --#        "Expected ineffective assignment to Unused_Data_Flow_Error_Flag";
                     FlowAnalyser.FlowAnalyse
                       (Dictionary.GetRegion (Current_Scope),
                        Stmt_Node,
                        Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                        GlobalComponentData,
                        TheHeap,
                        NodeTable,
                        Unused_Data_Flow_Error_Flag);
                     --# end accept;
                     ErrorHandler.Get_Error_Severity (Severity => ErrStatus);

                     if ErrStatus = ErrorHandler.No_Error then
                        ErrorHandler.Report_Success
                          (Position      => Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                           Subprog_Str   => LexTokenManager.Null_String,
                           ErrNum        => ErrorHandler.No_Error_Default);
                     end if;
                  end if;

               when SP_Symbols.subprogram_implementation =>
                  -- ASSUME Last_Node = subprogram_implementation
                  Stmt_Node := Child_Node (Current_Node => Last_Node);
                  while Stmt_Node /= STree.NullNode
                    and then Syntax_Node_Type (Node => Stmt_Node) /= SP_Symbols.sequence_of_statements
                  --# assert STree.Table = STree.Table~;
                  loop
                     Stmt_Node := Next_Sibling (Current_Node => Stmt_Node);
                  end loop;
                  if Stmt_Node /= STree.NullNode then
                     ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
                     Semantic_Error_In_Subprogram_Body  := not (ErrStatus = ErrorHandler.No_Error and then Stack.Top = False);
                     Data_Flow_Error_In_Subprogram_Body := False; -- default in case flow analyser not called below
                     if not Semantic_Error_In_Subprogram_Body then
                        -- don't do flow analysis in presence of semantic errors
                        FlowAnalyser.FlowAnalyse
                          (Dictionary.GetRegion (Current_Scope),
                           Stmt_Node,
                           Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                           GlobalComponentData,
                           TheHeap,
                           NodeTable,
                           -- to get
                           Data_Flow_Error_In_Subprogram_Body);

                        -- If flow=auto then issue a note informing the user which flow analysis mode was used.
                        -- If it's a function or a procedure with a derives, then info flow is used, otherwise
                        -- data flow is used. Don't need to worry about 83/95/2005 because flow=auto is not
                        -- allowed in 83 mode.
                        ErrNum_On_Success := ErrorHandler.No_Error_Default;
                        --# accept F, 41, "Stable expression OK here";
                        if CommandLineData.Content.Flow_Option = CommandLineData.Auto_Flow then
                           if (Dictionary.IsFunction (Dictionary.GetRegion (Current_Scope)) or else
                                 ((Dictionary.IsProcedure (Dictionary.GetRegion (Current_Scope)) or else
                                     Dictionary.IsTaskType (Dictionary.GetRegion (Current_Scope)) or else
                                     Dictionary.IsEntry (Dictionary.GetRegion (Current_Scope))) and then
                                    Dictionary.GetHasDerivesAnnotation (Dictionary.GetRegion (Current_Scope)))) then
                              -- information flow
                              ErrNum_On_Success := ErrorHandler.No_Error_Info_Flow;
                           else
                              -- data flow
                              ErrNum_On_Success := ErrorHandler.No_Error_Data_Flow;
                           end if;
                        end if;
                        --# end accept;

                        ErrorHandler.Get_Error_Severity (Severity => ErrStatus);

                        if ErrStatus = ErrorHandler.No_Error then
                           ErrorHandler.Report_Success
                             (Position      => Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                              Subprog_Str   => Dictionary.GetSimpleName (Dictionary.GetRegion (Current_Scope)),
                              ErrNum        => ErrNum_On_Success);
                        end if;
                        ErrorHandler.Flush_Echo_Messages;
                     end if;
                     -- Call VCG regardless of error status (used to be inside above IF statement).  Instead, we pass
                     -- the error status to the VCG which generates a false VC for subprograms
                     -- containing static semantic errors
                     VCG.Generate_VCs
                       (Start_Node                    => Stmt_Node,
                        Scope                         => Current_Scope,
                        Do_VCG                        => Do_VCG,
                        End_Position                  => Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                        Flow_Heap                     => TheHeap,
                        Semantic_Error_In_Subprogram  => Semantic_Error_In_Subprogram_Body,
                        Data_Flow_Error_In_Subprogram => Data_Flow_Error_In_Subprogram_Body,
                        Type_Check_Exports            => ErrorHandler.Possibly_Invalid_Values);
                  end if;

               when SP_Symbols.subprogram_body | SP_Symbols.not_overriding_subprogram_body =>
                  -- ASSUME Last_Node = subprogram_body OR not_overriding_subprogram_body
                  Up_Wf_Subprogram_Body (Node  => Last_Node,
                                         Scope => Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SP_Symbols.main_program_declaration =>
                  -- ASSUME Last_Node = main_program_declaration
                  if Dictionary.GetThePartition /= Dictionary.NullSymbol
                    and then Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Dictionary.GetThePartition) then
                     FlowAnalyser.FlowAnalysePartition (Last_Node, TheHeap);
                  end if;
               when others =>
                  null;
            end case;
         end loop;
      end if;
   end loop;

   Heap.ReportUsage (TheHeap);

   --# accept Flow, 33, Unused, "Expected Unused to be neither referenced nor exported" &
   --#        Flow, 33, Unused_Data_Flow_Error_Flag, "Expected Unused_Data_Flow_Error_Flag to be neither referenced nor exported";
end CompUnit;
