-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
function LookupSelectedItem
  (Prefix   : Symbol;
   Selector : LexTokenManager.Lex_String;
   Scope    : Scopes;
   Context  : Contexts)
  return     Symbol
is
   Item : Symbol;

   --------------------------------------------------------------------------------

   function LookupSelectedItemInPackage
     (ThePackage : Symbol;
      Name       : LexTokenManager.Lex_String;
      Scope      : Scopes;
      Context    : Contexts)
     return       Symbol
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in LexTokenManager.State;
   is
      Region, Item : Symbol;
      LibPackage   : Symbol;
      IsVisible    : Boolean;

      ------------------------------------------------------------------------------

      procedure LookupGlobalVariables
        (Abstraction : in     Abstractions;
         ThePackage  : in     Symbol;
         Name        : in     LexTokenManager.Lex_String;
         Subprogram  : in     Symbol;
         Context     : in     Contexts;
         Variable    :    out Symbol;
         IsVisible   :    out Boolean)
      --# global in Dict;
      --#        in LexTokenManager.State;
      --#        in Scope;
      --# derives IsVisible from Abstraction,
      --#                        Context,
      --#                        Dict,
      --#                        LexTokenManager.State,
      --#                        Name,
      --#                        Scope,
      --#                        Subprogram,
      --#                        ThePackage &
      --#         Variable  from Abstraction,
      --#                        Dict,
      --#                        LexTokenManager.State,
      --#                        Name,
      --#                        Subprogram,
      --#                        ThePackage;
      is
         GlobalVariable, TheVariable : Symbol;

         function FirstGlobal return Symbol
         --# global in Abstraction;
         --#        in Dict;
         --#        in Subprogram;
         is
            Result : Symbol;
         begin
            case RawDict.GetSymbolDiscriminant (Subprogram) is
               when SubprogramSymbol =>
                  Result := RawDict.GetSubprogramFirstGlobalVariable (Abstraction, Subprogram);
               when TypeSymbol => -- must be a task type
                  Result := RawDict.GetTaskTypeFirstGlobalVariable (Abstraction, Subprogram);
               when others =>
                  Result := NullSymbol;
            end case;
            return Result;
         end FirstGlobal;

         -- Two function lifted out to simplify Boolean expression in main procedure
         function IsDirectlyVisible (TheVariable, ThePackage : Symbol) return Boolean
         --# global in Dict;
         is
         begin
            return GetScope (TheVariable) = VisibleScope (ThePackage);
         end IsDirectlyVisible;

         function IsVisibleToAChild (TheVariable, ThePackage, TheLibraryPackage : Symbol) return Boolean
         --# global in Dict;
         is
         begin
            return GetScope (TheVariable) = PrivateScope (ThePackage)
              and then IsProperDescendent (TheLibraryPackage, ThePackage);
         end IsVisibleToAChild;

      begin --LookupGlobalVariables

         GlobalVariable := FirstGlobal;
         loop
            if GlobalVariable = NullSymbol then
               Variable  := NullSymbol;
               IsVisible := False;
               exit;
            end if;
            TheVariable := RawDict.GetGlobalVariableVariable (GlobalVariable);
            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (TheVariable),
                                                                    Lex_Str2 => Name) =
              LexTokenManager.Str_Eq
              and then RawDict.GetGlobalVariablePrefixNeeded (GlobalVariable)
              and then GetOwner (TheVariable) = ThePackage then
               Variable := TheVariable;
               if Context = ProofContext then
                  IsVisible := True;
               elsif IsDeclared (TheVariable) then
                  -- pna -- original two lines replaced by
                  --IsVisible := GetScope (TheVariable) = VisibleScope (ThePackage) or else
                  --  IsLocal (Scope, LocalScope (ThePackage));
                  -- pna -- these three ...
                  IsVisible := IsDirectlyVisible (TheVariable, ThePackage)
                    or else IsVisibleToAChild (TheVariable, ThePackage, GetLibraryPackage (Scope))
                    or else IsLocal (Scope, LocalScope (ThePackage));
               else
                  IsVisible := False;
               end if;
               exit;
            end if;
            GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
         end loop;

      end LookupGlobalVariables;

      ------------------------------------------------------------------------------

      function LookUpChildren
        (Sort       : PackageSort;
         ThePackage : Symbol;
         Name       : LexTokenManager.Lex_String;
         Scope      : Scopes;
         Context    : Contexts)
        return       Symbol
      --# global in Dict;
      --#        in LexTokenManager.State;
      is
         Result         : Symbol := NullSymbol;
         CurrentPackage : Symbol;

         -----------------------------------------

         function CheckIsInherited (PackSym : Symbol;
                                    Scope   : Scopes) return Symbol
         --# global in Dict;
         is
            CurrentScope : Scopes;
            Region       : Symbol;
            Result       : Symbol;

         begin --IsInherited
            CurrentScope := Scope;
            loop
               Region := GetRegion (CurrentScope);
               exit when IsPackage (Region);
               exit when IsMainProgram (Region);
               CurrentScope := GetEnclosingScope (CurrentScope);
            end loop;

            -- *NB* the inheritance check also must succeed in the case where
            --      a private child package is with'd by its parent; since
            --      the context required for this check is not available here,
            --      this case is handled during wf checking of the with clause
            --      by insertion of a 'fake' inherit reference.
            if IsInherited (PackSym, Region) then
               Result := PackSym;
            else
               Result := NullSymbol;
            end if;
            return Result;
         end CheckIsInherited;

         ----------------

         function CheckIsWithed
           (PackSym : Symbol;
            Scope   : Scopes;
            Context : Contexts)
           return    Symbol
         --# global in Dict;
         is
            Current  : Scopes;
            Last1    : Scopes;
            Ancestor : Symbol;
            Result   : Symbol;
         begin
            if Context = ProofContext then
               Result := PackSym;
            else
               Current := Scope;
               Last1   := Current;
               loop
                  exit when GetRegion (Current) = GetPredefinedPackageStandard;
                  exit when IsWithed (PackSym, Current);
                  Last1   := Current;
                  Current := GetEnclosingScope (Current);
               end loop;

               if GetRegion (Current) = GetPredefinedPackageStandard then
                  Result := NullSymbol;
                  if Last1 /= Current and then IsPackage (GetRegion (Last1)) then -- search through ancestors
                     Ancestor := RawDict.GetPackageParent (GetRegion (Last1));
                     loop
                        exit when Ancestor = NullSymbol;
                        exit when IsWithed (PackSym, VisibleScope (Ancestor));
                        Ancestor := RawDict.GetPackageParent (Ancestor);
                     end loop;
                     if Ancestor /= NullSymbol then
                        Result := PackSym;
                     end if;
                  end if;
               else
                  Result := PackSym;
               end if;
            end if;
            return Result;
         end CheckIsWithed;

         -----------------

      begin --LookUpChildren
         CurrentPackage := RawDict.GetPackageFirstChild (ThePackage, Sort);
         loop
            exit when CurrentPackage = NullSymbol;

            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (CurrentPackage),
                                                                    Lex_Str2 => Name) =
              LexTokenManager.Str_Eq then
               Result := CurrentPackage;
               exit;
            end if;

            CurrentPackage := RawDict.GetPackageSibling (CurrentPackage);
         end loop;

         if not IsGlobalScope (Scope) then
            Result := CheckIsInherited (Result, Scope);
         end if;

         if Result /= NullSymbol then
            Result := CheckIsWithed (Result, Scope, Context);
         end if;

         return Result;
      end LookUpChildren;

      ------------------------------------------------------------------------------

      -- CFR 806 Adds this function.  Needed so scan for enclosing region
      -- ignores enclosing loop statements.  SEPRS 889 and 1083
      function GetEnclosingNonLoopRegion (Scope : Scopes) return Symbol
      --# global in Dict;
      is
         CurrentScope : Scopes;
         Region       : Symbol;
      begin
         CurrentScope := Scope;
         loop
            Region := GetRegion (CurrentScope);
            exit when RawDict.GetSymbolDiscriminant (Region) /= LoopSymbol;
            CurrentScope := GetEnclosingScope (CurrentScope);
         end loop;
         return Region;
      end GetEnclosingNonLoopRegion;

      ------------------------------------------------------------------------------

   begin
      Trace_Lex_Str (Msg => "In LookUpSelectedItemInPackage seeking ",
                     L   => Name);

      Region := GetEnclosingNonLoopRegion (Scope);
      if IsSubprogram (Region) or else IsTaskType (Region) then
         LookupGlobalVariables (GetAbstraction (Region, Scope), ThePackage, Name, Region, Context, Item, IsVisible);
      else
         Item      := NullSymbol;
         IsVisible := False;
      end if;

      --# assert True;
      if Item = NullSymbol then
         LookupScope (Name, LexTokenManager.Null_String, VisibleScope (ThePackage), Scope, Context, Item, IsVisible);

         if Item = NullSymbol then
            LibPackage := GetLibraryPackage (Scope);
            if IsProperDescendent (LibPackage, ThePackage) then
               if Scope /= VisibleScope (LibPackage) or else DescendentIsPrivate (LibPackage, ThePackage) then
                  LookupScope (Name, LexTokenManager.Null_String, PrivateScope (ThePackage), Scope, Context, Item, IsVisible);
               end if;
            elsif IsLocal (Scope, LocalScope (ThePackage)) then
               Item      := LookupImmediateScope (Name, LocalScope (ThePackage), Context);
               IsVisible := Item /= NullSymbol;
            end if;
         elsif Context = ProofContext and then IsOwnVariable (Item) then
            LibPackage := GetLibraryPackage (Scope);
            if IsDescendentOfPrivateChild (LibPackage, ThePackage) then

               Item      := LookupImmediateScope (Name, PrivateScope (ThePackage), ProgramContext);
               IsVisible := Item /= NullSymbol;
            end if;
         end if;

         --# assert True;
         if IsVariable (Item) and then (IsSubprogram (Region) or else IsTaskType (Region)) then
            -- Region is equal to Scope.Region if not within a loop (see start of subprog)
            --   and second arg is equivalent to Scope
            -- otherwise it is nearest enclosing non-loop region (subprogram)
            IsVisible := IsLocal (VisibleScope (ThePackage), Scopes'(Scope.TypeOfScope, Region));
         elsif IsSubprogram (Item) then
            IsVisible := not IsRenamed (Item, Scope);
         end if;

      end if;

      --# assert True;
      if not IsVisible then
         Item := NullSymbol;
      end if;

      --# assert True;
      if Item = NullSymbol then

         case CommandLineData.Content.Language_Profile is
            when CommandLineData.SPARK83 =>
               null;

            when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

               -- look up children, if relevant
               LibPackage := GetLibraryPackage (Scope);
               if not IsProperDescendent (LibPackage, ThePackage) then
                  Item := LookUpChildren (Public, ThePackage, Name, Scope, Context);
                  if Item = NullSymbol and then IsGlobalScope (Scope) then
                     Item := LookUpChildren (PrivateChild, ThePackage, Name, Scope, Context);
                  end if;
               end if;
         end case;
      end if;

      return Item;

   end LookupSelectedItemInPackage;

   --------------------------------------------------------------------------------

   function LookupSelectedItemInTypeMark
     (TypeMark : Symbol;
      Name     : LexTokenManager.Lex_String;
      Context  : Contexts)
     return     Symbol
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
   begin
      Trace_Lex_Str (Msg => "In LookUpSelectedItemInTypeMark seeking ",
                     L   => Name);

      -- The GetRootType in the following statement is needed to handle
      -- lookup of items of protected subtypes and record subtypes.
      return LookupImmediateScope (Name, LocalScope (GetRootType (TypeMark)), Context);
   end LookupSelectedItemInTypeMark;

   --------------------------------------------------------------------------------

   function LookupSelectedItemInObject
     (Object  : Symbol;
      Name    : LexTokenManager.Lex_String;
      Context : Contexts)
     return    Symbol
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Result : Symbol;

      function LookupProtectedOperation (OpName           : LexTokenManager.Lex_String;
                                         TheProtectedType : Symbol) return Symbol
      --# global in Dict;
      --#        in LexTokenManager.State;
      is
         It     : Iterator;
         Result : Symbol := NullSymbol; -- default answer for failure case
      begin
         It := FirstVisibleSubprogram (TheProtectedType);
         while not IsNullIterator (It) loop
            if LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => GetSimpleName (CurrentSymbol (It)),
               Lex_Str2 => OpName) =
              LexTokenManager.Str_Eq then
               -- success
               Result := CurrentSymbol (It);
               exit;
            end if;
            It := NextSymbol (It);
         end loop;
         return Result;
      end LookupProtectedOperation;

   begin -- LookupSelectedItemInObject
      Trace_Lex_Str (Msg => "In LookUpSelectedItemInObject seeking ",
                     L   => Name);
      if IsConstant (Object) then
         Result := LookupSelectedItemInTypeMark (GetType (Object), Name, Context);
      else
         -- Must be a variable
         -- We need to distinguish between a record object where we will seek a field
         -- in local scope of the object and a protected object where we will need to
         -- seek an operation in the visible part of the type of the protected object
         if TypeIsRecord (GetType (Object)) then
            Result := LookupImmediateScope (Name, LocalScope (Object), Context);
         elsif TypeIsProtected (GetType (Object)) then
            Result := LookupProtectedOperation (Name, GetRootType (GetType (Object)));
         else
            Result := NullSymbol; -- to avoid DF error
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Symbol_Table,
               Msg     => "LookupSelectedItemInObject called with prefix which is not record or protected");
         end if;
      end if;
      return Result;
   end LookupSelectedItemInObject;

   --------------------------------------------------------------------------------

begin
   Trace_Lex_Str (Msg => "In LookUpSelectedItem seeking ",
                  L   => Selector);
   Trace_Sym (Msg   => "   in ",
              Sym   => Prefix,
              Scope => Scope);

   case RawDict.GetSymbolDiscriminant (Prefix) is
      when PackageSymbol =>
         Item := LookupSelectedItemInPackage (Prefix, Selector, Scope, Context);
      when TypeSymbol =>
         Item := LookupSelectedItemInTypeMark (Prefix, Selector, Context);
      when VariableSymbol | ConstantSymbol | SubcomponentSymbol | -- record object field may also have fields
        SubprogramParameterSymbol =>
         Item := LookupSelectedItemInObject (Prefix, Selector, Context);
      when SubprogramSymbol =>
         Item := LookupImmediateScope (Selector, LocalScope (Prefix), Context);
      when others =>
         -- following debug statements can be used to diagnose cause if FatalError called
         --Debug.PrintMsg ("GetSymbolDiscriminant (Prefix) is", False);
         --Debug.PrintMsg (SymbolDiscriminant'Image (RawDict.GetSymbolDiscriminant (Prefix)), True);
         Item := NullSymbol; -- to avoid DF error
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Symbol_Table,
            Msg     => "Attempt to LookUpSelectedItem with invalid prefix");
   end case;

   return Item;

end LookupSelectedItem;
