/*
** Copyright (C) 2001 Jeff Nathan <jeff@wwti.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**                
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spp_arpspoof.c,v 1.9 2001/11/19 21:37:54 cazz Exp $ */
/* Snort ARPspoof Preprocessor Plugin
*   by Jeff Nathan <jeff@wwti.com>
*   Version 0.1.1
*
* Purpose:
*
* This preprocessor looks for anomalies in ARP traffic and attempts to 
* maliciously overwrite  ARP cache information on hosts.
*
* Arguments:
*
* To check for unicast ARP requests use:
* arpspoof: -unicast
*
* WARNING: this can generate false positives as Linux systems send unicast 
* ARP requests repetatively for entries in their cache.
*
* This plugin also takes a list of IP addresses and MAC address in the form:
* arpspoof_detect_host: 10.10.10.10 29:a2:9a:29:a2:9a
* arpspoof_detect_host: 192.168.40.1 f0:0f:00:f0:0f:00
* and so forth...
*
* Effect:
* By comparing information in the Ethernet header to the ARP frame, obvious
* anomalies are detected.  Also, utilizing a user supplied list of IP addresses 
* and MAC addresses, ARP traffic appearing to have originated from any IP in 
* that list is carefully examined by comparing the source hardware address to 
* the user supplied hardware address.  If there is a mismatch, an alert is 
* generated as either an ARP request or REPLY can be used to overwrite cache 
* information on a remote host.  This should only be used for hosts/devices
* on the **same layer 2 segment** !!
*
* Bugs:
* This is a proof of concept ONLY.  It is clearly not complete.  Also, the 
* lookup function LookupIPMacEntryByIP is in need of optimization.  The
* arpspoof_detect_host functionality may false alarm in redundant environments. * Also, see the comment above pertaining to Linux systems.
*
* Thanks:
*
* First and foremost Patrick Mullen who sat beside me and helped every step of
* the way.  Andrew Baker for graciously supplying the tougher parts of this 
* code.  W. Richard Stevens for readable documentation, Kevin DePeugh for 
* his incredible depth of knowledge and dedication to research. And finally 
* Marty for being a badass.  All your packets are belong to Marty.
*
*/

#define MODNAME "spp_arpspoof"

#define WITHUNICAST "-unicast"

int check_unicast_arp = 0;

#include "spp_arpspoof.h"

#include <stdio.h>
#ifndef WIN32
    #include <sys/time.h>
#else
    #include <time.h>
#endif

struct spp_timeval
{
    time_t tv_sec;
    time_t tv_usec;
};

struct spp_timeval maxTime;

/* external globals from rules.c */
extern char *file_name;
extern int file_line;

/* from tcpdump source (where ARP frames are fully evaluated) */
u_int8_t bcast[6];

static IPMacEntryList *ipmel;

int AddIPMacEntryToList(IPMacEntryList *ip_mac_entry_list, IPMacEntry *ip_mac_entry)
{
    IPMacEntryListNode *newNode;

    if(ip_mac_entry == NULL || ipmel == NULL)
        return 1;

    newNode = (IPMacEntryListNode *)malloc(sizeof(IPMacEntryListNode));
    /* XXX: OOM Check */
    newNode->ip_mac_entry = ip_mac_entry;
    newNode->next = NULL;

    if(ipmel->head == NULL)
    {
        ipmel->head = newNode;
        ipmel->size = 1;
    }
    else
    {
        ipmel->tail->next = newNode;
        ipmel->size += 1;
    }
    ipmel->tail = newNode;
    return 0;
}


IPMacEntry *LookupIPMacEntryByIP(IPMacEntryList *ipmel, u_int32_t ipv4_addr)
{
    IPMacEntryListNode *current;

    if(ipmel == NULL)
        return NULL;

    for(current = ipmel->head; current != NULL; current = current->next)
    {
        if(current->ip_mac_entry->ipv4_addr == ipv4_addr)
            return current->ip_mac_entry;
    }
    return NULL;
}

void FreeIPMacEntryList(IPMacEntryList *ipmel)
{
    IPMacEntryListNode *prev;
    IPMacEntryListNode *current;

    if(ipmel == NULL)
        return;

    current = ipmel->head;
    while(current != NULL)
    {
        if(current->ip_mac_entry != NULL)
            free(current->ip_mac_entry);

        prev = current;
        current = current->next;
        free(prev);
    }
    ipmel->head = NULL;
    ipmel->size = 0;
}

void SetupARPspoof()
{
    RegisterPreprocessor("arpspoof", ARPspoofInit);
    RegisterPreprocessor("arpspoof_detect_host", ARPwatchArg);

    DebugMessage(DEBUG_INIT, "Preprocessor: ARPspoof is setup...\n");
}

void ParseARPspoofArgs(char *args)
{
    char **toks;
    int num_toks;
    int num;

    if (!args) return;
    toks = mSplit(args, " ", 2, &num_toks, '\\');

        if(num_toks > 1)
        {      
            FatalError(MODNAME ": ERROR: %s (%d) => ARPspoof configuration format: seconds \n", file_name, file_line);
        } 

    for(num = 0; num < num_toks; num++)
    {
        if(!strncasecmp(WITHUNICAST, toks[num], sizeof WITHUNICAST))
            check_unicast_arp = 1;
    }

/*        maxTime.tv_sec = atoi(toks[0]);
        maxTime.tv_usec = 0; */
}

void ARPspoofPreprocFunction(Packet *p)
{
    Event event;
 /*   char logMessage[180];  */
    IPMacEntry *ipme;
    u_int8_t addr[4];

    bzero(addr, sizeof(u_int8_t) * 4);
    if(p && (p->eh != NULL && p->ah != NULL))
    {
        memset(bcast, 0xff, 6);

        if ((ntohs(p->ah->ea_hdr.ar_hrd) != 0x0001) || 
           (ntohs(p->ah->ea_hdr.ar_pro) != ETHERNET_TYPE_IP))
        {
            /* hardware type is not Ethernet or protocol type is not IP */
            return;
        }
        switch(ntohs(p->ah->ea_hdr.ar_op))
        {
            case ARPOP_REQUEST:
                if (check_unicast_arp) 
                {
                    if (memcmp((u_char *)p->eh->ether_dst, 
                       (u_char *)bcast, 6) != 0)
                    {
                        SetEvent(&event, GENERATOR_SPP_ARPSPOOF, 
                                 ARPSPOOF_UNICAST_ARP_REQUEST, 1, 0, 0, 0);

                        CallAlertFuncs(NULL ,"unicast ARP request" , 
                                       NULL, &event);

                        #ifdef DEBUG
                        fprintf(stderr, "unicast ARP request\n");
                        #endif
                    }
                }
                else if (memcmp((u_char *)p->eh->ether_src, 
                        (u_char *)p->ah->arp_sha, 6) != 0) 
                {
                    SetEvent(&event, GENERATOR_SPP_ARPSPOOF, 
                             ARPSPOOF_ETHERFRAME_ARP_MISMATCH_SRC, 1, 0, 0, 0);

                    CallAlertFuncs(p, "Ethernet source/ARP sender address mismatch", NULL, &event);
                    #ifdef DEBUG
                    fprintf(stderr, "Ethernet/ARP mismatch request\n");
                    #endif
                }
                break;
            case ARPOP_REPLY:
                if (memcmp((u_char *)p->eh->ether_src, 
                   (u_char *)p->ah->arp_sha, 6) != 0)
                {
                    SetEvent(&event, GENERATOR_SPP_ARPSPOOF, 
                             ARPSPOOF_ETHERFRAME_ARP_MISMATCH_SRC, 1, 0, 0, 0);

                    CallAlertFuncs(p, "Ethernet source/ARP sender address mismatch", NULL, &event);
                    #ifdef DEBUG
                    fprintf(stderr, "Ethernet/ARP mismatch reply 1\n");
                    #endif
                }
                else if (memcmp((u_char *)p->eh->ether_dst, 
                        (u_char *)p->ah->arp_tha, 6) != 0)
                {
                    SetEvent(&event, GENERATOR_SPP_ARPSPOOF, 
                             ARPSPOOF_ETHERFRAME_ARP_MISMATCH_DST, 1, 0, 0, 0);

                    CallAlertFuncs(p, "Ethernet destination/ARP target address mismatch", NULL, &event);
                    #ifdef DEBUG
                    fprintf(stderr, "Ethernet/ARP mismatch reply 2\n");
                    #endif
                }
                break;
        }
        /* LookupIPMacEntryByIP() is too slow, will be fixed later */
        bcopy((void *)&p->ah->arp_spa, (void *)addr, sizeof(u_int8_t) * 4);
        if ((ipme = LookupIPMacEntryByIP(ipmel, *addr)) == NULL)
        {
            #ifdef DEBUG
            fprintf(stderr, "ipme was NULL\n");
            #endif
            return;
        }
        else
        {
            #ifdef DEBUG
            fprintf(stderr, "ipme not NULL\n");
            #endif
            if ((memcmp((u_int8_t *)p->eh->ether_src, 
               (u_int8_t *)ipme->mac_addr, 6) != 0) || 
               (memcmp((u_int8_t *)p->ah->arp_sha, 
               (u_int8_t *)ipme->mac_addr, 6) != 0))
            {
                SetEvent(&event, GENERATOR_SPP_ARPSPOOF, 
                         ARPSPOOF_ARP_CACHE_OVERWRITE_ATTACK, 1, 0, 0, 0);

                CallAlertFuncs(p, "Attempted ARP cache overwrite attack", 
                               NULL, &event);

                #ifdef DEBUG
                fprintf(stderr, "Attempted ARP cache overwrite attack\n");
                #endif
            }
       } 
    }
    else
    {
        /* p, eh or ah was NULL, either way it's a non-happy packet */
        return;
    }
}

void ARPspoofCleanExitFunction(int signal)
{
    FreeIPMacEntryList(ipmel);
    free(ipmel);
    ipmel = NULL;
}

void ARPspoofRestartFunction(int signal)
{
       /* restart code goes here */
}

void ARPspoofInit(u_char *args)
{
    DebugMessage(DEBUG_INIT, "Preprocessor: ARPspoof Initialized\n");

    /* parse the argument list from the rules file */
    ParseARPspoofArgs(args);
    AddFuncToPreprocList(ARPspoofPreprocFunction);

    /* Set the preprocessor function into the function list
    AddFuncToCleanExitList(ARPspoofCleanExitFunction);
    AddFuncToRestartList(ARPspoofRestartFunction); */

    ipmel = (IPMacEntryList *)malloc(sizeof(IPMacEntryList));
    bzero(ipmel, sizeof(IPMacEntryList));
}

void PrintIPMacEntryList(IPMacEntryList *ipmel)
{
    IPMacEntryListNode *current;
    int i;
    struct in_addr in;
    if(ipmel == NULL)
        return;

    current = ipmel->head;
    printf("IPMacEntry List\n");
    printf("  Size: %i\n", ipmel->size);
    while(current != NULL)
    {
        in.s_addr = current->ip_mac_entry->ipv4_addr;
        printf("%s -> ", inet_ntoa(in));
        for(i=0;i<6;++i)
        {
            printf("%02x", current->ip_mac_entry->mac_addr[i]);
            if(i != 5)
                printf(":");
        }
        printf("\n");
        current = current->next;
    }    
}


void ARPwatchArg(u_char *args)
{
    char **toks;
    int num_toks;
    int i;
    u_int32_t addr_tmp[6];
    struct in_addr IP_struct;
    IPMacEntry *ipme = NULL;

    if (ipmel == NULL)
    {
        fprintf(stderr, "Cannot initialize"
                " arpspoof_detect_host without arpspoof\n");
        return;
    }

    toks = mSplit(args, " ", 2, &num_toks, '\\');

    if (num_toks != 2)
    {
        fprintf(stderr, "arpspoof_detect_host: ERROR: invalid arguments,"
                " continuing\n");
        for(i=0;i<num_toks;i++)
            free(toks[i]);
        return;
    }

    /* Add entries */
    ipme = (IPMacEntry *)malloc(sizeof(IPMacEntry));
    /* XXX: OOM Check */
    bzero(ipme, sizeof(IPMacEntry));


    if ((IP_struct.s_addr = inet_addr(toks[0])) == -1)
    {
        fprintf(stderr, "arpspoof_detect_host: ERROR: non IP as first argument"
                " of IP/MAC pair\n");
        for(i=0;i<num_toks;i++)
            free(toks[i]);
        return;
    }

    ipme->ipv4_addr = (u_int32_t)IP_struct.s_addr;

    sscanf(toks[1], "%02X:%02X:%02X:%02X:%02X:%02X", &addr_tmp[0], 
           &addr_tmp[1], &addr_tmp[2], &addr_tmp[3], &addr_tmp[4], 
           &addr_tmp[5]);

    for(i=0;i<6;i++)
        ipme->mac_addr[i] = (u_int8_t)addr_tmp[i];

    AddIPMacEntryToList(ipmel, ipme);
#ifdef DEBUG
    PrintIPMacEntryList(ipmel);
#endif
}
