/*
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id$ */


#include "spo_snortnet.h"

/* my globals */

unsigned long SnortNetMaster;
unsigned long NodeID;
unsigned char *SnortNetName;
unsigned char *SnortNetDev;
unsigned int SnortNetPort;
unsigned int UseSsl;
int MasterSD;


//#define IAPDEBUG
#ifdef IAPDEBUG
extern FILE *iaplog;
#endif

/****************************************************************************
 * Function: SetupSnortNet()
 * Purpose: perform initial setup for SnortNet routines.
 * Arguments: none
 * Retuns: void
 ***************************************************************************/



void SetupSnortNet() {
	SetupAlertSnortNet();
	SetupLogSnortNet();
#ifdef IAPDEBUG
	iaplog = fopen("./iaplog","a");
#endif
}

/****************************************************************************
 * Function: SetupAlertSnortNet()
 * Purpose:  take care of AlertSnortNet part.
 * Arguments: none
 * Retuns: void
 ***************************************************************************/

void SetupAlertSnortNet() {

    RegisterOutputPlugin("alert_snortnet",NT_OUTPUT_ALERT,AlertSnortNetInit);

}


/****************************************************************************
 * Function: SetupLogSnortNet()
 * Purpose: take care of LogSnortNet part.
 * Arguments: none
 * Retuns: void
 ***************************************************************************/

void SetupLogSnortNet() {

    RegisterOutputPlugin("log_snortnet",NT_OUTPUT_LOG,LogSnortNetInit);
}



/****************************************************************************
 * Function: AlertSnortNetInit()
 * Purpose: Initialize AlertSnortNetInit stuff..
 * Arguments: none
 * Retuns: void
 ***************************************************************************/

void AlertSnortNetInit(u_char *args) {
#ifdef DEBUG
	fprintf(stderr,"AlertSnortNet activated (%s)\n",args);
#endif
	pv.log_plugin_active = 1;
	if (ParseSnortNetArgs(args) == -1) 
	   FatalError("AlertSnortNet plugin deactivated. Errors in params\n");
        if (SnortNetName) {  
	   if (MasterConnect(SnortNetMaster, UseSsl)<0) {
		   PrintError("connect");
		   FatalError("Unable to Connect to MasterServer\n");
	   }
	} else {
		/* should initialize file here */
	}

	/* if we get `broken pipe' probably means that mastersite went offline
	 * and we need to reconnect */ 
	signal(SIGPIPE,MasterReconnect);
        AddFuncToOutputList(AlertSnortNet, NT_OUTPUT_ALERT, NULL);
}



/****************************************************************************
 * Function: LogSnortNetInit()
 * Purpose: initialize LogSnortNet routines.
 * Arguments: none
 * Retuns: void
 ***************************************************************************/

void LogSnortNetInit(u_char *args) {
#ifdef DEBUG
fprintf(stderr,"LogSnortNet activated (%s)\n",args);
#endif
}


/****************************************************************************
 * Function: AlertSnortNet()
 * Purpose: Perform logging of Alerts
 * Arguments:  Packet p structure, message
 * Retuns: void
 ***************************************************************************/


void AlertSnortNet(Packet *p, char *msg, void *arg) {

	NetAlert na;
	unsigned long int msgsize;
	
	bzero(&na,sizeof(na));
	
	if (p->iph == NULL) { /* we can not monitor eth/tr/fddi specic attacks
		               * yet */
#ifdef DEBUG	
		printf("IP header - NULL. Ignored\n");
#endif
		return;
	}
	na.sensor_id = NodeID; /* not implemented */
	na.protocol = p->iph->ip_proto;
	na.src_ip = p->iph->ip_src.s_addr;
	na.dst_ip = p->iph->ip_dst.s_addr;
	switch (na.protocol) {
		case IPPROTO_TCP:
			na.sn_src_port = p->tcph->th_sport;
			na.sn_dst_port = p->tcph->th_dport;
			break;
		case IPPROTO_UDP:
			na.sn_src_port = p->udph->uh_sport;
			na.sn_dst_port = p->udph->uh_dport;
			break;
		case IPPROTO_ICMP:
			na.sn_icmp_type = p->icmph->type;
			na.sn_icmp_code = p->icmph->code;
			break;
               default:
			na.sn_src_port = -1;
			na.sn_dst_port = -1;
	        /* portless proto ;-) */
	} /* switch */
		 
	
	na.priority = ALERT_PRIORITY_HIGH;
	na.timestamp = p->pkth->ts;
	na.pkt_size = p->pkth->len;	
	if (msg!=NULL) 
		bcopy(msg,na.msg,strlen(msg)>STD_BUF?STD_BUF:strlen(msg));

	//msgsize = (sizeof(na) - (STD_BUF-strlen(na.msg)))/ 0xff;
	msgsize = (sizeof(na))/ 0xff;
	//if ((sizeof(na) - (STD_BUF - strlen(na.msg))) % 0xff)
	if ((sizeof(na)) % 0xff)
		msgsize++;
	fprintf(stderr,"message %s\n",msg);

	if(iap_content_send(MasterSD,(void *)&na,msgsize) < 1 ) {
		PrintError("snortnet write");
		/* we don't die? then we reconnect ;-) */
		MasterReconnect(1);
	}
	if (iap_get_response(MasterSD) != IAP_OK) {
		PrintError("IAP content send");
		MasterReconnect(1);
	}



}



/****************************************************************************
 * Function: LogSnortNet()
 * Purpose: Perform logging of Log
 * Arguments:  Packet p structure, message
 * Retuns: void
 ***************************************************************************/

void LogSnortNet(Packet *p, char *msg, void *arg) {

}



/****************************************************************************
 * Function: ParseSnortNetArgs()
 * Purpose: parse arguments
 * Arguments: char passed arguments
 * Retuns: void
 ***************************************************************************/

int ParseSnortNetArgs(char *args) {
    char *tmp, *tmp2;
    char *ssl;

    if (args == NULL) return -1;


    SnortNetName = NULL;
    SnortNetDev = NULL;
    UseSsl = 0;
    tmp=strtok(args, ",");
    while(tmp != NULL) {
	    /* get rid off leading spaces */
	    while(*tmp == ' ' || *tmp == '\t') tmp++;
	    if (!strncasecmp(tmp,"master",6)) {
		    if (SnortNetDev != NULL) return -1;
		    if((SnortNetName = index(tmp,'=')) == NULL) 
			    return -1;
		    SnortNetName++;
		    SnortNetMaster = ResolveHostname(SnortNetName);
		    if (SnortNetMaster == -1) return -1;

	    } else
	    
	    if (!strncasecmp(tmp,"file",4)) {
		    if (SnortNetName != NULL) return -1;
		    if((SnortNetDev = index(tmp,'=')) == NULL) 
			    return -1;
		    SnortNetDev++;
		    
	    } else

	    if (!strncasecmp(tmp,"id",2)) {
		    if((tmp2 = index(tmp,'=')) == NULL) 
			    return -1;
		    tmp2++;
		    NodeID = strtoul(tmp2,NULL,0);
		    
	    } else
	    
	    if (!strncasecmp(tmp,"ssl",3)) {
		    if((ssl = index(tmp,'=')) == NULL) 
			    return -1;
		    ssl++;
		    if (!strncasecmp(ssl,"y",1)) UseSsl = 1;
		    else UseSsl = 0;
	    } else {
		    ErrorMessage("ERROR: Unknown parameter: %s\n", tmp);
		    return -1;
	    }
            tmp = strtok(NULL,",");

    } /* while */
	    

    
return 0;    
}



/****************************************************************************
 * Function: ResolveHostname()
 * Purpose: Resolve given hostname or string into IP address.
 * Arguments: hostname
 * Retuns: unsigned long
 ***************************************************************************/

unsigned long ResolveHostname(char *name) {
	
	struct hostent *hostinfo;
	unsigned long addr;
	
        /* get the hostname and fill in the host_info struct */
        if ((hostinfo = gethostbyname(name)))
        {
            bcopy(hostinfo->h_addr, (char *)&addr, hostinfo->h_length);
        }
        else if ((addr = inet_addr(name)) == INADDR_NONE)
        {
            ErrorMessage("ERROR: Couldn't resolve hostname %s\n", 
                       name);
	    return -1;
        }

        return addr;
    }

int MasterConnect(unsigned long remote, int ssl) {

	struct sockaddr_in srv;
	
	if (ssl) return -1; /* not implemented yet */
	bzero(&srv, sizeof(srv));
	srv.sin_addr.s_addr = remote;

/* temporally */
#define MasterPort 1033
	srv.sin_port = htons(MasterPort);
	srv.sin_family = AF_INET;
	if ((MasterSD = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		PrintError("socket");
		return -1;
	}

	if (connect(MasterSD,(struct sockaddr *)&srv, sizeof(srv)) < 0) {
		PrintError("connect");
		return -1;
	}
	fprintf(stderr,"Connected!!\n");
	iap_connect_request(MasterSD, SnortNetName);
	if (iap_get_response(MasterSD) != IAP_OK) {
		PrintError("IAP protocol error on connect");
		return -1;
	}
	iap_upgrade_request(MasterSD);
	if (iap_get_response(MasterSD) != IAP_OK) {
		PrintError("IAP protocol error on upgrade");
		return -1;
	}
	iap_version_verify(MasterSD, IAPROLE_SENDER);
	if (iap_get_response(MasterSD) != IAP_OK) {
		PrintError("IAP protocol error on version verify");
		return -1;
	}
	
	return 0;
}


/****************************************************************************
 * Function: MasterReconnect()
 * Purpose: performs cleanup(?) and reinitiates connection uppon receivery
 * of sick signal(s) ;-)
 * Arguments: none
 * Retuns: void
 ***************************************************************************/

void MasterReconnect(int sig) {
	
	   ErrorMessage("Connection closed. Reconnecting .. \n");
	   close(MasterSD);
	   sleep(2);
	 
//		   FatalError("Unable to re-connect to MasterServer\n");
	   
	   while (MasterConnect(SnortNetMaster, UseSsl)<0) {
		   PrintError("connect");
	   	   close(MasterSD);
		   sleep(2);
		//   FatalError("Unable to re-connect to MasterServer\n");
	   }
}
