/***************************************************************************
    smb4kmounter.h  -  The core class, that mounts shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KMOUNTER_H
#define SMB4KMOUNTER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qptrqueue.h>

// KDE includes
#include <kprocess.h>
#include <kconfig.h>

// application specific includes
#include "smb4kshare.h"
#include "smb4kdefs.h"

/**
 * This is a core class. It mounts the shares.
 */
class Smb4KMounter : public QObject
{
  Q_OBJECT
  
  friend class Smb4KCore;

  public:
    /**
     * The constructor.
     */
    Smb4KMounter( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KMounter();
    /**
     * Aborts a running process.
     */
    void abort();
    /**
     * Enumeration for the state of the process. The members are also
     * used in the results() signal as return values.
     */
    enum State{ MountRecent, Import, Mount, Unmount, UnmountAll, ForceUnmount, Idle };
    /**
     * Does everything, that has to be done on exit. 
     */
    void exit();
    /**
     * Unmounts a share.
     *
     * @param mountpoint The mountpoint the share will be mounted to.
     *
     * @param uid        The UID the share possesses.
     *
     * @param gid        The GID the share possesses.
     *
     * @param noMessage  Determines whether this function should emit an error code in case of an error. The default value is FALSE.
     */
    void unmountShare( const QString &mountpoint, const QString &uid, const QString &gid, bool noMessage = false );
    /**
     * Unmounts all shares at once.
     */
    void unmountAllShares();
    /**
     * Mounts a share.
     */
    void mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share );
    /**
     * Unmounts a dead share using super/sudo and 'umount -l' command.
     */
    void forceUnmountShare( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Returns the unsorted list of mounted shares.
     */
    const QValueList<Smb4KShare *> &getShares() { return m_mounted_shares; };
    /**
     * Find a share in the list with its path.
     */
    Smb4KShare* findShareByPath( const QString &path );
    /**
     * Find a share in the list with its name
     */
    Smb4KShare* findShareByName( const QString &name );
    /**
     * This function returns TRUE if a share is mounted and FALSE otherwise.
     *
     * @param name The name of the share. It has to look like this: //HOST/SHARE.
     * 
     * @return TRUE if the share is mounted and FALSE otherwise.
     */
    bool isMounted( const QString &name );
    
  signals:
    /**
     * This signal is emitted, when a process starts or ends.
     *
     * @param mode  The so-called run mode. There are several defined for the Smb4KMounter
     *              class in the smb4kdefs.h header file.
     *
     * @param on    Is TRUE if a process starts and FALSE if it has stopped.
     */
    void running( int mode, bool on );
    /**
     * Tells the program, that the shares list has been updated.
     */
    void importedShares();
    /**
     * This signal is emitted in case of an error occurred.
     *
     * @param error_code      One of the error codes defined in the smb4kdefs.h header file.
     *
     * @param error_message   An error message. This is optional for some errors and mandatory
     *                        for others. Have a look at <code>Smb4KCore::slotShowErrorMessage()</code>
     *                        to figure this out.
     */
    void error( int error_code, const QString &error_message );

  protected:
    /**
     * Starts a process. It takes an interger, that determines,
     * which process has to be started.
     */
    void startProcess( int state );
    /**
     * Is called, when the process ended.
     */
    void endProcess();
    /**
     * Finishes the mounting of shares.
     */
    void processMount();
    /**
     * Finishes the unmounting of a single share.
     */
    void processUnmount();

  protected slots:
    /**
     * Is called, when the process exits.
     */
    void slotProcessExited( KProcess * );
    /**
     * Is called, if output is received at Stdout.
     */
    void slotReceivedStdout( KProcess *, char *buf, int len );
    /**
     * Is called, if output is received a Stderr.
     */
    void slotReceivedStderr( KProcess *, char *buf, int len );
    /**
     * Initiates the processing of the queue.
     */
    void init();
    /**
     * Initializes periodic checking. At the moment only the 
     * detection of mounted shares is supported.
     */
    void check();

  private:
    /**
     * Mount recently used shares.
     */
    void mountRecent();
    /**
     * Imports mounted shares.
     */
    void import();
    /**
     * Mounts a selected share.
     */
    void mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share );
    /**
     * Unmounts the selected item.
     */
    void unmount( const QString &mountpoint, const QString &uid, const QString &gid, bool noMessage );
    /**
     * Unmounts dead shares.
     */
    void forceUnmount( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Unmounts all shares at once.
     */
    void unmountAll();
    /**
     * This function returns the advanced option string that is needed to
     * mount the shares. The contents is determined by the choice of the 
     * user (config dialog).
     */
    const QString getMountOptions();
    /**
     * The KProcess object.
     */
    KProcess *m_proc;
    /**
     * The KConfig objects. With its help the options are read.
     */
    KConfig *m_config;
    /**
     * The buffer.
     */
    QString m_buffer;
    /**
     * Holds the mount point of a share.
     */
    QString m_path;
    /**
     * Holds the workgroup of the currently processed shares.
     */
    QString m_workgroup;
    /**
     * Holds the host's name of the currently processed share.
     */
    QString m_host;
    /**
     * Holds the IP address of the currently processed share.
     */
    QString m_ip;
    /**
     * Holds the name of the currently processed share.
     */
    QString m_share;
    /**
     * Holds the state the mounter is in.
     */
    int m_state;
    /**
     * Determines whether this is the last job of the mounter.
     */
    bool m_lastJob;
    /**
     * The queue, where the incoming requests are stored.
     */
    QPtrQueue<QString> m_queue;
    /**
     * The timer for the queue.
     */
    QTimer *m_timer1;
    /**
     * The timer for all the checks that should be done time by time.
     */
    QTimer *m_timer2;
    /**
     * Determines, whether the mounter is running or not.
     */
    bool m_working;
    /**
     * Holds the list of currently mounted shares as a pointer list.
     */
    QValueList<Smb4KShare *> m_mounted_shares;
    /**
     * Makes sure that the error message concerning the missing of
     * the file /proc/mounts is only shown once.
     */
    bool m_proc_error;
    /**
     * The file system of the share.
     */
    QString m_filesystem;
    /**
     * The CIFS login (user name).
     */
    QString m_cifs_login;
};

#endif
