/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KFILEIO_H
#define SMB4KFILEIO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qmap.h>
#include <qvaluelist.h>

// KDE includes
#include <kprocess.h>
#include <kconfig.h>

// application specific includes
#include "smb4kuser.h"


/**
 * This class handles all IO operations on system files, that
 * are needed by Smb4K.
 * @author Alexander Reinholdt
 */

class Smb4KFileIO : public QObject
{
  Q_OBJECT
  
  friend class Smb4KCore;

  public:
    /**
     * The constructor.
     */
    Smb4KFileIO( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor
     */
    ~Smb4KFileIO();
    /**
     * Writes the entries needed to gain super user rights to either the super.tab or
     * the sudoers file depending on the input values.
     *
     * @param program The name of the program that should be used to gain super user privileges.
     */
    bool writeSuperUserEntries( const QString &program );
    /**
     * Removes the entries of a user from the file /etc/super.tab.
     *
     * @return true in case of success and false otherwise.
     */
    bool removeSuperUserEntries();
    /**
     * Returns the map of global smb.conf options. You can reach the value
     * of each option by providing the lowercase option name as key, if it 
     * was defined in the global section of the smb.conf.
     */
    const QMap<QString, QString> readSMBOptions();
    /**
     * Returns the list of non-system users present on the system.
     */
    const QValueList<Smb4KUser *> getUsers();
    /**
     * Returns the papersize defined on the system. Therefore the 
     * /etc/papersize file is read. If this files does not exist, 
     * this function will return 'a4' as value, the default value 
     * of enscript.
     */
    const QString getPaperSize();
        /**
     * This class holds the contents and the path of a file.
     */
    class FileItem
    {
      public:
        /**
         * The contructor.
         *
         * @param path      The path of the file.
         *
         * @param contents  The contents of the file.
         *
         * @param exists    Whether the file exists or not.
        */
        FileItem( const QString &path, const QStringList &contents, bool exists ) : m_path(path), m_contents(contents), m_exists(exists) {}
        /**
         * The empty constructor.
         */
        FileItem() : m_path( QString::null ), m_contents( QStringList() ), m_exists( false ) {}
        /**
         * The destructor.
         */
        ~FileItem() {}
        /**
         * Returns the full path of the file.
         */
        QString path() const { return m_path; }
        /**
         * Retruns the contents of the file.
         */
        QStringList contents() const { return m_contents; }
        /**
         * Is TRUE if the file exists and FALSE if not.
         */
        bool exists() const { return m_exists; }
        /**
         * Replaces the old contents of the file item by the
         * new one.
         * @param contents New contents of the file
         */
        void replaceContents( const QStringList &contents ) { m_contents = contents; }
      
      private:
        QString m_path;
        QStringList m_contents;
        bool m_exists;
    };

  signals:
    /**
     * This signal emits the error code and an error message if 
     * an error occurred.
     */
    void error( int error_code, const QString &error_message );
    /**
     * This signal is emitted if something went wrong with writing the 
     * suid program stuff.
     */
    void no_suid_program();
    /**
     * This signal is emitted, when the writing of the SUID entries to the
     * config files finished.
     */
    void finished_suid_writing();

  protected slots:
    /**
     * This slot receives shell program output at Stderr.
     */
    void slotReceivedStderr( KProcess *, char *buf, int len );
    /**
     * This slot is called, when the process exited.
     */
    void slotProcessExited( KProcess * );

  private:
    /**
     * KConfig object.
     */
    KConfig *m_config;
    /**
     * Enumeration to specify what is done.
     */
    enum State{ writeSU, removeSU, copySudoers, Idle };
    /**
     * This enumeration determines what to do with the sudoers file.
     */
    enum Sudoers{ Add, Remove };
    /**
     * Holds a value of of the Sudoers enumeration.
     */
    int m_todo_sudoers;
    /**
     * Holds the state value.
     */
    int m_state;
    /**
     * The KProcess object.
     */
    KProcess *m_proc;
    /**
     * This is a temporary funtion and converts the entries in the super.tab 
     * file to the new format.
     */
    void convertSuperTab();
    /**
     * Locates a system config file, reads it contents and returns it. In case the
     * config file was not found, it emits and error and returns 0.
     *
     * @param fileName The name of the file.
     *
     * @param strip    Decides whether the white spaces should be stripped. Default is FALSE.
     *
     * @param quiet    If this is set to TRUE, there will be no error emitted if a config file 
     *                 could not be read.
     */
    Smb4KFileIO::FileItem readConfigFile( const QString &fileName, bool strip = false, bool quiet = false );
    /**
     * This function writes a file to its destination. It returns TRUE on success
     * and FALSE otherwise.
     *
     * @param item          The FileItem object that represents the file.
     *
     * @param permissions   This string holds the permissions the file should have.
     */
    bool writeFile( Smb4KFileIO::FileItem *item, const QString &permissions );
    /**
     * This function processes the sudoers file, i.e. is adds or removes entries.
     */
    void processSudoers();
};

#endif
