/***************************************************************************
    smb4knetworkbrowser_part  -  This Part encapsulates the network
    browser of Smb4K.
                             -------------------
    begin                : Fr Jan 5 2007
    copyright            : (C) 2007-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <QKeySequence>
#include <QEvent>
#include <QTreeWidget>
#include <QHeaderView>

// KDE includes
#include <kaboutdata.h>
#include <kaction.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <kcomponentdata.h>
#include <kicon.h>
#include <kactioncollection.h>
#include <kmenu.h>
#include <kapplication.h>
#include <kconfiggroup.h>

// application specific includes
#include <smb4knetworkbrowser_part.h>
#include <smb4knetworkbrowser.h>
#include <smb4knetworkbrowseritem.h>
#include <smb4knetworkbrowsertooltip.h>
#include <../dialogs/smb4kcustomoptionsdialog.h>
#include <../dialogs/smb4kmountdialog.h>
#include <../dialogs/smb4kprintdialog.h>
#include <../dialogs/smb4kpreviewdialog.h>
#include <core/smb4kcore.h>
#include <core/smb4kglobal.h>
#include <core/smb4ksettings.h>
#include <core/smb4kbookmark.h>
#include <core/smb4kdefs.h>
#include <core/smb4kwalletmanager.h>
#include <core/smb4kauthinfo.h>

using namespace Smb4KGlobal;

typedef KParts::GenericFactory<Smb4KNetworkBrowserPart> Smb4KNetworkBrowserPartFactory;
K_EXPORT_COMPONENT_FACTORY( libsmb4knetworkbrowser, Smb4KNetworkBrowserPartFactory )


Smb4KNetworkBrowserPart::Smb4KNetworkBrowserPart( QWidget *parentWidget, QObject *parent, const QStringList &args )
: KParts::Part( parent ), m_mode( Normal )
{
  // Parse arguments:
  for ( int i = 0; i < args.size(); ++i )
  {
    if ( args.at( i ).startsWith( "konqplugin" ) )
    {
      if ( QString::compare( args.at( i ).section( "=", 1, 1 ).trimmed(), "\"true\"" ) == 0 )
      {
        m_mode = KonqPlugin;
      }
      else
      {
        m_mode = Normal;
      }

      continue;
    }
    else
    {
      continue;
    }
  }

  // Set the XML file:
  setXMLFile( "smb4knetworkbrowser_part.rc" );

  // Set the widget of this part:
  m_widget = new Smb4KNetworkBrowser( parentWidget );
  setWidget( m_widget );

  // Set up the actions.
  // Do not put this before setWidget() or the shortcuts of the
  // actions will not be shown.
  setupActions();

  // Load the settings
  loadSettings();

  // Add some connections:
  connect( m_widget,             SIGNAL( customContextMenuRequested( const QPoint & ) ),
           this,                 SLOT( slotContextMenuRequested( const QPoint & ) ) );

  connect( m_widget,             SIGNAL( itemSelectionChanged() ),
           this,                 SLOT( slotItemSelectionChanged() ) );

  connect( m_widget,             SIGNAL( itemPressed( QTreeWidgetItem *, int ) ),
           this,                 SLOT( slotItemPressed( QTreeWidgetItem *, int ) ) );

  connect( m_widget,             SIGNAL( itemExpanded( QTreeWidgetItem * ) ),
           this,                 SLOT( slotItemExpanded( QTreeWidgetItem * ) ) );

  connect( m_widget,             SIGNAL( itemCollapsed( QTreeWidgetItem * ) ),
           this,                 SLOT( slotItemCollapsed( QTreeWidgetItem * ) ) );

  connect( m_widget,             SIGNAL( itemExecuted( QTreeWidgetItem *, int ) ),
           this,                 SLOT( slotItemExecuted( QTreeWidgetItem *, int ) ) );

  connect( m_widget->tooltip(),  SIGNAL( aboutToShow( Smb4KNetworkBrowserItem * ) ),
           this,                 SLOT( slotAboutToShowToolTip( Smb4KNetworkBrowserItem * ) ) );

  connect( m_widget->tooltip(),  SIGNAL( aboutToHide() ),
           this,                 SLOT( slotAboutToHideToolTip() ) );

  connect( Smb4KCore::scanner(), SIGNAL( workgroups( const QList<Smb4KWorkgroup *> & ) ),
           this,                 SLOT( slotWorkgroups( const QList<Smb4KWorkgroup *> & ) ) );

  connect( Smb4KCore::scanner(), SIGNAL( members( const QString &, const QList<Smb4KHost *> & ) ),
           this,                 SLOT( slotWorkgroupMembers( const QString &, const QList<Smb4KHost *> & ) ) );

  connect( Smb4KCore::scanner(), SIGNAL( shares( const QString &, const QList<Smb4KShare *> & ) ),
           this,                 SLOT( slotShares( const QString &, const QList<Smb4KShare *> & ) ) );

  connect( Smb4KCore::scanner(), SIGNAL( ipAddress( Smb4KHost * ) ),
           this,                 SLOT( slotAddIPAddress( Smb4KHost * ) ) );

  connect( Smb4KCore::scanner(), SIGNAL( info( Smb4KHost * ) ),
           this,                 SLOT( slotAddInformation( Smb4KHost * ) ) );

  connect( Smb4KCore::scanner(), SIGNAL( hostInserted( Smb4KHost * ) ),
           this,                 SLOT( slotInsertHost( Smb4KHost * ) ) );

  connect( Smb4KCore::mounter(), SIGNAL( updated() ),
           this,                 SLOT( slotMarkMountedShares() ) );

  connect( Smb4KCore::self(),    SIGNAL( runStateChanged() ),
           this,                 SLOT( slotRunStateChanged() ) );

  connect( kapp,                 SIGNAL( aboutToQuit() ),
           this,                 SLOT( slotAboutToQuit() ) );
}


Smb4KNetworkBrowserPart::~Smb4KNetworkBrowserPart()
{
}


void Smb4KNetworkBrowserPart::setupActions()
{
  KAction *rescan_action   = new KAction( KIcon( "view-refresh" ), i18n( "Scan Netwo&rk" ),
                             actionCollection() );
  rescan_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_R ) );
  connect( rescan_action, SIGNAL( triggered( bool ) ), this, SLOT( slotRescan( bool ) ) );

  KAction *abort_action    = new KAction( KIcon( "process-stop" ), i18n( "&Abort" ),
                             actionCollection() );
  abort_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_A ) );
  connect( abort_action, SIGNAL( triggered( bool ) ), this, SLOT( slotAbort( bool ) ) );

  KAction *manual_action   = new KAction( KIcon( "list-add" ), i18n( "M&ount Manually" ),
                             actionCollection() );
  manual_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_O ) );
  connect( manual_action, SIGNAL( triggered( bool ) ), this, SLOT( slotMountManually( bool ) ) );

  KAction *auth_action     = new KAction( KIcon( "dialog-password" ), i18n( "Au&thentication" ),
                             actionCollection() );
  auth_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_T ) );
  connect( auth_action, SIGNAL( triggered( bool ) ), this, SLOT( slotAuthentication( bool ) ) );

  KAction *custom_action   = new KAction( KIcon( "preferences-system-network" ), i18n( "&Custom Options" ),
                             actionCollection() );
  custom_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_C ) );
  connect( custom_action, SIGNAL( triggered( bool ) ), this, SLOT( slotCustomOptions( bool ) ) );

  KAction *bookmark_action = new KAction( KIcon( "bookmark-new" ), i18n( "Add &Bookmark" ),
                             actionCollection() );
  bookmark_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_B ) );
  connect( bookmark_action, SIGNAL( triggered( bool ) ), this, SLOT( slotAddBookmark( bool ) ) );

  KAction *preview_action  = new KAction( KIcon( "view-list-icons" ), i18n( "Pre&view" ),
                             actionCollection() );
  preview_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_V ) );
  connect( preview_action, SIGNAL( triggered( bool ) ), this, SLOT( slotPreview( bool ) ) );

  KAction *print_action    = new KAction( KIcon( "printer" ), i18n( "&Print File" ),
                             actionCollection() );
  print_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_P ) );
  connect( print_action, SIGNAL( triggered( bool ) ), this, SLOT( slotPrint( bool ) ) );

  KAction *mount_action    = new KAction( KIcon( "folder-remote" ), i18n( "&Mount" ),
                             actionCollection() );
  mount_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_M ) );
  connect( mount_action, SIGNAL( triggered( bool ) ), this, SLOT( slotMount( bool ) ) );

  actionCollection()->addAction( "rescan_action", rescan_action );
  actionCollection()->addAction( "abort_action", abort_action );
  actionCollection()->addAction( "mount_manually_action", manual_action );
  actionCollection()->addAction( "authentication_action", auth_action );
  actionCollection()->addAction( "custom_action", custom_action );
  actionCollection()->addAction( "bookmark_action", bookmark_action );
  actionCollection()->addAction( "preview_action", preview_action );
  actionCollection()->addAction( "print_action", print_action );
  actionCollection()->addAction( "mount_action", mount_action );

  // Enable/disable the actions:
  rescan_action->setEnabled( true );
  abort_action->setEnabled( false );
  manual_action->setEnabled( true );
  auth_action->setEnabled( false );
  custom_action->setEnabled( false );
  bookmark_action->setEnabled( false );
  preview_action->setEnabled( false );
  print_action->setEnabled( false );
  mount_action->setEnabled( false );

  // Plug the actions into the action menu:
  m_menu = new KActionMenu( this );
  m_menu_title = m_menu->menu()->addTitle( KIcon( "network-workgroup" ), i18n( "Network" ) );
  m_menu->addAction( rescan_action );
  m_menu->addAction( abort_action );
  m_menu->addSeparator();
  m_menu->addAction( manual_action );
  m_menu->addSeparator();
  m_menu->addAction( auth_action );
  m_menu->addAction( custom_action );
  m_menu->addAction( bookmark_action );
  m_menu->addAction( preview_action );
  m_menu->addAction( print_action );
  m_menu->addAction( mount_action );

  // If we are in Konqueror plugin mode, an unmount action
  // is also needed:
  if ( m_mode == KonqPlugin )
  {
    KAction *umount_action = new KAction( KIcon( "media-eject" ), i18n( "&Unmount" ),
                             actionCollection() );
    umount_action->setShortcut( QKeySequence( Qt::CTRL+Qt::Key_U ) );
    connect( umount_action, SIGNAL( triggered( bool ) ), this, SLOT( slotUnmount( bool ) ) );

    actionCollection()->addAction( "konq_umount_action", umount_action );

    umount_action->setEnabled( false );
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::loadSettings()
{
  // Show/hide columns:
  m_widget->setColumnHidden( Smb4KNetworkBrowser::IP, !Smb4KSettings::showIPAddress() );
  m_widget->setColumnHidden( Smb4KNetworkBrowser::Type, !Smb4KSettings::showType() );
  m_widget->setColumnHidden( Smb4KNetworkBrowser::Comment, !Smb4KSettings::showComment() );

  // Load and apply the positions of the columns.
  KConfigGroup configGroup( Smb4KSettings::self()->config(), "NetworkBrowserPart" );

  QMap<int, int> map;
  map.insert( configGroup.readEntry( "ColumnPositionNetwork", (int)Smb4KNetworkBrowser::Network ), Smb4KNetworkBrowser::Network );
  map.insert( configGroup.readEntry( "ColumnPositionType", (int)Smb4KNetworkBrowser::Type ), Smb4KNetworkBrowser::Type );
  map.insert( configGroup.readEntry( "ColumnPositionIP", (int)Smb4KNetworkBrowser::IP ), Smb4KNetworkBrowser::IP );
  map.insert( configGroup.readEntry( "ColumnPositionComment", (int)Smb4KNetworkBrowser::Comment ), Smb4KNetworkBrowser::Comment );

  QMap<int, int>::const_iterator it = map.constBegin();

  while ( it != map.constEnd() )
  {
    if ( it.key() != m_widget->header()->visualIndex( it.value() ) )
    {
      m_widget->header()->moveSection( m_widget->header()->visualIndex( it.value() ), it.key() );
    }
    else
    {
      // Do nothing
    }

    ++it;
  }

  // The rest of the settings will be applied on the fly.
}


void Smb4KNetworkBrowserPart::saveSettings()
{
  // Save the position of the columns.
  KConfigGroup configGroup( Smb4KSettings::self()->config(), "NetworkBrowserPart" );
  configGroup.writeEntry( "ColumnPositionNetwork", m_widget->header()->visualIndex( Smb4KNetworkBrowser::Network ) );
  configGroup.writeEntry( "ColumnPositionType", m_widget->header()->visualIndex( Smb4KNetworkBrowser::Type ) );
  configGroup.writeEntry( "ColumnPositionIP", m_widget->header()->visualIndex( Smb4KNetworkBrowser::IP ) );
  configGroup.writeEntry( "ColumnPositionComment", m_widget->header()->visualIndex( Smb4KNetworkBrowser::Comment ) );

  configGroup.sync();
}


KAboutData *Smb4KNetworkBrowserPart::createAboutData()
{
  KAboutData *aboutData = new KAboutData( "smb4knetworkbrowserpart",
                          "smb4k",
                          ki18n( "Smb4KNetworkBrowserPart" ),
                          "2.0",
                          ki18n( "The network browser KPart of Smb4K" ),
                          KAboutData::License_GPL_V2,
                          ki18n( "(c) 2007-2008, Alexander Reinholdt" ),
                          KLocalizedString(),
                          "http://smb4k.berlios.de",
                          "smb4k-bugs@lists.berlios.de" );

  return aboutData;
}


void Smb4KNetworkBrowserPart::customEvent( QEvent *e )
{
  switch ( e->type() )
  {
    case EVENT_LOAD_SETTINGS:
    {
      loadSettings();

      break;
    }
    case EVENT_SET_FOCUS:
    {
      if ( m_widget->topLevelItemCount() != 0 )
      {
        kDebug() << "Do we need to port the selection stuff?" << endl;
      }

      m_widget->setFocus( Qt::OtherFocusReason );

//       KListView *view = static_cast<KListView *>( m_widget );
//
//       if ( view->childCount() != 0 )
//       {
//         view->setSelected( !view->currentItem() ?
//                            view->firstChild() :
//                            view->currentItem(), true );
//       }

      break;
    }
    case EVENT_SCAN_NETWORK:
    {
      slotRescan( false ); // boolean is ignored

      break;
    }
    default:
    {
      break;
    }
  }

  KParts::Part::customEvent( e );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS (Smb4KNetworkBrowserPart)
/////////////////////////////////////////////////////////////////////////////

void Smb4KNetworkBrowserPart::slotContextMenuRequested( const QPoint &pos )
{
  QTreeWidgetItem *item = m_widget->itemAt( pos );

  m_menu->removeAction( m_menu_title );
  delete m_menu_title;

  if ( item )
  {
    m_menu_title = m_menu->menu()->addTitle( item->icon( Smb4KNetworkBrowserItem::Network ),
                                             item->text( Smb4KNetworkBrowserItem::Network ),
                                             actionCollection()->action( "rescan_action" ) );
  }
  else
  {
    m_menu_title = m_menu->menu()->addTitle( KIcon( "network-workgroup" ),
                                             i18n( "Network" ),
                                             actionCollection()->action( "rescan_action" ) );
  }

  m_menu->menu()->popup( m_widget->viewport()->mapToGlobal( pos ) );
}


void Smb4KNetworkBrowserPart::slotItemSelectionChanged()
{
  // Get the selected item.
  QList<QTreeWidgetItem *> items = m_widget->selectedItems();

  if ( items.size() == 1 )
  {
    Smb4KNetworkBrowserItem *browser_item = static_cast<Smb4KNetworkBrowserItem *>( items.first() );

    if ( browser_item )
    {
      switch ( m_mode )
      {
        case Normal:
        {
          switch ( browser_item->type() )
          {
            case Smb4KNetworkBrowserItem::Host:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Compute&r" ) );

              // Enable/disable the actions:
              actionCollection()->action( "bookmark_action" )->setEnabled( false );
              actionCollection()->action( "authentication_action" )->setEnabled( true );
              actionCollection()->action( "preview_action" )->setEnabled( false );
              actionCollection()->action( "mount_action" )->setEnabled( false );
              actionCollection()->action( "print_action" )->setEnabled( false );
              actionCollection()->action( "custom_action" )->setEnabled( true );

              break;
            }
            case Smb4KNetworkBrowserItem::Share:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Compute&r" ) );

              // Enable/disable the actions:
              actionCollection()->action( "authentication_action" )->setEnabled( true );

              if ( !browser_item->shareItem()->isPrinter() )
              {
                actionCollection()->action( "bookmark_action" )->setEnabled( true );
                actionCollection()->action( "preview_action" )->setEnabled( true );
                actionCollection()->action( "mount_action" )->setEnabled( true );
                actionCollection()->action( "print_action" )->setEnabled( false );
                actionCollection()->action( "custom_action" )->setEnabled( true );
              }
              else
              {
                actionCollection()->action( "bookmark_action" )->setEnabled( false );
                actionCollection()->action( "preview_action" )->setEnabled( false );
                actionCollection()->action( "mount_action" )->setEnabled( false );
                actionCollection()->action( "print_action" )->setEnabled( true );
                actionCollection()->action( "custom_action" )->setEnabled( false );
              }

              break;
            }
            default:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Wo&rkgroup" ) );

              actionCollection()->action( "bookmark_action" )->setEnabled( false );
              actionCollection()->action( "authentication_action" )->setEnabled( false );
              actionCollection()->action( "preview_action" )->setEnabled( false );
              actionCollection()->action( "mount_action" )->setEnabled( false );
              actionCollection()->action( "print_action" )->setEnabled( false );
              actionCollection()->action( "custom_action" )->setEnabled( false );

              break;
            }
          }

          break;
        }
        case KonqPlugin:
        {
          switch ( browser_item->type() )
          {
            case Smb4KNetworkBrowserItem::Host:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Compute&r" ) );

              // Enable/disable the actions:
              actionCollection()->action( "bookmark_action" )->setEnabled( false );
              actionCollection()->action( "authentication_action" )->setEnabled( true );
              actionCollection()->action( "preview_action" )->setEnabled( false );
              actionCollection()->action( "mount_action" )->setEnabled( false );
              actionCollection()->action( "konq_umount_action" )->setEnabled( false );
              actionCollection()->action( "print_action" )->setEnabled( false );
              actionCollection()->action( "custom_action" )->setEnabled( true );

              kDebug() << "Does the exchange of actions work?" << endl;

              // Get the action in the menu.
              QList<QAction *> actions = m_menu->menu()->actions();

              if ( actions.contains( actionCollection()->action( "konq_umount_action" ) ) )
              {
                m_menu->removeAction( actionCollection()->action( "konq_umount_action" ) );
                m_menu->insertAction( actionCollection()->action( "print_action" ),
                                      actionCollection()->action( "mount_action" ) );
              }
              else
              {
                // Nothing to do
              }

              break;
            }
            case Smb4KNetworkBrowserItem::Share:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Compute&r" ) );

              // Enable/disable the actions:
              actionCollection()->action( "authentication_action" )->setEnabled( true );

              // Get the action in the menu.
              QList<QAction *> actions = m_menu->menu()->actions();

              if ( !browser_item->shareItem()->isPrinter() )
              {
                actionCollection()->action( "bookmark_action" )->setEnabled( true );
                actionCollection()->action( "preview_action" )->setEnabled( true );
                actionCollection()->action( "mount_action" )->setEnabled( true );
                actionCollection()->action( "konq_umount_action" )->setEnabled( true );
                actionCollection()->action( "print_action" )->setEnabled( false );
                actionCollection()->action( "custom_action" )->setEnabled( true );

                if ( !browser_item->shareItem()->isMounted() )
                {
                  kDebug() << "Does the exchange of actions work?" << endl;

                  if ( actions.contains( actionCollection()->action( "konq_umount_action" ) ) )
                  {
                    m_menu->removeAction( actionCollection()->action( "konq_umount_action" ) );
                    m_menu->insertAction( actionCollection()->action( "print_action" ),
                                          actionCollection()->action( "mount_action" ) );
                  }
                  else
                  {
                    // Nothing to do
                  }
                }
                else
                {
                  kDebug() << "Does the exchange of actions work?" << endl;

                  if ( actions.contains( actionCollection()->action( "mount_action" ) ) )
                  {
                    m_menu->removeAction( actionCollection()->action( "mount_action" ) );
                    m_menu->insertAction( actionCollection()->action( "print_action" ),
                                          actionCollection()->action( "konq_umount_action" ) );
                  }
                  else
                  {
                    // Nothing to do
                  }
                }
              }
              else
              {
                actionCollection()->action( "bookmark_action" )->setEnabled( false );
                actionCollection()->action( "preview_action" )->setEnabled( false );
                actionCollection()->action( "mount_action" )->setEnabled( false );
                actionCollection()->action( "konq_umount_action" )->setEnabled( false );
                actionCollection()->action( "print_action" )->setEnabled( true );
                actionCollection()->action( "custom_action" )->setEnabled( false );

                kDebug() << "Does the exchange of actions work?" << endl;

                if ( actions.contains( actionCollection()->action( "konq_umount_action" ) ) )
                {
                  m_menu->removeAction( actionCollection()->action( "konq_umount_action" ) );
                  m_menu->insertAction( actionCollection()->action( "print_action" ),
                                        actionCollection()->action( "mount_action" ) );
                }
                else
                {
                  // Nothing to do
                }
              }

              break;
            }
            default:
            {
              // Change the text of the rescan action:
              actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Wo&rkgroup" ) );

              actionCollection()->action( "bookmark_action" )->setEnabled( false );
              actionCollection()->action( "authentication_action" )->setEnabled( false );
              actionCollection()->action( "preview_action" )->setEnabled( false );
              actionCollection()->action( "mount_action" )->setEnabled( false );
              actionCollection()->action( "konq_umount_action" )->setEnabled( false );
              actionCollection()->action( "print_action" )->setEnabled( false );
              actionCollection()->action( "custom_action" )->setEnabled( false );

              // Get the action in the menu.
              QList<QAction *> actions = m_menu->menu()->actions();

              kDebug() << "Does the exchange of actions work?" << endl;

              if ( actions.contains( actionCollection()->action( "konq_umount_action" ) ) )
              {
                m_menu->removeAction( actionCollection()->action( "konq_umount_action" ) );
                m_menu->insertAction( actionCollection()->action( "print_action" ),
                                      actionCollection()->action( "mount_action" ) );
              }
              else
              {
                // Nothing to do
              }

              break;
            }
          }

          break;
        }
        default:
        {
          break;
        }
      }
    }
    else
    {
      // Do nothing. This is managed elsewhere.
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotItemPressed( QTreeWidgetItem *item, int /*column*/ )
{
  // Enable/disable the actions.
  Smb4KNetworkBrowserItem *browser_item = static_cast<Smb4KNetworkBrowserItem *>( item );

  switch ( m_mode )
  {
    case Normal:
    {
      if ( !browser_item && m_widget->selectedItems().size() == 0 )
      {
        actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Netwo&rk" ) );

        actionCollection()->action( "bookmark_action" )->setEnabled( false );
        actionCollection()->action( "authentication_action" )->setEnabled( false );
        actionCollection()->action( "preview_action" )->setEnabled( false );
        actionCollection()->action( "mount_action" )->setEnabled( false );
        actionCollection()->action( "print_action" )->setEnabled( false );
        actionCollection()->action( "custom_action" )->setEnabled( false );
      }
      else
      {
        // See Smb4KNetworkBrowserPart::slotItemSelectionChanged()
      }

      break;
    }
    case KonqPlugin:
    {
      if ( !browser_item && m_widget->selectedItems().size() ==  0 )
      {
        actionCollection()->action( "rescan_action" )->setText( i18n( "Scan Netwo&rk" ) );

        actionCollection()->action( "bookmark_action" )->setEnabled( false );
        actionCollection()->action( "authentication_action" )->setEnabled( false );
        actionCollection()->action( "preview_action" )->setEnabled( false );
        actionCollection()->action( "mount_action" )->setEnabled( false );
        actionCollection()->action( "konq_umount_action" )->setEnabled( false );
        actionCollection()->action( "print_action" )->setEnabled( false );
        actionCollection()->action( "custom_action" )->setEnabled( false );

        QList<QAction *> actions = m_menu->menu()->actions();

        kDebug() << "Does the exchange of actions work?" << endl;

        if ( actions.contains( actionCollection()->action( "konq_umount_action" ) ) )
        {
          m_menu->removeAction( actionCollection()->action( "konq_umount_action" ) );
          m_menu->insertAction( actionCollection()->action( "print_action" ),
                                actionCollection()->action( "mount_action" ) );
        }
        else
        {
          // Nothing to do
        }
      }
      else
      {
        // See Smb4KNetworkBrowserPart::slotSelectionChanged()
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KNetworkBrowserPart::slotItemExpanded( QTreeWidgetItem *item )
{
  Smb4KNetworkBrowserItem *browserItem = static_cast<Smb4KNetworkBrowserItem *>( item );

  if ( browserItem )
  {
    switch ( browserItem->type() )
    {
      case Smb4KNetworkBrowserItem::Workgroup:
      {
        Smb4KCore::scanner()->getWorkgroupMembers( browserItem->workgroupItem() );
        break;
      }
      case Smb4KNetworkBrowserItem::Host:
      {
        Smb4KCore::scanner()->getShares( browserItem->hostItem() );
        break;
      }
      default:
        break;
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotItemCollapsed( QTreeWidgetItem *item )
{
  Smb4KNetworkBrowserItem *browserItem = static_cast<Smb4KNetworkBrowserItem *>( item );

  // Remove all children if we have a host item, because we
  // do not want shares already displayed before the user provided
  // the login data.
  if ( browserItem )
  {
    switch ( browserItem->type() )
    {
      case Smb4KNetworkBrowserItem::Host:
      {
        QList<QTreeWidgetItem *> children = browserItem->takeChildren();

        while ( children.size() != 0 )
        {
          delete children.takeFirst();
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotItemExecuted( QTreeWidgetItem *item, int /*column*/ )
{
  Smb4KNetworkBrowserItem *browserItem = static_cast<Smb4KNetworkBrowserItem *>( item );

  if ( browserItem )
  {
    switch ( browserItem->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        if ( !browserItem->shareItem()->isPrinter() )
        {
          slotMount( false );  // boolean is ignored
        }
        else
        {
          slotPrint( false );  // boolean is ignored
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotAboutToShowToolTip( Smb4KNetworkBrowserItem *item )
{
  if ( item )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Host:
      {
        // Check if additional information is needed and send a request to the scanner,
        // if necessary.
        if ( !item->hostItem()->infoChecked() )
        {
          Smb4KCore::scanner()->getInfo( item->hostItem() );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing --- BTW: Will this case occur at all?
  }
}


void Smb4KNetworkBrowserPart::slotAboutToHideToolTip()
{
  if ( Smb4KCore::scanner()->isRunning() &&
       Smb4KCore::scannerState() == SCANNER_QUERY_INFO )
  {
    Smb4KCore::scanner()->abort();
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotWorkgroups( const QList<Smb4KWorkgroup *> &list )
{
  if ( !list.isEmpty() )
  {
    // Check that the workgroup items in the tree widget are still
    // valid. Remove obsolete and new ones.
    if ( m_widget->topLevelItemCount() != 0 )
    {
      QTreeWidgetItemIterator it( m_widget );

      while ( *it )
      {
        Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );
        bool found_workgroup = false;

        switch ( item->type() )
        {
          case Smb4KNetworkBrowserItem::Workgroup:
          {
            for ( int i = 0; i < list.size(); ++i )
            {
              if ( QString::compare( item->workgroupItem()->name(), list.at( i )->name() ) == 0 )
              {
                // We found the workgroup. Now update it if necessary.
                // First check if the master browser is still the same. If it
                // is not, find the new one.
                if ( QString::compare( item->workgroupItem()->masterBrowserName(),
                     list.at( i )->masterBrowserName() ) != 0 )
                {
                  Smb4KNetworkBrowserItem *new_master_browser = NULL;

                  for ( int j = 0; j < item->childCount(); ++j )
                  {
                    Smb4KNetworkBrowserItem *child = static_cast<Smb4KNetworkBrowserItem *>( item->child( j ) );

                    if ( QString::compare( child->hostItem()->name(),
                         item->workgroupItem()->masterBrowserName() ) == 0 )
                    {
                      // This is the old master. Update it.
                      Smb4KHost *host = findHost( item->workgroupItem()->masterBrowserName(),
                                        item->workgroupItem()->name() );

                      if ( host )
                      {
                        // Update.
                        child->update( host );
                      }
                      else
                      {
                        // The old master is not available anymore.
                        // Remove it.
                        delete child;
                      }

                      continue;
                    }
                    else if ( QString::compare( child->hostItem()->name(),
                              list.at( i )->masterBrowserName() ) == 0 )
                    {
                      // This is the new master. Update it.
                      Smb4KHost *host = findHost( item->workgroupItem()->masterBrowserName(),
                                        item->workgroupItem()->name() );

                      if ( host )
                      {
                        // Update.
                        child->update( host );
                      }
                      else
                      {
                        // Huh...?
                      }

                      new_master_browser = child;

                      continue;
                    }
                    else
                    {
                      continue;
                    }
                  }

                  if ( !new_master_browser )
                  {
                    // The new master browser is not in the tree widget,
                    // so we have to put it there.
                    Smb4KHost *host = findHost( item->workgroupItem()->masterBrowserName(),
                                      item->workgroupItem()->name() );

                    (void) new Smb4KNetworkBrowserItem( item, host );
                  }
                  else
                  {
                    // Do nothing.
                  }
                }
                else
                {
                  // Do nothing.
                }

                item->update( list.at( i ) );

                found_workgroup = true;

                break;
              }
              else
              {
                continue;
              }
            }

            // Remove the workgroup from the tree widget if is not present
            // anymore.
            if ( !found_workgroup )
            {
              delete item;
            }
            else
            {
              // Do nothing
            }

            break;
          }
          default:
          {
            break;
          }
        }

        ++it;
      }

      // Add new workgroups to the tree widget.
      for ( int i = 0; i < list.size(); ++i )
      {
        QList<QTreeWidgetItem *> items = m_widget->findItems( list.at( i )->name(), Qt::MatchFixedString,
                                         Smb4KNetworkBrowser::Network );

        if ( items.isEmpty() )
        {
          // The workgroup is not in the tree widget. Add it.
          (void) new Smb4KNetworkBrowserItem( m_widget, list.at( i ) );

          continue;
        }
        else
        {
          continue;
        }
      }
    }
    else
    {
      // Put the items in the empty list view:
      for ( int i = 0; i < list.size(); ++i )
      {
        (void) new Smb4KNetworkBrowserItem( m_widget, list.at( i ) );
      }
    }
  }
  else
  {
    // Nothing was found: Clear the tree widget.
    m_widget->clear();
  }

  // Sort the items:
  m_widget->sortItems( Smb4KNetworkBrowser::Network, Qt::AscendingOrder );
}


void Smb4KNetworkBrowserPart::slotWorkgroupMembers( const QString &workgroup, const QList<Smb4KHost *> &list )
{
  if ( !workgroup.isEmpty() )
  {
    QTreeWidgetItemIterator it( m_widget );

    while ( *it )
    {
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );

      switch ( item->type() )
      {
        case Smb4KNetworkBrowserItem::Workgroup:
        {
          if ( QString::compare( item->workgroupItem()->name(), workgroup ) == 0 )
          {
            // We found the workgroup. Process the workgroup members.
            if ( !list.isEmpty() )
            {
              // Expand the workgroup item, if it is collapsed.
              if ( !item->isExpanded() )
              {
                m_widget->expandItem( item );
              }
              else
              {
                // Do nothing
              }

              // Update and add hosts.
              for ( int i = 0; i < list.size(); ++i )
              {
                bool found_item = false;

                for ( int j = 0; j < item->childCount(); ++j )
                {
                  if ( QString::compare( list.at( i )->workgroup(), item->workgroupItem()->name() ) == 0 )
                  {
                    Smb4KNetworkBrowserItem *hostItem = static_cast<Smb4KNetworkBrowserItem *>( item->child( j ) );

                    if ( QString::compare( list.at( i )->name(), hostItem->hostItem()->name() ) == 0 )
                    {
                      hostItem->update( list.at( i ) );

                      found_item = true;

                      break;
                    }
                    else
                    {
                      continue;
                    }
                  }
                  else
                  {
                    continue;
                  }
                }

                if ( !found_item &&
                     QString::compare( list.at( i )->workgroup(), item->workgroupItem()->name() ) == 0 )
                {
                  (void) new Smb4KNetworkBrowserItem( item, list.at( i ) );

                  continue;
                }
                else
                {
                  continue;
                }
              }

              // Delete obsolete hosts.
              for ( int i = 0; i < item->childCount(); ++i )
              {
                Smb4KNetworkBrowserItem *hostItem = static_cast<Smb4KNetworkBrowserItem *>( item->child( i ) );
                bool found_host = false;

                for ( int j = 0; j < list.size(); ++j )
                {
                  if ( QString::compare( list.at( j )->workgroup(), hostItem->hostItem()->workgroup() ) == 0 &&
                       QString::compare( list.at( j )->name(), hostItem->hostItem()->name() ) == 0 )
                  {
                    found_host = true;

                    break;
                  }
                  else
                  {
                    continue;
                  }
                }

                if ( !found_host )
                {
                  delete hostItem;

                  continue;
                }
                else
                {
                  continue;
                }
              }
            }
            else
            {
              // Collapse the item.
              m_widget->collapseItem( item );

              // Delete all host items in this workgroup:
              while ( item->childCount() != 0 )
              {
                delete item->child( 0 );
              }

              // And delete the item.
              delete item;
            }

            // Sort the items.
            m_widget->sortItems( Smb4KNetworkBrowser::Network, Qt::AscendingOrder );
          }
          else
          {
            // Do nothing
          }

          break;
        }
        default:
        {
          break;
        }
      }

      ++it;
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotShares( const QString &host, const QList<Smb4KShare *> &list )
{
  if ( !host.isEmpty() )
  {
    QTreeWidgetItemIterator it( m_widget );

    while ( *it )
    {
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );

      switch ( item->type() )
      {
        case Smb4KNetworkBrowserItem::Host:
        {
          if ( QString::compare( item->hostItem()->name(), host ) == 0 )
          {
            if ( !list.isEmpty() )
            {
              // Expand the host item, if it is collapsed.
              if ( !item->isExpanded() )
              {
                m_widget->expandItem( item );
              }
              else
              {
                // Do nothing
              }

              // Update and add the shares.
              for ( int i = 0; i < list.size(); ++i )
              {
                bool found_item = false;

                for ( int j = 0; j < item->childCount(); ++j )
                {
                  if ( QString::compare( list.at( i )->host(), item->hostItem()->name() ) == 0 )
                  {
                    Smb4KNetworkBrowserItem *shareItem = static_cast<Smb4KNetworkBrowserItem *>( item->child( j ) );

                    if ( QString::compare( list.at( i )->name(), shareItem->shareItem()->name() ) == 0 )
                    {
                      // We found the share. Now process it.
                      if ( !shareItem->shareItem()->isHidden() )
                      {
                        if ( !shareItem->shareItem()->isPrinter() || Smb4KSettings::showPrinterShares() )
                        {
                          shareItem->update( list.at( i ) );
                        }
                        else
                        {
                          // Do nothing. The item will be deleted below.
                        }

                        found_item = true;

                        break;
                      }
                      else
                      {
                        if ( Smb4KSettings::showHiddenShares() )
                        {
                          if ( (shareItem->shareItem()->isPrinter() && Smb4KSettings::showPrinterShares()) ||
                               (shareItem->shareItem()->isIPC() && Smb4KSettings::showHiddenIPCShares()) ||
                               (shareItem->shareItem()->isADMIN() && Smb4KSettings::showHiddenADMINShares()) )
                          {
                            shareItem->update( list.at( i ) );
                          }
                          else
                          {
                            // Do nothing. The item will be deleted below.
                          }
                        }
                        else
                        {
                          // Do nothing. The item will be deleted below.
                        }

                        found_item = true;

                        break;
                      }
                    }
                    else
                    {
                      continue;
                    }
                  }
                  else
                  {
                    continue;
                  }
                }

                if ( !found_item &&
                     QString::compare( list.at( i )->host(), item->hostItem()->name() ) == 0 )
                {
                  if ( !list.at( i )->isHidden() )
                  {
                    if ( !list.at( i )->isPrinter() || Smb4KSettings::showPrinterShares() )
                    {
                      (void) new Smb4KNetworkBrowserItem( item, list.at( i ) );
                    }
                    else
                    {
                      // Do nothing
                    }

                    continue;
                  }
                  else
                  {
                    if ( Smb4KSettings::showHiddenShares() )
                    {
                      if ( (!list.at( i )->isPrinter() && !list.at( i )->isIPC() && !list.at( i )->isADMIN()) ||
                           (list.at( i )->isPrinter() && Smb4KSettings::showPrinterShares()) ||
                           (list.at( i )->isIPC() && Smb4KSettings::showHiddenIPCShares()) ||
                           (list.at( i )->isADMIN() && Smb4KSettings::showHiddenADMINShares()) )
                      {
                        (void) new Smb4KNetworkBrowserItem( item, list.at( i ) );
                      }
                      else
                      {
                        // Do nothing
                      }
                    }
                    else
                    {
                      // Do nothing
                    }

                    continue;
                  }
                }
                else
                {
                  continue;
                }
              }

              // Delete obsolete shares.
              for ( int i = 0; i < item->childCount(); ++i )
              {
                Smb4KNetworkBrowserItem *shareItem = static_cast<Smb4KNetworkBrowserItem *>( item->child( i ) );
                bool found_share = false;

                for ( int j = 0; j < list.size(); ++j )
                {
                  if ( QString::compare( list.at( j )->host(), shareItem->shareItem()->host() ) == 0 &&
                       QString::compare( list.at( j )->name(), shareItem->shareItem()->name() ) == 0 )
                  {
                    // Test if the share needs to be deleted. If this is the case, we
                    // won't set found_share to TRUE.

                    // (a) Printer shares
                    if ( !Smb4KSettings::showPrinterShares() && shareItem->shareItem()->isPrinter() )
                    {
                      break;
                    }
                    else
                    {
                      // Do nothing.
                    }

                    // (b) Hidden shares
                    if ( !Smb4KSettings::showHiddenShares() && shareItem->shareItem()->isHidden() )
                    {
                      break;
                    }
                    else
                    {
                      if ( (!Smb4KSettings::showHiddenIPCShares() && shareItem->shareItem()->isIPC()) ||
                           (!Smb4KSettings::showHiddenADMINShares() && shareItem->shareItem()->isADMIN()) )
                      {
                        break;
                      }
                      else
                      {
                        // Do nothing
                      }
                    }

                    found_share = true;
                  }
                  else
                  {
                    continue;
                  }
                }

                if ( !found_share )
                {
                  delete shareItem;

                  continue;
                }
                else
                {
                  continue;
                }
              }
            }
            else
            {
              // Collapse the item.
              m_widget->collapseItem( item );

              // Delete all share items from this host:
              while ( item->childCount() != 0 )
              {
                delete item->child( 0 );
              }
            }

            // Sort the items.
            m_widget->sortItems( Smb4KNetworkBrowser::Network, Qt::AscendingOrder );
          }
          else
          {
            // Do nothing
          }

          break;
        }
        default:
        {
          break;
        }
      }

      ++it;
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotAddIPAddress( Smb4KHost *host )
{
  if ( host )
  {
    // Get the workgroup item where the host is located.
    Smb4KWorkgroup *workgroup = findWorkgroup( host->workgroup() );

    // Search the host and its workgroup and update them if necessary.
    QTreeWidgetItemIterator it( m_widget );

    while ( *it )
    {
      // Assign the correct item to workgroupItem. There should
      // only be one workgroup in the list.
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );

      switch ( item->type() )
      {
        case Smb4KNetworkBrowserItem::Workgroup:
        {
          if ( workgroup &&
               QString::compare( workgroup->name(), item->workgroupItem()->name(), Qt::CaseInsensitive ) == 0 )
          {
            item->update( workgroup );

            // If the tool tip is currently shown, update it.
            if ( m_widget->tooltip() && m_widget->tooltip()->isVisible() &&
                 m_widget->tooltip()->item() == item )
            {
              m_widget->tooltip()->update();
            }
            else
            {
              // Do nothing
            }
          }
          else
          {
            // Do nothing
          }

          break;
        }
        case Smb4KNetworkBrowserItem::Host:
        {
          if ( QString::compare( item->hostItem()->workgroup(), host->workgroup(), Qt::CaseInsensitive ) == 0 &&
               QString::compare( item->hostItem()->name(), host->name(), Qt::CaseInsensitive ) == 0 )
          {
            // Update the host item.
            item->update( host );

            // If the tool tip is currently shown, update it.
            if ( m_widget->tooltip() && m_widget->tooltip()->isVisible() &&
                 m_widget->tooltip()->item() == item )
            {
              m_widget->tooltip()->update();
            }
            else
            {
              // Do nothing
            }

            // Now adjust the IP address column, if it is not hidden.
            if ( !m_widget->isColumnHidden( Smb4KNetworkBrowser::IP ) )
            {
              m_widget->resizeColumnToContents( Smb4KNetworkBrowser::IP );
            }
            else
            {
              // Do nothing
            }
          }
          else
          {
            // Do nothing
          }

          break;
        }
        default:
        {
          break;
        }
      }

      ++it;
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotAddInformation( Smb4KHost *host )
{
  if ( host )
  {
    QTreeWidgetItemIterator it( m_widget );
    Smb4KNetworkBrowserItem *hostItem = NULL;

    while ( *it && !hostItem )
    {
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );

      switch ( item->type() )
      {
        case Smb4KNetworkBrowserItem::Host:
        {
          if ( QString::compare( item->hostItem()->workgroup(), host->workgroup() ) == 0 &&
               QString::compare( item->hostItem()->name(), host->name() ) == 0 )
          {
            hostItem = item;

            // Update the host item
            item->update( host );

            // Now update the tool tip in case it is shown:
            if ( m_widget->tooltip() && m_widget->tooltip()->isVisible() &&
                m_widget->tooltip()->item() == item )
            {
              m_widget->tooltip()->update();
            }
            else
            {
              // Do nothing
            }
          }
          else
          {
            // Do nothing
          }

          break;
        }
        default:
        {
          break;
        }
      }

      ++it;
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotInsertHost( Smb4KHost *host )
{
  if ( host )
  {
    Smb4KNetworkBrowserItem *workgroupItem = NULL;

    // Get the correct workgroup.
    QList<QTreeWidgetItem *> items = m_widget->findItems( host->workgroup(), Qt::MatchFixedString,
                                     Smb4KNetworkBrowser::Network );

    for ( int i = 0; i < items.size(); ++i )
    {
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( items.at( i ) );

      switch ( item->type() )
      {
        case Smb4KNetworkBrowserItem::Workgroup:
        {
          workgroupItem = item;

          break;
        }
        default:
        {
          break;
        }
      }
    }

    if ( workgroupItem )
    {
      // Check all children of this workgroup if the host is already
      // there.
      Smb4KNetworkBrowserItem *hostItem = NULL;

      for ( int i = 0; i < workgroupItem->childCount(); ++i )
      {
        Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( workgroupItem->child( i ) );

        switch( item->type() )
        {
          case Smb4KNetworkBrowserItem::Host:
          {
            if ( QString::compare( host->name(), item->hostItem()->workgroup() ) == 0 )
            {
              hostItem = item;
            }
            else
            {
              // Do nothing
            }

            break;
          }
          default:
          {
            break;
          }
        }
      }

      if ( !hostItem )
      {
        // The host item is not in the list. Add it.
        (void) new Smb4KNetworkBrowserItem( workgroupItem, host );
      }
      else
      {
        // Do nothing. The host is already there.
      }
    }
    else
    {
      // The workgroup is not present. Add it.
      Smb4KWorkgroup *workgroup = findWorkgroup( host->workgroup() );

      if ( workgroup )
      {
        workgroupItem = new Smb4KNetworkBrowserItem( m_widget, workgroup );

        (void) new Smb4KNetworkBrowserItem( workgroupItem, host );
      }
      else
      {
        // Do nothing (This should never happen...)
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotRescan( bool /* checked */)
{
  // The mouse is inside the viewport. Let's see what we have to do.
  if ( m_widget->currentItem() && m_widget->currentItem()->isSelected() )
  {
    Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Workgroup:
      {
        Smb4KCore::scanner()->getWorkgroupMembers( item->workgroupItem() );

        break;
      }
      case Smb4KNetworkBrowserItem::Host:
      {
        Smb4KCore::scanner()->getShares( item->hostItem() );

        break;
      }
      case Smb4KNetworkBrowserItem::Share:
      {
        Smb4KNetworkBrowserItem *parentItem = static_cast<Smb4KNetworkBrowserItem *>( item->parent() );
        Smb4KCore::scanner()->getShares( parentItem->hostItem() );

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // FIXME: Should we clear the tree widget here?

    Smb4KCore::scanner()->rescan();
  }
}


void Smb4KNetworkBrowserPart::slotAbort( bool /*checked*/ )
{
  if ( Smb4KCore::scanner()->isRunning() )
  {
    Smb4KCore::scanner()->abort();
  }
  else
  {
    // Do nothing
  }

  if ( Smb4KCore::mounter()->isRunning() &&
       Smb4KCore::mounterState() != MOUNTER_UNMOUNT )
  {
    Smb4KCore::mounter()->abort();
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotMountManually( bool /*checked*/ )
{
  Smb4KMountDialog *dlg = m_widget->findChild<Smb4KMountDialog *>();

  if ( !dlg )
  {
    dlg = new Smb4KMountDialog( m_widget );
  }
  else
  {
    // Do nothing
  }

  if ( !dlg->isVisible() )
  {
    dlg->setVisible( true );
  }
}


void Smb4KNetworkBrowserPart::slotAuthentication( bool /*checked*/ )
{
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

  if ( item )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Host:
      {
        Smb4KAuthInfo authInfo( item->hostItem() );
        Smb4KWalletManager::self()->showPasswordDialog( &authInfo, m_widget );

        break;
      }
      case Smb4KNetworkBrowserItem::Share:
      {
        Smb4KAuthInfo authInfo( item->shareItem() );
        Smb4KWalletManager::self()->showPasswordDialog( &authInfo, m_widget );

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotCustomOptions( bool /*checked*/ )
{
  Smb4KCustomOptionsDialog *dlg = m_widget->findChild<Smb4KCustomOptionsDialog *>();
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

  if ( !dlg && item )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Host:
      {
        dlg = new Smb4KCustomOptionsDialog( item->hostItem(), m_widget );

        break;
      }
      case Smb4KNetworkBrowserItem::Share:
      {
        dlg = new Smb4KCustomOptionsDialog( item->shareItem(), m_widget );

        break;
      }
      default:
      {
        break;
      }
    }
  }

  if ( dlg && !dlg->isVisible() )
  {
    if ( dlg->isInitialized() )
    {
      dlg->setVisible( true );
    }
    else
    {
      delete dlg;
    }
  }
}


void Smb4KNetworkBrowserPart::slotAddBookmark( bool /*checked*/ )
{
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

  if ( item )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        Smb4KCore::bookmarkHandler()->addBookmark( item->shareItem() );

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotPreview( bool /*checked*/ )
{
  // The user should be able to open several dialogs at a time, so
  // do not check for existing dialogs and use show() here.
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );
  Smb4KPreviewDialog *dlg = NULL;

  if ( item && !item->shareItem()->isPrinter() )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        dlg = new Smb4KPreviewDialog( item->shareItem(), m_widget );

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }

  if ( dlg && !dlg->isVisible() )
  {
    dlg->getPreview();
    dlg->setVisible( true );
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotPrint( bool /*checked*/ )
{
  Smb4KPrintDialog *dlg = m_widget->findChild<Smb4KPrintDialog *>();
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

  if ( !dlg && item )
  {
    switch( item->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        if ( item->shareItem()->isPrinter() )
        {
          dlg = new Smb4KPrintDialog( item->shareItem(), m_widget );
        }
        else
        {
          // Do nothing
        }

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }

  if ( dlg && !dlg->isVisible() )
  {
    dlg->setVisible( true );
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotMount( bool /*checked*/ )
{
  Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

  if ( item )
  {
    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        Smb4KCore::mounter()->mountShare( item->shareItem() );

        break;
      }
      default:
      {
        break;
      }
    }
  }
  else
  {
    // Do nothing
  }
}


void Smb4KNetworkBrowserPart::slotUnmount( bool /*checked*/ )
{
  switch ( m_mode )
  {
    case KonqPlugin:
    {
      Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( m_widget->currentItem() );

      if ( item )
      {
        switch ( item->type() )
        {
          case Smb4KNetworkBrowserItem::Share:
          {
            if ( item->shareItem()->isMounted() )
            {
              // FIXME: Implement forced unmounting?
              Smb4KCore::mounter()->unmountShare( item->shareItem(), false, false );
            }
            else
            {
              // Do nothing
            }

            break;
          }
          default:
          {
            break;
          }
        }
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KNetworkBrowserPart::slotMarkMountedShares()
{
  QTreeWidgetItemIterator it( m_widget );

  while ( *it )
  {
    Smb4KNetworkBrowserItem *item = static_cast<Smb4KNetworkBrowserItem *>( *it );

    switch ( item->type() )
    {
      case Smb4KNetworkBrowserItem::Share:
      {
        // Get the list of shares with the same UNC.
        QList<Smb4KShare *> list = findShareByUNC( item->shareItem()->unc() );

        if ( !list.isEmpty() )
        {
          for ( int i = 0; i < list.size(); ++i )
          {
            // Find the right share.
            if ( (!list.at( i )->isForeign() || Smb4KSettings::showAllShares()) &&
                 !item->shareItem()->isMounted() )
            {
              item->setMounted( list.at( i ), Smb4KNetworkBrowserItem::Mounted );

              break;
            }
            else if ( (list.at( i )->isForeign() && !Smb4KSettings::showAllShares()) &&
                      item->shareItem()->isMounted() )
            {
              item->setMounted( item->shareItem(), Smb4KNetworkBrowserItem::NotMounted );

              // This is a change due to a foreign share. Continue until
              // we hit the share owned by the user (if it exists).
              continue;
            }
            else
            {
              continue;
            }
          }
        }
        else
        {
          if ( item->shareItem()->isMounted() )
          {
            item->setMounted( item->shareItem(), Smb4KNetworkBrowserItem::NotMounted );
          }
          else
          {
            // Do nothing
          }
        }

        break;
      }
      default:
      {
        break;
      }
    }

    ++it;
  }
}


void Smb4KNetworkBrowserPart::slotRunStateChanged()
{
  switch ( Smb4KCore::currentState() )
  {
    case SCANNER_LOOKUP_DOMAINS:
    case SCANNER_QUERY_MASTER_BROWSER:
    case SCANNER_SCAN_BROADCAST_AREAS:
    case SCANNER_OPEN_WORKGROUP:
    case SCANNER_OPEN_HOST:
    case SCANNER_QUERY_INFO:
    {
      actionCollection()->action( "rescan_action" )->setEnabled( false );
      actionCollection()->action( "abort_action" )->setEnabled( true );

      break;
    }
    case MOUNTER_MOUNT:
    {
      actionCollection()->action( "abort_action" )->setEnabled( true );

      break;
    }
    case SCANNER_STOP:
    case MOUNTER_STOP:
    case CORE_STOP:
    {
      actionCollection()->action( "rescan_action" )->setEnabled( true );

      if ( Smb4KCore::scannerState() == SCANNER_STOP &&
           Smb4KCore::mounterState() != MOUNTER_MOUNT )
      {
        actionCollection()->action( "abort_action" )->setEnabled( false );
      }
      else
      {
        // Do nothing
      }

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KNetworkBrowserPart::slotAboutToQuit()
{
  saveSettings();
}

#include "smb4knetworkbrowser_part.moc"
