/*
 * Copyright (c) 1997 - 2001 Hansjrg Malthaner
 *
 * This file is part of the Simutrans project under the artistic licence.
 * (see licence.txt)
 */

/*
 * High-Level soundschnittstelle
 * von Hj. Maltahner, 1998, 2000
 */

#include <stdio.h>
#include <string.h>
#include "macros.h"
#include "music/music.h"
#include "besch/sound_besch.h"
#include "sound/sound.h"
#include "simsound.h"
#include "simsys.h"
#include "simio.h"
#include "simtools.h"
#include "simdebug.h"

#include "dataobj/umgebung.h"
#include "utils/simstring.h"


/**
 * max sound index
 * @author hj. Malthaner
 */
static int new_midi = 0;
static char *midi_title[MAX_MIDI];


/**
 * Gesamtlautstrke
 * @author hj. Malthaner
 */
static int global_volume = 127;


/**
 * MIDI Lautstrke
 * @author hj. Malthaner
 */
static int midi_volume = 127;

static bool shuffle_midi = false;

static int max_midi = -1; // number of MIDI files

static int current_midi = -1;  // Hajo: init with error condition,
                               // reset during loading


/**
 * setzt lautstarke fr alle effekte
 * @author Hj. Malthaner
 */
void sound_set_global_volume(int volume)
{
  // printf("Message: setting global volume to %d.\n", volume);
  global_volume = volume;
}


/**
 * ermittelt lautstarke fr alle effekte
 * @author Hj. Malthaner
 */
int sound_get_global_volume()
{
  return global_volume;
}



bool sound_get_shuffle_midi()
{
	return shuffle_midi;
}
void sound_set_shuffle_midi( bool shuffle )
{
	shuffle_midi = shuffle;
}



/**
 * spielt sound ab
 * @author Hj. Malthaner
 */
void
sound_play(const struct sound_info info)
{
	if(info.index!=(uint16)NO_SOUND) {
//DBG_MESSAGE("karte_t::interactive_event(event_t &ev)", "play sound %i",info.index);
		dr_play_sample(info.index, (info.volume*global_volume)>>8);
	}
}


/**
 * setzt Lautstrke fr MIDI playback
 * @param volume volume in range 0..255
 * @author Hj. Malthaner
 */
void sound_set_midi_volume(int volume)
{
  dr_set_midi_volume(volume);
  midi_volume = volume;
}


/**
 * Sets the MIDI volume variable - for internal use only
 * @param volume Volume in range 0-255
 * @author Owen Rudge
 */
extern "C" void sound_set_midi_volume_var(int volume)
{
	midi_volume = volume;
}


/**
 * ermittelt Lautstrke fr MIDI playback
 * @return volume in range 0..255
 * @author Hj. Malthaner
 */
int sound_get_midi_volume()
{
	return midi_volume;
}


/**
 * gets midi title
 * @author Hj. Malthaner
 */
const char * sound_get_midi_title(int index)
{
	if (index >= 0 && index <= max_midi) {
		return midi_title[index];
	}
	else {
		return "Invalid MIDI index!";
	}
}


/**
 * gets curent midi number
 * @author Hj. Malthaner
 */
int get_current_midi()
{
	return current_midi;
}


/**
 * Load MIDI files
 * By Owen Rudge
 */
int
midi_init(const char *directory)
{
	// read a list of soundfiles
	char full_path[1024];
	sprintf( full_path, "%smusic/music.tab", directory );
	FILE * file = fopen( full_path, "rb");
	if(file) {
		dr_init_midi();

		while(!feof(file)) {
			char buf[256];
			char title[256];
			long len;

			read_line(buf, 256, file);
			read_line(title, 256, file);
			if(!feof(file)) {
				len = strlen(buf);
				while(len>0  &&  buf[--len] <= 32) {
					buf[len] = 0;
				}

				if(len > 1) {
					sprintf( full_path, "%s%s", directory, buf );
					print("  Reading MIDI file '%s' - %s", full_path, title);
					max_midi = dr_load_midi(full_path);

					if(max_midi >= 0) {
						len = strlen(title);
						while(len>0  &&  title[--len] <= 32) {
							title[len] = 0;
						}
						midi_title[max_midi] = strdup(title);
					}
				}
			}
		}

		fclose(file);
		dr_set_midi_volume(midi_volume);
	} else {
		dbg->warning("midi_init()","can't open file 'music.tab' for reading, turning music off.");
	}

	if(max_midi >= 0) {
		current_midi = 0;
	}
	// success?
	return max_midi>=0;
}


void midi_play(const int no)
{
	if (no > max_midi) {
		dbg->warning("midi_play()", "MIDI index %d too high (total loaded: %d)", no, max_midi);
	} else {
		dr_play_midi(no);
	}
}


void midi_stop()
{
  dr_stop_midi();
}


/*
 * Check if need to play new MIDI
 */
void check_midi()
{
  if((dr_midi_pos() < 0 || new_midi == 1) && max_midi > -1) {
		if(shuffle_midi  &&  max_midi>1) {
			// shuffle songs
			int new_midi = simrand(max_midi-1);
			if(new_midi>=current_midi) {
				new_midi ++;
			}
			current_midi = new_midi;
		}
		else {
	    current_midi++;
			if (current_midi > max_midi) {
		    current_midi = 0;
			}
		}

    midi_play(current_midi);
    DBG_MESSAGE("check_midi()", "Playing MIDI %d", current_midi);
    new_midi = 0;
  }
}


/**
 * shuts down midi playing
 * @author Owen Rudge
 */
void close_midi()
{
  dr_destroy_midi();
}


void midi_next_track()
{
  new_midi = 1;
}


void midi_last_track()
{
	if (current_midi == 0) {
    current_midi = max_midi - 1;
	}
	else {
    current_midi = current_midi - 2;
	}
  new_midi = 1;
}
