/***************************************************************************
 *  Module:      $Id: io.c,v 2.20 1997/12/31 05:18:32 nemesis Exp nemesis $
 *  Description: Lower level io for PET protocol and modem
 *  Author:      maf, cjc
 *
 * Copyright (c) 1995 Mark Fullmer and The Ohio State University
 * Copyright (c) 1997 Cornelius Cook and Counterpoint Networking, Inc.
 * http://www.cpoint.net/projects/sendpage
 ***************************************************************************/
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
$Log: io.c,v $
Revision 2.20  1997/12/31 05:18:32  nemesis
CLOCAL for all, GCC fixes, WAIT_WORD finished

Revision 2.19  1997/12/29 02:45:13  nemesis
include strengthening.  works with db 2.0+1.85 compats.  compiles/runs on HPUX

Revision 2.18  1997/12/28 23:20:45  nemesis
include cleanups, configure additions/corrections

Revision 2.17  1997/12/26 02:56:22  nemesis
bug fixes

Revision 2.16  1997/12/26 02:37:31  nemesis
code clean up, b* -> mem* funcs, finding NL -> # bug

Revision 2.15  1997/12/25 08:30:52  nemesis
code cleanups

Revision 2.14  1997/12/25 06:44:34  nemesis
time_t declarations and "make install" fix

Revision 2.13  1997/12/24 21:50:04  nemesis
mailing list updates

Revision 2.12  1997/12/24 21:42:54  nemesis
0.8a released.

Revision 2.11  1997/12/24 21:02:02  nemesis
more changes

Revision 2.10  1997/12/24 20:56:03  nemesis
gearing up for 0.8a more

Revision 2.9  1997/12/24 20:45:35  nemesis
trying to make 0.8a release

Revision 2.8  1997/12/24 20:29:08  nemesis
fixed up autoconf modifications, cleaned up signal stuff

Revision 2.7  1997/12/24 20:15:13  nemesis
sendpage.h now mostly contained within 'configure'

Revision 2.6  1997/12/24 19:52:04  nemesis
fixing posix checking

Revision 2.5  1997/12/24 19:41:49  nemesis
posix additions, syslog autoconf'd

Revision 2.4  1997/12/24 19:33:03  nemesis
check for POSIX

Revision 2.3  1997/12/17 09:37:27  nemesis
more autoconf changes... mostly strerror.o

Revision 2.2  1997/12/17 08:24:05  nemesis
autoconf-ing

Revision 2.1  1997/12/17 08:03:38  nemesis
adjustments

Revision 2.0  1997/12/17 08:01:05  nemesis
setting up autoconf

Revision 1.9  1997/12/17 07:56:23  nemesis
starting on the autoconfing

Revision 1.8  1997/12/17 04:47:05  nemesis
still adjusting

Revision 1.7  1997/12/17 04:46:25  nemesis
adjusting version numbers

Revision 1.6  1997/12/17 04:44:36  nemesis
*** empty log message ***

Revision 1.5  1997/12/17 04:19:29  nemesis
more cpoint mods

Revision 1.4  1997/12/17 04:14:20  nemesis
cpoint mods

Revision 1.3  1997/12/17 04:04:54  nemesis
cpoint compatibility mods

Revision 1.2  1997/12/15 17:33:28  nemesis
patches by Steve Kennedy <steve@demon.net> added.
See http://www.gbnet.net/public/paging/sendpage-multi/

Revision 1.1  1997/12/15 15:56:58  nemesis
Initial revision

 * Revision 1.12  1996/02/11  06:02:42  maf
 * modem results, fix modem speed
 *
 * Revision 1.11  1995/12/18  00:23:31  maf
 * ignore RING response after dialing
 *
 * Revision 1.10  1995/12/18  00:01:14  maf
 * fix for modem hungup, debugging for modem locking
 *
 * Revision 1.9  1995/11/13  04:41:19  maf
 * solaris patch
 *
 * Revision 1.8  1995/11/13  03:28:48  maf
 * *** empty log message ***
 *
 * Revision 1.7  1995/10/09  01:40:11  maf
 * *** empty log message ***
 *
 * Revision 1.6  1995/08/28  03:40:43  maf
 * tabs, Linux patch for read() and EAGAIN
 *
 * Revision 1.5  1995/08/28  02:59:32  maf
 * ioctl.h
 *
 * Revision 1.4  1995/08/28  02:55:06  maf
 * osf/1 patches
 *
 * Revision 1.3  1995/05/24  00:56:04  maf
 * all unknown string sizes use %.512s for report()
 * merged in AIX contributed port daryl@tcomeng.com & mark@tcomeng.com
 * merged in HP/UX contributed port Mike Lutz & Heath Kehoe @ norand.com
 * I haven't tested the contributed ports...
 *
 * Revision 1.2  1995/03/15  04:40:53  maf
 * *** empty log message ***
 *
 * Revision 1.1  1995/01/10  01:43:05  maf
 * Initial revision
 *
*/

/*
#if defined(sun)
#  if defined(__SVR4) && !defined(sunos5)
#    define sunos5
#  endif
#endif
*/

/*
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
*/

/*
#ifdef sunos5
#include <termio.h>
#else
#include <termios.h>
#endif
*/

#include "sendpage.h"
#include "io.h"
#include "report.h"
#include "misc.h"

#ifdef HAVE_SYS_MODEM_H
#include <sys/modem.h>
#endif

/* the STRICT lookup is what my spec says.
 * the less strict lookup is what our paging central switched to
 * last week :(
 * also noted by by Adrian Collins <ade@manchester-computing-centre.ac.uk>
 * who just had problems with the goodbye message */
static struct pet_lookup pet_lookup[] = {

#ifdef PET_LESS_STRICT_LOOKUP

        { "\006\015",                   2},     /* accept */
        {"\025\015",                    2},     /* reject */
        {"\036\015",                    2},     /* abandon */
        {"\033\004\015",                3},     /* goodbye */
        {"\033[p\015",                  4}      /* goahead */

#else /* PET_LESS_STRICT_LOOKUP */

        {"\015\006\015",                3},     /* accept */
        {"\015\025\015",                3},     /* reject */
        {"\015\036\015",                3},     /* abandon */
        {"\015\033\004\015",            4},     /* goodbye */
        {"\033[p\015",                  4}      /* goahead */

#endif /* PET_LESS_STRICT_LOOKUP */
};

/*********************************************************************
 * Function: OpenModem
 *
 * Returns: 0 for good (the device was opened and initialized)
 *          !0  for bad 
 *
 *********************************************************************/
int OpenModem(struct pcinfo *pcinfo, int *fd) {
  extern int errno, debug;
#ifndef HAVE_TERMIOS_H
  struct termio t;
#else
  struct termios t;
#endif
  
#ifdef _POSIX_SOURCE
  int val;
#endif /* _POSIX_SOURCE */
  
#ifdef DEBUG
  if (debug > 2)
    report (LOG_INFO, "opening modem");
#endif /* DEBUG */
  
  /* open the modem device */
  if
#ifdef _POSIX_SOURCE
	 ((*fd = open (pcinfo->modemDev, O_RDWR | O_NONBLOCK, 0)) == -1)
#elif defined(hpux) || defined(__osf__) || defined(sunos5)
	 ((*fd = open (pcinfo->modemDev, O_RDWR | O_NOCTTY, 0)) == -1)
#else
	 ((*fd = open (pcinfo->modemDev, O_RDWR | O_NDELAY, 0)) == -1)
#endif /* hpux || osf */
    {
      report(LOG_ERR, "open (%.512s): %s", pcinfo->modemDev, strerror(errno));
      return 1; /* bad */
    }
  
  /* setup the device (baud rate, etc) */
  
  /* get the current device params */
  if
#ifndef HAVE_TERMIOS_H
	 (ioctl(*fd, TCGETA, &t))
#else
	 (tcgetattr(*fd, &t))
#endif
    {
      report (LOG_ERR, "tcgetattr (modem): %s", strerror(errno));
      close (*fd);
      return 1; /* bad */
    }
  
  /* input flags :
     ignore breaks, use XON/XOFF flow control
     ignore parity errors */
  t.c_iflag = IGNBRK|IXON|IXOFF|IGNPAR;
  
  /* strip the 8th bit if not on 8 bit mode */
  if (pcinfo->databits != CS8)
    t.c_iflag |= ISTRIP;
  
  /* output flags:
     (none) */
  t.c_oflag = 0;
  
  /* hardware control flags:
     enable receiver, modem control lines go low on close(), use RTS/CTS
     (hardwire) flow control to modem */
  t.c_cflag = pcinfo->databits|pcinfo->parity|pcinfo->stopbits|
    CREAD|C_FLAG;
  
  /* Is there a more portable way to do this?  First need to clear
   * out all speed bits, otherwise speed can be set to indeterminate
   * value when just blindly ORing desired speed into the structure
   */
  t.c_cflag &= (~31L);
  
  t.c_cflag |= pcinfo->speed;
  
  /* local flags:
     (none) */
  t.c_lflag = 0;
  
  /* allow read() to return 0 bytes (ie don't block), accept wait atleast */
  /* one second before returning 0 bytes */
  t.c_cc[VMIN] = 0;
#ifdef SLOW_PAGING_CENTRAL
  t.c_cc[VTIME] = 60;
#else
  t.c_cc[VTIME] = 10; /* 1/10ths of seconds */
#endif /* SLOW_PAGING_CENTRAL */
  
  /* set device params */
  if
#ifndef HAVE_TERMIOS_H
	 (ioctl(*fd, TCSETA, &t))
#else
	 (tcsetattr(*fd, TCSANOW, &t))
#endif
    {
      report (LOG_ERR, "tcsetattr (modem): %s", strerror(errno));
      close (*fd);
      return 1; /* bad */
    }
  
#ifdef _POSIX_SOURCE
  /**
   ** UVM mod alb 9/14/95 for AIX 4.1.3 (POSIX)
   **
   ** We needed to open the modem with O_NONBLOCK set, since an open
   ** of a port with a modem will wait for carrier before returning,
   ** which is clearly what we don't want to do when dialing out.
   ** However, with O_NONBLOCK set, the VTIME wait won't be honored,
   ** and read() will return immediately with no data; this is visible
   ** in a debug trace.  We need to unset the O_NONBLOCK flag prior
   ** to returning, which is what we'll do here.
   */
  
  if ( (val = fcntl(*fd, F_GETFL, 0)) < 0) {
    report (LOG_ERR, "fcntl F_GETFL error (modem): %s", strerror(errno));
    close (*fd);
    return 1; /* bad */
  }
  
  val &= ~O_NONBLOCK;   /* turn off nonblocking */
  
  if (fcntl(*fd, F_SETFL, val) < 0) {
    report (LOG_ERR, "fcntl F_SETFL error (modem): %s", strerror(errno));
    close (*fd);
    return 1; /* bad */
  }
  
  /**
   ** End UVM mod
   */
#endif /* _POSIX_SOURCE */
  
  return 0; /* good */
} /* OpenModem */

/*********************************************************************
 * Function: ResetModem
 *
 * Returns: 0 for good (the device was reset and is responding)
 *          !0  for bad 
 *
 *********************************************************************/
int ResetModem(int fd, struct cbuf *cbuf, struct pcinfo *pcinfo) {
  extern int errno, debug;
#ifdef HAVE_SYS_MODEM_H
  unsigned long modem;
#else
  int modem;
#endif /* hpux */
  

#ifdef TOGGLE_DTR
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Toggle DTR enter");
#endif /* DEBUG */

  /* first toggle DTR - some modems will go to command state when
     doing this */
  
  /* get the current modem control lines */
  if (debug >90)
    report (LOG_INFO, "Get current Modem Control lines");
  
#ifdef HAVE_SYS_MODEM_H
  if (ioctl(fd, MCGETA, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, MCGETA): %s", strerror(errno));
#else 
  if (ioctl(fd, TIOCMGET, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, TIOMGET): %s", strerror(errno));
#endif /* hpux */
    return 1; /* bad */
  }
	
  /* turn off DTR */
#ifdef HAVE_SYS_MODEM_H
  modem &= ~MDTR;
#else
  modem &= ~TIOCM_DTR;
#endif /* hpux */
  
  sleep (1);
  
	/* tell the modem */
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Tell Modem to Turn Off DTR");
#endif /* DEBUG */
  
#ifdef HAVE_SYS_MODEM_H
  if (ioctl(fd, MCSETA, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, MCSETA): %s", strerror(errno));
#else
# ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Setting current Modem Control lines");
# endif /* DEBUG */
  if (ioctl(fd, TIOCMSET, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, TIOCMSET): %s", strerror(errno));
#endif /* hpux */
    return 1; /* bad */
  }
    
  sleep (1);
    
#ifdef HAVE_SYS_MODEM_H
  modem |= MDTR;
#else
  modem |= TIOCM_DTR;
#endif /* hpux */

  /* tell the modem */
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Tell Modem to Turn On DTR");
#endif /* DEBUG */
  
#ifdef HAVE_SYS_MODEM_H
  if (ioctl(fd, MCSETA, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, MCSETA): %s", strerror(errno));
#else
  if (ioctl(fd, TIOCMSET, &modem) == -1) {
    report (LOG_ERR, "ioctl (modem, TIOCMSET): %s", strerror(errno));
#endif /* hpux */
    return 1; /* bad */
  }
  
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Toggle DTR exit");
#endif /* DEBUG */
  
#endif /* TOGGLE_DTR */
  
  /* give the modem a chance to recover */
  sleep (1);
  
#ifdef SEND_MODEM_PLUS
  /* now send the +++ wait ath */
  
  if (SendString (fd, " "))
    return 1;
  
  sleep (2);
  
  if (SendString(fd, MODEM_PLUS))
    return 1;
  
  sleep (3);
  
  if (SendString (fd, "\r"))
    return 1;
  
  /* depending on the previous state of the modem, different things
     could show up here - OK\r or the short number command for OK or
     whatever else some goofy clone decided to do.  Forget the return
     code here, instead just go ahead and send the init command */
  
  sleep (1);
#endif /* SEND_MODEM_PLUS */
  
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Send Modem init sequence");
#endif /* DEBUG */
  
  if (SendString(fd, pcinfo->modemInit))
    return 1;
  
  if (SendString(fd, "\r"))
    return 1;
  
#ifdef WAIT_WORD
  /* Now wait for the OK short result code */
  
  if (WaitString(fd, MODEM_WORD_OK, cbuf)) 
    return 1;
#else
  /* Or wait for the OK short result code */
  
  if (WaitString(fd, "0\r", cbuf))
    return 1;
#endif /* WAIT_WORD */
  
#ifndef WAIT_WORD
  
  /* the previous result code could of been from a few sources
   * sync up us to the modem by setting the return code to the word
   * form, then the number form */
  
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Send Modem Word Form ret code init");
#endif /* DEBUG */
  
  if (SendString (fd, MODEM_WORDS))
    return 1;
  
  /* should get an OK\r */
  if (WaitString (fd, MODEM_WORD_OK, cbuf))
    return 1;
  
#ifdef DEBUG
  if (debug >90)
    report (LOG_INFO, "Send Modem Number form ret code init");
#endif /* DEBUG */
  
  if (SendString (fd, MODEM_NUMS))
    return 1;
  
  /* should get an 0\r */
  if (WaitString (fd, MODEM_NUM_OK, cbuf))
    return 1;
  
#endif /* not WAIT_WORD */
  
  /* that's it */
  
  return 0; /* good */
  } /* Reset Modem */

/*********************************************************************
 * Function: SendString
 * Returns:  0  for good (string was sent)
 *          !0  for bad 
 *********************************************************************/
int SendString (int fd, char *string) {
  return SendBytes (fd, string, strlen(string));
} /* SendString */


/*********************************************************************
 * Function: WaitString
 *
 * Returns:  0  for good (string was received)
 *          !0  for bad 
 *
 *	It's assumed the fd is setup so that a read() will block for atleast
 *	1 second before returning 0 bytes.
 *
 *********************************************************************/
int WaitString (int fd, char *msg, struct cbuf* cbuf) {
  int msglen, goodbytes, x, n, retries, retries2, matchflag;
  extern int errno, debug;

  msglen = strlen(msg);
  retries = 5;		/* try the loop 5 times */
  retries2 = 50;		/* inner loop 50 times (just a safety) */
  goodbytes = 0;		/* no bytes matched yet */
  matchflag = 0;		/* no full match of msg */
  
#ifdef DEBUG
  if (debug > 2) 
    report (LOG_INFO, "WaitString (%d bytes) - %.512s",
	    msglen, PrintBytes (msg, msglen));
#endif /* DEBUG */
  
  /* try retries times to read when there is nothing to read */
  /* if there was something to read, then retries2 also gets counted */
  
  for (; (retries > 0) && (!matchflag); --retries) {
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "retry = %d", retries);
#endif /* DEBUG */
    
    
    if ((n = SafeRead(fd, cbuf->buf+cbuf->bufused,
		      CBUF_BYTES - cbuf->bufused)) == -1) 
      return 1;
    
    cbuf->bufused += n;
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "cbuf (%d bytes) - %.512s", cbuf->bufused,
	      PrintBytes (cbuf->buf, cbuf->bufused));
#endif
    
    /* try to match */
    for (x = goodbytes; x < cbuf->bufused; ++x) {
      
      /* if the byte doesn't match, discard the previous bytes
	 that matched, else setup to try next */
      if (msg[goodbytes] != cbuf->buf[x]) 
	goodbytes = 0;
      else
	++goodbytes;
      
      if (goodbytes == msglen) { /* full match */
	matchflag = 1;
	++x;
	break;
      }
    } /* for x */
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO,
	      "matchflag=%d, x=%d, bufused=%d, goodbytes=%d msglen=%d", 
	      matchflag, x, cbuf->bufused, goodbytes, msglen);
#endif /* DEBUG */
    
    /* save the end of the buffer that didn't match for next time */
    if (matchflag) {
      memcpy(cbuf->buf, cbuf->buf+x, cbuf->bufused-x);
      cbuf->bufused -= x;
    } else {
      memcpy(cbuf->buf, cbuf->buf+(x-goodbytes), goodbytes);
      cbuf->bufused = goodbytes;
    }
    
    /* give the device some time to talk */
    if (!matchflag && n)
      sleep (1);
    
    /* if bytes were actually read, keep trying a little longer */
    if (n && retries2) 
      --retries2, ++retries;
    
  } /* for retries */
  
#ifdef DEBUG
  if (debug > 2)
    report (LOG_INFO, "%smatch", matchflag ? "" : "no" );
#endif /* DEBUG */
  
  return (matchflag) ? 0 : 1;
} /* WaitString */
 
/*********************************************************************
 * Function: SafeRead
 *
 * Execute read with check and log
 *	Returns -1 for error.
 *          
 *
 *********************************************************************/
int SafeRead (int fd, char * buf, int len) {
  int n;
  
  if (((n = read(fd, buf, len)) == -1) && (errno != EAGAIN)) {
    report (LOG_ERR, "read(): %s", strerror(errno));
    return -1;
  }
  
  /* getting an EAGAIN is the same as returning 0 bytes */
  if ((n == -1) && (errno == EAGAIN))
    return 0;
  
  return n;
}

#ifdef DEBUG
/*********************************************************************
 * Function: PrintBytes
 *
 * Returns: pointer to upto 132 chars of string
 *          
 *
 *********************************************************************/
char *PrintBytes (char * buf, int len) {
  static char tmpBuf[132];
  char s[10], *c;
  int x;
  
  for (c = buf, x = 0; c != (buf + len); ++c) {
    
    if (isalnum((int)*c)) {
      if (x <= 129) {
	tmpBuf[x++] = *c; 
	tmpBuf[x++] = ' '; 
      } /* if */
      else break;
    } /* if */
    else {
      if (x <= 128) {
	sprintf(s, "%2.2x ", (int)*c);
	memcpy(&tmpBuf[x], s, 3);
	x += 3;
      } /* if */
      else break;
    } /* else */
    
  } /* for */
  
  tmpBuf[x] = 0;
  
  return tmpBuf;
} /* PrintBytes */
 
#endif /* DEBUG */
 
/*********************************************************************
 * Function: DialModem
 *
 * Returns: MODEM_RES_?? constants
 *
 *********************************************************************/
int DialModem(int fd, struct cbuf* cbuf, struct pcinfo* pcinfo) {
  extern int errno, debug;
  int x, res;
  char dialstring[PCINFO_STR_LEN*2+1];
  
  strcpy(dialstring, pcinfo->modemDial);
  strcat(dialstring, pcinfo->phone);
  strcat(dialstring, "\r");
  
  res = MODEM_RES_RING; /* ignore this it's a RING indicator */
  
  if (SendString (fd, dialstring))
    return MODEM_RES_INTER;
  
  while (res == MODEM_RES_RING) {
    
    /* GetResult in cbuf */
    if (WaitLine (fd, cbuf, pcinfo->answerTime))
      return MODEM_RES_INTER;
   
    /* parse and return the result */
    for (x = 0; x < cbuf->bufused; ++x) {
      
      if (cbuf->buf[x] == 0x0d) {
	cbuf->buf[x] = '\0';
#ifdef WAIT_WORD
	/* we've "used" these characters */
	x++;
#endif
	break;
      }
    } /* for */
    
#ifndef WAIT_WORD
    /* convert to int */
    res = atoi((char*)cbuf->buf);
#else /* def WAIT_WORD */
    if (strstr(cbuf->buf,"CONNECT")) res=MODEM_RES_C300;
#endif
    
#ifdef DEBUG
    if (debug > 5) 
      report (LOG_INFO, "Modem result: %s (%d)", pr_ModemResultCode(res),
	      res);
#endif /* DEBUG */
    
    
    /* discard what just got picked off */
    memcpy(cbuf->buf, cbuf->buf+x, cbuf->bufused-x);
    cbuf->bufused -= x;
  }
  
  return res;
  
#if 0 /* not WAIT_WORD */
  
  if (SendString (fd, dialstring))
    return MODEM_RES_INTER;
  
  /* Wait for "CONNECT" */
  if (Match (fd, "CONNECT", pcinfo->answerTime))
    return MODEM_RES_INTER;
  
# ifdef DEBUG
  
  if (debug > 5)
    report (LOG_INFO, "Modem result: %s", pr_ModemResultCode(MODEM_RES_C300));
  
# endif /* DEBUG */
  
  /* return a connect result */
  return MODEM_RES_C1200;

#endif /* not WAIT_WORD */

} /* DialModem */

/*********************************************************************
 * Function: WaitLine
 *
 * Returns:  0  for good (a line (string ending in 0x0d) was received)
 *          !0  for bad 
 *
 *********************************************************************/
int WaitLine (int fd, struct cbuf* cbuf, int retries) {
  int x, n, retries2, matchflag;
  extern int errno, debug;

  retries2 = 50;		/* inner loop 50 times (just a safety) */
  matchflag = 0;		/* got a line */
  
  /* try retries times to read when there is nothing to read */
  /* if there was something to read, then retries2 also gets counted */
  
  for (; (retries > 0) && (!matchflag); --retries) {
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "retry = %d", retries);
#endif /* DEBUG */
    
    /* leave enough space for a NULL */ 
    if ((n = SafeRead(fd, cbuf->buf+cbuf->bufused,
		      CBUF_BYTES - cbuf->bufused - 1)) == -1) 
      return 1;
    
    cbuf->bufused += n;
    /* make sure it's always NULL terminated */
    cbuf->buf[cbuf->bufused]='\0';
    
#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "cbuf (%d bytes) - %.512s", cbuf->bufused,
	      PrintBytes (cbuf->buf, cbuf->bufused));
#endif
    
    /* try to match */
    for (x = 0; x < cbuf->bufused; ++x) {
      
      if (cbuf->buf[x] == 0x0d) {
	matchflag = 1;
	break;
      }
      
    } /* for x */
    
    /* give the device some time to talk */
    if (!matchflag && n)
      sleep (1);
    
    /* if bytes were actually read, keep trying a little longer */
    if (n && retries2) 
      -- retries2, ++retries;
    
  } /* for retries */
  
  return (matchflag) ? 0 : 1;
} /* WaitLine */

	
/*********************************************************************
 * Function: StartPET
 *
 * Returns: 0 for good (The logon sequence succeeded) 
 *			1 for bad 
 *
 *********************************************************************/
int StartPET(int fd, struct cbuf*cbuf, struct pcinfo *pcinfo) {
  extern int errno;
  int retrystart, retrylogin, gotid, gotlogin, res=1, earlygoahead;
  char passwd[8];
  char *prot;
  
  gotid = 0;		/* got the PET_ID */
  gotlogin = 0;	        /* got through the login sequence */
  retrystart = 3;	/* number of times to try for PET_ID */
  retrylogin = 3;	/* number of times to try for */
  earlygoahead = 0;	/* got an early go ahead message */
  
  /* construct password line */
  strncpy(passwd, pcinfo->password, 6);
  passwd[6] = 0x0d;
  passwd[7] = 0;

#ifdef SLOW_PAGING_CENTRAL
  sleep (3);
#endif /* SLOW_PAGING_CENTRAL */

  /* Protocol requires 3 retries of sending CR, waiting for ID= */
  while (--retrystart >= 0) {
#ifdef SLOW_PAGING_CENTRAL
    sleep(3);
    if (SendString (fd, "\r")) return 1; /* bad */
    sleep(2);
#else
    if (SendString (fd, "\r")) return 1; /* bad */
#endif /* SLOW_PAGING_CENTRAL */
    if (!WaitString (fd, PET_ID, cbuf)) {
      gotid = 1;
      break;
    }
  }

  if (!gotid) {
    report (LOG_ERR, "Never got %s (initial logon sequence)", PET_ID);
    return 1; /* bad */
  }
  
  while (--retrylogin >= 0) {
    /* put remote side into automatic remote entry mode */
    
    if (pcinfo->protocol == PC_PET1)
      prot = PET_START_SEQ1;
    else if (pcinfo->protocol == PC_PET3)
      prot = PET_START_SEQ3;
    else
      prot = "";

    if (SendString (fd, prot)) return 1; /* bad */

    /* send the password followrd by 0x0d */
    if (SendString (fd, passwd)) return 1; /* bad */

    /* at this point, paging central can do a few things
       1) reply with ID= meaning send the PET_START_SEQ/passwd again 
       2) reply with <CR> <ACK> <CR> meaning ok, go ahead
       3) reply with <CR> <NAK> <CR> meaning try again
       4) reply with <CR> <ESC> <EOT> <CR> meaning get lost
       5) reply with something else, meaning who knows what */

    /* The ID= response is problematic, since it could of been an ID=
       that previously wasn't read.  For this reason, the ID= response
       is handled as case 5 (ie 2,3,4 weren't matched), and things
       start over with the PET_START_SEQ upto retrylogin times */

    /* Wait for a PET reply */
    res = WaitPETReply (fd, cbuf);

    /* accept PET_REPLY_ACCEPT */
    if (res == PET_REPLY_ACCEPT) {
      gotlogin = 1;
      break;
    }
    /* my spec says this is wrong, ofcourse that means nothing when
       paging central goes ahead and does it anyways */
    if (res == PET_REPLY_GOAHEAD) {
      gotlogin = 1;
      earlygoahead = 1;
      break;
    }

    report (LOG_INFO, "Unexpected reply during login, retrying - %d", res);
    
  } /* while retrylogin */
  
  if (!gotlogin) {
    report (LOG_ERR, "Never got through login sequence - reason %d", res);
    return 1; /* bad */
  }

  /* wait for PET_GOAHEAD message */
  if (!earlygoahead) {
    res = WaitPETReply (fd, cbuf);

    if (res != PET_REPLY_GOAHEAD) {
      report (LOG_ERR, "Never got go ahead message - reason %d", res);
      return 1; /* bad */
    }
  }
  
  return 0; /* good */
} /* StartPET */

/*********************************************************************
 * Function: WaitPETReply
 *
 * Returns: PET_REPLY_??
 *
 *********************************************************************/
int WaitPETReply (int fd, struct cbuf *cbuf) {
  int x, n, retries, retries2, pet, matchflag, goodbytes, j;
  extern int errno, debug;
  int pet_goodbytes[PET_REPLIES];

  retries = 10;		/* upto 10 seconds of idle time per protocol spec */
  retries2 = 50;	/* 50 times (just a safety) */
  matchflag = 0;
  goodbytes = 0;

  for (pet = 0; pet < PET_REPLIES; ++pet)
    pet_goodbytes[pet] = 0;

  /* try retries times to read when there is nothing to read */
  /* if there was something to read, then retries2 also gets counted */

  for (; (retries > 0) && (!matchflag); --retries) {

#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "retry = %d", retries);
#endif /* DEBUG */

    if ((n = SafeRead(fd, cbuf->buf+cbuf->bufused,
		      CBUF_BYTES - cbuf->bufused)) == -1) 
      return 1;
    
    cbuf->bufused += n;

#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "cbuf (%d bytes) - %.512s", cbuf->bufused,
	      PrintBytes (cbuf->buf, cbuf->bufused));
#endif
    
    
    /* try to match PET_REPLY_?? */
    for (x = goodbytes; x < cbuf->bufused; ++x) {

      /* protocol allows optional linefeeds after carriage returns -
	 just ignore them all. */
      if (cbuf->buf[x] == ASCII_LF) 
	continue;

      for (pet = 0; pet < PET_REPLIES; ++pet) {
	
	if (cbuf->buf[x] != pet_lookup[pet].msg[pet_goodbytes[pet]])
	  pet_goodbytes[pet] = 0;
	else 
	  ++pet_goodbytes[pet];
	
	if (pet_goodbytes[pet] == pet_lookup[pet].msglen) {
	  matchflag = 1;
	  ++x;
	  break;
	} /* if */
#ifdef DEBUG
	if (debug > 2)
	  report (LOG_INFO,
		  "pet=%d, matchflag=%d, pet_lookup[].msglen=%d, pet_goodbytes[]=%d",
		  pet, matchflag, pet_lookup[pet].msglen, pet_goodbytes[pet]);
#endif /* DEBUG */
	
      } /* for pet */
      
      if (matchflag)
	break;
      
    } /* for x */
    
    /* calculate the max # of bytes in this buf to save */
    for (j = 0, goodbytes = 0; j < PET_REPLIES; ++j) 
      if (pet_goodbytes[j] > goodbytes)
	goodbytes = pet_goodbytes[j];

#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "matchflag=%d, x=%d, bufused=%d, goodbytes=%d pet=%d",
	      matchflag, x, cbuf->bufused, goodbytes, pet);
#endif /* DEBUG */

    /* save the end of the buffer that didn't match for next time */
    if (matchflag) {
      memcpy(cbuf->buf, cbuf->buf+x, cbuf->bufused-x);
      cbuf->bufused -= x;
    } else {
      memcpy(cbuf->buf, cbuf->buf+(x-goodbytes), goodbytes);
      cbuf->bufused = goodbytes;
    }

#ifdef DEBUG
    if (debug > 2)
      report (LOG_INFO, "after memcpy buf (%d bytes) - %.512s", cbuf->bufused,
	      PrintBytes (cbuf->buf, cbuf->bufused));
#endif /* DEBUG */
    
    /* give the device some time to talk */
    if (!matchflag && n)
      sleep (1);

    /* if bytes were actually read, keep trying a little longer */
    if (n && retries2)
      -- retries2, ++retries;
		
  } /* for retries */
  
  return (matchflag) ? pet : PET_ERROR;
} /* WaitPETReply */

	
/*********************************************************************
 * Function: SendPETTransaction
 *
 * Returns: 0 for good (the transaction was accepted)
 * 	    1 for bad (some kind of error that should be retried)
 *	    2 for bad (page rejected)
 *	    3 for bad (all pages rejected)
 *
 *********************************************************************/
int SendPETTransaction(int fd, struct cbuf*cbuf, char *field1, char *field2,
		      int protocol) {
  extern int errno, debug;
  int retries, goodsend;
  char pet_packet [256];
  int pet_len, res, pagereject, abandon;
  
  pet_len = 256;
  retries = 3;
  goodsend = 0;
  pagereject = 0;
  abandon = 0;

  /* create the pet_packet */
  if (MakePETPacket(pet_packet, &pet_len, field1, field2, protocol))
    return 1;

  while ((--retries >= 0) && (!goodsend)) {
    
    if (SendBytes (fd, pet_packet, pet_len))
      return 1;
    res = WaitPETReply (fd, cbuf);
    
    switch (res) {
    case PET_REPLY_ACCEPT:
      report (LOG_INFO, "transaction accepted");
      goodsend = 1;
      break;
      
    case PET_REPLY_REJECT:
      report (LOG_WARNING, "transaction rejected, retrying");
      pagereject = 1;
      break;
      
    case PET_REPLY_ABANDON:
      report (LOG_ERR, "abandon reply, giving up");
      abandon = 1;
      retries = 0;
      break;

    case PET_REPLY_GOODBYE:
      report (LOG_ERR, "goodby after transaction -- did it send?");
      retries = 0;
      break;
      
    default:
      report (LOG_ERR, "unexpected response from WaitPETReply - %d",
	      res);
      break;
    } /* switch */

  } /* while --retries */

#ifdef DEBUG
  if (debug > 2)
    report (LOG_INFO, "SendPETTransaction exiting with goodsend=%d", 
	    goodsend);
#endif /* DEBUG */

  if (goodsend)
    return 0;
  else if (abandon)
    return 3;
  else if (pagereject)
    return 2;
  else
    return 1;

} /* SendPETTransaction */

/*********************************************************************
 * Function: MakePETPacket
 *
 * Returns: 0 for good (the packet was successfully constructed)
 *	    1 for bad 
 *
 *********************************************************************/
int MakePETPacket(char *buf, int *buflen, char *field1, char *field2, 
	          int protocol) {
  int len1, len2, x, i;
  int csum;

  len1 = strlen(field1);
  len2 = strlen(field2);
  csum = 0;

  if ((len1 + len2 + 8) > *buflen)
    return 1;

  /* start with a STX */
  buf[0] = ASCII_STX;
  i = 1;

  /* add field 1 */
  memcpy(&buf[i], field1, len1);
  i += len1;

  /* field 1 ends with CR */
  buf[i] = ASCII_CR;
  ++i;

  /* add field 2 */
  memcpy(&buf[i], field2, len2);
  i += len2;

  /* field 2 ends with CR */
  buf[i] = ASCII_CR;
  ++i;

  /* if the protocol is PC_PET3 need to add CR again for destination */
  if (protocol == PC_PET3) 
    buf[i++] = ASCII_CR;

  /* end fields with ETX */
  buf[i] = ASCII_ETX;
  ++i;

  /* 7 bit checksum follows */
  for (x = 0; x < i; ++x)
    csum += (buf[x] & 0x7f);

  /* the checksum is represented as 3 ascii characters having the values
     between 0x30 and 0x3f */
  /* tack it on backwards */
  buf[i+2] = 0x30 + (csum & 0x0f);
  csum >>= 4;
  
  buf[i+1] = 0x30 + (csum & 0x0f);
  csum >>= 4;

  buf[i] = 0x30 + (csum & 0x0f);
  
  /* end it all with a CR */
  buf[i+3] = ASCII_CR;
  
  *buflen = i+4;
  
  return 0;
} /* MakePETPacket */

/*********************************************************************
 * Function: SendBytes
 *
 * Returns: 0 for good (bytes were sent)
 *          !0  for bad 
 *
 *********************************************************************/
int SendBytes (int fd, char *bytes, int len) {
  extern int errno, debug;

#ifdef DEBUG
  if (debug > 2) 
    report (LOG_INFO, "SendBytes (%d bytes) -  %.512s", len,
	    PrintBytes (bytes, len));
#endif /* DEBUG */

  if (write (fd, bytes, len) == -1) {
    report(LOG_ERR, "write(): %s", strerror(errno));
    return 1; /* bad */
  }
  
  return 0; /* good */
} /* SendBytes */

/*********************************************************************
 * Function: EndPETTransaction
 *
 * Returns: 0 for good the (transaction was ended)
 *			1 for bad (unknown state)
 *			2 got ABANDON packet 
 *			  (this invalidates all previous transactions)
 *
 *********************************************************************/
int EndPETTransaction(int fd, struct cbuf *cbuf) {
  extern int errno;
  int res;

  if (SendString (fd, PET_END_SEQ))
    return 1;

  res = WaitPETReply(fd, cbuf);

  if (res == PET_REPLY_ABANDON)
    return 2;

  if (res == PET_REPLY_GOODBYE)
    return 0;
  
  /* be liberal on what we accept...*/
  report (LOG_WARNING, "unexpected termination reply %d", res);
  
  return 0;
} /* EndPETTransaction */


#ifdef UUCP_LOCKING

/*********************************************************************
 * Function: LockTTY
 *
 * Returns: 1 don't have lock
 *			0 got lock
 *
 * error reporting done via report()
 *
 *********************************************************************/
int LockTTY(char *ttyname) {
  extern int debug;
  char lockfile[_POSIX_PATH_MAX], tmpfile[_POSIX_PATH_MAX];
  char *tty;
  int fd, unlocked, isrunning;
  pid_t pid, pid2;
  char cpid[12]; /* 32 bits is 10 chars max + \n + terminator */
  FILE *FP;
  
  unlocked = 1;	/* don't have lock */
  fd = -1;
  
#ifdef DEBUG
  if (debug > 2)
    report (LOG_INFO, "locking modem device");
#endif /* DEBUG */

  /* point to the last component of the pathname for the tty */
  if ((tty = strrchr(ttyname, '/')))
    ++tty;
  else
    tty = ttyname;
  
  pid = getpid();
  
  /* create a temp file and lock file pathnames */
  sprintf(tmpfile, "%sLTMP.%u", TTY_LOCKDIR, (u_int)pid);
  sprintf(lockfile, "%sLCK..%s", TTY_LOCKDIR, tty);
  
  /* make sure there's not an extra one laying around */
  unlink(tmpfile);
  
  /* create the temp file */
  if ((fd = open(tmpfile, O_WRONLY|O_CREAT|O_EXCL, 0444)) == -1) {
    report(LOG_ERR, "open(%.512s): %s", tmpfile, strerror(errno));
    goto LockTTYout;
  }
  
  sprintf(cpid, "%u\n", (u_int)pid);
  
  /* stick our pid in the file */
  if (write(fd, cpid, strlen(cpid)) == -1) {
    report(LOG_ERR, "write(%.512s): %s", tmpfile, strerror(errno));
    goto LockTTYout;
  }
  
  if (close(fd) == -1) {
    fd = -1;
    goto LockTTYout;
  }
  
  fd = -1;
  
  /* try to link our temp file to the real one */
  
  if (!link(tmpfile, lockfile)) {
    unlocked = 0;
    goto LockTTYout1; /* good */
  }
  
  /* couldn't get the lock, but all is not lost.  The lockfile has
     a process id in it, Maybe that process died an untimely death and
     couldn't cleanup after itself */

  isrunning = 1;
  
  if ((FP = fopen(lockfile, "r"))) {
    fscanf(FP, "%lu", (long unsigned int*)&pid2);
    fclose(FP);
    
    if ((pid2) && (kill((int)pid2, 0))) 
      isrunning = 0;
  }
  
  if (!isrunning) {
    report(LOG_INFO, "ignoring stale lock file: %.512s", lockfile);
    unlink(lockfile);

    if (!link(tmpfile, lockfile)) {
      unlocked = 0;
      goto LockTTYout1; /* good */
    }
    
    report(LOG_ERR, "ignore lock file failed");
  }
  
  
LockTTYout1:

  /* remove the temp file */
  if (unlink(tmpfile) == -1) 
    report(LOG_WARNING, "unlink(%.512s): %s", tmpfile, strerror(errno));
  

LockTTYout:

  if (fd != -1)
    close (fd);

  return unlocked;
} /* LockTTY */

/*********************************************************************
 * Function: UnLockTTY
 *
 * Returns: 0 removed lock
 *			1 couldn't remove lock
 *
 * error reporting done via report()
 *
 *********************************************************************/
int UnLockTTY(char * ttyname) {
  extern int debug;
  char lockfile[_POSIX_PATH_MAX];
  char *tty;

#ifdef DEBUG
  if (debug > 2)
    report (LOG_INFO, "unlocking modem device");
#endif /* DEBUG */
  
  /* point to the last component of the pathname for the tty */
  if ((tty = strrchr(ttyname, '/')))
    ++tty;
  else
    tty = ttyname;
  
  sprintf(lockfile, "%sLCK..%s", TTY_LOCKDIR, tty);
  
  if (unlink(lockfile) == -1) {
    report (LOG_ERR, "can't remove lock for %.512s", tty);
    return 1; /* bad */
  }
  
  return 0; /* good */
} /* UnLockTTY */

#endif /* UUCP_LOCKING */

#if 0
/*********************************************************************
 * Function: Match
 *
 * Wait for a particular string from the modem
 * Returns: 0 - if a match was made
 *          1 - if no match is made
 *
 *********************************************************************/
int Match(int fd, char *str, int retries) {
  int len;
  char c;
  
  while (1) {
    c=0;
    len = read(fd, &c, 1);
    
    if (len) {
      if (c == *str) {
	str++;
      }
      if (!(*str)) break; /* matched all? Exit loop */
    }
    
    if (!retries--) {
      return 1;
    }
  } /* while */
  
  return 0;
} /* Match */

#endif /* WAIT_WORD */

/*********************************************************************
 * Function: HangUpModem
 *
 * Returns: 0 for good - it was able to send the modem hangup comand
 *			and get a response.
 *          !0  for bad 
 *
 *********************************************************************/
int HangUpModem(int fd, struct cbuf *cbuf) {

  /* give the modem a chance to recover */
  sleep (1);

  /* now send the +++ wait ath */
  if (SendString (fd, " "))
    return 1;

  sleep (2);

  if (SendString(fd, MODEM_PLUS))
    return 1;

  sleep (2);
  
  if (SendString (fd, MODEM_HANG))
    return 1;

#ifndef WAIT_WORD
  /* Now wait for the OK short result code */
  if (WaitString(fd, "0\r", cbuf)) 
    return 1;
#else
  /* Now wait for the OK long result code */
  if (WaitString(fd, "OK\r", cbuf)) 
    return 1;
#endif
  
  return 0;
}

/*********************************************************************
 * Function: send_pet_transaction
 *
 * Returns: 0 the transaction was accepted
 *  	    1 some kind of error that should be retried
 *	    2 page rejected
 *	    3 some other fatal error
 *
 *     fd	       	open file descriptor to read/write from
 *     cbuf		file descriptor buffer
 *     field_vec	array of char pointers to n fields, null terminated
 *     vec_len		length of field_vec
 *     protocol	        PC_PET*
 *			
 *
 *********************************************************************/
int send_pet_transaction(int fd, struct cbuf*cbuf, unsigned char * field_vec[],
		         int field_vec_len, int protocol) {
  extern int errno;
  char pet_block[256], *fields;
  int i, j, k, fields_len, pet_block_len;
  int block_fields_len, retries;
  int block_accepted=0, err, res, fatal_err;
  int block_abandon=0;
  
  fields = (char*)0L;
  fields_len = 0;
  err = 1;       /* bad */
  fatal_err = 0; /* good */
  
  /* calculate total bytes of fields */
  for (i = 0; i < field_vec_len; ++i)
    fields_len += strlen(field_vec[i]);
  
  /* CR between each field */
  fields_len += field_vec_len;
  
  /* allocate memory for all the fields buffer + null terminator */
  if (!(fields = (char*)malloc(fields_len + 1))) {
    report(LOG_ERR, "malloc() failed");
    goto out_send_pet_transaction;
  }
  
  /* contruct the fields in pet format -- each field is CR seperated */
  for (i = 0, k = 0; i < field_vec_len; ++i) {
    for (j = 0; j < strlen(field_vec[i]); ++j) {
      /* map out illegal characters */
      if (((field_vec[i][j] <= 127) &&
	   (field_vec[i][j] >= 32)) ||
	  (field_vec[i][j] == 0x0a)
	  )
	fields[k++] = field_vec[i][j];
      else
	fields[k++] = '#';
    }
    
    fields[k++] = ASCII_CR;
  }
  
  fields[k] = 0;
  
  /* the max fields len in the blocks is determined by the protocol */
  if (protocol == PC_PET3)
    block_fields_len = 249;
  else
    block_fields_len = 250;
  
  for (i = 0; i < fields_len; i += block_fields_len) {
    
    /* # of bytes to try to pack in */
    k = fields_len - i;
    if (k > block_fields_len)
      k = block_fields_len;
    
    /* will this loop execute again */
    j = (i + block_fields_len < fields_len) ? 1 : 0;
    
    if (make_pet_block(pet_block, &pet_block_len, fields + i,
		       k, protocol, j)) {
      fatal_err = 1;
      report (LOG_ERR, "make_pet_block() failed");
      goto out_send_pet_transaction;
    }
    
    
    retries = 3;		/* try to send this block n times */
    block_accepted = 0;	        /* block accepted ? */
    
    while (--retries >= 0) {
      
      if (SendBytes (fd, pet_block, pet_block_len))
	goto out_send_pet_transaction;
      
      res = WaitPETReply (fd, cbuf);
      
      block_accepted = 0;
      
      switch (res) {
	
      case PET_REPLY_ACCEPT:
	report (LOG_INFO, "block accepted");
	block_accepted = 1;
	retries = 0;
	break;
	
      case PET_REPLY_REJECT:
	report (LOG_WARNING, "block rejected, retrying");
	break;
	
      case PET_REPLY_ABANDON:
	report (LOG_ERR, "block abandon reply, giving up");
	retries = 0;
	block_abandon = 1;
	break;
	
      case PET_REPLY_GOODBYE:
	report (LOG_ERR, "block abandon -- did it send?");
	retries = 0;
	block_abandon = 1;
	break;
	
      default:
	report (LOG_ERR, "block unexpected WaitPETReply=%d", res);
	break;
	
      } /* switch */
      
    } /* while --retries */
    
    if (block_abandon)
      break;
    
    if (!block_accepted)
      break;
    
  } /* more blocks in this transaction */
  
  err = 0;

out_send_pet_transaction:

  /* release memory */
  if (fields)
    free (fields);
  
  /* exceptional exceptions */
  if (fatal_err)
    return 3;
  
  /* anything that fails here can be retried */
  if (err)
    return 1;
  
  /* everything went okay */
  if (block_accepted)
    return 0;
  
  return 2;
} /* send_pet_transaction */

/*********************************************************************
 * Function: make_pet_block
 *
 * Returns: 0 the block was successfully constructed
 *	    1 for error
 *
 *	buf	       	workspace to construct block, 256 bytes
 *		       	allocated by caller
 *	buf_len		length of constructed block
 *	data		fields of blocks.  individual fields must
 *			allready be CR seperated
 *	data_len	length of data buf, max 250 unless protocol
 *			is PC_PET3, then 249
 *	protocol	PC_*
 *
 *********************************************************************/
int make_pet_block(char *buf, int *buf_len, char *data, int data_len, 
	           int protocol, int more) {
  int i, x, csum;

  csum = i = 0;
  
  /* sanity check */
  if (((data_len > 249) && (protocol == PC_PET3)) || (data_len > 250) ||
      (!data_len))
    return 1;

  /* start with a STX */
  buf[i++] = ASCII_STX;
  
  /* add data */
  memcpy(&buf[i], data, data_len);
  i += data_len;

  /* if the protocol is PC_PET3 need to add CR again for destination */
  if ((protocol == PC_PET3) && (!more))
    buf[i++] = ASCII_CR;
  
  /*
    if more data to complete this field - end in ASCII_US,
    if more fields - end in ASCII_ETB
    else, end in ASCII_ETX
  */
  if (more) {
    /* if the data ends in a CR, then this block ends on a complete field */
    if (data[data_len - 1] == ASCII_CR)
      buf[i++] = ASCII_ETB;
    else
      buf[i++] = ASCII_US;
  } else
    buf[i++] = ASCII_ETX;
  
  /* 7 bit checksum follows */
  for (x = 0; x < i; ++x)
    csum += (buf[x] & 0x7f);
  
  /*
    The checksum is represented as 3 ascii characters between 0x30 and 0x3f
    tack it on backwards 
  */
  buf[i+2] = 0x30 + (csum & 0x0f);
  csum >>= 4;
  
  buf[i+1] = 0x30 + (csum & 0x0f);
  csum >>= 4;
  
  buf[i] = 0x30 + (csum & 0x0f);
  
  /* end it all with a CR */
  buf[i+3] = ASCII_CR;
  
  *buf_len = i+4;
  
  return 0;
} /* make_pet_block */
